#ifndef _MANAGER_SNMP_H
#define _MANAGER_SNMP_H

/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <tinysnmp/tinysnmp.h>
#include <tinysnmp/manager/udp.h>

#include <abz/typedefs.h>

/*
 * this structure is private. don't try to use anything in this
 * structure directly. some things, e.g. udp will disappear later
 * on.
 */
typedef struct
{
   octet_string_t community;
   uint16_t retries;
   udp_t udp;
} snmp_agent_t;

/*
 * Create a handle for use with functions below.
 *        agent           snmp agent info
 *        addr            address/port of agent
 *        community       community string used for authentication
 *        timeout         timeout in milliseconds for any operations. if timeout
 *                        is 0, the timeout is disabled (i.e. we don't use
 *                        non-blocking I/O).
 *        retries         number of retries before get routines fail. if retries
 *                        is 0, only one attempt is made to communicate with the
 *                        agent.
 *
 * Returns 0 if successful, or -1 if some error occurred. Check
 * errno to see what error occurred.
 */
extern int snmp_open (snmp_agent_t *agent,const struct sockaddr_in *addr,const char *community,uint32_t timeout,uint16_t retries);

/*
 * Destroy handle created by snmp_open().
 */
extern void snmp_close (snmp_agent_t *agent);

/*
 * Retrieve values from the agent.
 *
 *        agent           snmp agent info
 *        oid             list of ObjectID's to retrieve
 *        n               number of ObjectID's in list
 *
 * Return the values if successful, or NULL if some error occurred. Check
 * errno to see what error occurred.
 */
extern snmp_value_t *snmp_get (snmp_agent_t *agent,uint32_t *const *oid,size_t n);

/*
 * Retrieve next values from the agent.
 *
 *        agent           snmp agent info
 *        oid             list of (partial) ObjectID's to send to agent
 *        n               number of ObjectID's in list
 *
 * Return the values (and their corresponding ObjectID's) that lexigraphically
 * succeeds those specified if successful, or NULL if the query failed. Check
 * errno to see what error occurred.
 */
extern snmp_next_value_t *snmp_get_next (snmp_agent_t *agent,uint32_t *const *oid,size_t n);

/*
 * Retrieve tables of values from the agent. You should specify ObjectID's of the table
 * entries without their indexes. For example, if you want to retrieve the list of entries
 * in tableX.entryX, you should specify that, and this routine will return all the ObjectID's
 * of the form tableX.entryX.indexX. If the types of those entries differ, it is considered
 * to be an error.
 *
 *        agent           snmp agent info
 *        oid             list of table entry ObjectID's that define the tables
 *        n               number of ObjectID's in list
 *        rows            maximum number of rows to return (0 = unlimited)
 *
 * Return the entry lists that match specified entries if successful, or NULL if the query
 * failed. Check errno to see what error occurred.
 */
extern snmp_table_t *snmp_get_tables (snmp_agent_t *agent,uint32_t *const *oid,size_t n,size_t rows);

/*
 * Free memory allocated by snmp_get() routine.
 */
extern void snmp_free_values (snmp_value_t **values,size_t n);

/*
 * Free memory allocated by snmp_get_next() routine.
 */
extern void snmp_free_next_values (snmp_next_value_t **values,size_t n);

/*
 * Free memory allocated by snmp_get_tables() routine.
 */
extern void snmp_free_tables (snmp_table_t **table,size_t n);

#endif	/* #ifndef _MANAGER_SNMP_H */
