
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <debug/log.h>

#include "cmdline.h"

#define usage(p) help_stub(p,0)
#define help(p) help_stub(p,1)
static void help_stub (const char *progname,int verbose)
{
   fprintf (stderr,
			"usage: %s [OPTIONS] <configfile> <modulepath>\n"
			"       %s -h | --help\n",
			progname,progname);

   if (verbose)
	 {
		fprintf (stderr,
				 "\n"
				 "where OPTIONS can be any of the following:\n"
				 "\n"
				 "    -f | --logfile=FILENAME   Location of log file. By default, messages will be\n"
				 "                              logged to the console.\n"
				 "\n"
				 "    -l | --loglevel=LEVEL     The following levels can be specified (in increasing\n"
				 "                              order of verbosity):\n"
				 "\n"
				 "                                  quiet     No messages will be logged\n"
				 "                                  errors    Only errors will be logged\n"
				 "                                  warnings  Warnings & errors will be logged\n"
				 "                                  normal    Default messages and all of the above\n"
				 "                                  verbose   Verbose messages will also be logged\n"
				 "                                  debug     Log debug messages as well\n"
				 "                                  noisy     Noisy debug messages will be logged as well\n"
				 "\n"
				 "                              If a log file is specified, %s defaults to ``verbose'',\n"
				 "                              otherwise, ``noisy'' is selected.\n"
				 "\n"
				 "    -d | --daemon             Run in the background. You need to specify a log file if\n"
				 "                              you want to use this option.\n"
				 "\n"
				 "    -h | --help               Show this help message.\n"
				 "\n"
				 "    <configfile>              Location of %s configuration file.\n"
				 "\n"
				 "    <modulepath>              Path where mib libraries reside.\n"
				 "\n",
				 progname,progname);
	 }

   exit (EXIT_FAILURE);
}

void cmdline_parse (struct cmdline *cmdline,int argc,char *argv[])
{
   int i;

   (cmdline->progname = strrchr (argv[0],'/')) ? cmdline->progname++ : (cmdline->progname = argv[0]);

   /* is the user requesting help? */
   if (argc == 2 && (!strcmp (argv[1],"-h") || !strcmp (argv[1],"--help")))
	 help (cmdline->progname);

   if (argc < 3)
	 usage (cmdline->progname);

   cmdline->logfile = NULL;
   cmdline->loglevel = -1;
   cmdline->daemon = 0;

   for (i = 1; i < argc - 2; i++)
	 {
		if (!strcmp (argv[i],"-d") || !strcmp (argv[i],"--daemon"))
		  cmdline->daemon = 1;
		else if (!strcmp (argv[i],"-f") || !strncmp (argv[i],"--logfile=",10))
		  {
			 char *arg = strlen (argv[i]) == 2 ? argv[++i] : strchr (argv[i],'=') + 1;
			 if (i == argc - 1 || *arg == '\0') usage (cmdline->progname);
			 cmdline->logfile = arg;
		  }
		else if (!strcmp (argv[i],"-l") || !strncmp (argv[i],"--loglevel=",11))
		  {
			 char *arg = strlen (argv[i]) == 2 ? argv[++i] : strchr (argv[i],'=') + 1;
			 if (i == argc - 1 || *arg == '\0') usage (cmdline->progname);

			 if (!strcmp (arg,"quiet"))
			   cmdline->loglevel = LOG_QUIET;
			 else if (!strcmp (arg,"errors"))
			   cmdline->loglevel = LOG_ERROR;
			 else if (!strcmp (arg,"warnings"))
			   cmdline->loglevel = LOG_WARNING;
			 else if (!strcmp (arg,"normal"))
			   cmdline->loglevel = LOG_NORMAL;
			 else if (!strcmp (arg,"verbose"))
			   cmdline->loglevel = LOG_VERBOSE;
			 else if (!strcmp (arg,"debug"))
			   cmdline->loglevel = LOG_DEBUG;
			 else if (!strcmp (arg,"noisy"))
			   cmdline->loglevel = LOG_NOISY;
			 else usage (cmdline->progname);
		  }
		else usage (cmdline->progname);
	 }

   cmdline->configfile = argv[i];
   cmdline->modulepath = argv[i + 1];

   if (cmdline->daemon && cmdline->logfile == NULL)
	 usage (cmdline->progname);

   if (cmdline->loglevel == -1)
	 cmdline->loglevel = cmdline->logfile == NULL ? LOG_NOISY : LOG_VERBOSE;
}

#ifdef DEBUG
void cmdline_print_stub (const char *filename,int line,const char *function,int level,const struct cmdline *cmdline)
{
   static const char *loglevel[] =
	 {
		[LOG_QUIET]   "quiet",
		[LOG_ERROR]   "errors",
		[LOG_WARNING] "warnings",
		[LOG_NORMAL]  "normal",
		[LOG_VERBOSE] "verbose",
		[LOG_DEBUG]   "debug",
		[LOG_NOISY]   "noisy"
	 };

   log_puts_stub (filename,line,function,level,"# cmdline\n");

   if (cmdline->logfile != NULL)
	 log_printf_stub (filename,line,function,level,"logfile \"%s\"\n",cmdline->logfile);
   else
	 log_puts_stub (filename,line,function,level,"no logfile\n");

   log_printf_stub (filename,line,function,level,
					"loglevel %s\n"
					"%sdaemon\n"
					"configfile \"%s\"\n"
					"modulepath \"%s\"\n",
					loglevel[cmdline->loglevel],
					cmdline->daemon ? "" : "no ",
					cmdline->configfile,
					cmdline->modulepath);
}
#endif	/* #ifdef DEBUG */

