/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#ifndef INDII_CLUSTER_PEARSONDISTANCE_HPP
#define INDII_CLUSTER_PEARSONDISTANCE_HPP

#include "ClusterVector.hpp"

namespace indii {
/**
 * Pearson distance.
 *
 * @tparam T Datum element type.
 */
template<class T = float>
class PearsonDistance {
public:
  /**
   * Preprocess point.
   *
   * @param x Point.
   */
  static void prepare(typename ClusterVector<T>::type& x);

  /**
   * Calculate distance.
   *
   * @param x1 First point.
   * @param x2 Second point, should have same size as first.
   *
   * @return %Distance between the two points.
   */
  static T distance(const typename ClusterVector<T>::type& x1,
      const typename ClusterVector<T>::type& x2);

};
}

#include <cassert>

template<class T>
inline void indii::PearsonDistance<T>::prepare(
    typename ClusterVector<T>::type& x) {
  /* pre-condition */
  assert (x.size() == ClusterVector<T>::N);

  static const unsigned int N = ClusterVector<T>::N;
  
  T mu = boost::numeric::ublas::sum(x) / N;
  x -= boost::numeric::ublas::scalar_vector<T>(N,mu);
  T norm = boost::numeric::ublas::norm_2(x);
  if (norm > 0.0) {
    x /= norm;
  } else {
    x = boost::numeric::ublas::zero_vector<T>(N);
  }
}

template<class T>
inline T indii::PearsonDistance<T>::distance(
    const typename ClusterVector<T>::type& x1,
    const typename ClusterVector<T>::type& x2) {
  /* pre-condition */
  assert (x1.size() == x2.size());

  T d = T(0.5)*(T(1.0) - boost::numeric::ublas::inner_prod(x1, x2));
  
  /* relaxed IEEE compliance, so make sure this is bound as expected */
  //d = std::max((T)0.0, std::min(d, (T)1.0));
  
  return d;
}
 
#endif
