/* Hey EMACS -*- linux-c -*-
 *
 * tiser - low level driver for handling a serial link cable designed
 * for Texas Instruments graphing calculators (http://lpg.ticalc.org).
 * A part of the TiLP project.
 *
 * Copyright (C) 2000-2004, Romain Lievin <roms@lpg.ticalc.org>
 * under the terms of the GNU General Public License.
 *
 * Various fixes & clean-up from the Linux Kernel Mailing List
 * Pb: I/O region is always requested & locked by ttySx driver
 */

/* This driver should, in theory, work with any RS232 compliant serial port. 
 *
 *
 *
 * If this driver is built into the kernel, you can configure it using the
 * kernel command-line.  For example:
 *
 *      tiser=timeout,delay       (set timeout and delay)
 *
 * If the driver is loaded as a module, similar functionality is available
 * using module parameters.  The equivalent of the above commands would be:
 *
 *      # insmod tiser.o timeout=15 delay=10
 */

/* COMPATIBILITY WITH OLD KERNELS
 *
 * Usually, serial cables were bound to ports at
 * particular I/O addresses, as follows:
 *
 *      tiser0             0x3f8
 *      tiser1             0x2f8
 *      tiser2             0x3e8
 *      tiser3             0x2e8
 *
 * This driver, by default, binds tiser devices according to the minor number.
 * This means that if you do not have standard addresses for serial port,
 * you will have to manually edit the file and change structure values.
 */

#include <linux/config.h>

#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS
#endif

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/fcntl.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <asm/bitops.h>
#include <linux/devfs_fs_kernel.h>	/* DevFs support */
#include <linux/serial_reg.h>	/* Our code depends on UART registers */

/*
 * TI definitions
 */
#include "ticable.h"

/*
 * Version Information
 */
#define DRIVER_VERSION "1.19"
#define DRIVER_AUTHOR  "Romain Lievin <roms@tilp.info>"
#define DRIVER_DESC    "Device driver for TI/PC serial link cables"
#define DRIVER_LICENSE "GPL"

#define VERSION(ver,rel,seq) (((ver)<<16) | ((rel)<<8) | (seq))
#if LINUX_VERSION_CODE < VERSION(2,6,0)
# define minor(x) MINOR((x)->i_rdev)
# define need_resched() (current->need_resched)
#else
# define minor(x) iminor(x)
#endif

/* ----- global variables --------------------------------------------- */

/*
 * You must set these - there is no sane way to probe for this link cable.
 */

#define SP_NO 4
static int serial_table[SP_NO] = { 0x3f8, 0x2f8, 0x3e8, 0x2e8 };

static int delay = IO_DELAY;	/* inter-bit delay in microseconds */
static int timeout = TIMAXTIME;	/* timeout in tenth of seconds     */

#if LINUX_VERSION_CODE < VERSION(2,6,0)
static devfs_handle_t devfs_handle;
#endif

static unsigned long opened;	/* opened devices */

/* --- setting states on the D-bus with the right timing: ------------- */

static inline void
outbyte(int value, int minor)
{
	outb((value), serial_table[minor] + UART_MCR);
}

static inline int
inbyte(int minor)
{
	return (inb(serial_table[minor] + UART_MSR));
}

static inline void
init_ti_serial(int minor)
{
	outbyte(3, minor);
}

/* ----- global defines ----------------------------------------------- */

#define START(x) { x=jiffies+HZ/(timeout/10); }
#define WAIT(x) { if (!time_before(jiffies, (x))) return -1; \
 if (need_resched()) schedule(); }

/* ----- D-bus bit-banging functions ---------------------------------- */

/* D-bus protocol (45kbit/s max):
                    1                 0                      0
       _______        ______|______    __________|________    __________
Red  :        ________      |      ____          |        ____
       _        ____________|________      ______|__________       _____
White:  ________            |        ______      |          _______
*/

/* Try to transmit a byte on the specified port (-1 if error). */
static int
put_ti_serial(int minor, unsigned char data)
{
	int bit;
	unsigned long max;

	for (bit = 0; bit < 8; bit++) {
		if (data & 1) {
			outbyte(2, minor);
			START(max);
			do {
				WAIT(max);
			} while (inbyte(minor) & 0x10);

			outbyte(3, minor);
			START(max);
			do {
				WAIT(max);
			} while (!(inbyte(minor) & 0x10));
		} else {
			outbyte(1, minor);
			START(max);
			do {
				WAIT(max);
			} while (inbyte(minor) & 0x20);

			outbyte(3, minor);
			START(max);
			do {
				WAIT(max);
			} while (!(inbyte(minor) & 0x20));
		}

		data >>= 1;
		udelay(delay);

		if (need_resched())
			schedule();
	}

	return 0;
}

/* Receive a byte on the specified port or -1 if error. */
static int
get_ti_serial(int minor)
{
	int bit;
	unsigned char v, data = 0;
	unsigned long max;

	for (bit = 0; bit < 8; bit++) {
		START(max);
		do {
			WAIT(max);
		} while ((v = inbyte(minor) & 0x30) == 0x30);

		if (v == 0x10) {
			data = (data >> 1) | 0x80;
			outbyte(1, minor);
			START(max);
			do {
				WAIT(max);
			} while (!(inbyte(minor) & 0x20));
			outbyte(3, minor);
		} else {
			data = data >> 1;
			outbyte(2, minor);
			START(max);
			do {
				WAIT(max);
			} while (!(inbyte(minor) & 0x10));
			outbyte(3, minor);
		}

		udelay(delay);
		if (need_resched())
			schedule();
	}

	return (int) data;
}

/* Try to detect a parallel link cable on the specified port */
static int
probe_ti_serial(int minor)
{
	int i;
	int seq[] = { 0x00, 0x20, 0x00, 0x30 };

	for (i = 3; i >= 0; i--) {
		outbyte(3, minor);
		outbyte(i, minor);
		udelay(delay);
		/*printk("Probing -> %i: 0x%02x 0x%02x\n", i, data & 0x30, seq[i]); */
		if ((inbyte(minor) & 0x30) != seq[i]) {
			outbyte(3, minor);
			return -1;
		}
	}
	outbyte(3, minor);
	return 0;
}

/* ----- kernel module functions--------------------------------------- */

static int
tiser_open(struct inode *inode, struct file *file)
{
	unsigned int minor = minor(inode) - TISER_MINOR;

	if (minor >= SP_NO)
		return -ENXIO;

	if (test_and_set_bit(minor, &opened))
		return -EBUSY;
/*
	if (check_region(serial_table[minor], 8) >= 0)
		request_region(serial_table[minor], 8, "tiser");
	else
		return -ENXIO;
*/
	init_ti_serial(minor);

	return 0;
}

static int
tiser_close(struct inode *inode, struct file *file)
{
	unsigned int minor = minor(inode) - TISER_MINOR;

	clear_bit(minor, &opened);
	/*release_region(serial_table[minor], 8); */

	return 0;
}

static ssize_t
tiser_write(struct file *file, const char *buf, size_t count, loff_t * ppos)
{
	unsigned int minor =
	    minor(file->f_dentry->d_inode) - TISER_MINOR;
	ssize_t n;

	for (n = 0; n < count; n++) {
		unsigned char b;

		if (get_user(b, buf + n))
			return -EFAULT;

		if (put_ti_serial(minor, b) == -1) {
			init_ti_serial(minor);
			return -ETIMEDOUT;
		}
	}

	return n;
}

static ssize_t
tiser_read(struct file *file, char *buf, size_t count, loff_t * ppos)
{
	int b = 0;
	unsigned int minor =
	    minor(file->f_dentry->d_inode) - TISER_MINOR;
	ssize_t retval = 0;
	ssize_t n = 0;

	if (count == 0)
		return 0;

	if (ppos != &file->f_pos)
		return -ESPIPE;

	while (n < count) {
		b = get_ti_serial(minor);
		if (b == -1) {
			init_ti_serial(minor);
			retval = -ETIMEDOUT;
			goto out;
		} else {
			if (put_user(b, ((unsigned char *) buf) + n)) {
				retval = -EFAULT;
				break;
			} else
				retval = ++n;
		}

		/* Non-blocking mode : try again ! */
		if (file->f_flags & O_NONBLOCK) {
			retval = -EAGAIN;
			goto out;
		}

		/* Signal pending, try again ! */
		if (signal_pending(current)) {
			retval = -ERESTARTSYS;
			goto out;
		}

		if (need_resched())
			schedule();
	}

      out:
	return retval;
}

static int
tiser_ioctl(struct inode *inode, struct file *file,
	    unsigned int cmd, unsigned long arg)
{
	int retval = 0;

	switch (cmd) {
	case IOCTL_TISER_DELAY:
	  delay = (int)arg;    //get_user(delay, &arg);
	  break;
	case IOCTL_TISER_TIMEOUT:
	  timeout = (int)arg;  //get_user(timeout, &arg);
	  break;
	default:
		retval = -ENOTTY;
		break;
	}

	return retval;
}

/* ----- kernel module registering ------------------------------------ */

static struct file_operations tiser_fops = {
	owner:THIS_MODULE,
	llseek:no_llseek,
	read:tiser_read,
	write:tiser_write,
	ioctl:tiser_ioctl,
	open:tiser_open,
	release:tiser_close,
};

/* --- initialisation code ------------------------------------- */

#ifndef MODULE
/*      You must set these - there is no sane way to probe for this board.
 *      You can use tiser=timeout,delay to set these now. */
static int __init
tiser_setup(char *str)
{
	int ints[2];

	str = get_options(str, ARRAY_SIZE(ints), ints);

	if (ints[0] > 0) {
		timeout = ints[1];
		if (ints[0] > 1) {
			delay = ints[2];
		}
	}
	return 1;
}
#endif

int __init
tiser_init_module(void)
{
#if LINUX_VERSION_CODE < VERSION(2,6,0)
	char name[8];
#endif
	int nr;

	printk("tiser: serial link cable driver, version %s\n", DRIVER_VERSION);

#if LINUX_VERSION_CODE < VERSION(2,6,0)
	if (devfs_register_chrdev(TISER_MAJOR, "tiser", &tiser_fops)) {
#else
	  if (register_chrdev(TISER_MAJOR, "tiser", &tiser_fops)) {
#endif
		printk("tiser: unable to get major %d\n", TISER_MAJOR);
		return -EIO;
	}

	/* Use devfs, tree: /dev/ticables/ser/[0..3] */
#if LINUX_VERSION_CODE < VERSION(2,6,0)
	devfs_handle = devfs_mk_dir(NULL, "ticables/ser", NULL);
#else
	devfs_mk_dir("ticables/ser");
#endif

	for (nr = 0; nr < SP_NO; nr++) {
#if LINUX_VERSION_CODE < VERSION(2,6,0)
		sprintf(name, "%d", nr);
		printk
		    ("tiser: registering to devfs : major = %d, minor = %d, node = %s\n",
		     TISER_MAJOR, (TISER_MINOR + nr), name);
		devfs_register(devfs_handle, name, DEVFS_FL_DEFAULT,
			       TISER_MAJOR, TISER_MINOR + nr,
			       S_IFCHR | S_IRUGO | S_IWUGO, &tiser_fops, NULL);
#else
		devfs_mk_cdev(MKDEV(TISER_MAJOR, TISER_MINOR + nr),
			      S_IFCHR | S_IRUGO | S_IWUGO,
			      "ticables/ser/%d", nr);
#endif
	}
/*  
	for(nr=0; nr<SP_NO; nr++) {
		if (check_region(serial_table[nr], 8) >= 0) {
			request_region(serial_table[nr], 8, "tiser");
			if (probe_ti_serial(nr) != -1)
				printk("tiser%d: link cable found !\n", nr);
			else
				printk("tiser%d: link cable not found (did you plug the cable to calc ?).\n", nr);
			release_region(serial_table[nr], 8);
		}
		else
			printk("tiser%d: area locked, unable to probe !\n", nr);
	}
*/
	return 0;
}

void __exit
tiser_cleanup_module(void)
{
       int nr;

	printk("tiser: module unloaded.\n");

#if LINUX_VERSION_CODE < VERSION(2,6,0)
	devfs_unregister(devfs_handle);
	devfs_unregister_chrdev(TISER_MAJOR, "tiser");
#else
	unregister_chrdev(TISER_MAJOR, "tiser");
	for(nr = 0; nr < SP_NO; nr++)
	  devfs_remove("ticables/ser/%d", nr);
	devfs_remove("ticables/ser");
#endif
}

/* --------------------------------------------------------------------- */

__setup("tiser=", tiser_setup);
module_init(tiser_init_module);
module_exit(tiser_cleanup_module);

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE(DRIVER_LICENSE);

#if LINUX_VERSION_CODE < VERSION(2,6,0)
EXPORT_NO_SYMBOLS;
#endif

MODULE_PARM(timeout, "i");
MODULE_PARM_DESC(timeout, "Timeout, default=1.5 seconds");
MODULE_PARM(delay, "i");
MODULE_PARM_DESC(delay, "Inter-bit delay, default=10 microseconds");
