/* -*- mode: c; c-file-style: "gnu" -*-
 * gzip.c -- gzip related functions.
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file gzip.c
 * This file contains a deflate() wrapper, kind of gzwrite() on
 * steroids.
 *
 * The main problem with gzwrite() is that it does not work on
 * non-blocking file descriptors, which are a must for a
 * single-threaded webserver.
 *
 * This module implements thy_zlib_send(), a slightly misnamed
 * function, which does exactly what gzwrite() does, plust a bit more,
 * but works on non-blocking sockets too.
 */

#include <netinet/in.h>
#include <stdlib.h>
#include <string.h>

#include "options.h"

#if THY_OPTION_ZLIB
#include <zlib.h>

#include "compat/compat.h"

#include "config.h"
#include "gzip.h"
#include "os.h"
#include "session.h"
#include "thy.h"
#include "types.h"

/** @internal Static gzip header.
 * We are going the easy way: no filename, no timestamp, no
 * nothing. Just a plain dumb gzip header.
 */
static char gz_header[] = {0x1f, 0x8b, Z_DEFLATED, 0, 0, 0, 0, 0, 0, 0x03};

/** @internal Opaque type for compressed session data.
 */
typedef struct _thy_zlib_session_t
{
  char *gzdata; /**< The compressed data. */
  uLongf gzlength; /**< Length of the compressed data. */
  size_t gzoffs; /**< Current offset of the compressed data. */

  uLong datacrc; /**< CRC of the initial data. */
  size_t datasize; /**< Size of the initial data. */
} _thy_zlib_session_t;

/** Send a buffer out, compressed.
 *
 * Handles both setting up the opaque structure that holds the
 * compressed data, and the algorithm to push it out onto the network
 * in chunks. Each chunk is preceded by a static gzip header, and
 * followed by a CRC and datasize.
 *
 * @param session is the session to work with.
 * @param buf is the buffer to send out.
 * @param size is the size of the buffer.
 *
 * @returns The number of bytes written, or -1 on error (or when
 * done).
 */
int
thy_zlib_send (session_t *session, void *buf, size_t size)
{
  int w;

  if (!session->compress.gz_session)
    {
      thy_mappable_config_t *config =
	config_get_mapped (session->absuri, session->request->resolved);
      int level = config->encoding.level;
      unsigned char crc[4], dsize[4];
      uLong datacrc;
      size_t datasize;

      free (config);

      session->compress.gz_session =
	(thy_zlib_session_t *)bhc_malloc (sizeof (thy_zlib_session_t));

      session->compress.gz_session->datasize = size;
      session->compress.gz_session->datacrc = crc32 (0, Z_NULL, 0);
      session->compress.gz_session->datacrc =
	crc32 (session->compress.gz_session->datacrc, buf, size);

      session->compress.gz_session->gzlength = size * 2;
      session->compress.gz_session->gzdata = (char *)bhc_malloc (size * 2);
      session->compress.gz_session->gzoffs = 0;

      compress2 (session->compress.gz_session->gzdata,
		 &session->compress.gz_session->gzlength,
		 buf, size, level);
      session->compress.gz_session->gzlength -= 6;

      /* Construct gzip header */
      free (session->body.header.buff);
      session->body.header.buff =
	(char *)bhc_malloc (sizeof (gz_header) + 1);
      memcpy (session->body.header.buff, gz_header, sizeof (gz_header));
      session->body.header.size = sizeof (gz_header);
      session->body.header.offset = 0;

      /* Construct gzip footer */
      free (session->body.footer.buff);

      datacrc = htonl (session->compress.gz_session->datacrc);
      datasize = htonl (session->compress.gz_session->datasize);

      memcpy (crc, &datacrc, 4);
      memcpy (dsize, &datasize, 4);

      asprintf (&session->body.footer.buff, "%c%c%c%c%c%c%c%c",
		crc[3], crc[2], crc[1], crc[0],
		dsize[3], dsize[2], dsize[1], dsize[0]);
      session->body.footer.size = 8;
      session->body.footer.offset = 0;
    }

  /* Push header */
  if ((size_t)session->body.header.offset < session->body.header.size)
    {
      w = thy_sendbuffer
	(session->io.out,
	 &session->body.header.buff[session->body.header.offset],
	 (size_t)(session->body.header.size -
		  session->body.header.offset));
      if (w >= 0)
	{
	  session->body.header.offset += w;
	  return 0;
	}
    }

  /* Push footer or end compression */
  if (session->compress.gz_session->gzoffs ==
      session->compress.gz_session->gzlength)
    {
      /* Push footer */
      if ((size_t)session->body.footer.offset < session->body.footer.size)
	{
	  w = thy_sendbuffer
	    (session->io.out,
	     &session->body.footer.buff[session->body.footer.offset],
	     (size_t)(session->body.footer.size -
		      session->body.footer.offset));
	  if (w >= 0)
	    {
	      session->body.footer.offset += w;
	      return 0;
	    }
	}
      else
	return session->compress.gz_session->datasize;

      return -1;
    }

  /* Send the compressed body */
  w = thy_sendbuffer (session->io.out,
		      session->compress.gz_session->gzdata + 2 +
		      session->compress.gz_session->gzoffs,
		      session->compress.gz_session->gzlength -
		      session->compress.gz_session->gzoffs);
  if (w >= 0)
    session->compress.gz_session->gzoffs += w;
  else
    return -1;

  return 0;
}

/** Frees a compressed data structure.
 * @param session is the structure to free.
 */
void
thy_zlib_free (thy_zlib_session_t *session)
{
  if (!session)
    return;

  free (session->gzdata);
  free (session);
}

#endif /* THY_OPTION_ZLIB */
