# Copyright (c) 2003, 2004 by Intevation GmbH
# Authors:
# Jonathan Coles <jonathan@intevation.de>
# Frank Koormann <frank.koormann@intevation.de>
# Jan-Oliver Wagner <jan@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""Projection dialog"""

__version__ = "$Revision: 1.39.2.1 $"
# $Source: /thubanrepository/thuban/Thuban/UI/projdialog.py,v $
# $Id: projdialog.py,v 1.39.2.1 2004/12/16 10:56:17 bh Exp $

import os
from wxPython.wx import *

from Thuban import _

from Thuban.Model.proj import Projection, ProjFile

from Thuban.Model.resource import get_user_proj_file, get_system_proj_file, \
                                  read_proj_file, write_proj_file, \
                                  DEFAULT_PROJ_FILE, EPSG_PROJ_FILE, \
                                  EPSG_DEPRECATED_PROJ_FILE
from Thuban.UI.dialogs import NonModalNonParentDialog

from common import ThubanBeginBusyCursor, ThubanEndBusyCursor
from sizers import NotebookLikeSizer
from projlist import PROJ_SELECTION_CHANGED, ProjectionList
from common import ThubanBeginBusyCursor, ThubanEndBusyCursor



ID_PROJ_PROJCHOICE = 4002
ID_PROJ_ADDTOLIST    = 4003
ID_PROJ_NEW       = 4004
ID_PROJ_REVERT    = 4006
ID_PROJ_AVAIL     = 4009
ID_PROJ_SAVE      = 4010
ID_PROJ_IMPORT    = 4011
ID_PROJ_EXPORT    = 4012
ID_PROJ_REMOVE    = 4013
ID_PROJ_PROJNAME  = 4014

CLIENT_PROJ = 0
CLIENT_PROJFILE = 1

class ProjFrame(NonModalNonParentDialog):

    def __init__(self, parent, name, title, receiver):
        """Initialize the projection dialog.

        receiver -- An object that implements the following methods:
                        SetProjection(projection)
                        GetProjection()
        """
        NonModalNonParentDialog.__init__(self, parent, name, title)

        self.projection_panel_defs = [
            ("tmerc", _("Transverse Mercator"), TMPanel),
            ("utm", _("Universal Transverse Mercator"), UTMPanel),
            ("lcc", _("Lambert Conic Conformal"), LCCPanel),
            ("latlong", _("Geographic"), GeoPanel),
            ("longlat", _("Geographic"), GeoPanel)]#longlat is an alias of proj
        self.receiver = receiver
        self.haveTried = False
        self.curProjPanel = None
        self.__usrProjFile = None
        self._sys_proj_files = {}

        self.build_dialog()
        self.Layout()

        self.originalProjection = self.receiver.GetProjection()

        self.projection_list.SelectProjection(self.originalProjection)
        self.projection_list.SetFocus()

    def build_dialog(self):
        """Build the dialog's widgets and set the event handlers"""
        self.topBox = top_box = wxBoxSizer(wxVERTICAL)

        main_box = wxBoxSizer(wxHORIZONTAL)
        top_box.Add(main_box, 1, wxALL|wxEXPAND)

        #
        #    The projection list and associated controls
        #
        vbox = wxBoxSizer(wxVERTICAL)
        main_box.Add(vbox, 4, wxALL|wxEXPAND)

        #label = wxStaticText(self, -1, _("Available Projections:"))
        #vbox.Add(label, 0, wxLEFT|wxRIGHT|wxTOP, 4)

        hbox = wxBoxSizer(wxHORIZONTAL)
        vbox.Add(hbox, 1, wxALL|wxEXPAND)
        proj_files = [self.load_user_proj(),
                      self.load_system_proj(DEFAULT_PROJ_FILE)]
        self.projection_list = ProjectionList(self, proj_files,
                                              self.receiver.GetProjection())
        hbox.Add(self.projection_list, 1, wxALL|wxEXPAND|wxADJUST_MINSIZE, 4)
        self.projection_list.Subscribe(PROJ_SELECTION_CHANGED,
                                       self.proj_selection_changed)

        # Projection List specific actions (Import/Export/Remove)
        buttons = wxBoxSizer(wxVERTICAL)
        hbox.Add(buttons, 0, wxALL)
        self.button_import = wxButton(self, ID_PROJ_IMPORT, _("Import..."))
        EVT_BUTTON(self, ID_PROJ_IMPORT, self._OnImport)
        buttons.Add(self.button_import, 1, wxALL|wxEXPAND, 4)
        self.button_export = wxButton(self, ID_PROJ_EXPORT, _("Export..."))
        EVT_BUTTON(self, ID_PROJ_EXPORT, self._OnExport)
        buttons.Add(self.button_export, 1, wxALL|wxEXPAND, 4)
        buttons.Add(20, 20, 0, wxEXPAND, 0)
        self.button_remove = wxButton(self, ID_PROJ_REMOVE, _("Remove"))
        EVT_BUTTON(self, ID_PROJ_REMOVE, self._OnRemove)
        buttons.Add(self.button_remove, 1, wxALL|wxEXPAND, 4)

        buttons.Add(20, 20, 0, wxEXPAND, 0)
        label = wxStaticText(self, -1, _("Show EPSG:"))
        buttons.Add(label, 0, wxLEFT|wxRIGHT|wxTOP, 4)
        self.check_epsg = wxCheckBox(self, -1, _("Normal"))
        EVT_CHECKBOX(self, self.check_epsg.GetId(), self._OnShowEPSG)
        buttons.Add(self.check_epsg, 1, wxALL|wxEXPAND, 4)
        self.check_epsg_depr = wxCheckBox(self, -1, _("Deprecated"))
        EVT_CHECKBOX(self, self.check_epsg_depr.GetId(), self._OnShowEPSG)
        buttons.Add(self.check_epsg_depr, 1, wxALL|wxEXPAND, 4)

        # The file path
        self.projfilepath = wxStaticText(self, -1, "")
        vbox.Add(self.projfilepath, 0, wxALL|wxEXPAND)

        #
        #   The projection editor part
        #
        self.edit_box = wxStaticBox(self, -1, _("Edit"))
        sizer_edit = wxStaticBoxSizer(self.edit_box, wxHORIZONTAL)
        main_box.Add(sizer_edit, 5, wxALL|wxEXPAND)

        # Projection Specific Entries (Name/Projection)
        self.sizer_projctrls = wxBoxSizer(wxVERTICAL)
        sizer_edit.Add(self.sizer_projctrls, 1, wxALL|wxEXPAND)

        hbox = wxBoxSizer(wxHORIZONTAL)
        self.sizer_projctrls.Add(hbox, 0, wxALL|wxEXPAND)
        label = wxStaticText(self, -1, _("Name:"))
        hbox.Add(label, 0, wxALL|wxALIGN_CENTER_VERTICAL, 4)
        self.projname = wxTextCtrl(self, ID_PROJ_PROJNAME, "")
        EVT_TEXT(self, ID_PROJ_PROJNAME, self._OnProjName)
        hbox.Add(self.projname, 1, wxALL|wxEXPAND, 4)

        hbox = wxBoxSizer(wxHORIZONTAL)
        self.sizer_projctrls.Add(hbox, 0, wxALL|wxEXPAND)
        label = wxStaticText(self, -1, _("Projection:"))
        hbox.Add(label, 0, wxALL|wxALIGN_CENTER_VERTICAL, 4)
        self.projchoice = wxChoice(self, ID_PROJ_PROJCHOICE)
        self.projchoice.SetSelection(0)
        EVT_CHOICE(self, ID_PROJ_PROJCHOICE, self._OnProjChoice)
        hbox.Add(self.projchoice, 1, wxALL|wxEXPAND, 4)
        # Fill the projection choice list.
        self.nbsizer = NotebookLikeSizer()
        self.sizer_projctrls.Add(self.nbsizer, 1,
                                 wxALL|wxEXPAND|wxADJUST_MINSIZE, 3)
        self.projection_panels = []
        self.projchoice.Append(_("<Unknown>"), "")
        for proj_type, name, cls in self.projection_panel_defs:
            self.projchoice.Append(name, proj_type)
            panel = cls(self, self.receiver)
            panel.Hide()
            panel.projection_index = len(self.projection_panels)
            panel.projection_type = proj_type
            self.projection_panels.append(panel)
            self.nbsizer.Add(panel)
        self.unknown_projection_panel = UnknownProjPanel(self, self.receiver)
        self.unknown_projection_panel.Hide()
        self.nbsizer.Add(self.unknown_projection_panel)

        # Projection Specific actions (New/Save/Add)
        buttons = wxBoxSizer(wxVERTICAL)
        sizer_edit.Add(buttons, 0, wxALL)
        self.button_new = wxButton(self, ID_PROJ_NEW, _("New"))
        EVT_BUTTON(self, ID_PROJ_NEW, self._OnNew)
        buttons.Add(self.button_new, 0, wxEXPAND|wxALL, 4)
        self.button_add = wxButton(self, ID_PROJ_ADDTOLIST, _("Add to List"))
        EVT_BUTTON(self, ID_PROJ_ADDTOLIST, self._OnAddToList)
        buttons.Add(self.button_add, 0, wxEXPAND|wxALL, 4)
        buttons.Add(20, 20, 0, wxEXPAND, 0)
        self.button_save = wxButton(self, ID_PROJ_SAVE,_("Update"))
        EVT_BUTTON(self, ID_PROJ_SAVE, self._OnSave)
        buttons.Add(self.button_save, 0, wxEXPAND|wxALL|wxALIGN_BOTTOM, 4)

        #
        # Main Action buttons (Try/Revert/OK/Close)
        #
        buttons = wxBoxSizer(wxHORIZONTAL)
        top_box.Add(buttons, 0, wxALL|wxALIGN_RIGHT, 10)
        self.button_try = wxButton(self, wxID_APPLY, _("Try"))
        EVT_BUTTON(self, wxID_APPLY, self.OnApply)
        buttons.Add(self.button_try, 0, wxRIGHT, 10)
        self.button_revert = wxButton(self, ID_PROJ_REVERT, _("Revert"))
        EVT_BUTTON(self, ID_PROJ_REVERT, self._OnRevert)
        buttons.Add(self.button_revert, 0, wxRIGHT, 10)
        self.button_ok = wxButton(self, wxID_OK, _("OK"))
        EVT_BUTTON(self, wxID_OK, self.OnOK)
        self.button_ok.SetDefault()
        buttons.Add(self.button_ok, 0, wxRIGHT, 10)
        self.button_close = wxButton(self, wxID_CANCEL, _("Close"))
        EVT_BUTTON(self, wxID_CANCEL, self.OnCancel)
        buttons.Add(self.button_close, 0, wxRIGHT, 10)


        #
        # Automatic Layout
        #
        self.SetAutoLayout(1)
        self.SetSizer(top_box)
        top_box.Fit(self)
        top_box.SetSizeHints(self)

    def OnClose(self, event):
        self.projection_list.Unsubscribe(PROJ_SELECTION_CHANGED,
                                         self.proj_selection_changed)
        # Destroy the projection list explicitly so that it properly
        # unsubscribes everything. It would be cleaner if the projection
        # could do this by itself but wx doesn't always send destroy
        # events for non-top-level widgets
        self.projection_list.Destroy()
        NonModalNonParentDialog.OnClose(self, event)

    def OnApply(self, event):
        self.__SetProjection()
        self.haveTried = True

    def OnOK(self, event):
        self.__SetProjection()
        self.Close()

    def OnCancel(self, event):
        """Cancel just closes the dialog, but we call it cancel so we
        can overload the functionality of wxDialog.
        """
        self.Close()

    def _OnRevert(self, event):
        if self.haveTried:
            self.receiver.SetProjection(self.originalProjection)
            self.haveTried = False

    def _OnNew(self, event):

        self.projection_list.ClearSelection()
        self.projname.Clear()

        # supply a projection panel if there wasn't one
        if self.curProjPanel is None:
            self.projchoice.SetSelection(0)
            self.__DoOnProjChoice()

        if self.curProjPanel is not None:
            self.curProjPanel.Clear()

    def _OnSave(self, event):

        sel = self.projection_list.selected_projections()
        assert len(sel) == 1,  "button shouldn't be enabled"

        proj, projfile = sel[0]

        assert proj is not None and projfile is not None

        newproj = self.__GetProjection()

        if newproj is not None:
            # FIXME: we should only allow this for the user proj file.
            projfile.Replace(proj, newproj)
            self.write_proj_file(projfile)
            self.projection_list.SelectProjection(newproj)

    def _OnAddToList(self, event):

        proj = self.__GetProjection()
        if proj is not None:
            self.__usrProjFile.Add(proj)
            self.write_proj_file(self.__usrProjFile)
            self.projection_list.SelectProjection(proj)

    def show_warnings(self, title, filename, warnings):
        """Show the warnings (a list of strings) in a dialog

        If the list is empty no dialog will be shown.
        """
        if warnings:
            text = (_('Warnings when reading "%s":\n\n%s')
                    % (filename, "\n\n".join(warnings)))
            self.parent.RunMessageBox(title, text)

    def _OnImport(self, event):
        """Handler for the 'Import' button

        Ask the user for a filename, read the projections from that file
        add them to the user ProjFile object and write the user file
        back to disk.
        """
        dlg = wxFileDialog(self, _("Import"), 
                self.parent.application.Path("projection"), style = wxOPEN)

        if dlg.ShowModal() == wxID_OK:
            path = dlg.GetPath()

            ThubanBeginBusyCursor()
            try:
                try:
                    projFile, warnings = read_proj_file(path)
                except IOError, (errno, errstr):
                    self.__ShowError(dlg.GetPath(), errstr)
                else:
                    self.show_warnings(_("Warnings"), path, warnings)
                    for proj in projFile.GetProjections():
                        self.__usrProjFile.Add(proj)
                    self.write_proj_file(self.__usrProjFile)
                    self.parent.application.SetPath("projection", path)
            finally:
                ThubanEndBusyCursor()
        dlg.Destroy()

    def _OnExport(self, event):
        """Handler for the 'Export' button.

        Ask the user for a filename and write the selected projections
        to that file.
        """
        sel = self.projection_list.selected_projections()
        assert len(sel) != 0, "button should be disabled"

        dlg = wxFileDialog(self, _("Export"), 
                self.parent.application.Path("projection"),
                style=wxSAVE|wxOVERWRITE_PROMPT)

        if dlg.ShowModal() == wxID_OK:
            proj_file = ProjFile(dlg.GetPath())
            for proj, pf in sel:
                if proj is not None:
                    proj_file.Add(proj)
            self.write_proj_file(proj_file)
            self.parent.application.SetPath("projection", dlg.GetPath())

        dlg.Destroy()

    def _OnRemove(self, event):
        """Handler for the 'Remove' button

        Remove any selected projection that came from the user's
        ProjFile. If the user ProjFile was modified write it back to
        disk.
        """
        sel = self.projection_list.selected_projections()
        assert len(sel) != 0, "button should be disabled!"

        modified = False
        for proj, pf in sel:
            if proj is not None and pf is self.__usrProjFile:
                pf.Remove(proj)
                modified = True

        if modified:
            self.write_proj_file(self.__usrProjFile)

    def _OnShowEPSG(self, event):
        """Handler for the EVT_CHECKBOX events from the EPSG check button

        If the button is checked add the EPSG_PROJ_FILE to the list of
        projfiles shown by the projection list. Otherwise remove it
        """
        proj_files = [self.load_user_proj(),
                      self.load_system_proj(DEFAULT_PROJ_FILE)]
        if self.check_epsg.IsChecked():
            proj_files.append(self.load_system_proj(EPSG_PROJ_FILE))
        if self.check_epsg_depr.IsChecked():
            proj_files.append(self.load_system_proj(EPSG_DEPRECATED_PROJ_FILE))
        self.projection_list.SetProjFiles(proj_files)

    def _OnProjName(self, event):
        self.__VerifyButtons()

    def __ShowError(self, filename, errstr):
        wxMessageDialog(self,
            _("The following error occured:\n") +
            filename + "\n" + errstr,
            _("Error"), wxOK | wxICON_ERROR).ShowModal()

    def __VerifyButtons(self):
        """Update button sensitivity"""

        num_sel = self.projection_list.GetSelectedItemCount()

        self.button_import.Enable(True)
        self.button_export.Enable(True)
        self.button_save.Enable(True)
        self.button_remove.Enable(True)

        self.edit_box.Enable(True)

        for ctrl in [self.button_import,
                     self.button_export,
                     self.button_remove,
                     self.button_save,
                     self.button_add,
                     self.projchoice,
                     self.projname,
                     self.edit_box]:
            ctrl.Enable(True)

        if self.curProjPanel is not None:
            self.curProjPanel.Enable(True)

        if num_sel == 0:
            self.button_import.Enable(True)
            self.button_export.Enable(False)
            self.button_remove.Enable(False)
            self.button_save.Enable(False)

        elif num_sel == 1:

            selection = self.projection_list.selected_projections()
            proj, projFile = selection[0]

            self.button_save.Enable(len(self.projname.GetValue()) > 0)
            self.button_add.Enable(len(self.projname.GetValue()) > 0)

            if proj is None:
                # <None> is selected
                for ctrl in [self.button_export,
                             self.button_remove,
                             self.button_save,
                             self.button_add,
                             self.projchoice,
                             self.projname]:
                    ctrl.Enable(False)

                if self.curProjPanel is not None:
                    self.curProjPanel.Enable(False)

            elif proj is self.originalProjection:
                self.button_remove.Enable(False)

            if projFile is None:
                self.button_save.Enable(False)

        else:
            self.edit_box.Enable(False)

    def proj_selection_changed(self, projs):
        """Subscribed to the projection_list's PROJ_SELECTION_CHANGED message

        Update the dialog to reflect the new selection.
        """
        if len(projs) == 0:
            self.projfilepath.SetLabel(_("No Projections selected"))
        elif len(projs) == 1:
            proj, projfile = projs[0]
            if proj is None:
                # user selected <None>
                self.projname.Clear()
                self.projfilepath.SetLabel("")
            else:
                if projfile is not None:
                    filename = os.path.basename(projfile.GetFilename())
                    self.projfilepath.SetLabel(_("Source of Projection: %s")
                                               % filename)
                else:
                    # only None if the currently used projection is selected
                    self.projfilepath.SetLabel("")

                self.projname.SetValue(proj.Label())

                myProjType = proj.GetParameter("proj")
                i = 0
                for projType, name, cls in self.projection_panel_defs:
                    if myProjType == projType:
                        self.projchoice.Enable(True)
                        self.projchoice.SetSelection(i + 1)
                        self.__DoOnProjChoice()

                        #
                        # self.curProjPanel should not be null
                        # after a call to __DoOnProjChoice
                        #
                        assert self.curProjPanel is not None

                        self.curProjPanel.SetProjection(proj)
                        break
                    i += 1
                else:
                    self.projchoice.Select(0)
                    self.projchoice.Disable()
                    self._show_proj_panel(UnknownProjPanel)
                    assert self.curProjPanel is not None
                    self.curProjPanel.SetProjection(proj)
        else:
            self.projfilepath.SetLabel(_("Multiple Projections selected"))

        self.__VerifyButtons()

    def _OnProjChoice(self, event):
        self.__DoOnProjChoice()

    def __DoOnProjChoice(self):
        """Create and layout a projection panel based on the selected
        projection type.

        This is necessary to have in seperate method since calls to
        wxChoice.SetSelection() do not trigger an event.

        At the end of this method self.curProjPanel will not be None
        if there was a item selected.
        """
        choice = self.projchoice

        sel = choice.GetSelection()
        if sel != -1:
            proj_type = choice.GetClientData(sel)
            for t, name, cls in self.projection_panel_defs:
                if t == proj_type:
                    self._show_proj_panel(cls)
                    break
        # FIXME: what to do if sel == -1?

    def _show_proj_panel(self, panel_class):
        """Show the panel as the projection panel"""
        if panel_class is UnknownProjPanel:
            self.button_ok.Disable()
            self.button_try.Disable()
            self.edit_box.Disable()
            self.nbsizer.Activate(self.unknown_projection_panel)
            self.curProjPanel = self.unknown_projection_panel
        else:
            self.button_ok.Enable(True)
            self.button_try.Enable(True)
            self.edit_box.Enable(True)
            self.unknown_projection_panel.Hide()
            for panel in self.projection_panels:
                if panel.__class__ is panel_class:
                    self.nbsizer.Activate(panel)
                    self.curProjPanel = panel

    def __SetProjection(self):
        """Set the receiver's projection."""

        #
        # save the original projection only once in case 
        # we try to apply several different projections
        #
        self.receiver.SetProjection(self.__GetProjection())

    def __GetProjection(self):
        """Return a suitable Projection object based on the current
        state of the dialog box selections.

        Could be None.
        """

        assert self.projection_list.GetSelectedItemCount() < 2, \
               "button should be disabled"

        sel = self.projection_list.selected_projections()
        if len(sel) == 1:
            if sel[0][0] is None:
                # <None> is selected
                return None

        # self.curProjPanel should always contain the most relevant data
        # for a projection
        if self.curProjPanel is not None:
            parameters = self.curProjPanel.GetParameters()
            if parameters is not None:
                return Projection(parameters, self.projname.GetValue())

        return None

    def load_user_proj(self):
        """Return the user's ProjFile

        If the file has not yet been loaded by the dialog, load it first
        with get_user_proj_file and cache it in self.__usrProjFile.

        Show a busy cursor while loading the file.
    
        If the file is not available, leave a note to the console.
        """
        if self.__usrProjFile is None:
            ThubanBeginBusyCursor()
            try:
                projfile, warnings = get_user_proj_file()
                if warnings:
                    sys.stderr.write("".join(warnings))
                self.__usrProjFile = projfile
            finally:
                ThubanEndBusyCursor()
        return self.__usrProjFile

    def load_system_proj(self, name):
        """Load the system ProjFile with the given name.

        If the file has not been loaded yet, load it first with
        get_system_proj_file and put it into the cache. The name is
        simply forwarded to get_system_proj_file.

        Show a busy cursor while loading the file.
        """
        if name not in self._sys_proj_files:
            ThubanBeginBusyCursor()
            try:
                projfile, warnings = get_system_proj_file(name)
                self.show_warnings(_("Warnings"), projfile.GetFilename(),
                                   warnings)
                self._sys_proj_files[name] = projfile
            finally:
                ThubanEndBusyCursor()
        return self._sys_proj_files[name]

    def write_proj_file(self, proj_file):
        """Write the ProjFile object proj_file back to its file

        Show a busy cursor while writing and if an error occurs show a
        dialog with the error message.
        """
        try:
            ThubanBeginBusyCursor()
            try:
                write_proj_file(proj_file)
            finally:
                ThubanEndBusyCursor()
        except IOError, (errno, errstr):
            self.__ShowError(proj_file.GetFilename(), errstr)



class ProjPanel(wxPanel):
    """Base class for all projection panels."""

    def __init__(self, parent):
        wxPanel.__init__(self, parent, -1)

        self.__ellps = wxChoice(self, -1)
        self.ellpsData = [("", _("<Unknown>")),
                          ("airy"  , _("Airy")),
                          ("bessel", _("Bessel 1841")),
                          ("clrk66", _("Clarke 1866")),
                          ("clrk80", _("Clarke 1880")),
                          ("GRS80" , _("GRS 1980 (IUGG, 1980)")),
                          ("intl"  , _("International 1909 (Hayford)")),
                          ("WGS84" , _("WGS 84"))]

        for tag, name in self.ellpsData:
            self.__ellps.Append(name, tag)

        self.__ellps.SetSelection(0)
        
    def _DoLayout(self, childPanel = None):

        panelSizer = wxBoxSizer(wxVERTICAL)

        sizer = wxBoxSizer(wxHORIZONTAL)
        sizer.Add(wxStaticText(self, -1, _("Ellipsoid:")), 0,
                                    wxALL|wxALIGN_CENTER_VERTICAL, 4)
        sizer.Add(self.__ellps, 1, wxALL|wxALIGN_CENTER_VERTICAL, 4)
        panelSizer.Add(sizer, 0, wxALL|wxEXPAND, 4)

        if childPanel is not None:
            panelSizer.Add(childPanel, 0, wxEXPAND, 0)
            
        self.SetAutoLayout(1)
        self.SetSizer(panelSizer)
        panelSizer.Fit(self)
        panelSizer.SetSizeHints(self)
        self.Layout()

    def SetProjection(self, proj):
        if proj is not None:
            param = proj.GetParameter("ellps")
            i = 0
            for tag, name in self.ellpsData:
                if param == tag:
                    self.__ellps.SetSelection(i)
                    return # returning early!
                i += 1

        #
        # if proj is none, or the parameter couldn't be found...
        #
        self.__ellps.SetSelection(0)

    def GetParameters(self):
        ellps = self.__ellps.GetSelection()
        if ellps > 0:
            return ["ellps=" + self.__ellps.GetClientData(ellps)]
        return []


ID_TMPANEL_LAT = 4001
ID_TMPANEL_LONG = 4002
ID_TMPANEL_FASLE_EAST = 4003
ID_TMPANEL_FALSE_NORTH = 4004
ID_TMPANEL_SCALE = 4005

class UnknownProjPanel(ProjPanel):

    """Panel for unknown projection types"""

    def __init__(self, parent, receiver):
        ProjPanel.__init__(self, parent)

        self.__text = _("Thuban does not know the parameters\n"
                        "for the current projection and cannot\n"
                        "display a configuration panel.\n\n"
                        "The unidentified set of parameters is:\n\n")

        self.__textbox = wxTextCtrl(self, -1, self.__text, size=(100,200),
                            style=wxTE_READONLY|wxTE_MULTILINE|wxTE_LINEWRAP)
        self._DoLayout()

    def _DoLayout(self):
        sizer = wxBoxSizer(wxVERTICAL)

        sizer.Add(self.__textbox, 0, wxALL|wxEXPAND, 4)

        ProjPanel._DoLayout(self, sizer)

    def GetProjName(self):
        return "Unknown"

    def SetProjection(self, proj):
        """Append the available parameters to the info text."""
        text = self.__text
        for param in proj.GetAllParameters():
            text = text + '%s\n' % param
        self.__textbox.SetValue(text)

    def GetParameters(self):
        return None

class TMPanel(ProjPanel):
    """Projection panel for Transverse Mercator."""

    def __init__(self, parent, receiver):
        ProjPanel.__init__(self, parent)

        self.__latitude = wxTextCtrl(self, ID_TMPANEL_LAT)
        self.__longitude = wxTextCtrl(self, ID_TMPANEL_LONG)
        self.__falseEast = wxTextCtrl(self, ID_TMPANEL_FASLE_EAST)
        self.__falseNorth = wxTextCtrl(self, ID_TMPANEL_FALSE_NORTH)
        self.__scale = wxTextCtrl(self, ID_TMPANEL_SCALE)

        self._DoLayout()

    def _DoLayout(self):

        sizer = wxFlexGridSizer(4, 2, 0, 0)
        sizer.Add(wxStaticText(self, -1, _("Latitude:")), 0, wxALL, 4)
        sizer.Add(self.__latitude, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("Longitude:")), 0, wxALL, 4)
        sizer.Add(self.__longitude, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("False Easting:")), 0, wxALL, 4)
        sizer.Add(self.__falseEast, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("False Northing:")), 0, wxALL, 4)
        sizer.Add(self.__falseNorth, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("Scale Factor:")), 0, wxALL, 4)
        sizer.Add(self.__scale, 0, wxALL, 4)

        ProjPanel._DoLayout(self, sizer)

    def GetProjName(self):
        return _("Transverse Mercator")

    def SetProjection(self, proj):
        ProjPanel.SetProjection(self, proj)

        self.__latitude.SetValue(proj.GetParameter("lat_0"))
        self.__longitude.SetValue(proj.GetParameter("lon_0"))
        self.__falseEast.SetValue(proj.GetParameter("x_0"))
        self.__falseNorth.SetValue(proj.GetParameter("y_0"))
        self.__scale.SetValue(proj.GetParameter("k"))

        ProjPanel.SetProjection(self, proj)

    def GetParameters(self):
        params = ["proj=tmerc",
                  "lat_0=" + self.__latitude.GetValue(),
                  "lon_0=" + self.__longitude.GetValue(),
                  "x_0="   + self.__falseEast.GetValue(),
                  "y_0="   + self.__falseNorth.GetValue(),
                  "k="     + self.__scale.GetValue()]
        params.extend(ProjPanel.GetParameters(self))
        return params

    def Clear(self):
        self.__latitude.Clear()
        self.__longitude.Clear()
        self.__falseEast.Clear()
        self.__falseNorth.Clear()
        self.__scale.Clear()

        ProjPanel.Clear(self)

ID_UTMPANEL_ZONE = 4001
ID_UTMPANEL_SOUTH = 4002
ID_UTMPANEL_PROP = 4003

class UTMPanel(ProjPanel):
    """Projection Panel for Universal Transverse Mercator."""

    def __init__(self, parent, receiver):
        ProjPanel.__init__(self, parent)

        self.receiver = receiver

        self.__zone = wxSpinCtrl(self, ID_UTMPANEL_ZONE, "1", min=1, max=60)
        self.__propButton = wxButton(self, ID_UTMPANEL_PROP, _("Propose"))
        self.__south = wxCheckBox(self, ID_UTMPANEL_SOUTH, 
                                  _("Southern Hemisphere"))

        self._DoLayout()

        EVT_BUTTON(self, ID_UTMPANEL_PROP, self._OnPropose)

    def _DoLayout(self):

        sizer = wxBoxSizer(wxVERTICAL)
        psizer = wxBoxSizer(wxHORIZONTAL)
        psizer.Add(wxStaticText(self, -1, _("Zone:")), 0, wxALL, 4)
        psizer.Add(self.__zone, 0, wxALL, 4)
        psizer.Add(self.__propButton, 0, wxALL, 4)
        sizer.Add(psizer, 0, wxALL, 4)
        sizer.Add(self.__south, 0, wxALL, 4)

        ProjPanel._DoLayout(self, sizer)

    def GetProjName(self):
        return _("Universal Transverse Mercator")

    def SetProjection(self, proj):
        self.__zone.SetValue(int(proj.GetParameter("zone")))
        self.__south.SetValue(proj.GetParameter("south") != "")
        ProjPanel.SetProjection(self, proj)

    def GetParameters(self):
        params = ["proj=utm", "zone=" + str(self.__zone.GetValue())]
        if self.__south.IsChecked():
            params.append("south")

        params.extend(ProjPanel.GetParameters(self))
        return params

    def Clear(self):
        self.__zone.SetValue(1)
        self.__south.SetValue(False)
        ProjPanel.Clear(self)

    def _OnPropose(self, event):
        """Call the propose dialog.
        If the receiver (e.g. the current map) has no bounding box,
        inform the user accordingly.
        """
        bb = self.receiver.BoundingBox()
        if bb is None:
            dlg = wxMessageDialog(self,
                    _("Can not propose: No bounding box found."),
                    _("Projection: Propose UTM Zone"),
                    wxOK | wxICON_INFORMATION)
            dlg.CenterOnParent()
            result = dlg.ShowModal()
            dlg.Destroy()
            return

        dlg = UTMProposeZoneDialog(self, self.receiver.BoundingBox())
        if dlg.ShowModal() == wxID_OK:
            self.__zone.SetValue(dlg.GetProposedZone())

class LCCPanel(ProjPanel): 
    """Projection Panel for Lambert Conic Conformal."""

    def __init__(self, parent, receiver):
        ProjPanel.__init__(self, parent)
        
        self.__fspLatitude = wxTextCtrl(self, -1)
        self.__sspLatitude = wxTextCtrl(self, -1)
        self.__meridian    = wxTextCtrl(self, -1)
        self.__originLat   = wxTextCtrl(self, -1)
        self.__falseEast   = wxTextCtrl(self, -1)
        self.__falseNorth  = wxTextCtrl(self, -1)

        self._DoLayout()

    def _DoLayout(self):

        sizer = wxFlexGridSizer(6, 2, 0, 0)
        sizer.Add(wxStaticText(self, -1, 
            _("Latitude of first standard parallel:")))
        sizer.Add(self.__fspLatitude, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1,
            _("Latitude of second standard parallel:")))
        sizer.Add(self.__sspLatitude, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("Central Meridian:")))
        sizer.Add(self.__meridian, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("Latitude of origin:")))
        sizer.Add(self.__originLat, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("False Easting:")))
        sizer.Add(self.__falseEast, 0, wxALL, 4)
        sizer.Add(wxStaticText(self, -1, _("False Northing:")))
        sizer.Add(self.__falseNorth, 0, wxALL, 4)

        ProjPanel._DoLayout(self, sizer)

    def GetProjName(self):
        return _("Lambert Conic Conformal")
        
    def SetProjection(self, proj):
        self.__fspLatitude.SetValue(proj.GetParameter("lat_1"))
        self.__sspLatitude.SetValue(proj.GetParameter("lat_2"))
        self.__originLat.SetValue(proj.GetParameter("lat_0"))
        self.__meridian.SetValue(proj.GetParameter("lon_0"))
        self.__falseEast.SetValue(proj.GetParameter("x_0"))
        self.__falseNorth.SetValue(proj.GetParameter("y_0"))

        ProjPanel.SetProjection(self, proj)

    def GetParameters(self):
        params = ["proj=lcc", 
                  "lat_1=" + self.__fspLatitude.GetValue(),
                  "lat_2=" + self.__sspLatitude.GetValue(),
                  "lat_0=" + self.__originLat.GetValue(),
                  "lon_0=" + self.__meridian.GetValue(),
                  "x_0=" + self.__falseEast.GetValue(),
                  "y_0=" + self.__falseNorth.GetValue()]

        params.extend(ProjPanel.GetParameters(self))
        return params

    def Clear(self):
        self.__fspLatitude.Clear()
        self.__sspLatitude.Clear()
        self.__originLat.Clear()
        self.__meridian.Clear()
        self.__falseEast.Clear()
        self.__falseNorth.Clear()

        ProjPanel.Clear(self)

class GeoPanel(ProjPanel): 
    """Projection Panel for a Geographic Projection."""

    def __init__(self, parent, receiver):
        ProjPanel.__init__(self, parent)

        self.__choices = [(_("Degrees"), "0.017453"),
                          (_("Radians"), "1")]

        self.__scale = wxChoice(self, -1)
        for choice, value in self.__choices:
            self.__scale.Append(choice, value)

        self._DoLayout()

    def GetProjName(self):
        return _("Geographic")
        
    def SetProjection(self, proj):
        value = proj.GetParameter("to_meter")
        for i in range(len(self.__choices)):
            choice, data = self.__choices[i]
            if value == data:
                self.__scale.SetSelection(i)
        ProjPanel.SetProjection(self, proj)

    def GetParameters(self):
        params = ["proj=latlong", 
                  "to_meter=%s" % self.__scale.GetClientData(
                                  self.__scale.GetSelection())]

        params.extend(ProjPanel.GetParameters(self))
        return params

    def Clear(self):
        ProjPanel.Clear(self)

    def _DoLayout(self):
        sizer = wxBoxSizer(wxHORIZONTAL)

        sizer.Add(wxStaticText(self, -1, _("Source Data is in: ")), 
                  0, wxALL|wxALIGN_CENTER_VERTICAL, 4)
        sizer.Add(self.__scale, 1, wxEXPAND|wxALL, 4)

        self.__scale.SetSelection(0)

        ProjPanel._DoLayout(self, sizer)


ID_UTM_PROPOSE_ZONE_DIALOG_TAKE   = 4001
ID_UTM_PROPOSE_ZONE_DIALOG_CANCEL = 4002
class UTMProposeZoneDialog(wxDialog):

    """Propose a sensible Zone considering the current map extent."""

    def __init__(self, parent, (x, y, x2, y2)):
        wxDialog.__init__(self, parent, -1, _("Projection: Propose UTM Zone"),
                          wxDefaultPosition, wxSize(200, 100))
        self.parent = parent
        x = x + 180
        x2 = x2 + 180
        center = (x2 - x) / 2 + x
        self.proposedZone = int(center / 6 + 1)
        self.dialogLayout()

    def dialogLayout(self):
        topBox = wxBoxSizer(wxVERTICAL)

        textBox = wxBoxSizer(wxVERTICAL)
        textBox.Add(wxStaticText(self, -1, _("The current map extent center "
                                             "lies in UTM Zone")),
                    0, wxALIGN_CENTER|wxALL, 4)
        textBox.Add(wxStaticText(self, -1, str(self.proposedZone)),
                    0, wxALIGN_CENTER|wxALL, 4)

        topBox.Add(textBox, 1, wxEXPAND|wxALL, 4)

        buttonBox = wxBoxSizer(wxHORIZONTAL)
        buttonBox.Add(wxButton(self, ID_UTM_PROPOSE_ZONE_DIALOG_TAKE,
                      _("Take")), 0, wxALL, 4)
        buttonBox.Add(wxButton(self, ID_UTM_PROPOSE_ZONE_DIALOG_CANCEL,
                               _("Cancel")), 0, wxALL, 4)
        topBox.Add(buttonBox, 0, wxALIGN_CENTER_HORIZONTAL|wxALL, 10)
        EVT_BUTTON(self, ID_UTM_PROPOSE_ZONE_DIALOG_TAKE, self.OnTake)
        EVT_BUTTON(self, ID_UTM_PROPOSE_ZONE_DIALOG_CANCEL, self.OnCancel)

        self.SetAutoLayout(True)
        self.SetSizer(topBox)
        topBox.Fit(self)
        topBox.SetSizeHints(self)

    def OnTake(self, event):
        self.EndModal(wxID_OK)

    def OnCancel(self, event):
        self.EndModal(wxID_CANCEL)

    def GetProposedZone(self):
        return self.proposedZone
