# Copyright (c) 2001, 2002, 2003, 2004 by Intevation GmbH
# Authors:
#       Markus Rechtien <markus@intevation.de>
#
# This program is free software under the GPL (>=v2)
# Read the file COPYING coming with Thuban for details.

"""Tests for svgexport"""

__version__ = "$Revision: 1.2 $"
# $Source: /thubanrepository/thuban/Extensions/svgexport/test/test_svgmapwriter.py,v $
# $Id: test_svgmapwriter.py,v 1.2 2004/02/20 15:49:36 bh Exp $


import os
import sys
import unittest

# If run directly as a script, add Thuban's test directory to the path.
# Otherwise we assume that the importing code as already done it
if __name__ == "__main__":
    sys.path.append(os.path.join(os.path.abspath(os.path.dirname(sys.argv[0])),
                                 "..", "..", "..", "test"))
import support
support.initthuban()
import Thuban

# Now import needed SVG stuff
from Extensions.svgexport.svgmapwriter import VirtualDC, \
     Pen, Brush, SOLID,  Point, Font, TRANSPARENT_PEN, TRANSPARENT_BRUSH
# Color related classes from the model of thuban
from Thuban.Model.color import Color, Black

# We use xmlsupport to verify the SVG output
import xmlsupport

class VFile:
    '''A virtual file to show which contents would have been
    written to a file.'''
    
    def open(self, filename, mode="w"): 
        '''Compatibility function - initializes an empty string.'''
        self.str = ""

    def write(self, string):
        '''The file.write function to capture all input.'''
        if hasattr(self, "str"):
            self.str += string
        else: 
            self.str = ""
            self.str += string


class TestVirtualDC(unittest.TestCase):
    '''Tests the VirtualDC class that imitates a wxDC and writes
    SVGRenderer instead.'''

    def setUp(self):
        '''Initialize our tools.'''
        self.black = Black
        self.solid_pen = Pen(Color(1.0, 0.0, 1.0), 3, SOLID)
        self.trans_pen = TRANSPARENT_PEN
        self.solid_brush = Brush(Color(0.0, 1.0, 0.0), SOLID)
        self.trans_brush = TRANSPARENT_BRUSH
        self.polygon = [Point(5.6,5.5), Point(95.4,5.3), Point(95.2,95.1),
                Point(5.0,95.0), Point(5.0,5.0)]
        self.meta = {"Object":"test", "Label":"nothing"}
        self.font = Font("Helvetica", 12)
    
    def test_clippath(self):
        '''Define a clipping region and close it afterwards.'''
        data = '<clipPath id="testid">\n</clipPath>'
        file = VFile()
        dc = VirtualDC(file)
        dc.BeginClipPath('testid')
        dc.EndClipPath()
        self.assertEquals(xmlsupport.sax_eventlist(data = data), 
                          xmlsupport.sax_eventlist(data = file.str))
        
    def test_polygon_closed(self):
        '''Set drawing properties and draw a polygon onto the dc.'''
        data = ('<path style="stroke-linejoin:round; stroke:#ff00ff; '
                'stroke-width:3; stroke-linecap:round; fill:#00ff00" ' 
                'meta="Object:test; Label:nothing" d="M 5.6 5.5 '
                'L 95.4 5.3 L 95.2 95.1 L 5.0 95.0 L 5.0 5.0 Z"/>')
        file = VFile()
        dc = VirtualDC(file)
        dc.SetPen(self.solid_pen)
        dc.SetMeta(self.meta)
        dc.SetBrush(self.solid_brush)
        dc.DrawPolygon(self.polygon, 1)
        self.assertEquals(xmlsupport.sax_eventlist(data = data), 
                          xmlsupport.sax_eventlist(data = file.str))
    
    def test_polygon_opened(self):
        '''Set drawing properties and draw a polygon onto the dc.'''
        data = ('<path style="stroke-linejoin:round; stroke:#ff00ff; '
                'stroke-width:3; stroke-linecap:round; fill:#00ff00" ' 
                'meta="Object:test; Label:nothing" d="M 5.6 5.5 '
                'L 95.4 5.3 L 95.2 95.1 L 5.0 95.0 L 5.0 5.0 "/>')
        file = VFile()
        dc = VirtualDC(file)
        dc.SetPen(self.solid_pen)
        dc.SetMeta(self.meta)
        dc.SetBrush(self.solid_brush)
        dc.DrawPolygon(self.polygon, 0)
        self.assertEquals(xmlsupport.sax_eventlist(data = data), 
                          xmlsupport.sax_eventlist(data = file.str))
    
    def test_rect(self):
        '''Set drawing properties and draw a rectangle'''
        data = ('<rect x="5.5" y="5.4" width="90.3" height="90.2" '
                'style="stroke-linejoin:round; stroke:#ff00ff; '
                'stroke-width:3; stroke-linecap:round; fill:none" '
                'meta="Object:test; Label:nothing"/>')
        file = VFile()
        dc = VirtualDC(file)
        dc.SetPen(self.solid_pen)
        dc.SetMeta(self.meta)
        dc.SetBrush(self.trans_brush)
        dc.DrawRectangle(5.5, 5.4, 90.3, 90.2)
        self.assertEquals(xmlsupport.sax_eventlist(data = data), 
                          xmlsupport.sax_eventlist(data = file.str))
    
    def test_circle(self):
        '''Set drawing properties and draw a circle'''
        data = ('<circle cx="5.5" cy="5.3" r="90.1"  style="'
                'fill:#00ff00" meta="Object:test; Label:nothing"/>')
        file = VFile()
        dc = VirtualDC(file)
        dc.SetPen(self.trans_pen)
        dc.SetMeta(self.meta)
        dc.SetBrush(self.solid_brush)
        dc.DrawCircle(5.5, 5.3, 90.1)
        self.assertEquals(xmlsupport.sax_eventlist(data = data), 
                          xmlsupport.sax_eventlist(data = file.str))
    
    def test_ellipse(self):
        '''Set drawing properties and draw a circle'''
        data = ('<ellipse cx="5.5" cy="5.3" rx="90.1" ry="100.321"  style="'
                'fill:#00ff00" meta="Object:test; Label:nothing"/>')
        file = VFile()
        dc = VirtualDC(file)
        dc.SetPen(self.trans_pen)
        dc.SetMeta(self.meta)
        dc.SetBrush(self.solid_brush)
        dc.DrawEllipse(5.5, 5.3, 90.1, 100.321)
        self.assertEquals(xmlsupport.sax_eventlist(data = data), 
                          xmlsupport.sax_eventlist(data = file.str))
    
    def test_text(self):
        '''Set drawing properties and draw a circle'''
        data = ('<text x="123.321" y="1515.5151" style="font-size:12; '
                'font-family:Helvetica; fill:#000000" >Some text.</text>')
        file = VFile()
        dc = VirtualDC(file)
        dc.SetTextForeground(self.black)
        dc.SetFont(self.font)
        dc.DrawText('Some text.', 123.321, 1515.5151)
        self.assertEquals(xmlsupport.sax_eventlist(data = data), 
                          xmlsupport.sax_eventlist(data = file.str))
    
    def test_document(self):
        '''Set up a document with a dimension and 
        latin encoding'''
        data = ('<?xml version="1.0" encoding="ISO-8859-1" standalone'
                '="yes"?>\n<svg>\n</svg>')
        file = VFile()
        dc = VirtualDC(file)
        dc.BeginExport()
        dc.EndExport()
        self.assertEquals(xmlsupport.sax_eventlist(data = data), 
                          xmlsupport.sax_eventlist(data = file.str))


if __name__ == "__main__":
    support.run_tests()
