/***************************************************************************
                             th-main.c
                             ---------
    begin                : Mon Nov 08 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
 
#include "th-app-window.h"
#include "th-utils.h"

#include "gst-plugins/th-yuvscale.h"

#include <gtk/gtk.h>
#include <gst/gst.h>

#include <glib/gi18n.h>

#include <string.h>

/***************************************************************************
 *
 *   ensure_static_plugins
 *
 *   Make sure our static plugins are not optimised away by the linker
 *
 ***************************************************************************/

static gboolean
ensure_static_plugins (void)
{
	if (!th_yuvscale_get_type ())
		return FALSE;
	
	return TRUE;
}

/***************************************************************************
 *
 *   check_required_plugins
 *
 ***************************************************************************/

#define MIN_GST_PLUGINS_REQUIRED 0,8,10
#define MIN_GST_CORE_REQUIRED 0,8,10

static gboolean
check_required_plugins (void)
{
	struct {
		const gchar *plugin_name;
		const gchar *feature_name;
		guint        maj, min, micro;
	} req_plugins[] = { 
	 { "dvdreadsrc", "dvdreadsrc", MIN_GST_PLUGINS_REQUIRED },
	 { "mpegstream", "dvddemux", MIN_GST_PLUGINS_REQUIRED },
	 { "mpeg2dec", "mpeg2dec", MIN_GST_PLUGINS_REQUIRED }, 
	 { "a52dec", "a52dec", MIN_GST_PLUGINS_REQUIRED },
	 /* { "dtsdec", "dtsdec", MIN_GST_PLUGINS_REQUIRED }, */
	 /* { "dvdlpcmdec", "dvdlpcmdec", MIN_GST_PLUGINS_REQUIRED }, */
	 { "theora", "theoraenc", MIN_GST_PLUGINS_REQUIRED },
	 { "vorbis", "rawvorbisenc", MIN_GST_PLUGINS_REQUIRED },
	 { "ogg", "oggmux", MIN_GST_PLUGINS_REQUIRED },
	 { "ffmpegcolorspace", "ffmpegcolorspace", MIN_GST_PLUGINS_REQUIRED },
	 { "gstaudioconvert", "audioconvert", MIN_GST_PLUGINS_REQUIRED },
	 { "audioscale", "audioscale", MIN_GST_PLUGINS_REQUIRED },
	 { "videocrop", "videocrop", MIN_GST_PLUGINS_REQUIRED },
	 { "videorate", "videorate", MIN_GST_PLUGINS_REQUIRED },
	 { "deinterlace", "deinterlace", MIN_GST_PLUGINS_REQUIRED },
	 { "gstelements", "filesink", MIN_GST_CORE_REQUIRED }, /* core element */
	 { "gstelements", "identity", MIN_GST_CORE_REQUIRED }, /* core element */
	};
  
	GString *missing, *too_old;
	guint    i;
	
	missing = g_string_new (NULL);
	too_old = g_string_new (NULL);
	
	for (i = 0; i < G_N_ELEMENTS (req_plugins); ++i)
	{
		guint  major, minor, micro;

		if (!th_utils_get_plugin_version (req_plugins[i].plugin_name,
		                                  req_plugins[i].feature_name,
		                                  &major, &minor,&micro, NULL))
		{
			g_string_append_printf (missing, "  %s\n", req_plugins[i].feature_name);
			continue;
		}
		
		if ((major  > req_plugins[i].maj)
		 || (major == req_plugins[i].maj && minor  > req_plugins[i].min)
		 || (major == req_plugins[i].maj && minor == req_plugins[i].min && micro >= req_plugins[i].micro))
			continue;
		
		g_string_append_printf (too_old, "  %s (v%u.%u.%u &lt; %u.%u.%u)\n", 
		                        req_plugins[i].feature_name, major, minor, micro,
		                        req_plugins[i].maj, req_plugins[i].min, req_plugins[i].micro);
	}

	if (missing->len > 0 || too_old->len > 0)
	{
		GtkWidget *dlg;
		GString *errmsg;
		 
		errmsg = g_string_new (NULL);
		
		g_string_append_printf (errmsg, "\n<b>%s</b>\n\n", _("Missing or outdated GStreamer plugins"));
		g_string_append (errmsg, _("You seem to be missing some GStreamer plugins, or\n"
		                           "some of your GStreamer plugins are too old, or\n"
		                           "the GStreamer registry needs to be rebuilt by\n"
		                           "running 'gst-register' as root.\n"));
		
		g_string_append (errmsg, "\n\n");
		
		if (missing->len > 0)
		{
			g_string_append_printf (errmsg, "<b>%s</b>\n%s\n\n", _("Missing GStreamer plugins:"), missing->str);
			g_string_append (errmsg, "\n\n");
		}
		
		if (too_old->len > 0)
		{
			g_string_append_printf (errmsg, "<b>%s</b>\n%s\n\n", _("Outdated GStreamer plugins:"), too_old->str);
			g_string_append (errmsg, "\n\n");
		}

		dlg = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_MODAL,
		                                          GTK_MESSAGE_ERROR,
		                                          GTK_BUTTONS_CLOSE,
		                                          errmsg->str);
		
		(void) gtk_dialog_run (GTK_DIALOG (dlg));

		gtk_widget_destroy (dlg);

		g_string_free (missing, TRUE);
		g_string_free (too_old, TRUE);
		g_string_free (errmsg, TRUE);
		
		return FALSE;
	}
	
	g_string_free (missing, TRUE);
	g_string_free (too_old, TRUE);
	
	return TRUE;
}

/***************************************************************************
 *
 *   output_log_line
 *
 ***************************************************************************/

static void
output_log_line (const gchar *line, gboolean add_newline)
{
	gchar *s;

	s = g_strdup_printf (" :: %s%s", line, (add_newline) ? "\n" : "");

	g_printerr ("%s", s);

	th_log_add_line (s);

	g_free (s);
}

/***************************************************************************
 *
 *   th_main_log_func
 *
 ***************************************************************************/

static void
th_main_log_func (const gchar *domain, GLogLevelFlags f, const gchar *s, gpointer foo)
{
	gchar **p, **arr, *nl;

	nl = strstr (s, "\n");
	if (nl == NULL)
	{
		output_log_line (s, TRUE);
		return;
	}

	if (nl && *(nl+1) != 0x00)
	{
		arr = g_strsplit (s, "\n", -1);
		for (p = arr;  p && *p;  ++p)
			output_log_line (*p, TRUE);
		g_strfreev (arr);
	}
	else 
	{
		output_log_line (s, FALSE);
	}

	if (((f & G_LOG_LEVEL_INFO) == 0 && f != 0) || g_ascii_strcasecmp (domain, "thoggen"))
		g_log_default_handler (domain, f, s, NULL);
}

/***************************************************************************
 *
 *   th_main_print_func
 *
 ***************************************************************************/

static void
th_main_print_func (const gchar *s)
{
	th_main_log_func ("Thoggen", 0, s, NULL);
}


/***************************************************************************
 *
 *   print_system_info
 *
 ***************************************************************************/

static void
print_system_info (void)
{
	guint  maj, min, micro;
	gchar *out = NULL;

	gst_version (&maj, &min, &micro);

	th_log ("GStreamer core version: %u.%u.%u\n", maj, min, micro);

	if (g_spawn_command_line_sync ("uname -a", &out, NULL, NULL, NULL))
	{
		g_strdelimit (out, "\n", ' ');
		th_log ("System: %s\n", out);
		g_free (out);
	}
}

/***************************************************************************
 *
 *   main
 *
 ***************************************************************************/

int
main (int argc, char **argv)
{
	union {
		GtkWidget  *appwin;
		gpointer    appwin_ptr;
	} run_info;

	gtk_init (&argc, &argv);
	gst_init (&argc, &argv);

	if (!check_required_plugins ())
		g_error ("We are missing some required plugins. There's something wrong with the installation.\n");
	
	/* TODO: popup a dialog? */
	if (!ensure_static_plugins ())
		g_error ("Eeek, somehow our static plugins didn't get linked in or registered.\n");

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE "_iso_639", LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE "_iso_639", "UTF-8");

	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	g_log_set_default_handler (th_main_log_func, NULL);
	g_set_print_handler (th_main_print_func);

	print_system_info ();
	
	run_info.appwin = th_app_window_new ();

	g_object_add_weak_pointer (G_OBJECT (run_info.appwin), &run_info.appwin_ptr);

	th_app_window_run (TH_APP_WINDOW (run_info.appwin));

	if (run_info.appwin != NULL)
		gtk_widget_destroy (run_info.appwin);

	return EXIT_SUCCESS;
}

