/*

    File: photorec.c

    Copyright (C) 1998-2006 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License along
    with this program; if not, write the Free Software Foundation, Inc., 51
    Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include <stdarg.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>	/* unlink, ftruncate */
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#include <ctype.h>      /* tolower */
#ifdef HAVE_LOCALE_H
#include <locale.h>	/* setlocale */
#endif
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#include <errno.h>
#include "types.h"
#include "common.h"
#include "lang.h"
#include "intrf.h"
#include "godmode.h"
#include "fnctdsk.h"
#include "dir.h"
#include "fat.h"
#include "fat_dir.h"
#include "photorec.h"
#include "sessionp.h"
#include "hdcache.h"
#include "chgtype.h"
#include "list.h"
#include "fatp.h"
#include "ewf.h"

#define DEFAULT_RECUP_DIR "recup_dir"

typedef struct list_cluster_struct list_cluster_t;
struct list_cluster_struct
{
  struct list_head list;
  uint64_t offset;
  uint32_t cluster;
  t_file_data *dir_list;
};

enum photorec_status { STATUS_FIND_DIR, STATUS_FIND_OFFSET, STATUS_EXT2_ON, STATUS_EXT2_ON_BF, STATUS_EXT2_OFF, STATUS_EXT2_OFF_BF, STATUS_EXT2_ON_SAVE_EVERYTHING, STATUS_EXT2_OFF_SAVE_EVERYTHING, STATUS_QUIT };
typedef enum photorec_status t_photorec_status;

extern const t_arch_fnct arch_i386;
extern const t_arch_fnct arch_mac;
extern const t_arch_fnct arch_none;
extern const t_arch_fnct arch_sun;
extern const t_arch_fnct arch_xbox;
#define INTER_SELECT_X	0
#define INTER_SELECT_Y	23
#define INTER_SELECT	15
#define MAX_FILES_PER_DIR	500

extern const t_file_hint file_hint_7z;
extern const t_file_hint file_hint_aif;
extern const t_file_hint file_hint_all;
extern const t_file_hint file_hint_asf;
extern const t_file_hint file_hint_au;
extern const t_file_hint file_hint_bkf;
extern const t_file_hint file_hint_blend;
extern const t_file_hint file_hint_bmp;
extern const t_file_hint file_hint_bz2;
extern const t_file_hint file_hint_cam;
extern const t_file_hint file_hint_crw;
extern const t_file_hint file_hint_ctg;
extern const t_file_hint file_hint_cwk;
extern const t_file_hint file_hint_dbf;
extern const t_file_hint file_hint_dir;
extern const t_file_hint file_hint_djv;
extern const t_file_hint file_hint_doc;
extern const t_file_hint file_hint_dsc;
extern const t_file_hint file_hint_dv;
extern const t_file_hint file_hint_eps;
extern const t_file_hint file_hint_exe;
extern const t_file_hint file_hint_ext2_sb;
extern const t_file_hint file_hint_gif;
extern const t_file_hint file_hint_gz;
extern const t_file_hint file_hint_imb;
extern const t_file_hint file_hint_itunes;
extern const t_file_hint file_hint_jpg;
extern const t_file_hint file_hint_flac;
extern const t_file_hint file_hint_mdb;
extern const t_file_hint file_hint_mdf;
extern const t_file_hint file_hint_mid;
extern const t_file_hint file_hint_mov;
extern const t_file_hint file_hint_mp3;
extern const t_file_hint file_hint_mpg;
extern const t_file_hint file_hint_mrw;
extern const t_file_hint file_hint_mus;
extern const t_file_hint file_hint_mysql;
extern const t_file_hint file_hint_ogg;
extern const t_file_hint file_hint_orf;
extern const t_file_hint file_hint_qdf;
extern const t_file_hint file_hint_pap;
extern const t_file_hint file_hint_pcx;
extern const t_file_hint file_hint_pdf;
extern const t_file_hint file_hint_png;
extern const t_file_hint file_hint_ps;
extern const t_file_hint file_hint_prc;
extern const t_file_hint file_hint_psd;
extern const t_file_hint file_hint_pst;
extern const t_file_hint file_hint_qxd;
extern const t_file_hint file_hint_raf;
extern const t_file_hint file_hint_rar;
extern const t_file_hint file_hint_raw;
extern const t_file_hint file_hint_rdc;
extern const t_file_hint file_hint_riff;
extern const t_file_hint file_hint_rm;
extern const t_file_hint file_hint_rns;
extern const t_file_hint file_hint_sit;
extern const t_file_hint file_hint_stuffit;
extern const t_file_hint file_hint_swf;
extern const t_file_hint file_hint_tar;
extern const t_file_hint file_hint_tiff;
extern const t_file_hint file_hint_txt;
extern const t_file_hint file_hint_x3f;
extern const t_file_hint file_hint_xcf;
extern const t_file_hint file_hint_zip;


static void get_next_sector(t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset, const unsigned int blocksize);
static void get_next_header(t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset);
#ifdef DEAD_CODE
static void get_prev_sector(t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset, const unsigned int blocksize);
static void get_prev_header(t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset);
#endif
static int get_prev_file_header(t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset);
static int file_finish(t_file_recovery *file_recovery, const char *recup_dir, const int paranoid, unsigned int *file_nbr,
    const unsigned int blocksize, t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset,
    unsigned int *dir_num, const t_photorec_status status, const unsigned int sector_size, const t_param_disk *disk_car);
static void reset_file_stats(t_file_stat *file_stats);
static void reset_file_recovery(t_file_recovery *file_recovery);
static void write_stats_log(const t_file_stat *file_stats);
static void write_stats_stdout(const t_file_stat *file_stats);
static void update_stats(t_file_stat *file_stats, t_alloc_data *list_search_space);
static void interface_file_select(t_file_stat *file_stats, char**current_cmd);
static void interface_options_photorec(int *paranoid, int *allow_partial_last_cylinder, int *keep_corrupted_file, unsigned int *mode_ext2, unsigned int *expert, unsigned int *filenaming, unsigned int *lowmem, char**current_cmd);
static int do_curses_photorec(int debug, const char *recup_dir, const t_list_disk *list_disk, t_file_stat *file_stats, char *cmd_device, char*cmd_run);
static int photorec(t_param_disk *disk_car, t_partition *partition, const int debug, const int paranoid, const char *recup_dir, const int keep_corrupted_file, const int interface, t_file_stat *file_stats, const unsigned int mode_ext2, char **current_cmd, t_alloc_data *list_search_space, unsigned int blocksize, const unsigned int expert, const unsigned int filenaming, const unsigned int lowmem, const unsigned int carve_free_space_only);
static t_partition *new_whole_disk(const t_param_disk *disk_car);

static FILE *f_rapport=NULL;
static int f_status=0;
static unsigned int find_blocksize(t_alloc_data *list_file, const unsigned int default_blocksize, uint64_t *offset);
static t_alloc_data * update_blocksize(unsigned int blocksize, t_alloc_data *list_search_space, const uint64_t offset);
static t_alloc_data *update_search_space(const t_file_recovery *file_recovery, t_alloc_data *list_search_space, t_alloc_data **new_current_search_space, uint64_t *offset, const unsigned int blocksize);
static t_alloc_data *update_search_space_aux(t_alloc_data *list_search_space, uint64_t start, uint64_t end, t_alloc_data **new_current_search_space, uint64_t *offset);

static unsigned int photorec_mkdir(const char *recup_dir, const unsigned int initial_dir_num);
static int photorec_aux(t_param_disk *disk_car, t_partition *partition, const int debug, const int paranoid, const char *recup_dir, const int interface, t_file_stat *file_stats, unsigned int *file_nbr, unsigned int *blocksize, t_alloc_data *list_search_space, const time_t real_start_time, unsigned int *dir_num, const t_photorec_status status, const unsigned int pass, const unsigned int expert, const unsigned int filenaming, const unsigned int lowmem);
void free_list_search_space(t_alloc_data *list_search_space);
static void list_cluster_free(list_cluster_t *list_cluster);
static unsigned int find_blocksize_cluster(list_cluster_t *list_cluster, const unsigned int default_blocksize, uint64_t *offset);
static void list_space_used(const char *filename_out, const t_file_recovery *file_recovery, const unsigned int sector_size);
static void list_free_add(const t_file_recovery *file_recovery, t_alloc_data *list_search_space);
static void forget(t_alloc_data *list_search_space, t_alloc_data *current_search_space);
static int sort_file_stats(const void *p1, const void *p2);
static int photorec_progressbar(WINDOW *window, const unsigned int pass, const t_photorec_status status, const uint64_t offset, t_param_disk *disk_car, t_partition *partition, const unsigned int file_nbr, const time_t elapsed_time, const t_file_stat *file_stats);
static int ask_mode_ext2(const t_param_disk *disk_car, const t_partition *partition, unsigned int *mode_ext2, unsigned int *carve_free_space_only);

void ecrit_rapport_string(const char *string,const int max_length)
{
  int i;
  for(i=0;(string[i]!='\0')&&(i<max_length);i++)
    ecrit_rapport("%c",string[i]);
}

int ecrit_rapport(const char *_format, ...)
{
  int res=0;
  if(f_rapport!=NULL)
  {
    va_list ap;
    va_start(ap,_format);
    res=vfprintf(f_rapport,_format,ap);
    va_end(ap);
    if(res<0)
    {
      f_status=1;
    }
    /* flushing the outputs slow down the program, don't flush. Hope it's not a bad idea */
    /*
    fflush(stderr);
    if(fflush(f_rapport))
    {
      f_status=1;
    }
    */
  }
#ifdef DEBUG
  else
  {
    va_list ap;
    va_start(ap,_format);
    res=vfprintf(stderr,_format,ap);
    va_end(ap);
    if(res<0)
    {
      f_status=1;
    }
    if(fflush(stderr))
    {
      f_status=1;
    }
  }
#endif
  return res;
}

#ifdef HAVE_SIGACTION
void sighup_hdlr(int shup)
{
  ecrit_rapport("SIGHUP detected! PhotoRec has been killed.\n");
  if(f_rapport!=NULL)
  {
    fclose(f_rapport);
  }
  exit(1);
}
#endif

static void list_space_used(const char *filename_out, const t_file_recovery *file_recovery, const unsigned int sector_size)
{
  const t_alloc_list *element;
  uint64_t file_size=0;
  uint64_t file_size_on_disk=0;
  if(filename_out==NULL || file_recovery->file_size==0)
    return;
  ecrit_rapport("%s\t",filename_out);
  for(element=&file_recovery->location;element!=NULL;element=element->next)
  {
    file_size_on_disk+=(element->end-element->start+1);
    if(element->data>0)
    {
      ecrit_rapport(" %lu-%lu", (unsigned long)(element->start/sector_size), (unsigned long)(element->end/sector_size));
      file_size+=(element->end-element->start+1);
    }
    else
    {
      ecrit_rapport(" (%lu-%lu)", (unsigned long)(element->start/sector_size), (unsigned long)(element->end/sector_size));
    }
  }
  ecrit_rapport("\n");
//  ecrit_rapport("list file_size %lu, file_size_on_disk %lu\n",(unsigned long)file_size,(unsigned long)file_size_on_disk);
//  ecrit_rapport("file_size %lu, file_size_on_disk %lu\n",(unsigned long)file_recovery->file_size,(unsigned long)file_recovery->file_size_on_disk);
}

//#define DEBUG_FREE 1

static void list_free_add(const t_file_recovery *file_recovery, t_alloc_data *list_search_space)
{
  struct list_head *search_walker = NULL;
#ifdef DEBUG_FREE
  ecrit_rapport("list_free_add %lu\n",(long unsigned)(file_recovery->location.start/512));
#endif
  list_for_each(search_walker, &list_search_space->list)
  {
    t_alloc_data *current_search_space;
    current_search_space=list_entry(search_walker, t_alloc_data, list);
    if(current_search_space->start < file_recovery->location.start && file_recovery->location.start < current_search_space->end)
    {
      t_alloc_data *new_free_space;
      new_free_space=(t_alloc_data*)MALLOC(sizeof(*new_free_space));
      new_free_space->start=file_recovery->location.start;
      new_free_space->end=current_search_space->end;
      new_free_space->file_stat=NULL;
      current_search_space->end=file_recovery->location.start-1;
      list_add(&new_free_space->list, search_walker);
    }
    if(current_search_space->start==file_recovery->location.start)
    {
      current_search_space->file_stat=file_recovery->file_stat;
      return ;
    }
  }
}

static t_alloc_data *update_search_space(const t_file_recovery *file_recovery, t_alloc_data *list_search_space, t_alloc_data **new_current_search_space, uint64_t *offset, const unsigned int blocksize)
{
  const t_alloc_list *element;
  for(element=&file_recovery->location;element!=NULL;element=element->next)
  {
    uint64_t end=(element->end-(element->start%blocksize)+blocksize-1+1)/blocksize*blocksize+(element->start%blocksize)-1;
#ifdef DEBUG
    ecrit_rapport("update_search_space_aux %lu (%lu-%lu)\n", (long unsigned int)((*offset)/blocksize),
	(unsigned long int)(element->start/blocksize),
	(unsigned long int)(end/blocksize));
#endif
    list_search_space=update_search_space_aux(list_search_space, element->start, end, new_current_search_space, offset);
  }
  return list_search_space;
}

t_alloc_data *del_search_space(t_alloc_data *list_search_space, uint64_t start, uint64_t end)
{
  return update_search_space_aux(list_search_space, start, end, NULL, NULL);
}

static t_alloc_data *update_search_space_aux(t_alloc_data *list_search_space, uint64_t start, uint64_t end, t_alloc_data **new_current_search_space, uint64_t *offset)
{
  struct list_head *search_walker = NULL;
  if(start >= end)
    return list_search_space;
  list_for_each(search_walker, &list_search_space->list)
  {
    t_alloc_data *current_search_space;
    current_search_space=list_entry(search_walker, t_alloc_data, list);
    if(current_search_space->start==start)
    {
      if(current_search_space->end>end)
      {
	if(new_current_search_space!=NULL && current_search_space->start<=*offset && *offset<=end)
	  *offset=end+1;
	current_search_space->start=end+1;
	return list_search_space;
      }
      start=current_search_space->end+1;
      if(list_search_space==current_search_space)
	list_search_space=list_entry(current_search_space->list.next, t_alloc_data, list);
      if(new_current_search_space!=NULL && *new_current_search_space==current_search_space)
      {
	*new_current_search_space=list_entry(current_search_space->list.next, t_alloc_data, list);
	*offset=(*new_current_search_space)->start;
      }
      list_del(search_walker);
      free(current_search_space);
      return update_search_space_aux(list_search_space, start,end, new_current_search_space, offset);
    }
    if(current_search_space->end==end)
    {
      if(current_search_space->start<start)
      {
	current_search_space->end=start-1;
	return list_search_space;
      }
      end=current_search_space->start-1;
      if(list_search_space==current_search_space)
	list_search_space=list_entry(current_search_space->list.next, t_alloc_data, list);
      if(new_current_search_space!=NULL && *new_current_search_space==current_search_space)
      {
	*new_current_search_space=list_entry(current_search_space->list.next, t_alloc_data, list);
	*offset=(*new_current_search_space)->start;
      }
      list_del(search_walker);
      free(current_search_space);
      return update_search_space_aux(list_search_space, start,end, new_current_search_space, offset);
    }
    if(start < current_search_space->start && current_search_space->start <= end)
    {
      list_search_space=update_search_space_aux(list_search_space,
	  start, current_search_space->start-1,  new_current_search_space, offset);
      return update_search_space_aux(list_search_space, current_search_space->start, end, new_current_search_space, offset);
    }
    if(start <= current_search_space->end && current_search_space->end < end)
    {
      list_search_space=update_search_space_aux(list_search_space, start, current_search_space->end, new_current_search_space, offset);
      return update_search_space_aux(list_search_space, current_search_space->end+1, end, new_current_search_space, offset);
    }
    if(current_search_space->start < start && end < current_search_space->end)
    {
      t_alloc_data *new_free_space;
      new_free_space=(t_alloc_data*)MALLOC(sizeof(*new_free_space));
      new_free_space->start=start;
      new_free_space->end=current_search_space->end;
      new_free_space->file_stat=NULL;
      current_search_space->end=start-1;
      list_add(&new_free_space->list,search_walker);
      if(new_current_search_space!=NULL && new_free_space->start<=*offset && *offset<=new_free_space->end)
      {
	*new_current_search_space=new_free_space;
      }
      return update_search_space_aux(list_search_space, start, end, new_current_search_space, offset);
    }
  }
  return list_search_space;
}

static unsigned int menu_choose_blocksize(unsigned int blocksize, const unsigned int sector_size, uint64_t *offset)
{
  int command;
  unsigned int menu=0;
  const char *optionsBlocksize="51248736";
  static struct MenuItem menuBlocksize[]=
  {
	{'5',"512",""},
	{'1',"1024",""},
	{'2',"2048",""},
	{'4',"4096",""},
	{'8',"8192",""},
	{'7',"16384",""},
	{'3',"32768",""},
	{'6',"65536",""},
	{0,NULL,NULL}
  };
  switch(sector_size)
  {
    case 1024: optionsBlocksize+=1; break;
    case 2048: optionsBlocksize+=2; break;
    case 4096: optionsBlocksize+=3; break;
    case 8192: optionsBlocksize+=4; break;
    case 16384: optionsBlocksize+=5;break;
    case 32768: optionsBlocksize+=6; break;
    case 65536: optionsBlocksize+=7; break;
  }
  switch(blocksize)
  {
    case 512: menu=0; break;
    case 1024: menu=1; break;
    case 2048: menu=2; break;
    case 4096: menu=3; break;
    case 8192: menu=4; break;
    case 16384: menu=5; break;
    case 32768: menu=6; break;
    case 65536: menu=7; break;
  }
  aff_copy(stdscr);
  wmove(stdscr,INTER_PARTITION_Y-1,0);
  wdoprintf(stdscr,"Please select the block size, press Enter when done.");
  command = wmenuSelect_ext(stdscr,INTER_PARTITION_Y, INTER_PARTITION_X, menuBlocksize, 7,
      optionsBlocksize, MENU_VERT| MENU_BUTTON|MENU_VERT_WARN, &menu,NULL);
  switch(command)
  {
    case '5': blocksize=512; break;
    case '1': blocksize=1024; break;
    case '2': blocksize=2048; break;
    case '4': blocksize=4096; break;
    case '8': blocksize=8192; break;
    case '7': blocksize=16384; break;
    case '3': blocksize=32768; break;
    case '6': blocksize=65536; break;
  }
  if(*offset%sector_size!=0 || *offset>=blocksize)
    *offset=0;
  if(sector_size < blocksize)
  {
    unsigned int quit=0;
    aff_copy(stdscr);
    wmove(stdscr,INTER_PARTITION_Y-2,0);
    wdoprintf(stdscr,"Please select the offset (0 - %u). Press Up/Down to increase/decrease it,",blocksize-sector_size);
    wmove(stdscr,INTER_PARTITION_Y-1,0);
    wdoprintf(stdscr,"Enter when done.");
    do
    {
      wmove(stdscr,INTER_PARTITION_Y,0);
      wclrtoeol(stdscr);
      wdoprintf(stdscr,"Offset %u",(unsigned int)(*offset));
      switch(wgetch(stdscr))
      {
	case KEY_ENTER:
#ifdef PADENTER
	case PADENTER:
#endif
	case '\n':
	case '\r':
	  quit=1;
	  break;
	case KEY_PPAGE:
	case KEY_UP:
	case KEY_RIGHT:
	case '+':
	  if(*offset + sector_size < blocksize)
	    *offset+=sector_size;
	  break;
	case KEY_NPAGE:
	case KEY_DOWN:
	case KEY_LEFT:
	case '-':
	  if(*offset >= sector_size)
	    *offset-=sector_size;
	  break;
      }
    } while(quit==0);
  }
  ecrit_rapport("blocksize=%u,offset=%u\n",blocksize,(unsigned int)*offset);
  return blocksize;
}

static t_alloc_data *init_search_space(const t_partition *partition, const t_param_disk *disk_car)
{
  t_alloc_data *list_search_space;
  list_search_space=(t_alloc_data*)MALLOC(sizeof(*list_search_space));
  list_search_space->start=partition->part_offset;
  list_search_space->end=partition->part_offset+partition->part_size-1;
  if(list_search_space->end > disk_car->disk_size-1)
    list_search_space->end = disk_car->disk_size-1;
  if(list_search_space->end > disk_car->disk_real_size-1)
    list_search_space->end = disk_car->disk_real_size-1;
  list_search_space->file_stat=NULL;
  list_search_space->list.prev=&list_search_space->list;
  list_search_space->list.next=&list_search_space->list;
  return list_search_space;
}

void free_list_search_space(t_alloc_data *list_search_space)
{
  struct list_head *search_walker = NULL;
  struct list_head *search_walker_next = NULL;
  list_for_each_safe(search_walker,search_walker_next,&list_search_space->list)
  {
    t_alloc_data *current_search_space;
    current_search_space=list_entry(search_walker, t_alloc_data, list);
    list_del(search_walker);
    free(current_search_space);
  }
}

static void reset_file_stats(t_file_stat *file_stats)
{
  unsigned int i;
  for(i=0;file_stats[i].file_hint!=NULL;i++)
  {
    file_stats[i].not_recovered=0;
    file_stats[i].recovered=0;
  }
}

static void reset_file_recovery(t_file_recovery *file_recovery)
{
  file_recovery->filename[0]='\0';
  file_recovery->file_stat=NULL;
  file_recovery->handle=NULL;
  file_recovery->file_size=0;
  file_recovery->file_size_on_disk=0;
  file_recovery->location.prev=NULL;
  file_recovery->location.next=NULL;
  file_recovery->location.start=0;
  file_recovery->location.end=0;
  file_recovery->location.data=0;
}

static unsigned int photorec_mkdir(const char *recup_dir, const unsigned int initial_dir_num)
{
  char working_recup_dir[2048];
  int dir_ok=0;
  int dir_num=initial_dir_num;
#ifdef DJGPP
  int i=0;
#endif
  do
  {
    snprintf(working_recup_dir,sizeof(working_recup_dir)-1,"%s.%d",recup_dir,dir_num);
#ifdef HAVE_MKDIR
#ifdef __MINGW32__
    if(mkdir(working_recup_dir)!=0 && errno==EEXIST)
#else
      if(mkdir(working_recup_dir, 0700)!=0 && errno==EEXIST)
#endif
#else
#error You need a mkdir function!
#endif
      {
	dir_num++;
      }
      else
      {
	dir_ok=1;
      }
#ifdef DJGPP
  /* Avoid endless loop in Dos version of Photorec after 999 directories if working with short name */
    i++;
    if(dir_ok==0 && i==1000)
    {
      dir_num=initial_dir_num;
      dir_ok=1;
    }
#endif
  } while(dir_ok==0);
  return dir_num;
}

static void get_next_sector(t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset, const unsigned int blocksize)
{
  /*
  ecrit_rapport("get_next_sector     %lu (%lu-%lu)\n", (long unsigned int)((*offset)/blocksize),
      (unsigned long int)((*current_search_space)->start/blocksize),
      (unsigned long int)((*current_search_space)->end/blocksize));
  */
  if((*current_search_space) == list_search_space)
  {
    /*
    ecrit_rapport("get_next_sector en1 %lu (%lu-%lu)\n", (long unsigned int)((*offset)/blocksize),
	(unsigned long int)((*current_search_space)->start/blocksize),
	(unsigned long int)((*current_search_space)->end/blocksize));
    */
    return ;
  }
  if(! ((*current_search_space)->start <= *offset && (*offset)<=(*current_search_space)->end))
  {
    ecrit_rapport("BUG: get_next_sector stop everything\n");
    (*current_search_space)=list_search_space;
    return ;
  }
  if((*offset)+blocksize <= (*current_search_space)->end)
  {
    *offset+=blocksize;
    /*
    ecrit_rapport("get_next_sector en2 %lu (%lu-%lu)\n", (long unsigned int)((*offset)/blocksize),
	(unsigned long int)((*current_search_space)->start/blocksize),
	(unsigned long int)((*current_search_space)->end/blocksize));
    */
    return ;
  }
  get_next_header(list_search_space, current_search_space, offset);
  /*
  ecrit_rapport("get_next_sector en3 %lu (%lu-%lu)\n", (long unsigned int)((*offset)/blocksize),
      (unsigned long int)((*current_search_space)->start/blocksize),
      (unsigned long int)((*current_search_space)->end/blocksize));
  */
}

static void get_next_header(t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset)
{
  if((*current_search_space) == list_search_space)
    return ;
  *current_search_space=list_entry((*current_search_space)->list.next, t_alloc_data, list);
  if((*current_search_space) == list_search_space)
    return ;
  *offset=(*current_search_space)->start;
}

#ifdef DEAD_CODE
static void get_prev_sector(t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset, const unsigned int blocksize)
{
  /*
  ecrit_rapport("get_prev_sector %lu (%lu-%lu)\n", (long unsigned int)((*offset)/blocksize),
      (unsigned long int)((*current_search_space)->start/blocksize),
      (unsigned long int)((*current_search_space)->end/blocksize));
  */
  if((*current_search_space) == list_search_space)
    return ;
  if(! ((*current_search_space)->start <= *offset && (*offset)<=(*current_search_space)->end))
  {
    ecrit_rapport("BUG: stop everything\n");
    *current_search_space=list_search_space;
    return ;
  }
  if((*offset)-blocksize >= (*current_search_space)->start)
  {
    *offset-=blocksize;
    return ;
  }
  get_prev_header(list_search_space, current_search_space, offset);
}

static void get_prev_header(t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset)
{
  if((*current_search_space) == list_search_space)
    return ;
  *current_search_space=list_entry((*current_search_space)->list.prev, t_alloc_data, list);
  if((*current_search_space) == list_search_space)
    return ;
  *offset=(*current_search_space)->start;
}
#endif

static int get_prev_file_header(t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset)
{
  uint64_t prev_offset;
  int nbr=0;
  struct list_head *search_walker = NULL;
  t_alloc_data *file_space=*current_search_space;
  prev_offset=(*current_search_space)->start;
  for(search_walker=(*current_search_space)->list.prev;
      search_walker!=&list_search_space->list && file_space->file_stat==NULL;
      search_walker=search_walker->prev)
  {
    file_space=list_entry(search_walker, t_alloc_data, list);
    if(file_space->end+1<prev_offset)
      nbr++;
  }
#ifdef DEBUG
  ecrit_rapport("get_prev_file_header offset=%lu -> %lu nbr=%u\n", (unsigned long int)((*offset)/512),
      (unsigned long int)(list_search_space->start/512), nbr);
#endif
  if(file_space!=list_search_space && file_space->file_stat!=NULL && nbr<5)
  {
    *current_search_space=file_space;
    *offset=file_space->start;
    return 0;
  }
  return -1;
}

static int file_finish(t_file_recovery *file_recovery, const char *recup_dir, const int paranoid, unsigned int *file_nbr,
    const unsigned int blocksize, t_alloc_data *list_search_space, t_alloc_data **current_search_space, uint64_t *offset,
    unsigned int *dir_num, const t_photorec_status status, const unsigned int sector_size, const t_param_disk *disk_car)
{
  int file_recovered=0;
#ifdef DEBUG
  ecrit_rapport("file_finish start %lu (%lu-%lu)\n", (long unsigned int)((*offset)/blocksize),
      (unsigned long int)((*current_search_space)->start/blocksize),
      (unsigned long int)((*current_search_space)->end/blocksize));
#endif
  if(file_recovery->handle)
  {
    if(status!=STATUS_EXT2_ON_SAVE_EVERYTHING && status!=STATUS_EXT2_OFF_SAVE_EVERYTHING)
    {
      if(file_recovery->file_stat!=NULL && file_recovery->file_stat->file_hint->file_check!=NULL && paranoid>0)
      {
	file_recovery->file_stat->file_hint->file_check(file_recovery);
      }
      if(file_recovery->file_stat!=NULL && file_recovery->file_size> 0 &&
	  file_recovery->file_size < file_recovery->file_stat->file_hint->min_filesize)
      { 
	ecrit_rapport("File too small ( %llu < %llu), reject it\n",
	    (long long unsigned)file_recovery->file_size,
	    (long long unsigned) file_recovery->file_stat->file_hint->min_filesize);
	file_recovery->file_size=0;
	file_recovery->file_size_on_disk=0;
      }
      /* FIXME: need to adapt read_size to volume size to avoid this */
      if(file_recovery->file_size > disk_car->disk_size)
	file_recovery->file_size = disk_car->disk_size;
      if(file_recovery->file_size > disk_car->disk_real_size)
	file_recovery->file_size = disk_car->disk_real_size;
#ifdef HAVE_FTRUNCATE
      fflush(file_recovery->handle);
      if(ftruncate(fileno(file_recovery->handle), file_recovery->file_size)<0)
      {
	ecrit_rapport("ftruncate failed.\n");
      }
#endif
    }
    fclose(file_recovery->handle);
    file_recovery->handle=NULL;
//    ecrit_rapport("%s %llu\n",file_recovery->filename,(long long unsigned)file_recovery->file_size);
    if(file_recovery->file_size>0)
    {
      if((++(*file_nbr))%MAX_FILES_PER_DIR==0)
      {
	*dir_num=photorec_mkdir(recup_dir,*dir_num+1);
      }
      if(paranoid==0 || (status!=STATUS_EXT2_ON_SAVE_EVERYTHING && status!=STATUS_EXT2_OFF_SAVE_EVERYTHING))
	file_recovery->file_stat->recovered++;
    }
    else
    {
      unlink(file_recovery->filename);
    }
  }
  if(file_recovery->file_stat!=NULL)
  {
    list_truncate(&file_recovery->location,file_recovery->file_size);
    if(status!=STATUS_FIND_OFFSET)
      list_space_used(file_recovery->filename,file_recovery,sector_size);
    if(file_recovery->file_size==0)
    {
      /* File hasn't been sucessfully recovered, remember where it begins */
      list_free_add(file_recovery, list_search_space);
      if((*current_search_space)!=list_search_space &&
	  !((*current_search_space)->start <= *offset && *offset <= (*current_search_space)->end))
	*current_search_space=list_entry((*current_search_space)->list.next, t_alloc_data, list);
    }
    else if(status!=STATUS_EXT2_ON_SAVE_EVERYTHING && status!=STATUS_EXT2_OFF_SAVE_EVERYTHING && status!=STATUS_FIND_OFFSET)
    {
      list_search_space=update_search_space(file_recovery,list_search_space,current_search_space,offset,blocksize);
      file_recovered=1;
    }
    list_delete(file_recovery->location.next);
    file_recovery->location.next=NULL;
    file_recovery->file_stat=NULL;
  }
  reset_file_recovery(file_recovery);
#ifdef DEBUG
  ecrit_rapport("file_finish end %lu (%lu-%lu)\n\n", (long unsigned int)((*offset)/blocksize),
      (unsigned long int)((*current_search_space)->start/blocksize),
      (unsigned long int)((*current_search_space)->end/blocksize));
#endif
  return file_recovered;
}

static void forget(t_alloc_data *list_search_space, t_alloc_data *current_search_space)
{
  struct list_head *search_walker = NULL;
  struct list_head *prev= NULL;
  int nbr=0;
  if(current_search_space==list_search_space)
    return ;
  for(search_walker=current_search_space->list.prev;
      search_walker!=&list_search_space->list;
      search_walker=prev)
  {
    t_alloc_data *tmp;
    tmp=list_entry(search_walker, t_alloc_data, list);
    prev=search_walker->prev;
    if(nbr>10000)
    {
      list_del(&tmp->list);
      free(tmp);
    }
    else
      nbr++;
  }
}

static void list_cluster_free(list_cluster_t *list_cluster)
{
  struct list_head *dir_walker = NULL;
  struct list_head *dir_walker_next = NULL;
  list_for_each_safe(dir_walker,dir_walker_next,&list_cluster->list)
  {
    list_cluster_t *info;
    info=list_entry(dir_walker, list_cluster_t, list);
    delete_list_file(info->dir_list);
    list_del(dir_walker);
    free(info);
  }
}

static void photorec_info(WINDOW *window, const t_file_stat *file_stats)
{
  unsigned int i;
  unsigned int nbr;
  unsigned int others=0;
  t_file_stat *new_file_stats;
  for(i=0;file_stats[i].file_hint!=NULL;i++);
  nbr=i;
  new_file_stats=(t_file_stat*)MALLOC(nbr*sizeof(t_file_stat));
  memcpy(new_file_stats, file_stats, nbr*sizeof(t_file_stat));
  qsort(new_file_stats, nbr, sizeof(t_file_stat), sort_file_stats);
  for(i=0;i<nbr && new_file_stats[i].recovered>0;i++)
  {
    if(i<10)
    {
      wmove(window,11+i,0);
      wclrtoeol(window);
      wdoprintf(window, "%s: %u recovered\n",
	  (new_file_stats[i].file_hint->extension!=NULL?
	   new_file_stats[i].file_hint->extension:""),
	  new_file_stats[i].recovered);
    }
    else
      others+=new_file_stats[i].recovered;
  }
  if(others>0)
  {
    wmove(window,11+10,0);
    wclrtoeol(window);
    wdoprintf(window, "others: %u recovered\n", others);
  }
  free(new_file_stats);
}

static int photorec_progressbar(WINDOW *window, const unsigned int pass, const t_photorec_status status, const uint64_t offset, t_param_disk *disk_car, t_partition *partition, const unsigned int file_nbr, const time_t elapsed_time, const t_file_stat *file_stats)
{
  wmove(window,9,0);
  wclrtoeol(window);
  wdoprintf(window, "Pass %u - ", pass);
  if(status==STATUS_FIND_DIR)
    wdoprintf(window,"Reading sector %10lu/%lu, %u directories found\n",(unsigned long)((offset-partition->part_offset)/disk_car->sector_size),(unsigned long)(partition->part_size/disk_car->sector_size), file_nbr);
  else if(status==STATUS_FIND_OFFSET)
    wdoprintf(window,"Reading sector %10lu/%lu, %u/10 headers found\n",(unsigned long)((offset-partition->part_offset)/disk_car->sector_size),(unsigned long)(partition->part_size/disk_car->sector_size), file_nbr);
  else
    wdoprintf(window,"Reading sector %10lu/%lu, %u files found\n",(unsigned long)((offset-partition->part_offset)/disk_car->sector_size),(unsigned long)(partition->part_size/disk_car->sector_size), file_nbr);
  wmove(window,10,0);
  wclrtoeol(window);
  wdoprintf(window,"Elapsed time %uh%02um%02us - Estimated time for achievement %uh%02um%02u\n",
      (unsigned)(elapsed_time/60/60),
      (unsigned)(elapsed_time/60%60),
      (unsigned)(elapsed_time%60),
      (unsigned)((partition->part_offset+partition->part_size-1-offset)*elapsed_time/(offset-partition->part_offset)/3600),
      (unsigned)(((partition->part_offset+partition->part_size-1-offset)*elapsed_time/(offset-partition->part_offset)/60)%60),
      (unsigned)((partition->part_offset+partition->part_size-1-offset)*elapsed_time/(offset-partition->part_offset))%60);
  photorec_info(window, file_stats);
  wrefresh(window);
  return check_enter_or_s(window);
}

static int photorec_aux(t_param_disk *disk_car, t_partition *partition, const int debug, const int paranoid, const char *recup_dir, const int interface, t_file_stat *file_stats, unsigned int *file_nbr, unsigned int *blocksize, t_alloc_data *list_search_space, const time_t real_start_time, unsigned int *dir_num, const t_photorec_status status, const unsigned int pass, const unsigned int expert, const unsigned int filenaming, const unsigned int lowmem)
{
  uint64_t offset=0;
  unsigned char *double_buffer;
  unsigned char *buffer;
  const char *current_extension=NULL;
  time_t start_time;
  time_t previous_time;
  int ind_stop=0;
  unsigned int read_size;
  t_alloc_data *current_search_space;
  t_file_recovery file_recovery;
  static t_alloc_data list_file={
    .list = LIST_HEAD_INIT(list_file.list)
  };
  static list_cluster_t list_cluster= {
    .list = LIST_HEAD_INIT(list_cluster.list)
  };
  read_size=((*blocksize)>8192?(*blocksize):8192);
  double_buffer=MALLOC((*blocksize)+read_size);
  buffer=double_buffer+(*blocksize);
  reset_file_recovery(&file_recovery);

  start_time=time(NULL);
  previous_time=start_time;
  memset(double_buffer,0,2*(*blocksize));
  if(debug>0)
  {
    struct list_head *search_walker = NULL;
    ecrit_rapport("Explore ");
    list_for_each(search_walker, &list_search_space->list)
    {
      t_alloc_data *cur_free_space;
      cur_free_space=list_entry(search_walker, t_alloc_data, list);
      ecrit_rapport(" %lu-%lu",(long unsigned)(cur_free_space->start/disk_car->sector_size),
	  (long unsigned)(cur_free_space->end/disk_car->sector_size));
    }
    ecrit_rapport("\n");
  }
  current_search_space=list_entry(list_search_space->list.next, t_alloc_data, list);
  if(current_search_space!=list_search_space)
    offset=current_search_space->start;
  while(current_search_space!=list_search_space)
  {
    int file_recovered=0;
    uint64_t old_offset=offset;
    if(lowmem>0 && pass>0)
      forget(list_search_space,current_search_space);
    if(debug>1)
    {
      ecrit_rapport("Reading sector %10lu/%lu\n",(unsigned long)((offset-partition->part_offset)/disk_car->sector_size),(unsigned long)((partition->part_size-1)/disk_car->sector_size));
    }
    disk_car->read(disk_car,read_size, buffer, offset);
//    dump_log(buffer,512);
    if(interface!=0 && offset%(123*0x200)==0 && offset!=partition->part_offset)
    {
      time_t current_time;
      current_time=time(NULL);
      if(current_time>previous_time)
      {
	previous_time=current_time;
	ind_stop=photorec_progressbar(stdscr, pass, status, offset, disk_car, partition, *file_nbr, current_time-real_start_time, file_stats);
      }
    }
    {
      if(file_recovery.file_stat!=NULL &&
	  file_recovery.file_stat->file_hint->min_header_distance > 0 &&
	  file_recovery.file_size<=file_recovery.file_stat->file_hint->min_header_distance)
      {
      }
      else if(file_recovery.file_stat!=NULL && file_recovery.file_stat->file_hint==&file_hint_tar &&
	  (current_extension=file_hint_tar.header_check(double_buffer+(*blocksize)-0x200,0x200,0,&file_recovery))!=NULL)
      {
	if(debug>1)
	{
	  ecrit_rapport("Currently saving a tar file, sector %lu.\n",
	      (unsigned long)((offset-partition->part_offset)/disk_car->sector_size));
	}
      }
      else
      {
	unsigned int i;
	t_file_stat *file_stat_found=NULL;
	if(status==STATUS_FIND_DIR)
	{
	  for(i=0;file_stats[i].file_hint!=NULL;i++)
	    if(file_stats[i].file_hint==&file_hint_dir &&
		((current_extension=file_stats[i].file_hint->header_check(buffer,read_size,0,&file_recovery))!=NULL))
	      file_stat_found=&file_stats[i];
	}
	else
	{
	  for(i=0;file_stats[i].file_hint!=NULL && ( file_stats[i].enable==0 || file_stats[i].ifnothingfound==1 ||
		(current_extension=file_stats[i].file_hint->header_check(buffer,read_size,0,&file_recovery))==NULL);i++);
	  file_stat_found=&file_stats[i];
	  if(file_stat_found->file_hint==NULL)
	  {
	    for(i=0;file_stats[i].file_hint!=NULL && ( file_stats[i].enable==0 || file_stats[i].ifnothingfound==0 ||
		  (current_extension=file_stats[i].file_hint->header_check(buffer,read_size,(file_recovery.file_stat!=NULL || status==STATUS_FIND_OFFSET),&file_recovery))==NULL);i++);
	    file_stat_found=&file_stats[i];
	  }
	}

	if(file_stat_found->file_hint!=NULL)
	{
	  if(file_finish(&file_recovery,recup_dir,paranoid,file_nbr,*blocksize,list_search_space,&current_search_space, &offset, dir_num,status,disk_car->sector_size,disk_car))
	    file_recovered=1;
	  if(file_recovered>0 && (offset!=current_search_space->start || file_stat_found->file_hint==&file_hint_txt) && get_prev_file_header(list_search_space, &current_search_space, &offset)>=0)
	  {
	    file_recovered=2;
	    memset(double_buffer,0,(*blocksize));
	  }
	  else
	  {
	    file_recovery.file_stat=file_stat_found;
	    file_recovery.location.start=offset;
	    if(lowmem>0)
	      forget(list_search_space,current_search_space);
	    if(debug>1)
	    {
	      ecrit_rapport("%s header found at sector %lu\n",
		  ((current_extension!=NULL && current_extension[0]!='\0')?
		   current_extension:file_stat_found->file_hint->description),
		  (unsigned long)((offset-partition->part_offset)/disk_car->sector_size));
	    }

	    if(status==STATUS_FIND_OFFSET)
	    {
	      /* Backup file offset */
	      t_alloc_data *new_file_alloc;
	      new_file_alloc=(t_alloc_data*)MALLOC(sizeof(*new_file_alloc));
	      new_file_alloc->start=file_recovery.location.start;
	      new_file_alloc->end=0;
	      list_add_tail(&new_file_alloc->list,&list_file.list);
	      (*file_nbr)++;
	    }
	    if(file_stat_found->file_hint==&file_hint_dir)
	    {
	      t_file_data *dir_list;
	      dir_list=dir_fat_aux(buffer,read_size,0);
	      if(dir_list!=NULL)
	      {
		if(debug>0)
		{
		  dir_aff_log(disk_car, partition, NULL, dir_list);
		}
		if(status==STATUS_FIND_DIR)
		{
		  list_cluster_t *info=MALLOC(sizeof(*info));
		  info->offset=file_recovery.location.start;
		  info->cluster=dir_list->filestat.st_ino;
		  info->dir_list=dir_list;
		  list_add_tail(&info->list,&list_cluster.list);
		  (*file_nbr)++;
		}
		else
		  delete_list_file(dir_list);
	      }
	    }
	  }
	}
      }
      if(file_recovery.file_stat!=NULL && file_recovery.handle==NULL)
      {
	if(filenaming==0)
	{
	  if(current_extension==NULL || current_extension[0]=='\0')
	  {
	    snprintf(file_recovery.filename,sizeof(file_recovery.filename)-1,"%s.%u/%c%u",recup_dir,
		*dir_num,(status==STATUS_EXT2_ON_SAVE_EVERYTHING||status==STATUS_EXT2_OFF_SAVE_EVERYTHING?'b':'f'),
		*file_nbr+1);
	  }
	  else
	  {
	    snprintf(file_recovery.filename,sizeof(file_recovery.filename)-1,"%s.%u/%c%u.%s",recup_dir,
		*dir_num,(status==STATUS_EXT2_ON_SAVE_EVERYTHING||status==STATUS_EXT2_OFF_SAVE_EVERYTHING?'b':'f'),
		*file_nbr+1,current_extension);
	  }
	}
	else
	{
	  if(current_extension==NULL || current_extension[0]=='\0')
	  {
	    snprintf(file_recovery.filename,sizeof(file_recovery.filename)-1,"%s.%u/%c%u",recup_dir,
		*dir_num,(status==STATUS_EXT2_ON_SAVE_EVERYTHING||status==STATUS_EXT2_OFF_SAVE_EVERYTHING?'b':'f'),
		(unsigned int)((file_recovery.location.start-partition->part_offset)/disk_car->sector_size));
	  }
	  else
	  {
	    snprintf(file_recovery.filename,sizeof(file_recovery.filename)-1,"%s.%u/%c%u.%s",recup_dir,
		*dir_num,(status==STATUS_EXT2_ON_SAVE_EVERYTHING||status==STATUS_EXT2_OFF_SAVE_EVERYTHING?'b':'f'),
		(unsigned int)((file_recovery.location.start-partition->part_offset)/disk_car->sector_size), current_extension);
	  }
	}
	if(file_recovery.file_stat->file_hint->recover==1 && status!=STATUS_FIND_OFFSET)
	{
	  if(!(file_recovery.handle=fopen(file_recovery.filename,"w+b")))
	  { 
	    ecrit_rapport("Can't create file\n");
	    ind_stop=2;
	  }
	}
      }
    }
    /* EXT2_NDIR_BLOCKS=12, try to skip indirect block */
    if((status==STATUS_EXT2_ON || status==STATUS_EXT2_ON_SAVE_EVERYTHING) &&
	file_recovery.file_stat!=NULL &&
	((file_recovery.file_size_on_disk==12*(*blocksize)) ||
	 (file_recovery.file_size_on_disk==(12+1+(*blocksize)/4)*(*blocksize)) ||
	 (file_recovery.file_size_on_disk%(*blocksize)==0 && file_recovery.file_size_on_disk>(12+1+(*blocksize)/4)*(*blocksize) && ((file_recovery.file_size_on_disk/(*blocksize))-(12+1))%((*blocksize)/4+1)==0)) &&
	(file_recovery.location.start>=current_search_space->start)
      )
    {
      list_append_block(&file_recovery.location,offset,*blocksize,0);
      file_recovery.file_size_on_disk+=*blocksize;
      if(debug>1)
      {
	ecrit_rapport("Skipping sector %10lu/%lu\n",(unsigned long)((offset-partition->part_offset)/disk_car->sector_size),(unsigned long)(partition->part_size/disk_car->sector_size));
      }
    }
    else
    {
      if(file_recovery.handle!=NULL)
      {
	if(fwrite(buffer,*blocksize,1,file_recovery.handle)<1)
	{ 
	  ecrit_rapport("Can't write file\n");
	  ind_stop=3;
	}
      }
      if(file_recovery.file_stat!=NULL)
      {
	list_append_block(&file_recovery.location,offset,*blocksize,1);
	if(file_recovery.file_stat->file_hint->data_check!=NULL && file_recovery.file_stat->file_hint->data_check(double_buffer,2*(*blocksize),&file_recovery)==2)
	{
	  /* EOF found */
	  if(file_finish(&file_recovery,recup_dir,paranoid,file_nbr, *blocksize, list_search_space, &current_search_space, &offset, dir_num,status,disk_car->sector_size,disk_car))
	    file_recovered=1;
	  if(lowmem>0)
	    forget(list_search_space,current_search_space);
	}
	else
	{
	  file_recovery.file_size+=*blocksize;
	  file_recovery.file_size_on_disk+=*blocksize;
	}
      }
      memcpy(double_buffer,buffer,*blocksize);
    }
    if(file_recovery.file_stat!=NULL && file_recovery.file_stat->file_hint->max_filesize>0 && file_recovery.file_size>=file_recovery.file_stat->file_hint->max_filesize)
    {
      ecrit_rapport("File should not be bigger than %llu, stop adding data\n",
	  (long long unsigned)file_recovery.file_stat->file_hint->max_filesize);
      if(file_finish(&file_recovery,recup_dir,paranoid,file_nbr,*blocksize, list_search_space, &current_search_space, &offset, dir_num,status,disk_car->sector_size,disk_car))
	file_recovered=1;
      if(lowmem>0)
	forget(list_search_space,current_search_space);
    }
    if(ind_stop>0)
    {
      ecrit_rapport("PhotoRec has been stopped\n");
      current_search_space=list_search_space;
    }
    else if(*file_nbr>=10 && status==STATUS_FIND_OFFSET)
    {
      current_search_space=list_search_space;
    }
    else if(file_recovery.handle==NULL && file_recovered>1)
    {
    }
    else if(file_recovery.handle==NULL && pass==1)
    {
      if(file_recovered>0 && get_prev_file_header(list_search_space, &current_search_space, &offset)>=0)
      {
	memset(double_buffer,0,(*blocksize));
      }
      else if(old_offset==offset)
	get_next_sector(list_search_space, &current_search_space,&offset,*blocksize);
    }
    else if(file_recovery.handle==NULL && pass>1)
    {
      if(file_recovered>0 && get_prev_file_header(list_search_space, &current_search_space, &offset)>=0)
      {
	memset(double_buffer,0,(*blocksize));
      }
      else if(old_offset==offset)
      {
	get_next_header(list_search_space, &current_search_space, &offset);
	memset(double_buffer,0,(*blocksize));
      }
    }
    else if(old_offset==offset)
      get_next_sector(list_search_space, &current_search_space,&offset,*blocksize);
    if(current_search_space==list_search_space)
    {
      if(file_finish(&file_recovery,recup_dir,paranoid,file_nbr,*blocksize,list_search_space, &current_search_space, &offset, dir_num,status,disk_car->sector_size,disk_car))
	file_recovered=1;
      if(lowmem>0)
	forget(list_search_space,current_search_space);
      if(file_recovered>0 && get_prev_file_header(list_search_space, &current_search_space, &offset)>=0)
      {
	memset(double_buffer,0,(*blocksize));
      }
    }
  }
  session_save(list_search_space, disk_car, partition, file_stats, *blocksize, debug);
  if(status==STATUS_FIND_OFFSET)
  {
    uint64_t start_offset;
    *blocksize=find_blocksize(&list_file,disk_car->sector_size, &start_offset);
    if(expert>0)
      *blocksize=menu_choose_blocksize(*blocksize, disk_car->sector_size, &start_offset);
    update_blocksize(*blocksize,list_search_space, start_offset);
    free_list_search_space(&list_file);
  }
  if(status==STATUS_FIND_DIR)
  {
    uint64_t start_offset;
    *blocksize=find_blocksize_cluster(&list_cluster,disk_car->sector_size, &start_offset);
    if(expert>0)
      *blocksize=menu_choose_blocksize(*blocksize, disk_car->sector_size, &start_offset);
    /* TODO remove sectors that starts before start_offset */
    update_blocksize(*blocksize,list_search_space, start_offset);
    /* TODO:
       - remove info that doesn't match
       - merge and sort list of files
     list_cluster=list_cluster_fix(list_cluster,blocksize,start_offset);
     */
     list_cluster_free(&list_cluster);
  }
  free(double_buffer);
  photorec_info(stdscr, file_stats);
  return ind_stop;
}

static int photorec(t_param_disk *disk_car, t_partition *partition, const int debug, const int paranoid, const char *recup_dir, const int keep_corrupted_file, const int interface, t_file_stat *file_stats, unsigned int mode_ext2, char **current_cmd, t_alloc_data *list_search_space, unsigned int blocksize, const unsigned int expert, const unsigned int filenaming, const unsigned int lowmem, const unsigned int carve_free_space_only)
{
  time_t real_start_time;
  unsigned int file_nbr=0;
  unsigned int dir_num=1;
  int ind_stop=0;
  unsigned int pass;
  unsigned int blocksize_is_known=0;
  t_photorec_status status;
  aff_buffer(BUFFER_RESET,"Q");
  reset_file_stats(file_stats);
  ecrit_rapport("\nAnalyse ");
  aff_part_rapport(disk_car,partition);
  if(blocksize==0 || list_empty(&list_search_space->list))
  {
    blocksize=disk_car->sector_size;
    blocksize_is_known=0;
  }
  else
    blocksize_is_known=1;

  if(list_empty(&list_search_space->list))
  {
    t_alloc_data *tmp=init_search_space(partition,disk_car);
    list_add_tail(&tmp->list, &list_search_space->list);
    if(mode_ext2==0 && carve_free_space_only>0)
    {
      blocksize=remove_used_space(disk_car, partition, list_search_space);
      if(blocksize==0)
	blocksize=disk_car->sector_size;
      else
	blocksize_is_known=1;
    }
  }
  else
  { /* Correct the values */
    struct list_head *search_walker = NULL;
    list_for_each(search_walker, &list_search_space->list)
    {
      t_alloc_data *current_search_space;
      current_search_space=list_entry(search_walker, t_alloc_data, list);
      current_search_space->start=current_search_space->start*disk_car->sector_size;
      current_search_space->end=current_search_space->end*disk_car->sector_size+disk_car->sector_size-1;
    }
  }

  real_start_time=time(NULL);
  dir_num=photorec_mkdir(recup_dir,dir_num);
  status=STATUS_FIND_OFFSET;
  for(pass=0;status!=STATUS_QUIT;pass++)
  {
    unsigned int old_file_nbr=file_nbr;
    ecrit_rapport("Pass %u (blocksize=%u) ",pass,blocksize);
    switch(status)
    {
      case STATUS_FIND_DIR:			ecrit_rapport("STATUS_FIND_DIR\n");	break;
      case STATUS_FIND_OFFSET:			ecrit_rapport("STATUS_FIND_OFFSET\n");	break;
      case STATUS_EXT2_ON:			ecrit_rapport("STATUS_EXT2_ON\n");	break;
      case STATUS_EXT2_ON_BF:			ecrit_rapport("STATUS_EXT2_ON_BF\n");	break;
      case STATUS_EXT2_OFF:			ecrit_rapport("STATUS_EXT2_OFF\n");	break;
      case STATUS_EXT2_OFF_BF:			ecrit_rapport("STATUS_EXT2_OFF_BF\n");	break;
      case STATUS_EXT2_ON_SAVE_EVERYTHING:	ecrit_rapport("STATUS_EXT2_ON_SAVE_EVERYTHING\n");	break;
      case STATUS_EXT2_OFF_SAVE_EVERYTHING:	ecrit_rapport("STATUS_EXT2_OFF_SAVE_EVERYTHING\n");	break;
      case STATUS_QUIT :			ecrit_rapport("STATUS_QUIT\n");			break;
    }
    if(interface)
    {
      aff_copy(stdscr);
      wmove(stdscr,4,0);
      wdoprintf(stdscr,"%s",disk_car->description_short(disk_car));
      mvwaddstr(stdscr,5,0,msg_PART_HEADER_LONG);
      wmove(stdscr,6,0);
      aff_part(stdscr,AFF_PART_ORDER,disk_car,partition);
      wmove(stdscr,22,0);
      wattrset(stdscr,A_STANDOUT);
      waddstr(stdscr,"  Stop  ");
      wattroff(stdscr,A_STANDOUT);
      wrefresh(stdscr);
    }
    if(status==STATUS_FIND_OFFSET && blocksize_is_known>0)
      ind_stop=0;
    else if(status==STATUS_EXT2_ON_BF || status==STATUS_EXT2_OFF_BF)
    {
//      ind_stop=photorec_bf(disk_car, partition, debug, paranoid, recup_dir, interface, file_stats, &file_nbr, blocksize, list_search_space, real_start_time, &dir_num, status, pass, expert, filenaming);
      ind_stop=0;
    }
    else
      ind_stop=photorec_aux(disk_car, partition, debug, paranoid, recup_dir, interface, file_stats, &file_nbr, &blocksize, list_search_space, real_start_time, &dir_num, status, pass,expert, filenaming, lowmem);
    if(ind_stop>0)
      status=STATUS_QUIT;
    else if(paranoid>0)
    {
      switch(status)
      {
	case STATUS_FIND_DIR:
	case STATUS_FIND_OFFSET:
	  status=(mode_ext2>0?STATUS_EXT2_ON:STATUS_EXT2_OFF);
	  file_nbr=0;
	  break;
	case STATUS_EXT2_ON:
	    status=(paranoid>1?STATUS_EXT2_ON_BF:STATUS_EXT2_OFF);
	  break;
	case STATUS_EXT2_ON_BF:
	  status=STATUS_EXT2_OFF;
	  break;
	case STATUS_EXT2_OFF:
	  if(paranoid>1)
	  {
	    status=STATUS_EXT2_OFF_BF;
	  }
	  else
	  {
	    if(keep_corrupted_file>0)
	      status=(mode_ext2>0?STATUS_EXT2_ON_SAVE_EVERYTHING:STATUS_EXT2_OFF_SAVE_EVERYTHING);
	    else
	    {
	      status=STATUS_QUIT;
	      unlink("photorec.ses");
	    }
	  }
	  break;
	case STATUS_EXT2_OFF_BF:
	  if(keep_corrupted_file>0)
	    status=(mode_ext2>0?STATUS_EXT2_ON_SAVE_EVERYTHING:STATUS_EXT2_OFF_SAVE_EVERYTHING);
	  else
	  {
	    status=STATUS_QUIT;
	    unlink("photorec.ses");
	  }
	  break;
	case STATUS_EXT2_ON_SAVE_EVERYTHING:
	  status=STATUS_EXT2_OFF_SAVE_EVERYTHING;
	  break;
	default:
	  status=STATUS_QUIT;
	  unlink("photorec.ses");
	  break;
      }
    }
    else
    {
      switch(status)
      {
	case STATUS_FIND_OFFSET:
	  status=(mode_ext2>0?STATUS_EXT2_ON_SAVE_EVERYTHING:STATUS_EXT2_OFF_SAVE_EVERYTHING);
	  file_nbr=0;
	  break;
	default:
	  status=STATUS_QUIT;
	  unlink("photorec.ses");
	  break;
      }
    }
    {
      time_t current_time;
      current_time=time(NULL);
      ecrit_rapport("Elapsed time %uh%02um%02us\n",
	  (unsigned)((current_time-real_start_time)/60/60),
	  (unsigned)((current_time-real_start_time)/60%60),
	  (unsigned)((current_time-real_start_time)%60));
    }
    update_stats(file_stats,list_search_space);
    ecrit_rapport("Pass %u +%u file%s\n",pass,file_nbr-old_file_nbr,(file_nbr-old_file_nbr<=1?"":"s"));
    write_stats_log(file_stats);
    if(interface==0)
    {
      printf("Pass %u +%u file%s\n",pass,file_nbr-old_file_nbr,(file_nbr-old_file_nbr<=1?"":"s"));
      write_stats_stdout(file_stats);
      fflush(stdout);
    }
  }
  {
    struct list_head *search_walker = NULL;
    struct list_head *search_walker_next = NULL;
    unsigned long int nbr_headers=0;
    uint64_t sectors_with_unknown_data=0;
    /* Free memory */
    list_for_each_safe(search_walker,search_walker_next,&list_search_space->list)
    {
      t_alloc_data *current_search_space;
      current_search_space=list_entry(search_walker, t_alloc_data, list);
      if(current_search_space->file_stat!=NULL)
      {
	nbr_headers++;
	current_search_space->file_stat->not_recovered++;
      }
      sectors_with_unknown_data+=(current_search_space->end-current_search_space->start+disk_car->sector_size-1)/disk_car->sector_size;
      if(debug>0)
      {
	ecrit_rapport("%lu-%lu: %s\n",(long unsigned)(current_search_space->start/disk_car->sector_size),
	    (long unsigned)(current_search_space->end/disk_car->sector_size),
	    (current_search_space->file_stat!=NULL && current_search_space->file_stat->file_hint!=NULL?
	     (current_search_space->file_stat->file_hint->extension?
	      current_search_space->file_stat->file_hint->extension:""):
	     "(null)"));
      }
      list_del(search_walker);
      free(current_search_space);
    }
    ecrit_rapport("%llu sectors contains unknown data, %lu invalid files found %s.\n",
	(long long unsigned)sectors_with_unknown_data, (long unsigned)nbr_headers,
	(keep_corrupted_file>0?"but saved":"and rejected"));
  }
  if(interface)
  {
    wmove(stdscr,9,0);
    wclrtoeol(stdscr);
    wdoprintf(stdscr,"%u files saved in %s directory.\n",file_nbr,recup_dir);
    wmove(stdscr,10,0);
    wclrtoeol(stdscr);
    switch(ind_stop)
    {
      case 0:
	wdoprintf(stdscr,"Recovery completed.");
        break;
      case 1:
	wdoprintf(stdscr,"Recovery aborted by the user.");
        break;
      case 2:
	wdoprintf(stdscr,"Can't create file in current directory.");
        break;
      case 3:
	wdoprintf(stdscr,"Can't write file, no space left.");
        break;
    }
    if(*current_cmd==NULL)
    {
      int quit=0;
      wmove(stdscr,22,0);
      wclrtoeol(stdscr);
      wattrset(stdscr,A_STANDOUT);
      wdoprintf(stdscr,"[ Quit ]");
      wattroff(stdscr,A_STANDOUT);
      wrefresh(stdscr);
      do
      {
	switch(wgetch(stdscr))
	{
	  case KEY_ENTER:
#ifdef PADENTER
	  case PADENTER:
#endif
	  case '\n':
	  case '\r':
	  case 'q':
	  case 'Q':
	    quit=1;
	    break;
	}
      } while(quit==0);
    }
  }
  return 0;
}

static void update_stats(t_file_stat *file_stats, t_alloc_data *list_search_space)
{
  struct list_head *search_walker = NULL;
  int i;
  /* Reset */
  for(i=0;file_stats[i].file_hint!=NULL;i++)
    file_stats[i].not_recovered=0;
  /* Update */
  list_for_each(search_walker, &list_search_space->list)
  {
    t_alloc_data *current_search_space;
    current_search_space=list_entry(search_walker, t_alloc_data, list);
    if(current_search_space->file_stat!=NULL)
    {
      current_search_space->file_stat->not_recovered++;
    }
  }
}

static void write_stats_log(const t_file_stat *file_stats)
{
  unsigned int file_nbr=0;
  unsigned int i;
  unsigned int nbr;
  t_file_stat *new_file_stats;
  for(i=0;file_stats[i].file_hint!=NULL;i++);
  nbr=i;
  new_file_stats=(t_file_stat*)MALLOC(nbr*sizeof(t_file_stat));
  memcpy(new_file_stats, file_stats, nbr*sizeof(t_file_stat));
  qsort(new_file_stats, nbr, sizeof(t_file_stat), sort_file_stats);
  for(i=0;i<nbr;i++)
  {
    if(new_file_stats[i].recovered+new_file_stats[i].not_recovered>0)
    {
      file_nbr+=new_file_stats[i].recovered;
      ecrit_rapport("%s: %u/%u recovered\n",
	  (new_file_stats[i].file_hint->extension!=NULL?
	   new_file_stats[i].file_hint->extension:""),
	  new_file_stats[i].recovered, new_file_stats[i].recovered+new_file_stats[i].not_recovered);
    }
  }
  free(new_file_stats);
  if(file_nbr>1)
  {
    ecrit_rapport("Total: %u files found\n\n",file_nbr);
  }
  else
  {
    ecrit_rapport("Total: %u file found\n\n",file_nbr);
  }
}

static int sort_file_stats(const void *p1, const void *p2)
{
  const t_file_stat *f1=(const t_file_stat *)p1;
  const t_file_stat *f2=(const t_file_stat *)p2;
  /* bigest to lowest */
  if(f1->recovered < f2->recovered)
    return 1;
  if(f1->recovered > f2->recovered)
    return -1;
  return 0;
}

static void write_stats_stdout(const t_file_stat *file_stats)
{
  int i;
  unsigned int file_nbr=0;
  for(i=0;file_stats[i].file_hint!=NULL;i++)
  {
    if(file_stats[i].recovered+file_stats[i].not_recovered>0)
    {
      file_nbr+=file_stats[i].recovered;
      printf("%s: %u/%u recovered\n",
	  (file_stats[i].file_hint->extension!=NULL?
	   file_stats[i].file_hint->extension:""),
	  file_stats[i].recovered, file_stats[i].recovered+file_stats[i].not_recovered);
    }
  }
  if(file_nbr>1)
  {
    printf("Total: %u files found\n\n",file_nbr);
  }
  else
  {
    printf("Total: %u file found\n\n",file_nbr);
  }
}

static int menu_photorec(t_param_disk *disk_car, int debug, const char *recup_dir, t_file_stat *file_stats, char **current_cmd, t_alloc_data*list_search_space)
{
  int command;
  int done=0;
  int offset=0;
  int allow_partial_last_cylinder=0;
  int paranoid=1;
  unsigned int menu=0;
  int keep_corrupted_file=0;
  int current_element_num=0;
  unsigned int mode_ext2=0;
  unsigned int blocksize=0;
  unsigned int expert=0;
  unsigned int filenaming=1;
  unsigned int lowmem=0;
  unsigned int carve_free_space_only=0;
  t_list_part *list_part;
  t_list_part *element;
  t_list_part *current_element;
  static struct MenuItem menuMain[]=
  {
	{'S',"Search","Start file recovery"},
	{'O',"Options","Modify options"},
	{'F',"File Opt","Modify file options"},
	{'G',"Geometry", "Change disk geometry" },
	{'Q',"Quit","Return to disk selection"},
	{0,NULL,NULL}
  };
  list_part=disk_car->arch->read_part(disk_car,debug,0);
  list_part=insert_new_partition(list_part,new_whole_disk(disk_car));
  current_element=list_part;
  for(element=list_part;element!=NULL;element=element->next)
  {
    aff_part_rapport(disk_car,element->part);
  }

  /* ncurses interface */
  while(done==0)
  {
    unsigned int i;
    aff_copy(stdscr);
    wmove(stdscr,4,0);
    wdoprintf(stdscr,"%s",disk_car->description_short(disk_car));
    mvwaddstr(stdscr,6,0,msg_PART_HEADER_LONG);
    for(i=0,element=list_part;(element!=NULL) && (i<offset);element=element->next,i++);
    for(i=offset;(element!=NULL) && ((i-offset)<INTER_SELECT);i++,element=element->next)
    {
      wmove(stdscr,5+2+i-offset,0);
      wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
      if(element==current_element)
      {
	wattrset(stdscr,A_STANDOUT);
	aff_part(stdscr,AFF_PART_ORDER,disk_car,element->part);
	wattroff(stdscr,A_STANDOUT);
      } else
      {
	aff_part(stdscr,AFF_PART_ORDER,disk_car,element->part);
      }
    }

    if(*current_cmd!=NULL)
    {
      int keep_asking;
      command='q';
      do
      {
	keep_asking=0;
	while(*current_cmd[0]==',')
	  (*current_cmd)++;
	if(*current_cmd[0]=='\0')
	{
	}
	else if(strncmp(*current_cmd,"search",6)==0)
	{
	  (*current_cmd)+=6;
	  command='s';
	}
	else if(strncmp(*current_cmd,"options",7)==0)
	{
	  (*current_cmd)+=7;
	  command='O';
	}
	else if(strncmp(*current_cmd,"fileopt",7)==0)
	{
	  (*current_cmd)+=7;
	  command='F';
	}
	else if(strncmp(*current_cmd,"blocksize",9)==0)
	{
	  (*current_cmd)+=9;
	  while(*current_cmd[0]!=',' && *current_cmd[0]!='\0')
	    (*current_cmd)++;
	  blocksize=atoi(*current_cmd);
	  while(*current_cmd[0]!=',' && *current_cmd[0]!='\0')
	    (*current_cmd)++;
	}
	else if(isdigit(*current_cmd[0]))
	{
	  unsigned int order;
	  order= atoi(*current_cmd);
	  while(*current_cmd[0]!=',' && *current_cmd[0]!='\0')
	    (*current_cmd)++;
	  for(element=list_part;element!=NULL && element->part->order!=order;element=element->next);
	  if(element!=NULL)
	  {
	    current_element=element;
	    keep_asking=1;
	  }
	}
	else
	{
	  ecrit_rapport("error >%s<\n",*current_cmd);
	  command='Q';
	  while(*current_cmd[0]!='\0')
	    (*current_cmd)++;
	}
      } while(keep_asking>0);
    }
    else
      command = wmenuSelect(stdscr,INTER_SELECT_Y, INTER_SELECT_X, menuMain, 8,
	  (expert==0?"SOFQ":"SOFGQ"), MENU_HORIZ | MENU_BUTTON | MENU_ACCEPT_OTHERS, menu);
    switch(command)
    {
      case KEY_UP:
	if(current_element!=NULL)
	{
	  if(current_element->prev!=NULL)
	  {
	    current_element=current_element->prev;
	    current_element_num--;
	  }
	  if(current_element_num<offset)
	    offset--;
	}
	break;
      case KEY_DOWN:
	if(current_element!=NULL)
	{
	  if(current_element->next!=NULL)
	  {
	    current_element=current_element->next;
	    current_element_num++;
	  }
	  if(current_element_num>=offset+INTER_SELECT)
	    offset++;
	}
	break;
      case 's':
      case 'S':
	if(*current_cmd==NULL)
	{
	  ask_mode_ext2(disk_car, current_element->part, &mode_ext2, &carve_free_space_only);
	}
	{
	  char *res;
	  menu=0;
	  if(recup_dir!=NULL)
	    res=recup_dir;
	  else
	  {
	    res=ask_location("Do you want to save recovered files in %s%s ? [Y/N]","");
	    if(res!=NULL)
	    {
	      char *new_recup_dir=MALLOC(strlen(res)+1+strlen(DEFAULT_RECUP_DIR)+1);
	      strcpy(new_recup_dir,res);
	      strcat(new_recup_dir,"/");
	      strcat(new_recup_dir,DEFAULT_RECUP_DIR);
	      free(res);
	      res=new_recup_dir;
	    }
	  }
	  if(res!=NULL)
	    photorec(disk_car, current_element->part, debug, paranoid, res, keep_corrupted_file,1,file_stats,mode_ext2,current_cmd,list_search_space,blocksize,expert, filenaming, lowmem, carve_free_space_only);
	}
	break;
      case 'o':
      case 'O':
	{
	  int old_allow_partial_last_cylinder=allow_partial_last_cylinder;
	  interface_options_photorec(&paranoid, &allow_partial_last_cylinder,
	      &keep_corrupted_file, &mode_ext2, &expert, &filenaming, &lowmem, current_cmd);
	  if(old_allow_partial_last_cylinder!=allow_partial_last_cylinder)
	    hd_update_geometry(disk_car,allow_partial_last_cylinder,debug);
	  menu=1;
	}
	break;
      case 'f':
      case 'F':
	interface_file_select(file_stats,current_cmd);
	menu=2;
	break;
      case 'g':
      case 'G':
	if(expert!=0)
	  change_geometry(disk_car,current_cmd);
	break;
      case 'q':
      case 'Q':
	done = 1;
	break;
    }
  }
  ecrit_rapport("\n");
  delete_list_part(list_part);
  return 0;
}

static int do_curses_photorec(int debug, const char *recup_dir, const t_list_disk *list_disk, t_file_stat *file_stats, char *cmd_device, char *cmd_run)
{
  int command;
  int real_key;
  int allow_partial_last_cylinder=0;
  int done=0;
  unsigned int menu=0;
  int offset=0;
  int pos_num=0;
  const t_list_disk *element_disk;
  const t_list_disk *current_disk=NULL;
  static struct MenuItem menuMain[]=
  {
    { 'P', "Previous",""},
    { 'N', "Next","" },
    { 'O',"Proceed",""},
    { 'Q',"Quit","Quit program"},
    { 0,NULL,NULL}
  };
  char *current_cmd=cmd_run;
  t_alloc_data list_search_space={
    .list = LIST_HEAD_INIT(list_search_space.list)
  };
  if(cmd_device==NULL)
  {
    session_load(&cmd_device, &current_cmd,&list_search_space);
    if(cmd_device!=NULL && current_cmd!=NULL && !list_empty(&list_search_space.list) && ask_confirmation("Continue previous session ? (Y/N)")!=0)
    {
      /* yes */
    }
    else
    {
      if(cmd_device!=NULL)
	free(cmd_device);
      cmd_device=NULL;
      if(current_cmd!=NULL)
	free(current_cmd);
      current_cmd=NULL;
      free_list_search_space(&list_search_space);
    }
  }
  if(cmd_device!=NULL)
  {
    for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
    {
      if(strcmp(element_disk->disk->device,cmd_device)==0)
	current_disk=element_disk;
    }
  }
  else
      current_disk=list_disk;
  if(current_disk==NULL)
  {
    intrf_no_disk("PhotoRec");
    return 0;
  }
  /* ncurses interface */
  while(done==0)
  {
    const char *options;
    int i;
    aff_copy(stdscr);
    wmove(stdscr,4,0);
    wdoprintf(stdscr,"  PhotoRec is free software, and");
    wmove(stdscr,5,0);
    wdoprintf(stdscr,"comes with ABSOLUTELY NO WARRANTY.");
    wmove(stdscr,7,0);
    wdoprintf(stdscr,"Select a media (use Arrow keys, then press Enter):");
    for(i=0,element_disk=list_disk;(element_disk!=NULL) && (i<offset);element_disk=element_disk->next,i++);
    for(;element_disk!=NULL && (i-offset)<10;i++,element_disk=element_disk->next)
    {
      wmove(stdscr,8+i-offset,0);
      if(element_disk!=current_disk)
	wdoprintf(stdscr,"%s\n",element_disk->disk->description_short(element_disk->disk));
      else
      {
	wattrset(stdscr,A_STANDOUT);
	wdoprintf(stdscr,"%s\n",element_disk->disk->description_short(element_disk->disk));
	wattroff(stdscr,A_STANDOUT);
      }
    }
    if(i<=10 && element_disk==NULL)
      options="OQ";
    else
      options="PNOQ";
    {
      int line=20;
#if defined(__CYGWIN__) || defined(__MINGW32__)
#else
#ifndef DJGPP
#ifdef HAVE_GETEUID
      if(geteuid()!=0)
      {
	wmove(stdscr,line++,0);
	wdoprintf(stdscr,"Note: Some disks won't appear unless you're root user.");
      }
#endif
#endif
#endif
      wmove(stdscr,line++,0);
      if(line==22)
	wdoprintf(stdscr,"Disk capacity must be correctly detected for a successful recovery.");
      else
	wdoprintf(stdscr,"Note: Disk capacity must be correctly detected for a successful recovery.");
      wmove(stdscr,line++,0);
      wdoprintf(stdscr,"If a disk listed above has incorrect size, check HD jumper settings, BIOS");
      wmove(stdscr,line++,0);
      wdoprintf(stdscr,"detection, and install the latest OS patches and disk drivers."); 
    }
    if(current_cmd!=NULL)
    {
      while(current_cmd[0]==',')
	current_cmd++;
      if(current_cmd[0]=='\0')
	command='Q';
      else if(strncmp(current_cmd,"inter",4)==0)
      {
	cmd_device=NULL;
	current_cmd=NULL;
      }
      else
	command='O';
    }
    if(current_cmd==NULL)
      command = wmenuSelect_ext(stdscr,INTER_MAIN_Y, INTER_MAIN_X, menuMain, 8,
	    options, MENU_HORIZ | MENU_BUTTON | MENU_ACCEPT_OTHERS, &menu,&real_key);
    switch(command)
    {
      case KEY_UP:
      case 'P':
	if(current_disk->prev!=NULL)
	{
	  current_disk=current_disk->prev;
	  pos_num--;
	}
	if(pos_num<offset)
	  offset--;
	break;
      case KEY_DOWN:
      case 'N':
	if(current_disk->next!=NULL)
	{
	  current_disk=current_disk->next;
	  pos_num++;
	}
	if(pos_num>=offset+10)
	  offset++;
	break;
      case 'o':
      case 'O':
	if(interface_partition_type(current_disk->disk, allow_partial_last_cylinder, debug, &current_cmd)==0)
	  menu_photorec(current_disk->disk, debug, recup_dir, file_stats, &current_cmd, &list_search_space);
	break;
      case 'q':
      case 'Q':
	done=1;
	break;
    }
  }
  end_ncurses();
  ecrit_rapport("\n");
  return 0;
}

static void interface_options_photorec(int *paranoid, int *allow_partial_last_cylinder, int *keep_corrupted_file, unsigned int *mode_ext2, unsigned int *expert, unsigned int *filenaming, unsigned int *lowmem, char**current_cmd)
{
  if(*current_cmd!=NULL)
  {
    int keep_asking=1;
    do
    {
      while(*current_cmd[0]==',')
	(*current_cmd)++;
      if(strncmp(*current_cmd,"mode_ext2",9)==0)
      {
	(*current_cmd)+=9;
	*mode_ext2=1;
      }
      else if(strncmp(*current_cmd,"expert",6)==0)
      {
	(*current_cmd)+=6;
	*expert=1;
      }
      else if(strncmp(*current_cmd,"lowmem",6)==0)
      {
	(*current_cmd)+=6;
	*lowmem=1;
      }
      else
	keep_asking=0;
    } while(keep_asking>0);
  }
  else
  {
    int done = FALSE;
    unsigned int menu = 6;
    while (done==FALSE)
    {
      int car;
      int real_key;
      struct MenuItem menuOptions[]=
      {
	{ 'P', NULL, "Check JPG files" },
	{ 'A',NULL,"" },
	{ 'K',NULL,"Keep corrupted files"},
	{ 'S',NULL,"Try to skip indirect block"},
	{ 'E',NULL,"Provide additional controls"},
	{ 'L',NULL,"Low memory"},
	{ 'Q',"Quit","Return to main menu"},
	{ 0, NULL, NULL }
      };
      menuOptions[0].name=*paranoid?"Paranoid : Yes":"Paranoid : No";
      menuOptions[1].name=*allow_partial_last_cylinder?"Allow partial last cylinder : Yes":"Allow partial last cylinder : No";
      menuOptions[2].name=*keep_corrupted_file?"Keep corrupted files : Yes":"Keep corrupted files : No";
      menuOptions[3].name=*mode_ext2?"Mode ext2/ext3 : Yes":"Mode ext2/ext3 : No";
      menuOptions[4].name=*expert?"Mode expert : Yes":"Mode expert : No";
      menuOptions[5].name=*lowmem?"Low memory: Yes":"Low memory: No";
      /* Jpg
	 Mov
	 Mpg
	 Minolta MRW
	 Canon CRW
	 Signa/Foveon X3F
	 Fuji RAF
	 Rollei RDC
	 MP3

       */
      aff_copy(stdscr);
      car=wmenuSelect_ext(stdscr,INTER_OPTION_Y, INTER_OPTION_X, menuOptions, 0, "PAKELQ", MENU_VERT|MENU_VERT_ARROW2VALID, &menu,&real_key);
      switch(car)
      {
	case 'p':
	case 'P':
	  *paranoid=!*paranoid;
	  break;
	case 'a':
	case 'A':
	  *allow_partial_last_cylinder=!*allow_partial_last_cylinder;
	  break;
	case 'k':
	case 'K':
	  *keep_corrupted_file=!*keep_corrupted_file;
	  break;
	case 's':
	case 'S':
	  *mode_ext2=!*mode_ext2;
	  break;
	case 'e':
	case 'E':
	  *expert=!*expert;
	  break;
	case 'l':
	case 'L':
	  *lowmem=!*lowmem;
	  break;
	case key_ESC:
	case 'q':
	case 'Q':
	  done = TRUE;
	  break;
      }
    }
  }
  /* write new options to log file */
  ecrit_rapport("New options :\n Paranoid : %s\n", *paranoid?"Yes":"No");
  ecrit_rapport(" Allow partial last cylinder : %s\n Keep corrupted files : %s\n Mode ext2/ext3 : %s\n Mode expert : %s\n Low memory : %s\n",
      *allow_partial_last_cylinder?"Yes":"No",
      *keep_corrupted_file?"Yes":"No",
      *mode_ext2?"Yes":"No",
      *expert?"Yes":"No",
      *lowmem?"Yes":"No");
}

static t_partition *new_whole_disk(const t_param_disk *disk_car)
{
  t_partition *fake_partition;
  fake_partition=partition_new();
  fake_partition->part_offset=0;
  fake_partition->part_size=disk_car->disk_size;
  strncpy(fake_partition->name,"Whole disk",sizeof(fake_partition->name)-1);
  return fake_partition;
}

static void interface_file_select(t_file_stat *file_stats, char**current_cmd)
{
  ecrit_rapport("\nInterface File Select\n");
  if(*current_cmd!=NULL)
  {
    int keep_asking;
    do
    {
      unsigned int cmd_length=0;
      int i;
      keep_asking=0;
      while(*current_cmd[0]==',')
	(*current_cmd)++;
      while((*current_cmd)[cmd_length]!='\0' && (*current_cmd)[cmd_length]!=',')
	cmd_length++;
      for(i=0;file_stats[i].file_hint!=NULL;i++)
      {
	if(file_stats[i].file_hint->extension!=NULL &&
	    strncmp(file_stats[i].file_hint->extension,*current_cmd,cmd_length)==0)
	{
	  keep_asking=1;
	  while(*current_cmd[0]!='\0' && *current_cmd[0]!=',')
	    (*current_cmd)++;
	  while(*current_cmd[0]==',')
	    (*current_cmd)++;
	  if(strncmp(*current_cmd,"enable",6)==0)
	  {
	    (*current_cmd)+=6;
	    file_stats[i].enable=1;
	  }
	  else if(strncmp(*current_cmd,"disable",7)==0)
	  {
	    (*current_cmd)+=7;
	    file_stats[i].enable=0;
	  }
	  else
	  {
	    ecrit_rapport("Syntax error %s\n",*current_cmd);
	    return;
	  }
	}
      }
    } while(keep_asking>0);
    return ;
  }
  {
    int quit;
    int offset=0;
    int current_element_num=0;
    int rewrite=1;
    unsigned int menu=0;
    do
    {
      static struct MenuItem menuAdv[]=
      {
	{'q',"Quit","Return to main menu"},
	{0,NULL,NULL}
      };
      int i;
      int command;
      if(rewrite!=0)
      {
	aff_copy(stdscr);
	wmove(stdscr,5,0);
	wdoprintf(stdscr,"PhotoRec will try to locate the following files");
	rewrite=0;
      }
      wmove(stdscr,5+1,4);
      wclrtoeol(stdscr);
      if(offset>0)
	wdoprintf(stdscr,"Previous");
      for(i=offset;file_stats[i].file_hint!=NULL && ((i-offset)<INTER_SELECT);i++)
      {
	wmove(stdscr,5+2+i-offset,0);
	wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
	if(i==current_element_num)
	{
	  wattrset(stdscr,A_STANDOUT);
	  wdoprintf(stdscr,"[%c] %-4s %s", (file_stats[i].enable==0?' ':'X'),
	      (file_stats[i].file_hint->extension!=NULL?
	       file_stats[i].file_hint->extension:""),
	      file_stats[i].file_hint->description);
	  wattroff(stdscr,A_STANDOUT);
	} else
	{
	  wdoprintf(stdscr,"[%c] %-4s %s", (file_stats[i].enable==0?' ':'X'),
	      (file_stats[i].file_hint->extension!=NULL?
	       file_stats[i].file_hint->extension:""),
	      file_stats[i].file_hint->description);
	}
      }
      wmove(stdscr,5+2+INTER_SELECT,4);
      wclrtoeol(stdscr);	/* before addstr for BSD compatibility */
      if(file_stats[i].file_hint!=NULL)
	wdoprintf(stdscr,"Next");
      quit=0;
      command = wmenuSelect(stdscr,INTER_SELECT_Y, INTER_SELECT_X, menuAdv, 8,
	  "q", MENU_BUTTON | MENU_ACCEPT_OTHERS, menu);
      switch(command)
      {
	case KEY_UP:
	  if(current_element_num>0)
	  {
	    current_element_num--;
	    if(current_element_num<offset)
	      offset--;
	  }
	  break;
	case KEY_PPAGE:
	  for(i=0;(i<INTER_SELECT) && (current_element_num>0);i++)
	  {
	    current_element_num--;
	    if(current_element_num<offset)
	      offset--;
	  }
	  break;
	case KEY_DOWN:
	  if(file_stats[current_element_num+1].file_hint!=NULL)
	  {
	    current_element_num++;
	    if(current_element_num>=offset+INTER_SELECT)
	      offset++;
	  }
	  break;
	case KEY_NPAGE:
	  for(i=0;(i<INTER_SELECT) && (file_stats[current_element_num+1].file_hint!=NULL);i++)
	  {
	    current_element_num++;
	    if(current_element_num>=offset+INTER_SELECT)
	      offset++;
	  }
	  break;
	case KEY_RIGHT:
	case '+':
	case ' ':
	case KEY_LEFT:
	case '-':
	case 'x':
	case 'X':
	  file_stats[current_element_num].enable=1-file_stats[current_element_num].enable;
	  break;
	case 'q':
	case 'Q':
	  quit=1;
	  break;
      }
    } while(quit==0);
  }
}

void aff_copy(WINDOW *window)
{
  wclear(window);
  keypad(window, TRUE); /* Need it to get arrow key */
  wmove(window,0,0);
  wdoprintf(window, "PhotoRec %s, Data Recovery Utility, %s\n",VERSION,TESTDISKDATE);
  wmove(window,1,0);
  wdoprintf(window, "Christophe GRENIER <grenier@cgsecurity.org>");
  wmove(window,2,0);
  wdoprintf(window, "http://www.cgsecurity.org");
}

typedef struct info_cluster_offset t_cluster_offset;

struct info_cluster_offset
{
  unsigned int cluster_size;
  unsigned long int offset;
  unsigned int nbr;
};

static unsigned int find_blocksize_cluster(list_cluster_t *list_cluster, const unsigned int default_blocksize, uint64_t *offset)
{
  unsigned int cluster_size;
  unsigned int cluster_size_best=default_blocksize;
  unsigned int nbr_max=0;
  for(cluster_size=default_blocksize;cluster_size<=128*512;cluster_size*=2)
  {
    struct list_head *dir_walker = NULL;
    t_cluster_offset cluster_offset[1000];
    unsigned int nbr_sol=0;
    list_for_each(dir_walker,&list_cluster->list)
    {
      list_cluster_t *info;
      info=list_entry(dir_walker, list_cluster_t, list);
      if(info->cluster>=2 && (info->cluster-2)*cluster_size<info->offset)
      {
	unsigned int sol_cur;
	unsigned int found=0;
	unsigned int offset_tmp;
	offset_tmp=info->offset-(info->cluster-2)*cluster_size;
	for(sol_cur=0;sol_cur<nbr_sol && !found;sol_cur++)
	{
	  if(cluster_offset[sol_cur].offset==offset_tmp)
	  {
	    cluster_offset[sol_cur].nbr++;
	    found=1;
	  }
	}
	if(!found && nbr_sol<1000)
	{
	  cluster_offset[nbr_sol].offset=offset_tmp;
	  cluster_offset[nbr_sol].nbr=1;
	  nbr_sol++;
	}
      }
    }
    {
      unsigned int sol_cur;
      for(sol_cur=0;sol_cur<nbr_sol;sol_cur++)
	if(nbr_max<cluster_offset[sol_cur].nbr)
	{
	  nbr_max=cluster_offset[sol_cur].nbr;
	  cluster_size_best=cluster_size;
	  *offset=cluster_offset[sol_cur].offset;
	}
    }
  }
  return cluster_size_best;
}

static unsigned int find_blocksize(t_alloc_data *list_file, const unsigned int default_blocksize, uint64_t *offset)
{
  int blocksize_ok=0;
  unsigned int blocksize;
  *offset=0;
  if(list_empty(&list_file->list))
    return default_blocksize;
  for(blocksize=128*512;blocksize>=default_blocksize && blocksize_ok==0;blocksize=blocksize>>1)
  {
    struct list_head *search_walker = NULL;
    blocksize_ok=1;
    {
      t_alloc_data *tmp;
      tmp=list_entry(list_file->list.next, t_alloc_data, list);
      *offset=tmp->start%blocksize;
    }
    for(search_walker=list_file->list.next;
      search_walker!=&list_file->list && blocksize_ok!=0;
      search_walker=search_walker->next)
    {
      t_alloc_data *current_file;
      current_file=list_entry(search_walker, t_alloc_data, list);
      if(current_file->start%blocksize!=*offset)
	blocksize_ok=0;
    }
  }
  blocksize=blocksize<<1;
  return blocksize;
}

static t_alloc_data * update_blocksize(unsigned int blocksize, t_alloc_data *list_search_space, const uint64_t offset)
{
  struct list_head *search_walker = NULL;
  struct list_head *search_walker_next = NULL;
  list_for_each_safe(search_walker,search_walker_next,&list_search_space->list)
  {
    t_alloc_data *current_search_space;
    current_search_space=list_entry(search_walker, t_alloc_data, list);
    current_search_space->start=(current_search_space->start-offset%blocksize+blocksize-1)/blocksize*blocksize+offset%blocksize;
    if(current_search_space->start>current_search_space->end)
    {
      list_del(search_walker);
      if(list_search_space==current_search_space)
	list_search_space=list_entry(search_walker_next, t_alloc_data, list);
      free(current_search_space);
    }
  }
  return list_search_space;
}

int main( int argc, char **argv )
{
  int i;
  int help=0, create_log=0, debug=0;
  int run_setlocale=1;
  int testdisk_mode=TESTDISK_O_RDONLY|TESTDISK_O_READAHEAD_32K;
  const char *recup_dir=NULL;
  t_list_disk *list_disk=NULL;
  t_list_disk *element_disk;
  char *cmd_device=NULL;
  char *cmd_run=NULL;
#ifdef TARGET_SOLARIS
  const t_arch_fnct *arch=&arch_sun;
#elif defined __APPLE__
  const t_arch_fnct *arch=&arch_mac;
#else
  const t_arch_fnct *arch=&arch_i386;
#endif
#ifdef HAVE_SIGACTION
  struct sigaction action, old_action;
#endif
  t_file_stat file_stats[]=
  {
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_png  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_7z   },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_aif  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_all  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_asf  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_au   },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_bkf  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_blend },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_bmp  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_bz2  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_cam  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_crw  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_ctg  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_cwk  },
    { .enable=0, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_dbf  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_dir  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_djv  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_doc  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_dsc  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_dv   },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_eps  },
    { .enable=0, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_exe  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_ext2_sb },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_gif  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_gz   },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_imb  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_itunes  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_jpg  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_flac },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mdb  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mdf  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mid  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mov  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mp3  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mpg  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mrw  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mus  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_mysql },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_ogg  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_orf  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_qdf  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_pcx  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_pdf  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_pap  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_ps   },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_prc  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_psd  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_pst  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_qxd  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_raf  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_rar  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_raw  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_rdc  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_riff },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_rm   },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_rns  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_sit  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_stuffit  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_swf  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_tar  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_tiff },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_txt  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_x3f  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_xcf  },
    { .enable=1, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=&file_hint_zip  },
    { .enable=0, .ifnothingfound=0, .not_recovered=0, .recovered=0, .file_hint=NULL }
  };

#ifdef TESTING
  srand(1);
#endif
#ifdef HAVE_SIGACTION
  /* set up the signal handler for SIGHUP */
  action.sa_handler  = sighup_hdlr;
  action.sa_flags = 0;
  if(sigaction(SIGHUP, &action, &old_action)==-1)
  {
    printf("Error on SIGACTION call\n");
    return -1;
  }
#endif
  printf("PhotoRec %s, Data Recovery Utility, %s\nChristophe GRENIER <grenier@cgsecurity.org>\nhttp://www.cgsecurity.org\n",VERSION,TESTDISKDATE);
  for(i=1;i<argc;i++)
  {
    if((strcmp(argv[i],"/log")==0) ||(strcmp(argv[i],"-log")==0))
      create_log=1;
    else if((strcmp(argv[i],"/debug")==0) || (strcmp(argv[i],"-debug")==0))
    {
      debug++;
      create_log=1;
    }
    else if(((strcmp(argv[i],"/d")==0)||(strcmp(argv[i],"-d")==0)) &&(i+1<argc))
    {
      int len=strlen(argv[i+1]);
      if(argv[i+1][len-1]=='\\' || argv[i+1][len-1]=='/')
      {
	char *new_recup_dir=MALLOC(len+strlen(DEFAULT_RECUP_DIR)+1);
	strcpy(new_recup_dir,argv[i+1]);
	strcat(new_recup_dir,DEFAULT_RECUP_DIR);
	recup_dir=new_recup_dir;	/* small memory leak */
      }
      else
	recup_dir=argv[i+1];
      i++;
    }
    else if((strcmp(argv[i],"/direct")==0) || (strcmp(argv[i],"-direct")==0))
      testdisk_mode|=TESTDISK_O_DIRECT;
    else if((strcmp(argv[i],"/nosetlocale")==0) || (strcmp(argv[i],"-nosetlocale")==0))
      run_setlocale=0;
    else if((strcmp(argv[i],"/help")==0) || (strcmp(argv[i],"-help")==0) || (strcmp(argv[i],"--help")==0) ||
      (strcmp(argv[i],"/h")==0) || (strcmp(argv[i],"-h")==0))
      help=1;
    else if(strcmp(argv[i],"/cmd")==0)
    {
      if(i+2>=argc)
	help=1;
      else
      {
	t_param_disk *disk_car;
	cmd_device=argv[++i];
	cmd_run=argv[++i];
	/* There is no log currently */
	disk_car=file_test_availability(cmd_device,debug,arch,testdisk_mode);
	if(disk_car==NULL)
	{
	  printf("\nUnable to open file or device %s\n",cmd_device);
	  help=1;
	}
	else
	  list_disk=insert_new_disk(list_disk,disk_car);
      }
    }
    else
    {
      t_param_disk *disk_car=file_test_availability(argv[i],debug,arch,testdisk_mode);
      if(disk_car==NULL)
      {
	printf("\nUnable to open file or device %s\n",argv[i]);
	help=1;
      }
      else
	list_disk=insert_new_disk(list_disk,disk_car);
    }
  }
  if(help!=0)
  {
    printf("\nUsage: photorec [/log] [/debug] [/d recup_dir] [file or device]\n"\
	    "\n" \
	    "/log          : create a photorec.log file\n" \
	    "/debug        : add debug information\n" \
	    "\n" \
	    "PhotoRec searches various file formats (JPEG, Office...), it stores them\n" \
	    "in recup_dir directory.\n" \
	    "\n" \
	    "If you have problems with PhotoRec or bug reports, please contact me.\n");
    return 0;
  }
  if(create_log>0)
    f_rapport=log_open("photorec.log",(create_log==1?"a":"w"),"PhotoRec",argc,argv);
  if(f_rapport!=NULL)
  {
    ecrit_rapport("PhotoRec %s, Data Recovery Utility, %s\nChristophe GRENIER <grenier@cgsecurity.org>\nhttp://www.cgsecurity.org\n",VERSION,TESTDISKDATE);
    ecrit_rapport(TESTDISK_OS);
    ecrit_rapport(" (ewf lib: %s)\n", td_ewf_version());
    /* libjpeg ? */
    ecrit_rapport("\n");
  }
  printf("Please wait...\n");
#ifdef HAVE_SETLOCALE
  if(run_setlocale>0)
  {
    const char *locale;
    locale = setlocale (LC_ALL, "");
    if (locale==NULL) {
      locale = setlocale (LC_ALL, NULL);
      ecrit_rapport("Failed to set locale, using default '%s'.\n", locale);
    } else {
      ecrit_rapport("Using locale '%s'.\n", locale);
    }
  }
#endif
  aff_buffer(BUFFER_RESET,"Q");
  /* Scan for available device only if no device or image has been supplied in parameter */
  if(list_disk==NULL)
    list_disk=hd_parse(list_disk,debug,arch,testdisk_mode);
#ifdef DJGPP
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
  {
    printf("%s\n",element_disk->disk->description(element_disk->disk));
  }
#endif
  hd_update_all_geometry(list_disk,0,debug);
  /* Activate the cache, even if photorec has its own */
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
    element_disk->disk=new_diskcache(element_disk->disk,testdisk_mode);
  /* save disk parameters to rapport */
  ecrit_rapport("Hard disk list\n");
  for(element_disk=list_disk;element_disk!=NULL;element_disk=element_disk->next)
  {
    printf("%s, sector size=%u\n",element_disk->disk->description(element_disk->disk),element_disk->disk->sector_size);
    ecrit_rapport("%s, sector size=%u\n",element_disk->disk->description(element_disk->disk),element_disk->disk->sector_size);
  }
  printf("\n");
  ecrit_rapport("\n");
  if(start_ncurses("PhotoRec", argv[0])==0)
  {
    do_curses_photorec(debug,recup_dir,list_disk,file_stats,cmd_device,cmd_run);
    end_ncurses();
  }
  delete_list_disk(list_disk);
  if(f_rapport!=NULL)
  {
    ecrit_rapport("PhotoRec exited normally.\n");
    if(fclose(f_rapport))
    {
      f_status=1;
    }
  }
  if(f_status!=0)
  {
    printf("PhotoRec: Log file corrupted!\n");
  }
  else
  {
    printf("PhotoRec exited normally.\n");
  }
  return 0;
}

void file_search_footer(t_file_recovery *file_recovery, const unsigned char*footer, const unsigned int footer_length)
{
  const unsigned int read_size=4096;
  unsigned char*buffer;
  int64_t file_size;
  if(footer_length==0)
    return ;
  buffer=(unsigned char*)MALLOC(read_size+footer_length-1);
  file_size=file_recovery->file_size;
  memset(buffer+read_size,0,footer_length-1);
  do
  {
    int i;
    int taille;
    if(file_size%read_size!=0)
      file_size=file_size-(file_size%read_size);
    else
      file_size-=read_size;
    if(fseek(file_recovery->handle,file_size,SEEK_SET)<0)
      return;
    taille=fread(buffer,1,read_size,file_recovery->handle);
    for(i=taille-1;i>=0;i--)
    {
      if(buffer[i]==footer[0] && memcmp(buffer+i,footer,footer_length)==0)
      {
	file_recovery->file_size=file_size+i+footer_length;
	free(buffer);
	return;
      }
    }
    memcpy(buffer+read_size,buffer,footer_length-1);
  } while(file_size>0);
  file_recovery->file_size=0;
  free(buffer);
}

void file_search_lc_footer(t_file_recovery *file_recovery, const unsigned char*footer, const unsigned int footer_length)
{
  const unsigned int read_size=4096;
  unsigned char*buffer;
  int64_t file_size;
  if(footer_length==0)
    return ;
  buffer=(unsigned char*)MALLOC(read_size+footer_length-1);
  file_size=file_recovery->file_size;
  memset(buffer+read_size,0,footer_length-1);
  do
  {
    int i;
    int taille;
    if(file_size%read_size!=0)
      file_size=file_size-(file_size%read_size);
    else
      file_size-=read_size;
    if(fseek(file_recovery->handle,file_size,SEEK_SET)<0)
      return;
    taille=fread(buffer,1,read_size,file_recovery->handle);
    for(i=0;i<taille;i++)
      buffer[i]=tolower(buffer[i]);
    for(i=taille-1;i>=0;i--)
    {
      if(buffer[i]==footer[0] && memcmp(buffer+i,footer,footer_length)==0)
      {
	file_recovery->file_size=file_size+i+footer_length;
	free(buffer);
	return;
      }
    }
    memcpy(buffer+read_size,buffer,footer_length-1);
  } while(file_size>0);
  file_recovery->file_size=0;
  free(buffer);
}

static int ask_mode_ext2(const t_param_disk *disk_car, const t_partition *partition, unsigned int *mode_ext2, unsigned int *carve_free_space_only)
{
  static struct MenuItem menuMode[]=
    {
      {'E',"EXT2/EXT3","EXT2/EXT3 filesystem"},
      {'O',"Other","FAT/NTFS/HFS+/ReiserFS/..."},
      {0,NULL,NULL}
    };
  static struct MenuItem menuFAT16[]=
  {
    {'F',"Free", "Scan for files from FAT16 unallocated space only"},
    {'W',"Whole","Extract files from whole partition"},
    {0,NULL,NULL}
  };
  static struct MenuItem menuFAT32[]=
  {
    {'F',"Free", "Scan for file from FAT32 unallocated space only"},
    {'W',"Whole","Extract files from whole partition"},
    {0,NULL,NULL}
  };
  const char *options="EO";
  WINDOW *window;
  unsigned int menu;
  int command;
  if(partition->upart_type==UP_EXT2 ||
      partition->upart_type==UP_EXT3)
    menu=0;
  else
    menu=1;
  window=newwin(0,0,0,0);	/* full screen */
  aff_copy(window);
  wmove(window,4,0);
  aff_part(window, AFF_PART_ORDER,disk_car,partition);
  wmove(window,6,0);
  waddstr(window,"To recover lost files, PhotoRec need to know the filesystem type where the");
  wmove(window,7,0);
  waddstr(window,"file were stored:");
  command = wmenuSelect_ext(window,8, 0, menuMode, 11,
      options, MENU_VERT | MENU_VERT_WARN | MENU_BUTTON, &menu,NULL);
  *mode_ext2=(command=='E' || command=='e');
  if(*mode_ext2>0)
  {
    ecrit_rapport("Mode EXT/EXT3 activated.\n");
  }
  *carve_free_space_only=0;
  if((*mode_ext2)!=0)
    return 0;
  {
    menu=0;
    options="FW";
    wmove(window,6,0);
    wclrtoeol(window);
    wmove(window,7,0);
    wclrtoeol(window);
    waddstr(window,"Please choose if all space need to be analysed:");
    if(partition->upart_type==UP_FAT16)
      command = wmenuSelect_ext(window,8, 0, menuFAT16, 11,
	  options, MENU_VERT | MENU_VERT_WARN | MENU_BUTTON, &menu,NULL);
    else if(partition->upart_type==UP_FAT32)
      command = wmenuSelect_ext(window,8, 0, menuFAT32, 11,
	  options, MENU_VERT | MENU_VERT_WARN | MENU_BUTTON, &menu,NULL);
    else
      command='W';
    if(command=='F' || command=='f')
      *carve_free_space_only=1;
    if(*carve_free_space_only>0)
    {
      ecrit_rapport("Carve free space only.\n");
    }
  }
  delwin(window);
  return 0;
}
