/*

    File: savehdr.c

    Copyright (C) 2004-2005 Christophe GRENIER <grenier@cgsecurity.org>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
 
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include <errno.h>
#include "types.h"
#include "common.h"
#include "fnctdsk.h" /* get_LBA_part */
#include "analyse.h"
#include "savehdr.h"
#define BACKUP_MAXSIZE 5120

static t_backup_disk *insert_backup(t_backup_disk *list,t_backup_disk *element2insert);

int save_header(t_param_disk *disk_car,t_partition *partition, const int debug)
{
  unsigned char *buffer;
  FILE *f_backup;
  if(debug>1)
  {
    ecrit_rapport("save_header\n");
  }
  f_backup=fopen("header.log","ab");
  if(!f_backup)
  {
    ecrit_rapport("Can't create header.log file: %s\n",strerror(errno));
    return -1;
  }
  buffer=MALLOC(256*DEFAULT_SECTOR_SIZE);
  memset(buffer,0,DEFAULT_SECTOR_SIZE);
  {
    char status='D';
    switch(partition->status)
    {
      case STATUS_PRIM:           status='P'; break;
      case STATUS_PRIM_BOOT:      status='*'; break;
      case STATUS_EXT:            status='E'; break;
      case STATUS_EXT_IN_EXT:     status='X'; break;
      case STATUS_LOG:            status='L'; break;
      case STATUS_DELETED:        status='D'; break;
    }
    snprintf((char*)buffer,256*DEFAULT_SECTOR_SIZE,"%s\n%2u %c Sys=%02X %5u %3u %2u %5u %3u %2u %10lu\n",
	disk_car->description(disk_car), partition->order,status,disk_car->arch->get_part_type(partition),
	offset2cylinder(disk_car,partition->part_offset), offset2head(disk_car,partition->part_offset),offset2sector(disk_car,partition->part_offset),
	offset2cylinder(disk_car,partition->part_offset+partition->part_size-disk_car->sector_size), offset2head(disk_car,partition->part_offset+partition->part_size-disk_car->sector_size),offset2sector(disk_car,partition->part_offset+partition->part_size-disk_car->sector_size),
	(unsigned long)(partition->part_size/disk_car->sector_size));
  }
  fwrite(buffer,DEFAULT_SECTOR_SIZE,1,f_backup);
  if(disk_car->read(disk_car,256*DEFAULT_SECTOR_SIZE, buffer, partition->part_offset)!=0)
  {
    fclose(f_backup);
    FREE(buffer);
    return -1;
  }
  fwrite(buffer,DEFAULT_SECTOR_SIZE,256,f_backup);
  fclose(f_backup);
  FREE(buffer);
  return 0;
}

static t_backup_disk *insert_backup(t_backup_disk *list,t_backup_disk *element2insert)
{
  if(element2insert==NULL)
    return list;
  {
    t_backup_disk *prev;
    for(prev=list;prev!=NULL && prev->next!=NULL;prev=prev->next);
    if(prev!=NULL)
    {
      prev->next=element2insert;
    }
    element2insert->prev=prev;
    element2insert->next=NULL;
  }
  return (list!=NULL?list:element2insert);
}

t_backup_disk *partition_load(const t_param_disk *disk_car, const int debug)
{
  FILE *f_backup;
  char *buffer;
  char *pos=NULL;
  int taille;
  t_backup_disk *list_backup=NULL;
  t_backup_disk *new_backup=NULL;
  if(debug>1)
  {
    ecrit_rapport("partition_load\n");
  }
  f_backup=fopen("backup.log","r");
  if(!f_backup)
  {
    ecrit_rapport("Can't open backup.log file: %s\n",strerror(errno));
    return list_backup;
  }
  buffer=MALLOC(BACKUP_MAXSIZE);
  taille=fread(buffer,1,BACKUP_MAXSIZE,f_backup);
  buffer[(taille<BACKUP_MAXSIZE?taille:BACKUP_MAXSIZE-1)]='\0';
  if(debug>1)
  {
    ecrit_rapport("partition_load backup.log size=%d\n",taille);
  }
  for(pos=buffer;pos<buffer+taille;pos++)
  {
    if(*pos=='\n')
    {
      *pos='\0';
    }
  }
  pos=buffer;
  while(pos!=NULL && pos<buffer+taille)
  {
    if(*pos=='#')
    {
      pos++;
      if(debug>1)
      {
	ecrit_rapport("new disk: %s\n",pos);
      }
      list_backup=insert_backup(list_backup,new_backup);
      new_backup=(t_backup_disk*)MALLOC(sizeof(*new_backup));
      new_backup->description[0]='\0';
      new_backup->list_part=NULL;
      new_backup->my_time=strtol(pos,&pos,10);
      if(pos!=NULL)
      {
	strncpy(new_backup->description,++pos,sizeof(new_backup->description));
      }
    }
    else if(new_backup!=NULL)
    {
      t_partition *new_partition=partition_new();
      char status;
      unsigned int part_type;
      unsigned long part_size;
      unsigned long part_offset;
      if(debug>1)
      {
	ecrit_rapport("new partition\n");
      }
      if(sscanf(pos,"%2u : start=%10lu, size=%10lu, Id=%02X, %c\n",
	  &new_partition->order, &part_offset,
	  &part_size,&part_type,&status)==5)
      {
	new_partition->part_offset=(uint64_t)part_offset*disk_car->sector_size;
	new_partition->part_size=(uint64_t)part_size*disk_car->sector_size;
	disk_car->arch->set_part_type(new_partition,part_type);
	switch(status)
	{
	  case 'P':	new_partition->status=STATUS_PRIM; break;
	  case '*':	new_partition->status=STATUS_PRIM_BOOT; break;
	  case 'L':	new_partition->status=STATUS_LOG; break;
	  default:	new_partition->status=STATUS_DELETED; break;
	}
	if(new_partition->status!=STATUS_DELETED)
	{
	  new_backup->list_part=insert_new_partition(new_backup->list_part,new_partition);
	}
	else
	{
	  FREE(new_partition);
	}
      }
      else
      {
	ecrit_rapport("partition_load: sscanf failed\n");
	FREE(new_partition);
	pos=NULL;
      }
    }
    if(pos!=NULL)
    {
      while(*pos!='\0' && pos<buffer+taille)
	pos++;
      pos++;
    }
  }
  list_backup=insert_backup(list_backup,new_backup);
  fclose(f_backup);
  FREE(buffer);
  return list_backup;
}

int partition_save(t_param_disk *disk_car, t_list_part *list_part, const int debug)
{
  t_list_part *parts;
  FILE *f_backup;
  if(debug>0)
  {
    ecrit_rapport("partition_save\n");
  }
  f_backup=fopen("backup.log","a");
  if(!f_backup)
  {
    ecrit_rapport("Can't create backup.log file: %s\n",strerror(errno));
    return -1;
  }
  fprintf(f_backup,"#%u %s\n",(unsigned int)time(NULL), disk_car->description(disk_car));
  for(parts=list_part;parts!=NULL;parts=parts->next)
  {
    switch(parts->part->status)
    {
      case STATUS_PRIM:
      case STATUS_PRIM_BOOT:
      case STATUS_EXT:
      case STATUS_LOG:
	{
	  char status='D';
	  switch(parts->part->status)
	  {
	    case STATUS_PRIM:           status='P'; break;
	    case STATUS_PRIM_BOOT:      status='*'; break;
	    case STATUS_EXT:            status='E'; break;
	    case STATUS_EXT_IN_EXT:     status='X'; break;
	    case STATUS_LOG:            status='L'; break;
	    case STATUS_DELETED:        status='D'; break;
	  }
	  fprintf(f_backup,"%2d : start=%9lu, size=%9lu, Id=%02X, %c\n",
	      parts->part->order, (unsigned long)(parts->part->part_offset/disk_car->sector_size),
	      (unsigned long)(parts->part->part_size/disk_car->sector_size),
	      disk_car->arch->get_part_type(parts->part), status);
	}
	break;
      default:
	break;
    }
  }
  fclose(f_backup);
  return 0;
}

