/*
 * gibber-r-multicast-packet.c - Source for GibberRMulticastPacket
 * Copyright (C) 2007 Collabora Ltd.
 * @author Sjoerd Simons <sjoerd.simons@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <arpa/inet.h>

#include "gibber-r-multicast-packet.h"

static void gibber_r_multicast_packet_sender_info_free(
    GibberRMulticastPacketSenderInfo *sender_info);

G_DEFINE_TYPE(GibberRMulticastPacket, gibber_r_multicast_packet, G_TYPE_OBJECT)

/* private structure */
typedef struct _GibberRMulticastPacketPrivate GibberRMulticastPacketPrivate;

struct _GibberRMulticastPacketPrivate
{
  gboolean dispose_has_run;
  /* Actually needed data size untill this point */
  gsize size;

  guint8 *data;
  /* Maximum data size */
  gsize max_data;
};

#define GIBBER_R_MULTICAST_PACKET_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), GIBBER_TYPE_R_MULTICAST_PACKET, \
      GibberRMulticastPacketPrivate))

static void
gibber_r_multicast_packet_init (GibberRMulticastPacket *obj)
{
  /* allocate any data required by the object here */
}

static void gibber_r_multicast_packet_dispose (GObject *object);
static void gibber_r_multicast_packet_finalize (GObject *object);

static void
gibber_r_multicast_packet_class_init (GibberRMulticastPacketClass *gibber_r_multicast_packet_class)
{
  GObjectClass *object_class = G_OBJECT_CLASS (gibber_r_multicast_packet_class);

  g_type_class_add_private (gibber_r_multicast_packet_class, sizeof (GibberRMulticastPacketPrivate));

  object_class->dispose = gibber_r_multicast_packet_dispose;
  object_class->finalize = gibber_r_multicast_packet_finalize;

}

void
gibber_r_multicast_packet_dispose (GObject *object)
{
  GibberRMulticastPacket *self = GIBBER_R_MULTICAST_PACKET (object);
  GibberRMulticastPacketPrivate *priv = GIBBER_R_MULTICAST_PACKET_GET_PRIVATE (self);

  if (priv->dispose_has_run)
    return;

  priv->dispose_has_run = TRUE;

  /* release any references held by the object here */

  if (G_OBJECT_CLASS (gibber_r_multicast_packet_parent_class)->dispose)
    G_OBJECT_CLASS (gibber_r_multicast_packet_parent_class)->dispose (object);
}

void
gibber_r_multicast_packet_finalize (GObject *object)
{
  GibberRMulticastPacket *self = GIBBER_R_MULTICAST_PACKET (object);
  GibberRMulticastPacketPrivate *priv =
      GIBBER_R_MULTICAST_PACKET_GET_PRIVATE (self);

  /* free any data held directly by the object here */
  switch (self->type) {
    case PACKET_TYPE_WHOIS_REPLY:
      g_free(self->data.whois_reply.sender_name);
      break;
    case PACKET_TYPE_DATA:
      g_list_foreach(self->data.data.depends,
          (GFunc)gibber_r_multicast_packet_sender_info_free, NULL);
      g_list_free(self->data.data.depends);
      g_free(self->data.data.payload);
      break;
    case PACKET_TYPE_SESSION:
      g_list_foreach(self->data.session.senders,
          (GFunc)gibber_r_multicast_packet_sender_info_free, NULL);
      g_list_free(self->data.session.senders);
      break;
    default:
      /* Nothing specific to free */;
  }
  g_free(priv->data);

  G_OBJECT_CLASS (gibber_r_multicast_packet_parent_class)->finalize (object);
}

static GibberRMulticastPacketSenderInfo *
gibber_r_multicast_packet_sender_info_new(guint32 sender_id,
   guint32 expected_packet)
{
  GibberRMulticastPacketSenderInfo *result
      = g_slice_new(GibberRMulticastPacketSenderInfo);
  result->sender_id = sender_id;
  result->packet_id = expected_packet;

  return result;
}

static void
gibber_r_multicast_packet_sender_info_free(
    GibberRMulticastPacketSenderInfo *sender_info) {
  g_slice_free(GibberRMulticastPacketSenderInfo, sender_info);
}

/* Start a new packet */
GibberRMulticastPacket *
gibber_r_multicast_packet_new(GibberRMulticastPacketType type,
                              guint32 sender,
                              gsize max_size) {
  GibberRMulticastPacket *result = g_object_new(GIBBER_TYPE_R_MULTICAST_PACKET,
                                                NULL);
  GibberRMulticastPacketPrivate *priv =
      GIBBER_R_MULTICAST_PACKET_GET_PRIVATE(result);

  /* Fixme do this using properties */
  result->type = type;
  result->sender = sender;

  priv->max_data = max_size;

  return result;
}

gboolean
gibber_r_multicast_packet_add_sender_info(GibberRMulticastPacket *packet,
                                          guint32 sender_id,
                                          guint32 packet_id,
                                          GError **error) {
  GibberRMulticastPacketSenderInfo *s =
      gibber_r_multicast_packet_sender_info_new(sender_id, packet_id);
  GibberRMulticastPacketPrivate *priv =
      GIBBER_R_MULTICAST_PACKET_GET_PRIVATE (packet);

  g_assert(priv->data == NULL);
  g_assert(packet->type == PACKET_TYPE_DATA
      || packet->type == PACKET_TYPE_SESSION);

  if (packet->type == PACKET_TYPE_DATA) {
    packet->data.data.depends = g_list_append(packet->data.data.depends, s);
  } else {
    packet->data.session.senders =
        g_list_append(packet->data.session.senders, s);
  }

  return TRUE;
}

void
gibber_r_multicast_packet_set_data_info(GibberRMulticastPacket *packet,
    guint32 packet_id, guint8 stream_id, guint8 part, guint8 total)
{
  g_assert(part < total);
  g_assert(packet->type == PACKET_TYPE_DATA);

  packet->data.data.packet_part = part;
  packet->data.data.packet_total = total;
  packet->data.data.packet_id = packet_id;
  packet->data.data.stream_id = stream_id;
}

void
gibber_r_multicast_packet_set_repair_request_info (
     GibberRMulticastPacket *packet, guint32 sender_id, guint32 packet_id)
{
  g_assert (packet->type == PACKET_TYPE_REPAIR_REQUEST);

  packet->data.repair_request.packet_id = packet_id;
  packet->data.repair_request.sender_id = sender_id;
}

void
gibber_r_multicast_packet_set_whois_request_info (
    GibberRMulticastPacket *packet,
    guint32 sender_id)
{
  g_assert (packet->type == PACKET_TYPE_WHOIS_REQUEST);

  packet->data.whois_request.sender_id = sender_id;
}

void
gibber_r_multicast_packet_set_whois_reply_info(GibberRMulticastPacket *packet,
   const gchar *name)
{
  g_assert (packet->type == PACKET_TYPE_WHOIS_REPLY);

  packet->data.whois_reply.sender_name = g_strdup (name);
}


static gsize
gibber_r_multicast_packet_calculate_size(GibberRMulticastPacket *packet)
{

  gsize result = 6; /* 8 bit type, 8 bit version, 32 bit sender */

  switch (packet->type) {
    case PACKET_TYPE_WHOIS_REQUEST:
      /* 32 bit sender id */
      result += 4;
      break;
    case PACKET_TYPE_WHOIS_REPLY:
      g_assert(packet->data.whois_reply.sender_name != NULL);
      result += 1 + strlen(packet->data.whois_reply.sender_name);
      break;
    case PACKET_TYPE_DATA:
      /* 8 bit part, 8 bit total,
         32 bit packet id, 8 bit stream id,
         8 bit nr sender info */
      result += 8;
      /* 32 bit sender id, 32 bit packet id */
      result += 8 * g_list_length(packet->data.data.depends);
      result += packet->data.data.payload_size;
      break;
    case PACKET_TYPE_REPAIR_REQUEST:
      /* 32 bit packet id and 32 sender id*/
      result += 8;
      break;
    case PACKET_TYPE_SESSION:
         /* 8 bit nr sender info + N times 32 bit sender id, 32 bit packet id
          */
      result += 1 + 8 * g_list_length(packet->data.session.senders);
      break;
    case PACKET_TYPE_BYE:
    case PACKET_TYPE_INVALID:
      /* Nothing to add */;
  }

  return result;
}

static void
add_guint8(guint8 *data, gsize length, gsize *offset, guint8 i) {
  g_assert(*offset + 1 <= length);
  *(data + *offset) = i;
  (*offset)++;
}

static guint8
get_guint8(const guint8 *data, gsize length, gsize *offset) {
  guint8 i;
  g_assert(*offset + 1 <= length);
  i = *(data + *offset);
  (*offset)++;
  return i;
}

static void
add_guint32(guint8 *data, gsize length, gsize *offset, guint32 i) {
  guint32 ni = htonl(i);

  g_assert(*offset + 4 <= length);

  memcpy(data + *offset, &ni, 4);
  (*offset) += 4;
}

static guint32
get_guint32(const guint8 *data, gsize length, gsize *offset) {
  guint32 ni;

  g_assert(*offset + 4 <= length);

  memcpy(&ni, data + *offset, 4);
  (*offset) += 4;
  return ntohl(ni);
}

static void
add_string(guint8 *data, gsize length, gsize *offset, const gchar *str) {
  gsize len = strlen(str);

  g_assert(len < G_MAXUINT8);
  add_guint8(data, length, offset, len);

  g_assert(*offset + len <= length);
  memcpy(data + *offset, str, len);
  (*offset) += len;
}

static gchar *
get_string(const guint8 *data, gsize length, gsize *offset) {
  gsize len;
  gchar *str;

  len = get_guint8(data, length, offset);

  g_assert(*offset + len <= length);

  str = g_strndup((gchar *)data + *offset, len);
  (*offset) += len;
  return str;
}

static void
add_sender_info(guint8 *data, gsize length, gsize *offset, GList *senders)
{
  guint nr_items;
  GList *l;

  nr_items = g_list_length(senders);
  add_guint8(data, length, offset, nr_items);

  for (l = senders; l != NULL; l = g_list_next(l))
    {
      GibberRMulticastPacketSenderInfo *info =
          (GibberRMulticastPacketSenderInfo *)l->data;
      add_guint32(data, length, offset, info->sender_id);
      add_guint32(data, length, offset, info->packet_id);
    }
}

static GList *
get_sender_info(guint8 *data, gsize length, gsize *offset) {
  GList *l = NULL;
  guint8 nr_items;

  for (nr_items = get_guint8(data, length, offset); nr_items > 0; nr_items--) {
    GibberRMulticastPacketSenderInfo *sender_info;
    guint32 sender_id;
    guint32 packet_id;

    sender_id = get_guint32(data, length, offset);
    packet_id = get_guint32(data, length, offset);
    sender_info =
        gibber_r_multicast_packet_sender_info_new(sender_id, packet_id);
    l = g_list_prepend(l, sender_info);
  }

  return l;
}

static void
gibber_r_multicast_packet_build(GibberRMulticastPacket *packet) {
  GibberRMulticastPacketPrivate *priv =
     GIBBER_R_MULTICAST_PACKET_GET_PRIVATE (packet);
  gsize needed_size;

  if (priv->data != NULL) {
    /* Already serialized return cached version */
    return;
  }

  needed_size = gibber_r_multicast_packet_calculate_size(packet);

  g_assert(needed_size <= priv->max_data);

  /* Trim down the maximum data size to what we actually need */
  priv->max_data = needed_size;
  priv->data = g_malloc0(priv->max_data);
  priv->size = 0;

  add_guint8 (priv->data, priv->max_data, &(priv->size), packet->type);
  add_guint8 (priv->data, priv->max_data, &(priv->size), packet->version);
  add_guint32 (priv->data, priv->max_data, &(priv->size), packet->sender);

  switch (packet->type) {
    case PACKET_TYPE_WHOIS_REQUEST:
      add_guint32 (priv->data, priv->max_data, &(priv->size),
          packet->data.whois_request.sender_id);
      break;
    case PACKET_TYPE_WHOIS_REPLY:
      add_string(priv->data, priv->max_data, &(priv->size),
          packet->data.whois_reply.sender_name);
      break;
    case PACKET_TYPE_DATA:
      add_guint32 (priv->data, priv->max_data, &(priv->size),
            packet->data.data.packet_id);
      add_guint8 (priv->data, priv->max_data, &(priv->size),
          packet->data.data.packet_part);
      add_guint8 (priv->data, priv->max_data, &(priv->size),
          packet->data.data.packet_total);
      add_guint8 (priv->data, priv->max_data, &(priv->size),
          packet->data.data.stream_id);
      add_sender_info (priv->data, priv->max_data, &(priv->size),
          packet->data.data.depends);

      g_assert(priv->size + packet->data.data.payload_size == priv->max_data);

      memcpy(priv->data + priv->size, packet->data.data.payload,
          packet->data.data.payload_size);
      priv->size += packet->data.data.payload_size;
      break;
    case PACKET_TYPE_REPAIR_REQUEST:
      add_guint32 (priv->data, priv->max_data, &(priv->size),
            packet->data.repair_request.sender_id);
      add_guint32 (priv->data, priv->max_data, &(priv->size),
            packet->data.repair_request.packet_id);
      break;
    case PACKET_TYPE_SESSION:
      add_sender_info (priv->data, priv->max_data, &(priv->size),
          packet->data.session.senders);
      break;
    case PACKET_TYPE_BYE:
      /* Not implemented, fall through */
    default:
      g_assert_not_reached();
  }

  /* If this fails our size precalculation is buggy */
  g_assert(priv->size == priv->max_data);
}

gsize
gibber_r_multicast_packet_add_payload(GibberRMulticastPacket *packet,
                                      const guint8 *data, gsize size) {
  GibberRMulticastPacketPrivate *priv =
     GIBBER_R_MULTICAST_PACKET_GET_PRIVATE (packet);
  gsize avail;

  g_assert(packet->type == PACKET_TYPE_DATA);
  g_assert(packet->data.data.payload == NULL);
  g_assert(priv->data == NULL);

  avail = MIN(size, priv->max_data -
    gibber_r_multicast_packet_calculate_size(packet));

  packet->data.data.payload = g_memdup(data, avail);
  packet->data.data.payload_size = avail;

  return avail;
}

/* Create a packet by parsing raw data, packet is immutable afterwards */
GibberRMulticastPacket *
gibber_r_multicast_packet_parse(const guint8 *data, gsize size,
    GError **error) {
  GibberRMulticastPacket *result = g_object_new(GIBBER_TYPE_R_MULTICAST_PACKET,
                                                NULL);
  GibberRMulticastPacketPrivate *priv =
      GIBBER_R_MULTICAST_PACKET_GET_PRIVATE(result);

  priv->data = g_memdup(data, size);
  priv->size = 0;
  priv->max_data = size;

  result->type = get_guint8 (priv->data, priv->max_data, &(priv->size));
  result->version = get_guint8 (priv->data, priv->max_data, &(priv->size));
  result->sender = get_guint32 (priv->data, priv->max_data, &(priv->size));

  switch (result->type) {
    case PACKET_TYPE_WHOIS_REQUEST:
      result->data.whois_request.sender_id = get_guint32 (priv->data,
          priv->max_data, &(priv->size));
      break;
    case PACKET_TYPE_WHOIS_REPLY:
      result->data.whois_reply.sender_name = get_string(priv->data,
          priv->max_data, &(priv->size));
      break;
    case PACKET_TYPE_DATA:
      result->data.data.packet_id = get_guint32 (priv->data,
           priv->max_data, &(priv->size));
      result->data.data.packet_part =
          get_guint8 (priv->data, priv->max_data, &(priv->size));
      result->data.data.packet_total =
          get_guint8 (priv->data, priv->max_data, &(priv->size));
      result->data.data.stream_id =
          get_guint8 (priv->data, priv->max_data, &(priv->size));
      result->data.data.depends =
          get_sender_info (priv->data, priv->max_data, &(priv->size));

      result->data.data.payload_size = priv->max_data - priv->size;
      result->data.data.payload = g_memdup(priv->data + priv->size,
          result->data.data.payload_size);
      break;
    case PACKET_TYPE_REPAIR_REQUEST:
      result->data.repair_request.sender_id =
          get_guint32 (priv->data, priv->max_data, &(priv->size));
      result->data.repair_request.packet_id =
          get_guint32 (priv->data, priv->max_data, &(priv->size));
      break;
    case PACKET_TYPE_SESSION:
      result->data.session.senders =
          get_sender_info(priv->data, priv->max_data, &(priv->size));
      break;
    case PACKET_TYPE_BYE:
      /* Not implemented, fall through */
    default:
      g_assert_not_reached();
  }

  return result;
}

/* Get the packets payload */
guint8 *
gibber_r_multicast_packet_get_payload(GibberRMulticastPacket *packet,
                                      gsize *size) {
  g_assert (packet->type == PACKET_TYPE_DATA);
  g_assert (size != NULL);

  *size = packet->data.data.payload_size;

  return packet->data.data.payload;
}

/* Get the packets raw data, packet is immutable after this call */
guint8 *
gibber_r_multicast_packet_get_raw_data(GibberRMulticastPacket *packet,
                                       gsize *size) {
  GibberRMulticastPacketPrivate *priv =
     GIBBER_R_MULTICAST_PACKET_GET_PRIVATE (packet);

 /* Ensure the packet is serialized */
 gibber_r_multicast_packet_build(packet);

 *size = priv->size;

 return priv->data;
}

gint32
gibber_r_multicast_packet_diff(guint32 from, guint32 to) {
  if (from > (G_MAXUINT32 - 0xffff) && to < 0xffff) {
    return G_MAXUINT32 - from + to + 1;
  }
  if (to > (G_MAXUINT32 - 0xffff) && from < 0xffff) {
    return - from - (G_MAXUINT32 - to) - 1;
  }
  if (from > to) {
    return -MIN(from - to, G_MAXINT);
  }

  return MIN(to - from, G_MAXINT);
}
