/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test properties managers.
 *
 * ti-properties.c:
 * GObject wrapper for D-Bus method calls to org.freedesktop.Telepathy.Properties
 * 
 * Copyright (C) 2007 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */


#include "ti-properties.h"
#include "ti-constants.h"
#include "ti-signals-marshal.h"
#include "ti-util.h"

#include <dbus/dbus-glib.h>

struct _TIPropertiesClass {
    GObjectClass parent;

    // "properties-changed" signal id
    // org.freedesktop.Telepathy.Properties - PropertiesChanged signal
    guint properties_changed_id;

    // "property-flags-changed" signal id
    // org.freedesktop.Telepathy.Properties - PropertyFlagsChanged signal
    guint property_flags_changed_id;
};

G_DEFINE_TYPE (TIProperties, ti_properties, G_TYPE_OBJECT);

/*
 * Instance private data.
 */
struct _TIPropertiesPrivate {
    DBusGProxy* dbus_proxy;
};
typedef struct _TIPropertiesPrivate TIPropertiesPrivate;

#define TI_PROPERTIES_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_PROPERTIES, TIPropertiesPrivate))

// Function prototypes.
static void _on_properties_changed (gpointer proxy, GPtrArray* props, TIProperties* properties);
static void _on_property_flags_changed (gpointer proxy, GPtrArray* flags, TIProperties* properties);

/*
 * Drop all references to other objects.
 */
static void
ti_properties_dispose (GObject *object)
{
    TIProperties* properties = TI_PROPERTIES (object);
    TIPropertiesPrivate* priv = TI_PROPERTIES_GET_PRIVATE (properties);

    TI_OBJ_UNREF (priv->dbus_proxy);

    G_OBJECT_CLASS (ti_properties_parent_class)->dispose (object);
}

/*
 * Finalizes the object, marking the memory as ready for reuse
 */
/*static void
ti_properties_finalize (GObject *object)
{
    TIProperties* properties = TI_PROPERTIES (object);
    TIPropertiesPrivate* priv = TI_PROPERTIES_GET_PRIVATE (properties);

    G_OBJECT_CLASS (ti_properties_parent_class)->finalize (object);
}*/

/*
 * Class initialization.
 */
static void
ti_properties_class_init (TIPropertiesClass *ti_properties_class)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (ti_properties_class);

    /* override base object methods */
    gobject_class->dispose = ti_properties_dispose;
    //gobject_class->finalize = ti_properties_finalize;

    /* Add private */
    g_type_class_add_private (ti_properties_class, sizeof (TIPropertiesPrivate));

    ti_properties_class->properties_changed_id =
        g_signal_new ("properties-changed",
                      G_OBJECT_CLASS_TYPE (ti_properties_class),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__BOXED,
                      G_TYPE_NONE,
                      1, dbus_g_type_get_collection ("GPtrArray", dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_VALUE, G_TYPE_INVALID)) );

    ti_properties_class->property_flags_changed_id =
        g_signal_new ("property-flags-changed",
                      G_OBJECT_CLASS_TYPE (ti_properties_class),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__BOXED,
                      G_TYPE_NONE,
                      1, dbus_g_type_get_collection ("GPtrArray", dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID)) );
}

/*
 * Instance initialization.
 */
static void
ti_properties_init (TIProperties *ti_properties)
{
    TIPropertiesPrivate *priv = TI_PROPERTIES_GET_PRIVATE (ti_properties);

    priv->dbus_proxy = NULL;
}

/*
 * Creates a new properties
 */
TIProperties*
ti_properties_new (const gchar* service_name, const gchar* obj_path)
{
    TIProperties* properties;
    TIPropertiesPrivate* priv;
    TIPropertiesClass* klass;
    DBusGConnection* dbus_conn;
    GError* error;

    error = NULL;
    dbus_conn = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (dbus_conn == NULL)
    {
        g_printerr ("Failed to open connection to bus: %s\n", error->message);
        g_error_free (error);
        return NULL;
    }

    properties = g_object_new (TI_TYPE_PROPERTIES, NULL);
    priv = TI_PROPERTIES_GET_PRIVATE (properties);
    klass = TI_PROPERTIES_GET_CLASS (properties);

    priv->dbus_proxy = dbus_g_proxy_new_for_name (dbus_conn, service_name, obj_path,
                                                  "org.freedesktop.Telepathy.Properties");

    /* Tell DBus what the type signature of the signal callback is; this
     * allows us to sanity-check incoming messages before invoking the
     * callback.  You need to do this once for each proxy you create,
     * not every time you want to connect to the signal.
     */

    dbus_g_proxy_add_signal (priv->dbus_proxy, "PropertiesChanged",
                             dbus_g_type_get_collection ("GPtrArray", dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_VALUE, G_TYPE_INVALID)), G_TYPE_INVALID);

    dbus_g_proxy_add_signal (priv->dbus_proxy, "PropertyFlagsChanged",
                             dbus_g_type_get_collection ("GPtrArray", dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT, G_TYPE_UINT, G_TYPE_INVALID)), G_TYPE_INVALID);

    /* Actually connect to the signal.  Note you can call
     * dbus_g_proxy_connect_signal multiple times for one invocation of
     * dbus_g_proxy_add_signal.
     */

    dbus_g_proxy_connect_signal (priv->dbus_proxy, "PropertiesChanged", G_CALLBACK (_on_properties_changed),
                                 properties, NULL);

    dbus_g_proxy_connect_signal (priv->dbus_proxy, "PropertyFlagsChanged", G_CALLBACK (_on_property_flags_changed),
                                 properties, NULL);

    return properties;
}

/*
 * On Properties Changed
 */
static void
_on_properties_changed (gpointer proxy, GPtrArray* props, TIProperties* properties)
{
    TIPropertiesClass* klass = TI_PROPERTIES_GET_CLASS (properties);

    g_signal_emit (properties, klass->properties_changed_id, 0, props);
}

/*
 * On Property Flags Changed
 */
static void _on_property_flags_changed (gpointer proxy, GPtrArray* flags, TIProperties* properties)
{
    TIPropertiesClass* klass = TI_PROPERTIES_GET_CLASS (properties);

    g_signal_emit (properties, klass->property_flags_changed_id, 0, flags);
}

/*
 * Get Properties
 */
GPtrArray*
ti_properties_get_properties (TIProperties* self, GArray* prop_ids, GError** error)
{
    TIPropertiesPrivate* priv = TI_PROPERTIES_GET_PRIVATE(self);
    static GType prop_gtype = G_TYPE_INVALID;
    GPtrArray* props = NULL;

    if (prop_gtype == G_TYPE_INVALID)
    {
        prop_gtype = dbus_g_type_get_struct ("GValueArray",
                                             G_TYPE_UINT,
                                             G_TYPE_VALUE,
                                             G_TYPE_INVALID);
    }

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "GetProperties", error,
                            DBUS_TYPE_G_UINT_ARRAY, prop_ids, G_TYPE_INVALID,
                            dbus_g_type_get_collection ("GPtrArray", prop_gtype), &props, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return props;
}

/*
 * List Properties
 */
GPtrArray* ti_properties_list_properties (TIProperties* self, GError** error)
{
    TIPropertiesPrivate* priv = TI_PROPERTIES_GET_PRIVATE(self);
    static GType desc_gtype = G_TYPE_INVALID;
    GPtrArray* props = NULL;

    if (desc_gtype == G_TYPE_INVALID)
    {
        desc_gtype = dbus_g_type_get_struct ("GValueArray",
                                             G_TYPE_UINT,
                                             G_TYPE_STRING,
                                             G_TYPE_STRING,
                                             G_TYPE_UINT,
                                             G_TYPE_INVALID);
    }

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "ListProperties", error,
                            G_TYPE_INVALID,
                            dbus_g_type_get_collection ("GPtrArray", desc_gtype), &props, G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }

    return props;
}

/*
 * Set Properties
 */
void ti_properties_set_properties (TIProperties* self, GPtrArray* props, GError** error)
{
    TIPropertiesPrivate* priv = TI_PROPERTIES_GET_PRIVATE(self);
    static GType prop_gtype = G_TYPE_INVALID;

    if (prop_gtype == G_TYPE_INVALID)
    {
        prop_gtype = dbus_g_type_get_struct ("GValueArray",
                                             G_TYPE_UINT,
                                             G_TYPE_VALUE,
                                             G_TYPE_INVALID);
    }

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "SetProperties", error,
                            dbus_g_type_get_collection ("GPtrArray", prop_gtype), props, G_TYPE_INVALID,
                            G_TYPE_INVALID))
    {
        if (*error != NULL)
        {
            g_printerr ("Error: %s\n", (*error)->message);
        }
    }
}
