/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-message-bus.c:
 * D-Bus' message bus.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <string.h>
#include <glib.h>
#include <dbus/dbus-glib.h>

#include "ti-message-bus.h"

G_DEFINE_TYPE (TIMessageBus, ti_message_bus, G_TYPE_OBJECT);

/* Function prototypes */
static gboolean _ti_parse_active_conn (TIActiveConn* active_conn, const gchar* service_suffix);

/***
 * Instance private data.
 */
struct _TIMessageBusPrivate {
    gboolean disposed;

    DBusGProxy* dbus_proxy;
};
typedef struct _TIMessageBusPrivate TIMessageBusPrivate;


#define TI_MESSAGE_BUS_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_MESSAGE_BUS, TIMessageBusPrivate))

/***
 * Drop all references to other objects.
 */
static void
ti_message_bus_dispose (GObject *object)
{
    TIMessageBus* message_bus = TI_MESSAGE_BUS (object);
    TIMessageBusPrivate* priv = TI_MESSAGE_BUS_GET_PRIVATE (message_bus);

    if (priv->disposed)
        return;
    else
        priv->disposed = TRUE;

    if (priv->dbus_proxy != NULL)
    {
        g_object_unref (priv->dbus_proxy);
        priv->dbus_proxy = NULL;
    }

    G_OBJECT_CLASS (ti_message_bus_parent_class)->dispose (object);
}

/***
 * Finalizes the object, marking the memory as ready for reuse
 */
static void
ti_message_bus_finalize (GObject *object)
{
//	TIMessageBus* message_bus = TI_MESSAGE_BUS (object);
//  TIMessageBusPrivate* priv = TI_MESSAGE_BUS_GET_PRIVATE (message_bus);

	G_OBJECT_CLASS (ti_message_bus_parent_class)->finalize (object);
}

/**
 * Class initialization.
 */
static void
ti_message_bus_class_init (TIMessageBusClass *ti_message_bus_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_message_bus_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_message_bus_dispose;
    gobject_class->finalize = ti_message_bus_finalize;
		
	/* Add private */
	g_type_class_add_private (ti_message_bus_class, sizeof (TIMessageBusPrivate));
}

/***
 * Instance initialization.
 */
static void
ti_message_bus_init (TIMessageBus *ti_message_bus)
{
    TIMessageBusPrivate *priv = TI_MESSAGE_BUS_GET_PRIVATE (ti_message_bus);

    priv->disposed = FALSE;
    priv->dbus_proxy = NULL;
}

/**
 * Returns a new instance.
 */
TIMessageBus*
ti_message_bus_new ()
{
    TIMessageBus* message_bus = NULL;
    TIMessageBusPrivate* priv;
    DBusGConnection* dbus_conn;
    GError* error;

    error = NULL;
    dbus_conn = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
    if (dbus_conn == NULL)
    {
        g_printerr ("Failed to open connection to bus: %s\n", error->message);
        g_error_free (error);
        return NULL;
    }

	message_bus = g_object_new (TI_TYPE_MESSAGE_BUS, NULL);
    priv = TI_MESSAGE_BUS_GET_PRIVATE (message_bus);

    priv->dbus_proxy = dbus_g_proxy_new_for_name (dbus_conn,
                                                  DBUS_SERVICE_DBUS,
                                                  DBUS_PATH_DBUS,
                                                  DBUS_INTERFACE_DBUS);

    return message_bus;
}

/**
 * List Names
 */
GStrv
ti_message_bus_list_names (TIMessageBus* message_bus, GError** error)
{
    TIMessageBusPrivate* priv = TI_MESSAGE_BUS_GET_PRIVATE (message_bus);
    gchar** services;

    *error = NULL;
    if (!dbus_g_proxy_call (priv->dbus_proxy, "ListNames", error,
                            G_TYPE_INVALID,
                            G_TYPE_STRV, &services, G_TYPE_INVALID))
    {
        g_printerr ("Error: %s\n", (*error)->message);
    }

    return services;
}


/**
 * List Active Connections
 */
GArray*
ti_message_bus_list_active_conns (TIMessageBus* message_bus, GError** error)
{
    gchar** services = NULL;
    guint i;
    GArray* active_conns = NULL;
    gchar* service_suffix = NULL;
    TIActiveConn active_conn;
    gboolean ok;

    services = ti_message_bus_list_names (message_bus, error);
    if (*error != NULL)
        goto CLEAN_UP;

    active_conns = g_array_new (FALSE, FALSE, sizeof (TIActiveConn));

    for (i = 0; services[i] != NULL; i++)
    {
        if (!g_str_has_prefix (services[i], "org.freedesktop.Telepathy.Connection."))
            continue;

        service_suffix = &(services[i][strlen ("org.freedesktop.Telepathy.Connection.")]);
        ok = _ti_parse_active_conn (&active_conn, service_suffix);
        if (ok)
        {
            g_array_append_val (active_conns, active_conn);
        }
    }

    CLEAN_UP:
    g_strfreev (services);

    return active_conns;
}

/**
 * Parse Active Connection
 */
static gboolean
_ti_parse_active_conn (TIActiveConn* active_conn, const gchar* service_suffix)
{
    gchar** str_array = NULL;
    gboolean ok = FALSE;

    str_array = g_strsplit (service_suffix, ".", 10);
    if (g_strv_length (str_array) != 3)
        goto CLEAN_UP;

    active_conn->conn_man = g_strdup (str_array [0]);
    active_conn->protocol = g_strdup (str_array [1]);
    active_conn->account =  g_strdup (str_array [2]);

    ok = TRUE;

    CLEAN_UP:
    g_strfreev(str_array);

    return ok;
}
