/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-handle-list-editor.c:
 * A widget to create and edit a handle list of a single handle type.
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-handle-list-editor.h"
#include "ti-util.h"
#include "ti-constants.h"
#include "ti-preferences.h"

#include <string.h>

enum {
    TI_COLUMN_HANDLES_LIST_HANDLE = 0,
    TI_COLUMN_HANDLES_LIST_NAME
};

struct _TIHandleListEditorClass {
    GtkHBoxClass parent;
    /* class members */
};

/* Interface to entry handle numbers */
struct _TIHandleListEditorNumberUI {
    GtkEntry* entry;
    GtkButton* button_add;
    GtkWidget* vbox_buttons_and_entry;
};
typedef struct _TIHandleListEditorNumberUI TIHandleListEditorNumberUI;

/* Interface to entry handle names */
struct _TIHandleListEditorNameUI {
    GtkButton* button_add;
    GtkTreeSelection* selection;
    GtkWidget* vbox_buttons_and_list;
};
typedef struct _TIHandleListEditorNameUI TIHandleListEditorNameUI;

G_DEFINE_TYPE (TIHandleListEditor, ti_handle_list_editor, GTK_TYPE_HBOX);

/**
 * Instance private data.
 */
struct _TIHandleListEditorPrivate {

    TIHandleMapper* handle_mapper;
    TIPreferences* preferences;

    TIHandleListEditorNumberUI number_ui;
    TIHandleListEditorNameUI name_ui;

    /* Handle list */
    GtkTreeView* tree_view;
    GtkTreeSelection* selection;
    GtkListStore* list_store;
    GtkButton* button_remove;
};
typedef struct _TIHandleListEditorPrivate TIHandleListEditorPrivate;

#define TI_HANDLE_LIST_EDITOR_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_HANDLE_LIST_EDITOR, TIHandleListEditorPrivate))

/* Private functions */
static void _ti_handle_list_editor_init_number_ui (TIHandleListEditor* self);
static void _ti_handle_list_editor_init_name_ui (TIHandleListEditor* self);
static void _ti_handle_list_editor_build_tree_view (TIHandleListEditor* self);
static void _ti_handle_list_editor_add (TIHandleListEditor* self);
static void _ti_handle_list_editor_remove (TIHandleListEditor* self);
static void _ti_handle_list_editor_update_button_add_sensitivity (TIHandleListEditor* self);
static void _ti_handle_list_editor_update_button_remove_sensitivity (TIHandleListEditor* self);
static void _ti_handle_list_editor_name_ui_add (TIHandleListEditor* self);
static void _ti_handle_list_editor_name_ui_update_button_add_sensitivity (TIHandleListEditor* self);
static void _ti_handle_list_editor_contact_handle_display_mode_changed (TIHandleListEditor* self, guint contact_handle_display_mode);
static void _ti_handle_list_editor_change_tree_view_display_mode (TIHandleListEditor* self, guint contact_handle_display_mode);

/**
 * Drop all references to other objects.
 */
static void
ti_handle_list_editor_dispose (GObject *object)
{
    TIHandleListEditorPrivate *priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (object);

    if (priv->preferences != NULL)
    {
        g_object_unref (priv->preferences);
        priv->preferences = NULL;
    }

    if (priv->handle_mapper != NULL)
    {
        g_object_unref (priv->handle_mapper);
        priv->handle_mapper = NULL;
    }

    if (priv->number_ui.vbox_buttons_and_entry != NULL)
    {
        g_object_unref (priv->number_ui.vbox_buttons_and_entry);
        priv->number_ui.vbox_buttons_and_entry = NULL;
    }

    if (priv->name_ui.vbox_buttons_and_list != NULL)
    {
        g_object_unref (priv->name_ui.vbox_buttons_and_list);
        priv->name_ui.vbox_buttons_and_list = NULL;
    }

    G_OBJECT_CLASS (ti_handle_list_editor_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_handle_list_editor_class_init (TIHandleListEditorClass* handle_list_editor_class)
{
	GObjectClass* gobject_class = G_OBJECT_CLASS (handle_list_editor_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_handle_list_editor_dispose;
		
	/* Add private */
	g_type_class_add_private (handle_list_editor_class, sizeof (TIHandleListEditorPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_handle_list_editor_init (TIHandleListEditor* self)
{
    TIHandleListEditorPrivate* priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);

    priv->preferences = ti_preferences_new ();
    priv->handle_mapper = NULL;
    priv->number_ui.vbox_buttons_and_entry = NULL;
    priv->name_ui.vbox_buttons_and_list = NULL;
}

/**
 * New
 */
GtkWidget*
ti_handle_list_editor_new (TIHandleMapper* handle_mapper)
{
    TIHandleListEditor* handle_list_editor = NULL;
    TIHandleListEditorPrivate* priv;
    GtkWidget* scrolled_wnd;
    GtkWidget* buttons;
    GtkWidget* vbox_list_and_buttons;

    handle_list_editor = g_object_new (TI_TYPE_HANDLE_LIST_EDITOR, NULL);
    priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (handle_list_editor);

    priv->handle_mapper = handle_mapper;

    _ti_handle_list_editor_init_number_ui (handle_list_editor);
    _ti_handle_list_editor_init_name_ui (handle_list_editor);
    _ti_handle_list_editor_build_tree_view (handle_list_editor);

    g_signal_connect_swapped (priv->preferences, "contact-handle-display-changed",
                              G_CALLBACK (_ti_handle_list_editor_contact_handle_display_mode_changed), handle_list_editor);

    // Button Remove
    priv->button_remove = GTK_BUTTON (gtk_button_new_from_stock (GTK_STOCK_REMOVE));
    gtk_widget_set_sensitive (GTK_WIDGET (priv->button_remove), FALSE);
    g_signal_connect_swapped (priv->button_remove, "clicked", G_CALLBACK (_ti_handle_list_editor_remove), handle_list_editor);

    // Scrolled Window
    scrolled_wnd = gtk_scrolled_window_new (NULL, NULL);
    gtk_container_set_border_width  (GTK_CONTAINER (scrolled_wnd), 4);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_wnd), GTK_SHADOW_IN);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_wnd), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_container_add (GTK_CONTAINER (scrolled_wnd), GTK_WIDGET (priv->tree_view));

    // Buttons
    buttons = gtk_hbutton_box_new ();
    gtk_button_box_set_layout (GTK_BUTTON_BOX (buttons), GTK_BUTTONBOX_SPREAD);
    gtk_box_pack_start (GTK_BOX (buttons), GTK_WIDGET (priv->button_remove), FALSE, TRUE, 0);

    // VBox List and Buttons
    vbox_list_and_buttons = gtk_vbox_new (FALSE, 2);
    gtk_box_pack_start (GTK_BOX (vbox_list_and_buttons), scrolled_wnd, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vbox_list_and_buttons), buttons, TRUE, TRUE, 0);

    // Widget Root
    gtk_box_set_homogeneous (GTK_BOX (handle_list_editor), TRUE);

    if (ti_preferences_get_contact_handle_display_mode (priv->preferences) ==
        TI_PREFERENCES_CONTACT_HANDLE_DISPLAY_HANDLE)
    {
        gtk_box_pack_start (GTK_BOX (handle_list_editor), priv->number_ui.vbox_buttons_and_entry, FALSE, TRUE, 0);
    }
    else
    {
        gtk_box_pack_start (GTK_BOX (handle_list_editor), priv->name_ui.vbox_buttons_and_list, TRUE, TRUE, 0);
    }

    gtk_box_pack_start (GTK_BOX (handle_list_editor), vbox_list_and_buttons, TRUE, TRUE, 0);

    return GTK_WIDGET (handle_list_editor);
}

/**
 * Init Number UI
 */
static void
_ti_handle_list_editor_init_number_ui (TIHandleListEditor* self)
{
    TIHandleListEditorPrivate* priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);
    GtkWidget* buttons;

    priv->number_ui.entry = GTK_ENTRY (gtk_entry_new ());
    g_signal_connect_swapped (priv->number_ui.entry, "key-release-event", G_CALLBACK (_ti_handle_list_editor_update_button_add_sensitivity), self);

    priv->number_ui.button_add = GTK_BUTTON (gtk_button_new_from_stock (GTK_STOCK_ADD));
    gtk_widget_set_sensitive (GTK_WIDGET (priv->number_ui.button_add), FALSE);
    g_signal_connect_swapped (priv->number_ui.button_add, "clicked", G_CALLBACK (_ti_handle_list_editor_add), self);

    buttons = gtk_vbutton_box_new ();
    gtk_button_box_set_layout (GTK_BUTTON_BOX (buttons), GTK_BUTTONBOX_SPREAD);
    gtk_box_pack_start (GTK_BOX (buttons), GTK_WIDGET (priv->number_ui.button_add), FALSE, TRUE, 0);

    priv->number_ui.vbox_buttons_and_entry = gtk_vbox_new (TRUE, 0);
    g_object_ref_sink (priv->number_ui.vbox_buttons_and_entry);
    gtk_container_set_border_width  (GTK_CONTAINER (priv->number_ui.vbox_buttons_and_entry), 4);
    gtk_box_pack_start (GTK_BOX (priv->number_ui.vbox_buttons_and_entry), GTK_WIDGET (priv->number_ui.entry), FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (priv->number_ui.vbox_buttons_and_entry), buttons, TRUE, TRUE, 0);
}

/**
 * Init Name UI
 */
static void
_ti_handle_list_editor_init_name_ui (TIHandleListEditor* self)
{
    TIHandleListEditorPrivate* priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);
    GtkCellRenderer* renderer;
    GtkTreeViewColumn* column;
    GtkTreeView* tree_view;
    GtkTreeModel* tree_model;
    GtkWidget* scrolled_wnd;
    GtkWidget* buttons;

    // Tree View
    tree_model = GTK_TREE_MODEL (ti_handle_mapper_get_contact_handle_list_store (priv->handle_mapper));

    tree_view = GTK_TREE_VIEW (gtk_tree_view_new_with_model (tree_model));
    gtk_tree_view_set_headers_visible (tree_view, FALSE);

    renderer = gtk_cell_renderer_text_new ();

    column = gtk_tree_view_column_new_with_attributes ("Name",
                                                       renderer,
                                                       "text", TI_HANDLES_LIST_COLUMN_NAME,
                                                       NULL);

    gtk_tree_view_append_column (tree_view, column);

    priv->name_ui.selection = gtk_tree_view_get_selection (tree_view);

    g_signal_connect_swapped (priv->name_ui.selection, "changed",
                              G_CALLBACK (_ti_handle_list_editor_name_ui_update_button_add_sensitivity),
                              self);

    gtk_tree_selection_set_mode (priv->name_ui.selection, GTK_SELECTION_MULTIPLE);

    // Scrolled Window
    scrolled_wnd = gtk_scrolled_window_new (NULL, NULL);
    gtk_container_set_border_width  (GTK_CONTAINER (scrolled_wnd), 4);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled_wnd), GTK_SHADOW_IN);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_wnd), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_container_add (GTK_CONTAINER (scrolled_wnd), GTK_WIDGET (tree_view));

    // Button Add
    priv->name_ui.button_add = GTK_BUTTON (gtk_button_new_from_stock (GTK_STOCK_ADD));
    gtk_widget_set_sensitive (GTK_WIDGET (priv->name_ui.button_add), FALSE);
    g_signal_connect_swapped (priv->name_ui.button_add, "clicked", G_CALLBACK (_ti_handle_list_editor_name_ui_add), self);

    // Buttons
    buttons = gtk_vbutton_box_new ();
    gtk_button_box_set_layout (GTK_BUTTON_BOX (buttons), GTK_BUTTONBOX_SPREAD);
    gtk_box_pack_start (GTK_BOX (buttons), GTK_WIDGET (priv->name_ui.button_add), FALSE, TRUE, 0);

    // VBox Buttons and List
    priv->name_ui.vbox_buttons_and_list = gtk_vbox_new (FALSE, 0);
    g_object_ref_sink (priv->name_ui.vbox_buttons_and_list);
    gtk_container_set_border_width  (GTK_CONTAINER (priv->name_ui.vbox_buttons_and_list), 4);
    gtk_box_pack_start (GTK_BOX (priv->name_ui.vbox_buttons_and_list), scrolled_wnd, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (priv->name_ui.vbox_buttons_and_list), buttons, FALSE, TRUE, 0);
}

/**
 * Build Tree View
 */
static void
_ti_handle_list_editor_build_tree_view (TIHandleListEditor* self)
{
    TIHandleListEditorPrivate* priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);
    GtkCellRenderer* renderer;
    GtkTreeViewColumn* column;

    priv->list_store = gtk_list_store_new (2, G_TYPE_UINT,    // handle
                                              G_TYPE_STRING); // name

    priv->tree_view = GTK_TREE_VIEW (gtk_tree_view_new_with_model (GTK_TREE_MODEL (priv->list_store)));
    gtk_tree_view_set_headers_visible (priv->tree_view, FALSE);

    renderer = gtk_cell_renderer_text_new ();

    if (ti_preferences_get_contact_handle_display_mode (priv->preferences) ==
        TI_PREFERENCES_CONTACT_HANDLE_DISPLAY_HANDLE)
    {
        column = gtk_tree_view_column_new_with_attributes ("Handle",
                                                           renderer,
                                                           "text", TI_COLUMN_HANDLES_LIST_HANDLE,
                                                           NULL);
    }
    else
    {
        column = gtk_tree_view_column_new_with_attributes ("Name",
                                                           renderer,
                                                           "text", TI_COLUMN_HANDLES_LIST_NAME,
                                                           NULL);
    }
    gtk_tree_view_append_column (priv->tree_view, column);

    priv->selection = gtk_tree_view_get_selection (priv->tree_view);

    g_signal_connect_swapped (priv->selection, "changed",
                              G_CALLBACK (_ti_handle_list_editor_update_button_remove_sensitivity),
                              self);

    gtk_tree_selection_set_mode (priv->selection, GTK_SELECTION_MULTIPLE);
}

/**
 * Add
 */
static void
_ti_handle_list_editor_add (TIHandleListEditor* self)
{
    TIHandleListEditorPrivate *priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);
    const gchar* handle_str;
    guint handle;
    int result;
    GtkTreeIter iter;
    
    handle_str = gtk_entry_get_text (priv->number_ui.entry);

    result = sscanf (handle_str, "%u", &handle);
    if (result != 1)
        return;

    gtk_list_store_append (priv->list_store, &iter);
    gtk_list_store_set (priv->list_store, &iter,
                        0, handle,
                        -1);
}

/**
 * Remove
 */
static void
_ti_handle_list_editor_remove (TIHandleListEditor* self)
{
    TIHandleListEditorPrivate* priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);

    ti_remove_selected_elements (priv->selection);
}


/**
 * Update Button Remove Sensitivity
 */
static void
_ti_handle_list_editor_update_button_remove_sensitivity (TIHandleListEditor* self)
{
    TIHandleListEditorPrivate* priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);
    gint rows_count;

    rows_count = gtk_tree_selection_count_selected_rows (priv->selection);

    gtk_widget_set_sensitive (GTK_WIDGET (priv->button_remove), rows_count > 0);
}

/**
 * Set the handle list
 */
void
ti_handle_list_editor_set (TIHandleListEditor* self, GArray* handles)
{
    TIHandleListEditorPrivate* priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);
    GtkTreeIter iter;
    guint i;
    guint handle_number;
    gchar* handle_name;

    gtk_list_store_clear (priv->list_store);

    if (handles == NULL || handles->len == 0)
        return;

    // Fill the list.
    for (i = 0; i < handles->len; i++)
    {
        handle_number = g_array_index (handles, guint, i);

        handle_name = ti_handle_mapper_get_contact_handle_name (priv->handle_mapper, handle_number);
        if (handle_name == NULL)
        {
            handle_name = g_strdup_printf ("%u", handle_number);
        }

        gtk_list_store_append (priv->list_store, &iter);
        gtk_list_store_set (priv->list_store, &iter,
                            TI_COLUMN_HANDLES_LIST_HANDLE, handle_number,
                            TI_COLUMN_HANDLES_LIST_NAME, handle_name,
                            -1);

        g_free (handle_name);
    }
}

/**
 * Get the handle list
 */
GArray* ti_handle_list_editor_get (TIHandleListEditor* self)
{
    TIHandleListEditorPrivate* priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);

    return ti_get_tree_model_elements (GTK_TREE_MODEL (priv->list_store), TI_COLUMN_HANDLES_LIST_HANDLE, G_TYPE_UINT);
}

/**
 * Update Button Add Sensitivity
 */
static void
_ti_handle_list_editor_update_button_add_sensitivity (TIHandleListEditor* self)
{
    TIHandleListEditorPrivate* priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);
    const gchar* handle_str;

    handle_str = gtk_entry_get_text (priv->number_ui.entry);

    gtk_widget_set_sensitive (GTK_WIDGET (priv->number_ui.button_add), strlen (handle_str) > 0);
}

/**
 * Name UI Add
 */
static void
_ti_handle_list_editor_name_ui_add (TIHandleListEditor* self)
{
    TIHandleListEditorPrivate *priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);
    GList* selected_rows_list = NULL;
    GList* list_item = NULL;
    GtkTreeModel* tree_model = NULL;
    GtkTreeIter iter;
    guint handle_number;
    gchar* handle_name;
    GtkTreePath* tree_path;

    selected_rows_list = gtk_tree_selection_get_selected_rows (priv->name_ui.selection, &tree_model);

    list_item = selected_rows_list;
    while (list_item != NULL)
    {
        tree_path = (GtkTreePath*) list_item->data;

        gtk_tree_model_get_iter (tree_model, &iter, tree_path);
        gtk_tree_model_get (tree_model, &iter,
                            TI_HANDLES_LIST_COLUMN_NUMBER, &handle_number,
                            TI_HANDLES_LIST_COLUMN_NAME, &handle_name,
                            -1);

        gtk_list_store_append (priv->list_store, &iter);
        gtk_list_store_set (priv->list_store, &iter,
                            TI_COLUMN_HANDLES_LIST_HANDLE, handle_number,
                            TI_COLUMN_HANDLES_LIST_NAME, handle_name,
                            -1);

        g_free (handle_name);

        list_item = list_item->next;
    }

    // Clean up
    if (selected_rows_list != NULL)
    {
        g_list_foreach (selected_rows_list, (GFunc) gtk_tree_path_free, NULL);
        g_list_free (selected_rows_list);
    }
}

/**
 * Update Name UI Button Add Sensitivity
 */
static void
_ti_handle_list_editor_name_ui_update_button_add_sensitivity (TIHandleListEditor* self)
{
    TIHandleListEditorPrivate *priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);
    gint count;

    count = gtk_tree_selection_count_selected_rows (priv->name_ui.selection);

    gtk_widget_set_sensitive (GTK_WIDGET (priv->name_ui.button_add), count > 0);
}

/**
 * Contact Handle Display Mode Changed
 */
static void
_ti_handle_list_editor_contact_handle_display_mode_changed (TIHandleListEditor* self, guint contact_handle_display_mode)
{
    TIHandleListEditorPrivate *priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);

    // Remove the existing UI
    gtk_container_remove (GTK_CONTAINER (self), priv->name_ui.vbox_buttons_and_list);
    gtk_container_remove (GTK_CONTAINER (self), priv->number_ui.vbox_buttons_and_entry);

    // Add one according to the global preferences
    if (contact_handle_display_mode == TI_PREFERENCES_CONTACT_HANDLE_DISPLAY_HANDLE)
    {
        gtk_box_pack_start (GTK_BOX (self), priv->number_ui.vbox_buttons_and_entry, TRUE, TRUE, 0);
        gtk_box_reorder_child (GTK_BOX (self), priv->number_ui.vbox_buttons_and_entry, 0);
    }
    else
    {
        gtk_box_pack_start (GTK_BOX (self), priv->name_ui.vbox_buttons_and_list, TRUE, TRUE, 0);
        gtk_box_reorder_child (GTK_BOX (self), priv->name_ui.vbox_buttons_and_list, 0);
    }

    _ti_handle_list_editor_change_tree_view_display_mode (self, contact_handle_display_mode);
}

/**
 * Change Tree View Display Mode
 */
static void
_ti_handle_list_editor_change_tree_view_display_mode (TIHandleListEditor* self, guint contact_handle_display_mode)
{
    TIHandleListEditorPrivate *priv = TI_HANDLE_LIST_EDITOR_GET_PRIVATE (self);
    GtkTreeViewColumn* contact_handle_column;
    GtkCellRenderer* renderer;
    GList* renderers_list = NULL;

    contact_handle_column = gtk_tree_view_get_column (priv->tree_view, 0); // It's the first and only column.

    renderers_list = gtk_tree_view_column_get_cell_renderers (contact_handle_column);
    g_assert (g_list_length (renderers_list) == 1);

    renderer = GTK_CELL_RENDERER (renderers_list->data);

    if (contact_handle_display_mode == TI_PREFERENCES_CONTACT_HANDLE_DISPLAY_HANDLE)
    {
        gtk_tree_view_column_set_title (contact_handle_column, "Handle");

        gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
                                             "text", TI_COLUMN_HANDLES_LIST_HANDLE,
                                             NULL);
    }
    else // TI_PREFERENCES_CONTACT_HANDLE_DISPLAY_NAME
    {
        gtk_tree_view_column_set_title (contact_handle_column, "Name");

        gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
                                             "text", TI_COLUMN_HANDLES_LIST_NAME,
                                             NULL);
    }

    // Clean up
    g_list_free (renderers_list);
}
