/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-dlg-preferences.c:
 * Preferences Dialog
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-dlg-preferences.h"
#include "ti-config.h"
#include "ti-preferences.h"
#include "ti-constants.h"

#include <glade/glade.h>

G_DEFINE_TYPE (TIDlgPreferences, ti_dlg_preferences, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIDlgPreferencesPrivate {
    gboolean disposed;

    TIPreferences* preferences;

    GtkWindow* parent;

    GladeXML* glade_xml;

    GtkWidget* dialog;

    GtkWidget* radiobutton_contact_display_handle;
    GtkWidget* checkbutton_auto_inspect_handles;
};
typedef struct _TIDlgPreferencesPrivate TIDlgPreferencesPrivate;

#define TI_DLG_PREFERENCES_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_DLG_PREFERENCES, TIDlgPreferencesPrivate))

/* Function prototypes */
static void _button_ok_clicked (GtkButton *button, gpointer user_data);
static void _button_cancel_clicked (GtkButton *button, gpointer user_data);
static void _ti_dlg_preferences_setup (TIDlgPreferences* dlg_preferences);
static void _ti_dlg_preferences_apply (TIDlgPreferences* self);
static void _ti_dlg_preferences_setup_initial_values (TIDlgPreferences* self);

/**
 * Drop all references to other objects.
 */
static void
ti_dlg_preferences_dispose (GObject *object)
{
    TIDlgPreferences *dlg_preferences = TI_DLG_PREFERENCES (object);
    TIDlgPreferencesPrivate *priv = TI_DLG_PREFERENCES_GET_PRIVATE (dlg_preferences);

    if (priv->disposed)
        return;
    else
        priv->disposed = TRUE;

    if (priv->preferences != NULL)
    {
        g_object_unref (priv->preferences);
        priv->preferences = NULL;
    }

    if (priv->glade_xml != NULL)
    {
        g_object_unref (priv->glade_xml);
        priv->glade_xml = NULL;
    }

    G_OBJECT_CLASS (ti_dlg_preferences_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_dlg_preferences_class_init (TIDlgPreferencesClass *ti_dlg_preferences_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_dlg_preferences_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_dlg_preferences_dispose;
		
	/* Add private */
	g_type_class_add_private (ti_dlg_preferences_class, sizeof (TIDlgPreferencesPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_dlg_preferences_init (TIDlgPreferences *ti_dlg_preferences)
{
    TIDlgPreferencesPrivate *priv = TI_DLG_PREFERENCES_GET_PRIVATE (ti_dlg_preferences);

    priv->disposed = FALSE;

    priv->preferences = ti_preferences_new ();
    g_assert (priv->preferences != NULL);
}

/**
 * Returns a new instance.
 */
TIDlgPreferences*
ti_dlg_preferences_new (GtkWindow* parent) 
{
    TIDlgPreferences* dialog = NULL;
    TIDlgPreferencesPrivate* priv = NULL;
    gchar* glade_file_path = NULL;

	dialog = g_object_new (TI_TYPE_DLG_PREFERENCES, NULL);
    priv = TI_DLG_PREFERENCES_GET_PRIVATE (dialog);

    priv->parent = parent;

    glade_file_path = g_strdup_printf ("%s%s", TI_DATA_DIR_PREFIX, "dlg-preferences.xml");
    priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);
    if (priv->glade_xml == NULL)
    {
        g_critical ("Error loading glade file \"%s\".", glade_file_path);
        g_object_unref (dialog);
        dialog = NULL;
        goto CLEAN_UP;
    }

    priv->dialog = glade_xml_get_widget (priv->glade_xml, "dialog_preferences");
    g_assert (GTK_IS_DIALOG (priv->dialog));

    gtk_window_set_transient_for (GTK_WINDOW (priv->dialog), GTK_WINDOW (priv->parent));
    gtk_window_set_position (GTK_WINDOW (priv->dialog), GTK_WIN_POS_CENTER_ON_PARENT);

    _ti_dlg_preferences_setup (dialog);

    CLEAN_UP:
    g_free (glade_file_path);

    return dialog;
}

/**
 * Run
 */
void ti_dlg_preferences_run (TIDlgPreferences* self)
{
    TIDlgPreferencesPrivate* priv = TI_DLG_PREFERENCES_GET_PRIVATE (self);
    gint result;

    _ti_dlg_preferences_setup_initial_values (self);

    gtk_widget_show_all (priv->dialog);
    result = gtk_dialog_run (GTK_DIALOG (priv->dialog));
    gtk_widget_hide_all (priv->dialog);

    if (result == GTK_RESPONSE_OK)
    {
        _ti_dlg_preferences_apply (self);
    }
}

/**
 * Setup initial values
 */
static void
_ti_dlg_preferences_setup_initial_values (TIDlgPreferences* self)
{
    TIDlgPreferencesPrivate *priv = TI_DLG_PREFERENCES_GET_PRIVATE (self);
    guint display_mode;

    display_mode = ti_preferences_get_contact_handle_display_mode (priv->preferences);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->radiobutton_contact_display_handle),
                                  display_mode == TI_PREFERENCES_CONTACT_HANDLE_DISPLAY_HANDLE);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->checkbutton_auto_inspect_handles),
                                  ti_preferences_get_auto_inspect_handles (priv->preferences));
}

/**
 * Called when OK button is clicked.
 */
static void
_button_ok_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_OK);
}

/**
 * Called when "Cancel" button is clicked.
 */
static void
_button_cancel_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_CANCEL);
}

/**
 * Setup dialog
 * Connect signals, get widgets from gladexml, etc.
 */
static void
_ti_dlg_preferences_setup (TIDlgPreferences* dlg_preferences)
{
    TIDlgPreferencesPrivate *priv = TI_DLG_PREFERENCES_GET_PRIVATE (dlg_preferences);

    glade_xml_signal_connect_data (priv->glade_xml, "button_ok_clicked", G_CALLBACK(_button_ok_clicked), priv->dialog);
    glade_xml_signal_connect_data (priv->glade_xml, "button_cancel_clicked", G_CALLBACK(_button_cancel_clicked), priv->dialog);

    // Contact handles display mode
    priv->radiobutton_contact_display_handle = glade_xml_get_widget (priv->glade_xml, "radiobutton_contact_display_handle");
    g_assert (GTK_IS_RADIO_BUTTON (priv->radiobutton_contact_display_handle));

    // Auto inspect handles
    priv->checkbutton_auto_inspect_handles = glade_xml_get_widget (priv->glade_xml, "checkbutton_auto_inspect_handles");
    g_assert (GTK_IS_CHECK_BUTTON (priv->checkbutton_auto_inspect_handles));
    
}

/**
 * Apply
 */
static void
_ti_dlg_preferences_apply (TIDlgPreferences* self)
{
    TIDlgPreferencesPrivate *priv = TI_DLG_PREFERENCES_GET_PRIVATE (self);
    gboolean is_active;

    // Display mode
    is_active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->radiobutton_contact_display_handle));

    if (is_active)
    {
        ti_preferences_set_contact_handle_display_mode (priv->preferences, TI_PREFERENCES_CONTACT_HANDLE_DISPLAY_HANDLE);
    }
    else
    {
        ti_preferences_set_contact_handle_display_mode (priv->preferences, TI_PREFERENCES_CONTACT_HANDLE_DISPLAY_NAME);
    }

    // Auto inspect handles
    is_active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->checkbutton_auto_inspect_handles));
    ti_preferences_set_auto_inspect_handles (priv->preferences, is_active);
}
