/*
 * This file is part of telepathy-idle
 * 
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License 
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "idle-dns-resolver.h"

#define SU_ROOT_MAGIC_T IdleDNSResolver
#include <sofia-sip/su_source.h>
#include <sofia-sip/su_wait.h>
#define HAVE_SU_WAIT_H 1
#include <sofia-sip/sresolv.h>

#include <glib.h>

#define idle_dns_result_new() \
	(g_slice_new(IdleDNSResult))
#define idle_dns_result_new0() \
	(g_slice_new0(IdleDNSResult))

void idle_dns_result_destroy(IdleDNSResult *result)
{
	g_free(result->ai_addr);

	if (result->ai_next)
	{
		idle_dns_result_destroy(result->ai_next);
	}
	
	g_slice_free(IdleDNSResult, result);
}

struct _IdleDNSResolver
{
	sres_resolver_t *resolver;
	su_root_t *root;

	GHashTable *queries;

	guint source_id;

	guint serial;
};

typedef struct _IdleDNSQueryData IdleDNSQueryData;

struct _IdleDNSQueryData
{
	IdleDNSResultCallback callback;
	guint port;
	gpointer user_data;
	guint query_id;
};

#define idle_dns_query_data_new() \
	(g_slice_new(IdleDNSQueryData))
	
#define idle_dns_query_data_new0() \
	(g_slice_new0(IdleDNSQueryData))

static void idle_dns_query_data_destroy(IdleDNSQueryData *data)
{
	g_slice_free(IdleDNSQueryData, data);
}

IdleDNSResolver *idle_dns_resolver_new()
{
	IdleDNSResolver *ret = g_slice_new(IdleDNSResolver);
	GSource *source;

	ret->root = su_root_source_create(ret);
	g_assert(ret->root);
	su_root_threading(ret->root, FALSE);
	
	source = su_root_gsource(ret->root);
	g_assert(source != NULL);
	ret->source_id = g_source_attach(source, NULL);
	
	ret->resolver = sres_resolver_create(ret->root, NULL, TAG_END());
	ret->queries = g_hash_table_new_full(g_direct_hash, 
										 g_direct_equal, 
										 NULL, 
										 (GDestroyNotify)(idle_dns_query_data_destroy));

	return ret;
}

void idle_dns_resolver_destroy(IdleDNSResolver *res)
{
	if (res->resolver)
	{
		sres_resolver_unref(res->resolver);
		res->resolver = NULL;
	}

	if (res->root)
	{
		su_root_destroy(res->root);
		res->root = NULL;
	}
	
	if (res->source_id)
	{
		g_source_remove(res->source_id);
		res->source_id = 0;
	}
	
	if (res->queries)
	{
		g_hash_table_destroy(res->queries);
		res->queries = NULL;
	}

	g_slice_free(IdleDNSResolver, res);
}

struct _call_callback_helper
{
	guint id;
	IdleDNSResult *results;
	IdleDNSResultCallback callback;
	gpointer user_data;
};

static gboolean call_callback_idle(gpointer user_data)
{
	struct _call_callback_helper *helper = user_data;

	helper->callback(helper->id, helper->results, helper->user_data);

	g_free(user_data);
	return FALSE;
}

static void sres_answer_callback(sres_context_t *ctx, sres_query_t *query, sres_record_t **answers)
{
	IdleDNSResolver *resolver = (IdleDNSResolver *)(ctx);
	IdleDNSQueryData *data;
	IdleDNSResult *results = NULL;
	IdleDNSResult *tail = NULL;
	int i;
	IdleDNSResultCallback callback;
	gpointer user_data;
	struct _call_callback_helper *helper = g_new0(struct _call_callback_helper, 1);
	guint id;
	guint port;

	data = g_hash_table_lookup(resolver->queries, query);

	if (!data)
	{
		g_debug("%s: invalid or cancelled context %p, ignoring", G_STRFUNC, ctx);
		return;
	}

	callback = data->callback;
	user_data = data->user_data;
	port = data->port;
	id = data->query_id;

	g_hash_table_remove(resolver->queries, query);
	
	sres_sort_answers(resolver->resolver, answers);

	for (i = 0; answers && answers[i] != NULL; i++)
	{
		IdleDNSResult *result;
		int ai_family;
		int ai_socktype = SOCK_STREAM;
		int ai_protocol = 0;
		struct sockaddr *ai_addr;
		socklen_t ai_addrlen;

		switch (answers[i]->sr_record->r_type)
		{
			case sres_type_a:
			{
				struct sockaddr_in *sin;
				
				sin = g_new(struct sockaddr_in, 1);
				
				sin->sin_family = ai_family = AF_INET;
				sin->sin_port = port;
				sin->sin_addr = answers[i]->sr_a->a_addr;
				
				ai_addrlen = sizeof(struct sockaddr_in);

				ai_addr = (struct sockaddr *)(sin);
			};
			break;
			case sres_type_a6:
			{
				g_debug("%s: FIXME IMPLEMENTME IAMTHEIPV6IGNORERBASTARD", G_STRFUNC);
				continue;
			};
			break;
			default:
			{
				g_debug("%s: unsupported address family %u encountered, ignoring", G_STRFUNC, answers[i]->sr_record->r_type);
				continue;
			}
			break;
		}

		result = idle_dns_result_new0();

		result->ai_family = ai_family;
		result->ai_socktype = ai_socktype;
		result->ai_protocol = ai_protocol;

		result->ai_addr = ai_addr;
		result->ai_addrlen = ai_addrlen;
		
		if (tail)
		{
			tail->ai_next = result;
		}

		if (!results)
		{
			results = result;
		}

		tail = result;
	}
	
	sres_free_answers(resolver->resolver, answers);

	/*
	 * FIXME this sucks. We have to return from this function before we can destroy the resolver (which calling of callback can lead to) so we need to trampoline it with g_idle_add 
	 */

	helper->callback = callback;
	helper->id = id;
	helper->results = results;
	helper->user_data = user_data;

	g_idle_add(call_callback_idle, helper);
}

guint idle_dns_resolver_query(IdleDNSResolver *resolver, const gchar *name, guint port, IdleDNSResultCallback callback, gpointer user_data)
{
	IdleDNSQueryData *data;
	guint query_id = resolver->serial++;
	sres_query_t *sofia_query;

	g_debug("%s: resolving %s:%u", G_STRFUNC, name, port);

	sofia_query = sres_query(resolver->resolver,
							 sres_answer_callback, 
							 (sres_context_t *)(resolver),
							 sres_type_a,
							 name);

	data = idle_dns_query_data_new();

	data->callback = callback;
	data->user_data = user_data;
	data->query_id = query_id;
	data->port = htons((unsigned short)(port));

	g_hash_table_insert(resolver->queries, sofia_query, data);

	return query_id;
}

static gboolean queries_remove_foreach_func(gpointer key, gpointer value, gpointer user_data)
{
	IdleDNSQueryData *data = (IdleDNSQueryData *)(value);

	return (data->query_id = GPOINTER_TO_UINT(user_data));
}

void idle_dns_resolver_cancel_query(IdleDNSResolver *resolver, guint query_id)
{
	g_hash_table_foreach_remove(resolver->queries, queries_remove_foreach_func, GUINT_TO_POINTER(query_id));
}
