#include <string.h>
#include <lua.h>
#include <lauxlib.h>
#include <stdlib.h>
#include <time.h>

#include "platform.h"
#include "path.h"
#include "context.h"

#ifdef BAM_FAMILY_WINDOWS
	#define WIN32_LEAN_AND_MEAN
	#define VC_EXTRALEAN
	#include <windows.h>
	#include <sys/types.h>
	#include <sys/stat.h>
	#include <signal.h>

	#define _stat stat

	/* Windows code */
	static int listdir(lua_State *L)
	{
		WIN32_FIND_DATA finddata;
		HANDLE handle;
		const char *path = 0;
		char buffer[1024*2];
		int i = 1;

		/* create the table */
		lua_newtable(L);

		if (strlen(lua_tostring(L, 1)) < 1)
			path = context_get_path(L);
		else
			path = lua_tostring(L, 1);

		strcpy(buffer, path);
		strcat(buffer, "/*");

		handle = FindFirstFileA(buffer, &finddata);

		if (handle == INVALID_HANDLE_VALUE)
			return 0;

		/* add all the entries */
		do
		{
			path_join(path, finddata.cFileName, buffer, 1024*2);
			lua_pushstring(L, buffer);
			lua_rawseti(L, -2, i++);
		} while (FindNextFileA(handle, &finddata));

		FindClose(handle);

		return 1;
	}

	/* signals. should be moved to platform.c or similar? */
	void install_signals(void (*abortsignal)(int))
	{
		signal(SIGINT, abortsignal);
		signal(SIGBREAK, abortsignal);
	}

	static CRITICAL_SECTION criticalsection;
	void platform_init()
	{
		InitializeCriticalSection(&criticalsection);
	}

	void platform_shutdown()
	{
		DeleteCriticalSection(&criticalsection);
	}

	void criticalsection_enter()
	{
		EnterCriticalSection(&criticalsection);
	}

	void criticalsection_leave()
	{
		LeaveCriticalSection(&criticalsection);
	}

	void *threads_create(void (*threadfunc)(void *), void *u)
	{
		return CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE)threadfunc, u, 0, NULL);
	}

	void threads_wait(void *thread)
	{
		WaitForSingleObject((HANDLE)thread, INFINITE);
	}

	void threads_destroy(void *thread)
	{
	}

	void threads_yield()
	{
		Sleep(1);
	}

#else
	#include <dirent.h>
	#include <unistd.h>
	#include <sys/types.h>
	#include <sys/signal.h>
	#include <sys/stat.h>
	#include <pthread.h>

	/* NIX code */
	static int listdir(lua_State *L)
	{
		DIR *dir;
		struct dirent *entry;
		char buffer[1024*2];
		int i;
		char *startpoint;

		/* create the table */
		lua_newtable(L);

		/* add all the entries */
		i = 1;
		if(strlen(lua_tostring(L, 1)) < 1)
		{
			strcpy(buffer, context_get_path(L));
			dir = opendir(buffer);
		}
		else
		{
			path_join(context_get_path(L), lua_tostring(L,1), buffer, 1024*2);
			dir = opendir(buffer);
		}

		if(!dir)
			return 0;
		
		/* get starting point and append a slash */
		startpoint = buffer + strlen(buffer);
		*startpoint = '/';
		startpoint++;
			
		while((entry = readdir(dir)) != NULL)
		{
			/* make the path absolute */
			strcpy(startpoint, entry->d_name);
			path_normalize(startpoint);
			
			/* push the string and continue the search */
			lua_pushstring(L, buffer);
			lua_rawseti(L, -2, i++);
		}

		/* close the directory and return */
		closedir(dir);
		return 1;
	}

	/* signals. should be moved to platform.c or similar? */
	void install_signals(void (*abortsignal)(int))
	{
		signal(SIGHUP, abortsignal);
		signal(SIGINT, abortsignal);
		signal(SIGKILL, abortsignal);
	}

	static pthread_mutex_t lock_mutex = PTHREAD_MUTEX_INITIALIZER;

	void platform_init()
	{
	}

	void platform_shutdown()
	{
	}

	void criticalsection_enter()
	{
		pthread_mutex_lock(&lock_mutex);
	}

	void criticalsection_leave()
	{
		pthread_mutex_unlock(&lock_mutex);
	}
	void *threads_create(void (*threadfunc)(void *), void *u)
	{
		pthread_t id;
		pthread_create(&id, NULL, (void *(*)(void*))threadfunc, u);
		return (void*)id;
	}

	void threads_wait(void *thread)
	{
		pthread_join((pthread_t)thread, NULL);
	}

	void threads_destroy(void *thread)
	{
	}

	void threads_yield()
	{
		sched_yield();
	}	
#endif

int timestamp() { return time(NULL); }
	
#ifdef BAM_PLATFORM_MACOSX
	/* Mac OS X version */
	int file_timestamp(const char *filename)
	{
		struct stat s;
		if(stat(filename, &s) == 0)
			return s.st_mtimespec.tv_sec;
		return 0;
	}
#else
	/* NIX and Windows version */
	int file_timestamp(const char *filename)
	{
		struct stat s;
		if(stat(filename, &s) == 0)
			return s.st_mtime;
		return 0;
	}
#endif

/* general */
static int simple_print(lua_State *L)
{
	printf(lua_tostring(L, 1));
	fflush(stdout);
	return 0;
}

int file_exist(const char *filename)
{
	struct stat s;
	if(stat(filename, &s) == 0)
		return 1;
	return 0;
}

static const luaL_reg supportlib[] ={
	{"listdir",		listdir},
	{"simpleprint", simple_print},
	{NULL,NULL}
};

int support_luaopen(lua_State *L)
{
	luaL_openlib(L, "support", supportlib, 0);
	return 0;
}
