/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *  @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#include "config.h"

#include "QtTapioca/Handle"

using namespace QtTapioca;

Handle::Handle(org::freedesktop::Telepathy::Connection *conn, uint type, const QString &name, QObject *parent)
    : QObject(parent),
      m_conn(conn),
      m_type(type),
      m_id(request(name)),
      m_isValid(m_id != 0)
{
    /* no need to hold the handle as we are requesting it */
}

Handle::Handle(org::freedesktop::Telepathy::Connection *conn, uint type, uint id, QObject *parent)
    : QObject(parent),
      m_conn(conn),
      m_type(type),
      m_id(id),
      m_isValid(id != 0)
{
    hold();
}

Handle::~Handle()
{
    release();
}

bool Handle::isValid() const
{
    return m_isValid;
}

QString Handle::inspect()
{
    if (!m_isValid)
       return QString();

    QList<uint> handleList;
    handleList << m_id;

    QStringList ls = m_conn->InspectHandles(static_cast<uint>(m_type), handleList);

    return (ls.isEmpty() ? "" : ls.first());
}

uint Handle::id() const
{
    return m_id;
}

uint Handle::type() const
{
    return m_type;
}

void Handle::hold()
{
    if (!m_isValid)
        return;

    QList<uint> handleList;
    handleList << m_id;
    m_conn->HoldHandles(static_cast<uint>(m_type), handleList);
}

void Handle::release()
{
    if (!m_isValid)
        return;

    QList<uint> handleList;
    handleList << m_id;
    m_conn->ReleaseHandles(static_cast<uint>(m_type), handleList);
}

uint Handle::request(const QString &name)
{
    uint handle = 0;

    QStringList names(name);
    QDBusReply<QList<uint> > handleList = m_conn->RequestHandles(static_cast<uint>(m_type), names);
    if (!handleList.isValid())
        return 0;

    if (static_cast<QList<uint> >(handleList).size() > 0)
        handle = static_cast<QList<uint> >(handleList).first();

    return handle;
}

bool Handle::operator==(const Handle &handle) const
{
    return ((m_id == handle.id()) && (m_type == handle.type()));
}

