#ifndef TAGCOLL_OPTIONS_H
#define TAGCOLL_OPTIONS_H

/*
 * Commandline parser for tagcoll
 *
 * Copyright (C) 2003,2004,2005,2006  Enrico Zini
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/Commandline.h>

namespace Tagcoll {
namespace commandline {

struct TagcollOptions : public MainParser<CommandParser>
{
	struct HelpGroup : public OptionGroup
	{
		BoolOption* help;
		BoolOption* version;

		HelpGroup()
		{
			add(help = new BoolOption("help", 'h', "help"));
			add(version = new BoolOption("version", 'V', "version"));
			help->shortNames.push_back('?');
			help->description = "print an help message and exit";
			version->description = "print the program version and exit";
			description = "Help options";
		}
		~HelpGroup()
		{
			delete help; delete version;
		}
	} helpGroup;

	struct InputGroup : public OptionGroup
	{
		ExistingFileOption* derived;
		ExistingFileOption* extimpl;
		ExistingFileOption* rename;
		ExistingFileOption* patch;
		BoolOption* rmunfaceted;
		StringOption* rmtags;

		InputGroup()
		{
			add(derived		= new ExistingFileOption("derived-tags-from", 0, "derived"));
			add(extimpl		= new ExistingFileOption("extimpl", 0, "implications-from"));
			add(rename		= new ExistingFileOption("rename", 0, "rename-from"));
			add(patch		= new ExistingFileOption("patch", 'p', "patch-with"));
			add(rmunfaceted	= new BoolOption("rmunfaceted", 0, "remove-unfaceted"));
			add(rmtags		= new StringOption("rmtags", 0, "remove-tags"));

			derived->description = "expand derived tags using the given list";
			extimpl->description = "use an external list of implications";
			rename->description = "rename tags using the given mapping list";
			patch->description = "apply patches from the given tag patch file";
			patch->longNames.push_back("patch");
			rmunfaceted->description = "while parsing, remove all tags with no facet part";
			rmtags->usage = "<expression>";
			rmtags->description = "while parsing, remove all tags matching the given tag expression";
			description = "Options controlling transformations of input data";
		}
		~InputGroup()
		{
			delete derived; delete extimpl; delete rename; delete patch;
			delete rmunfaceted; delete rmtags;
		}
	} inputGroup;

	struct OutputGroup : public OptionGroup
	{
		BoolOption* group;
		BoolOption* redundant;
		BoolOption* itemsOnly;
		
		OutputGroup()
		{
			add(group		= new BoolOption("group", 'g', "group"));
			add(redundant	= new BoolOption("redundant", 0, "redundant"));
			add(itemsOnly	= new BoolOption("items", 'i', "items"));
			group->description = "group items with the same tagset in the output collection";
			group->longNames.push_back("group-items");
			redundant->description = "when implications are provided, expand them explicitly in the output";
			itemsOnly->description = "output only the names of the items, without the tags";
			description = "Options controlling transformations of output data";
		}
		~OutputGroup()
		{
			delete group;
			delete redundant;
		}
	} outputGroup;

	struct HierarchyGroup : public OptionGroup
	{
		IntOption* flatten;
		IntOption* filter;

		HierarchyGroup()
		{
			add(flatten = new IntOption("flatten", 0, "flatten-threshold"));
			add(filter = new IntOption("filter", 'f', "filter"));

			flatten->description = "set the number of total items below which a branch is flattened when using the \"hierarchy\" command (defaults to 0, meaning \"don't flatten\")";
			filter->description = "filter out the tags with cardinality less than the given value (defaults to not filter; currently only works when building hierarchies)";

			description = "Options controlling generation of tag hierarchies";
		}
		~HierarchyGroup()
		{
			delete flatten;
			delete filter;
		}
	} hierarchyGroup;

	struct Generic : public OptionParser
	{
		Generic(TagcollOptions* cp) : OptionParser("")
		{
			add(&cp->helpGroup);
		}
	} generic;
	struct Help : public OptionParser
	{
		Help(TagcollOptions* cp) : OptionParser("help")
		{
			usage = "[command]";
			description = "print help informations";
		}
	} help;
	struct Copy : public OptionParser
	{
		Copy(TagcollOptions* cp) : OptionParser("copy")
		{
			usage = "[files...]";
			description = "output the collection";
			longDescription = 
				"Output the normalized collection on standard output, applying transformations "
				"if requested.  This is the default action if no other switches are provided.  "
				"A normalized collection is a collection in which an item appears in just one "
				"line.";
			add(&cp->inputGroup);
			add(&cp->outputGroup);
			add(&cp->helpGroup);
			aliases.push_back("cat");
		}
	} copy;
	struct Reverse : public OptionParser
	{
		StringOption* untaggedTag;

		Reverse(TagcollOptions* cp) : OptionParser("reverse")
		{
			add(untaggedTag = new StringOption("untagged-tag", 0, "untagged-tag"));
			untaggedTag->usage = "<tag>";
			untaggedTag->description = "set item name to use for associating untagged items when using the \"reverse\" command.  If not specified, untagged items are not included in the output";
				
			usage = "[files...]";
			description = "\"reverse\" the collection, outputting one with items associated to tags";
			longDescription =
				"Output the inbound collection \"reversed\" from the tags point of view, that is, "
				"associating to each tag the list of items associated to it in the input.\n"
				"The --untagged-tag switch can be used to provide a name to which untagged "
				"items will be associated in the output.";
			add(&cp->inputGroup);
			add(&cp->outputGroup);
			add(&cp->helpGroup);
		}
		~Reverse()
		{
			delete untaggedTag;
		}
	} reverse;
	struct Diff : public OptionParser
	{
		Diff(TagcollOptions* cp) : OptionParser("diff")
		{
			usage = "<file1> <file2>";
			description = "output a tag patch file with the differences between two files";
			longDescription =
				"Output a tag patch file with the differences between two files (requires two "
				"file arguments).\n"
				"The output tag patch file can then be applied when reading a collection with "
				"the --patch-with option.";
			add(&cp->helpGroup);
		}
	} diff;
	struct Related : public OptionParser
	{
		IntOption* distance;
		
		Related(TagcollOptions* cp) : OptionParser("related")
		{
			add(distance = new IntOption("distance", 'd', "distance"));
			distance->description = "set the maximum distance to use for the \"related\" command (defaults to 0)";
			
			usage = "<item> [files...]";
			description = "print a list of items related to the given one";
			longDescription =
				"Output a list of the items that are related to the given item or list of items.  "
				"If more than one items are to be specified, separate them with commas.\n"
				"The --distance option can be used to control how closely related the output "
				"items shold be from the item(s) specified.";
			examples = "tagcoll related mutt,mozilla-browser -";

			add(&cp->inputGroup);
			add(&cp->helpGroup);
		}
		~Related()
		{
			delete distance;
		}
	} related;
	struct Implications : public OptionParser
	{
		Implications(TagcollOptions* cp) : OptionParser("implications")
		{
			usage = "[files...]";
			description = "compute a list of tag implications";
			longDescription =
				"Output a list of all implicit implications between tags contained in the "
				"hierarchy.  Implication is defined such that tag A implies tag B if every item "
				"tagged with A is also tagged with B.\n"
				"Implications can be used to discover implicit hierarchical relationships "
				"between tags.\n"
				"The output is one line per tag, with just tags that have implications, with the "
				"name of the package, a colon and a comma-separated list of all implied tags.";
			examples =
				"C:devel,languages\n"
				"ada:devel,languages\n"
				"apachemodules:net,servers,web\n"
				"browsers:net,web\n";
			add(&cp->inputGroup);
			add(&cp->helpGroup);
		}
	} implications;
	struct Hierarchy : public OptionParser
	{
		Hierarchy(TagcollOptions* cp) : OptionParser("hierarchy")
		{
			usage = "[files...]";
			description = "build a smart hierarchy with the collection data";
			longDescription =
				"Organize the collection in an intuitively navigable hierarchy.  The "
				"output is one line per package, with the package name prepended by the "
				"path to the item in the hierarchy.\n"
				"A detailed description of the hierarchy generation algorithm is found in the "
				"tagbk-draft.pdf draft paper available in this package; if you want to "
				"understand what are the goals of the algorithm and how it works, please give it "
				"a read.";
			examples =
				"/net/clients/mail: mutt\n"
				"/net/filters/mail: procmail\n";
			add(&cp->inputGroup);
			add(&cp->hierarchyGroup);
			add(&cp->helpGroup);
		}
	} hierarchy;

	struct CleanHierarchy : public OptionParser
	{
		CleanHierarchy(TagcollOptions* cp) : OptionParser("cleanhierarchy")
		{
			usage = "[files...]";
			description = "build a cleaned smart hierarchy with the collection data";
			longDescription =
				"Like hiearchy, but in every node it merges tags which are attached to the "
				"same set of items.";
			add(&cp->inputGroup);
			add(&cp->hierarchyGroup);
			add(&cp->helpGroup);
		}
	} cleanhierarchy;
	struct FindSpecials : public OptionParser
	{
		FindSpecials(TagcollOptions* cp) : OptionParser("findspecials")
		{
			usage = "[files...]";
			description =
				"generate a smart hierarchy and print, for each toplevel tag, "
				"what are the items that make it toplevel instead of going below "
				"another tag";
			add(&cp->inputGroup);
			add(&cp->hierarchyGroup);
			add(&cp->helpGroup);
		}
	} findspecials;
	struct Grep : public OptionParser
	{
		Grep(TagcollOptions* cp) : OptionParser("grep")
		{
			usage = "<expression> [files...]";
			description = "output the collection of tags that match the given tag expression";
			add(&cp->inputGroup);
			add(&cp->outputGroup);
			add(&cp->helpGroup);
		}
	} grep;

	TagcollOptions() : MainParser<CommandParser>("tagcoll"),
		generic(this),
		help(this), copy(this), reverse(this), diff(this), related(this), implications(this),
		hierarchy(this), cleanhierarchy(this), findspecials(this), grep(this)
	{
		add(generic);
		add(help);
		add(copy);
		add(reverse);
		add(diff);
		add(related);
		add(implications);
		add(hierarchy);
		add(cleanhierarchy);
		add(findspecials);
		add(grep);

		usage = "<command> [options and arguments]";
		description = "Perform various operations on a tagged collection";
	}

#if 0
		//opts.add("verbose", 'v', "verbose", "enable verbose output");
		//opts.add("debug", 0, "debug", "enable debugging output (including verbose output)");
#endif
};

}
}

// vim:set ts=4 sw=4:
#endif
