/**
 * @file test-utils.h
 * @author Peter Rockai (mornfall) <mornfall@danill.sk>
 * @brief Utility functions for the unit tests
 */
#include <tut.h>

#include <string>
#include <tagcoll/stringf.h>

#define TEST_TAGCOLL

#ifdef TEST_TAGCOLL
#include <tagcoll/Collection.h>
#endif
/*
#include <apt-front/cache.h>
#include <apt-pkg/configuration.h>
#include <apt-pkg/error.h>
#include <apt-pkg/pkgsystem.h>
#include <apt-pkg/init.h>
#include <iostream>
*/

#define TESTGRP(name) \
typedef test_group<name ## _shar> tg; \
typedef tg::object to; \
tg name ## _tg (#name);


namespace tut_tagcoll {
using namespace std;
using namespace stringf;
using namespace Tagcoll;
using namespace tut;

template<class ITEM, class TAG>
class TestConsumer : public Tagcoll::Consumer<ITEM, TAG>
{
protected:
	virtual void consumeItemUntagged(const ITEM& item) { items++; }

	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags)
	{
		items++;
		this->tags += tags.size();
	}
	
public:
	int items;
	int tags;

	TestConsumer() : items(0), tags(0) {}
};

class Location
{
	string file;
	int line;
	string str;
	string testfile;
	int testline;
	string teststr;
public:
	Location(const std::string& file, int line, const std::string& str)
		: file(file), line(line), str(str) {}
	Location(const Location& loc,
		 const std::string& testfile, int testline, const std::string& str) :
		file(loc.file), line(loc.line), str(loc.str),
		testfile(testfile), testline(testline), teststr(str) {}
	string locstr() const
	{
		std::stringstream ss;
		ss << file << ":" << line << "(" << str << ")";
		if (!testfile.empty())
			ss << "[" << testfile << ":" << testline << "(" << teststr << ")]";
		ss << ": ";
		return ss.str();
	}
	string msg(const std::string m) const
	{
		return locstr() + ": " + m;
	}
};

void outputCollection(const std::string& str, Tagcoll::Consumer<string, string>& cons);

#define gen_ensure(x) _gen_ensure(Location(__FILE__, __LINE__, #x), (x))
#define inner_ensure(x) _gen_ensure(Location(loc, __FILE__, __LINE__, #x), (x))
void _gen_ensure(const Location& loc, bool res);

#define gen_ensure_equals(x, y) my_ensure_equals(Location(__FILE__, __LINE__, #x " == " #y), (x), (y))
#define inner_ensure_equals(x, y) my_ensure_equals(Location(loc, __FILE__, __LINE__, #x " == " #y), (x), (y))
template <class T,class Q>
void my_ensure_equals(const Location& loc, const Q& actual, const T& expected)
{
	if( expected != actual )
	{
		std::stringstream ss;
		ss << "expected " << expected << " actual " << actual;
		throw failure(loc.msg(ss.str()));
	}
}

#define gen_ensure_coll_equals(a, b) \
	__tc_ensure_coll_equals(Location(__FILE__, __LINE__, #a " == " #b), a, b)
#define inner_ensure_coll_equals(a, b) \
	__tc_ensure_coll_equals(Location(loc, __FILE__, __LINE__, #a " == " #b), a, b)
void __tc_ensure_coll_equals(const Location& loc,
		const Tagcoll::ReadonlyCollection<string, string>& c1,
		const Tagcoll::ReadonlyCollection<string, string>& c2);

#define gen_ensure_contains(a, b) \
	_ensure_contains(Location(__FILE__, __LINE__, #a " contains " #b), a, b)
#define inner_ensure_contains(a, b) \
	_ensure_contains(Location(loc, __FILE__, __LINE__, #a " contains " #b), a, b)
template<class T>
void _ensure_contains(const Location& loc, const OpSet<T>& s, const T& item)
{
	if (!s.contains(item))
	{
		std::stringstream ss;
		ss << "tagset ";
		for (typename OpSet<T>::const_iterator i = s.begin();
				i != s.end(); i++)
			if (i == s.begin())
				ss << *i;
			else
				ss << ", " << *i;
		ss << " does not contain " << item;
		
		throw failure(loc.msg(ss.str()));
	}
}

#define gen_ensure_not_contains(a, b) \
	_ensure_not_contains(Location(__FILE__, __LINE__, #a " contains " #b), a, b)
#define inner_ensure_not_contains(a, b) \
	_ensure_not_contains(Location(loc, __FILE__, __LINE__, #a " contains " #b), a, b)
template<class T>
void _ensure_not_contains(const Location& loc, const OpSet<T>& s, const T& item)
{
	if (s.contains(item))
	{
		std::stringstream ss;
		ss << "tagset ";
		for (typename OpSet<T>::const_iterator i = s.begin();
				i != s.end(); i++)
			if (i == s.begin())
				ss << *i;
			else
				ss << ", " << *i;
		ss << " does not contain " << item;
		
		throw failure(loc.msg(ss.str()));
	}
}

#ifdef TEST_TAGCOLL

// Output a test collection for later testing with test_tagged_collection_ro

#define output_test_collection(x) (__output_test_collection(Location(__FILE__, __LINE__, #x), (x)))
#define inner_output_test_collection(x) (__output_test_collection(Location(loc, __FILE__, __LINE__, $x), (x)))
void __output_test_collection(const Location& loc, Consumer<string, string>& tc);

#define test_readonly_collection(x) (__test_readonly_collection(Location(__FILE__, __LINE__, #x), (x)))
#define inner_test_readonly_collection(x) (__test_readonly_collection(Location(loc, __FILE__, __LINE__, #x), (x)))
void __test_readonly_collection(const Location& loc, ReadonlyCollection<string, string>& tc);


#define test_collection(x) (__test_collection(Location(__FILE__, __LINE__, #x), (x)))
void __test_collection(const Location& loc, Collection<string, string>& tc);

#endif
}

/*
namespace tut {
    static void aptInit () {
        pkgInitConfig (*_config);
        _config->Set("Dir", CACHE_DIR);
        _config->Set("Dir::Cache", "cache");
        _config->Set("Dir::State", "state");
        _config->Set("Dir::Etc", "etc");
        _config->Set("Dir::State::status", CACHE_DIR "dpkg-status");
        pkgInitSystem (*_config, _system);
        // _config -> Set ("Capture::Cache::UseExtState", extstate);
    }
}
*/
