#ifndef TAGCOLL_OPSET_H
#define TAGCOLL_OPSET_H

/** \file
 * std::set with operators overridden with set operations
 */

/*
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

/* TODO: replace + with | and ^ with &, since logical operators better
 * correspond to set operations */

#include <set>

namespace Tagcoll
{

/**
 * OpSet is just the standard std::set extended with set operations.
 *
 * The reason for not using a plain std::set is that the libtagcoll code
 * involves a lot of set operations, and overridden operators greatly help
 * in having cleaner code.
 *
 * Example:
 * \code
 *  OpSet<string> myfavs;
 *  OpSet<string> yourfavs;
 *  myfavourite += "pear";
 *  myfavourite += "banana";
 *  yourfavourite += "apple";
 *  yourfavourite += "pear";
 *  OpSet<string> ourfavs = myfavs ^ yourfavs;
 *  OpSet<string> interesting = myfavs + yourfavs;
 *  OpSet<string> myonlyfavs = myfavs - yourfavs;
 *  for (OpSet<string>::const_iterator i = ourfavs.begin();
 *       i != ourfavs.end(); i++)
 *     cout << *i << endl;
 * \endcode
 */
template<class T>
class OpSet : public std::set<T>
{
public:
	using std::set<T>::begin;
	using std::set<T>::end;

	OpSet() : std::set<T>() {}

	template<typename A, typename B>
	OpSet(A a, B b) : std::set<T>(a, b) {}

	/** Return true if the tag set contains tag, else false */
	bool contains(const T& item) const { return this->find(item) != this->end(); }

	/** Return true if the tag set contains ts, else false */
	bool contains(const OpSet<T>& ts) const;

	/** 
	 * @brief Calculates the distance between two tagsets.
	 * 
	 * The distance between A and B is defined by infinity if the 
	 * intersection between A und B is empty, else it is 
	 * \f$|(A \cup B) \setminus (A \cap B)|\f$
	 * @returns the distance between the two tagsets or -1 if the distance 
	 * is infinity
	 */
	int distance(const OpSet<T>& ts) const;

	/**
	 * Singleton union
	 * @return the set union of this set and the singleton set {tag}
	 */
	OpSet<T> operator+(const T& tag) const;

	/**
	 * Singleton union
	 * @return the set union of this set and the singleton set {tag}
	 */
	OpSet<T>& operator+=(const T& ts);

	/**
	 * Set union
	 * @return the set union of this set and the set ts
	 */
	OpSet<T> operator+(const OpSet<T>& ts) const;

	/**
	 * Singleton union
	 * @return the set union of this set and the singleton set {tag}
	 */
	OpSet<T>& operator+=(const OpSet<T>& ts);

	/**
	 * Singleton difference
	 * @return the set difference of this set and the singleton set {tag}
	 */
	OpSet<T> operator-(const T& tag) const;

	/**
	 * Singleton difference
	 * @return the set difference of this set and the singleton set {tag}
	 */
	OpSet<T>& operator-=(const T& tag);

	/**
	 * Set difference
	 * @return the set difference of this set and the set ts
	 */
	OpSet<T> operator-(const OpSet<T>& ts) const;

	/**
	 * Set difference
	 * @return the set difference of this set and the set ts
	 */
	OpSet<T>& operator-=(const OpSet<T>& ts);

	/**
	 * Set intersection
	 * @return the set intersection of this set and the set ts
	 */
	OpSet<T> operator^(const OpSet<T>& ts) const;

	/**
	 * Set intersection
	 * @return the set intersection of this set and the set ts
	 */
	OpSet<T>& operator^=(const OpSet<T>& ts);
};

};

// vim:set ts=4 sw=4:
#endif
