#ifndef TAGCOLL_DERIVEDTAGS_H
#define TAGCOLL_DERIVEDTAGS_H

/** \file
 * Model a collection of derived tags and a TagcollFilter to add them to a
 * collection
 */

/*
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/Filter.h>
#include <tagcoll/Expression.h>

#include <map>
#include <string>

#include <stdio.h>

namespace Tagcoll
{

/**
 * List of changes to apply to tag names with a tagged collection is being
 * parsed
 */
class DerivedTags : public std::map<std::string, Expression>
{
public:
	virtual ~DerivedTags() {}

	/**
	 * Add a new derived tag to the list
	 */
	void add(const std::string& tag, const Expression& expr)
	{
		insert(make_pair(tag, expr));
	}

	/**
	 * Return the tags in `tags', plus all the derived tags that matched `tags'
	 */
	OpSet<std::string> derivedTags(const OpSet<std::string>& tags) const
	{
		OpSet<std::string> res;
		for (const_iterator i = begin(); i != end(); i++)
		{
			TagexprContext context(tags, *this);
			if (i->second(context))
				res += i->first;
		}
		return res;

	}
};

/**
 * Filter that adds derived tags to a collection
 */
template <class ITEM>
class AddDerived : public Filter<ITEM, std::string>
{
protected:
	DerivedTags dtags;

	virtual void consumeItemUntagged(const ITEM& item)
	{
		this->consumer->consume(item, dtags.derivedTags(OpSet<std::string>()));
	}
	virtual void consumeItem(const ITEM& item, const OpSet<std::string>& tags)
	{
		this->consumer->consume(item, tags + dtags.derivedTags(tags));
	}
	virtual void consumeItemsUntagged(const OpSet<ITEM>& items)
	{
		this->consumer->consume(items, dtags.derivedTags(OpSet<std::string>()));
	}
	virtual void consumeItems(const OpSet<ITEM>& items, const OpSet<std::string>& tags)
	{
		this->consumer->consume(items, tags + dtags.derivedTags(tags));
	}
	
public:
	AddDerived() {}
	AddDerived(Consumer<ITEM, std::string>& cons) : Filter<ITEM, std::string>(cons) {}
	AddDerived(const DerivedTags& dtags) : dtags(dtags) {}
	AddDerived(Consumer<ITEM, std::string>& cons, const DerivedTags& dtags)
		: Filter<ITEM, std::string>(cons), dtags(dtags) {}
	virtual ~AddDerived() {}

	/**
	 * Access the internal DerivedTags table
	 */
	DerivedTags& derivedTags() { return dtags; }

	/**
	 * Access the internal DerivedTags table (const version)
	 */
	const DerivedTags& derivedTags() const { return dtags; }
};

/**
 * Filter that removes redundant derived tags from a collection
 */
template <class ITEM>
class RemoveDerived : public Filter<ITEM, std::string>
{
protected:
	DerivedTags dtags;

	virtual void consumeItemUntagged(const ITEM& item)
	{
		this->consumer->consume(item);
	}
	virtual void consumeItem(const ITEM& item, const OpSet<std::string>& tags)
	{
		this->consumer->consume(item, tags - dtags.derivedTags(tags));
	}
	virtual void consumeItemsUntagged(const OpSet<ITEM>& items)
	{
		this->consumer->consume(items);
	}
	virtual void consumeItems(const OpSet<ITEM>& items, const OpSet<std::string>& tags)
	{
		this->consumer->consume(items, tags - dtags.derivedTags(tags));
	}

public:
	RemoveDerived() {}
	RemoveDerived(Consumer<ITEM, std::string>& cons) : Filter<ITEM, std::string>(cons) {}
	RemoveDerived(const DerivedTags& dtags) : dtags(dtags) {}
	RemoveDerived(Consumer<ITEM, std::string>& cons, const DerivedTags& dtags)
		: Filter<ITEM, std::string>(cons), dtags(dtags) {}
	virtual ~RemoveDerived() {}

	/**
	 * Access the internal DerivedTags table
	 */
	DerivedTags& derivedTags() { return dtags; }

	/**
	 * Access the internal DerivedTags table (const version)
	 */
	const DerivedTags& derivedTags() const { return dtags; }
};

};

// vim:set ts=4 sw=4:
#endif
