/*
 * Fast index for tag data
 *
 * Copyright (C) 2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/Serializer.h>
#include <tagcoll/IntDiskIndex.h>
#include <tagcoll/StringIndex.h>
#include <tagcoll/BasicStringDiskIndex.h>
#include <tagcoll/BasicStringDiskIndex.h>
#include <unistd.h>
#include <errno.h>
#include <iostream>

using namespace std;
using namespace Tagcoll;


BasicStringDiskIndex::BasicStringDiskIndex(const std::string& file)
	: MasterMMapIndex(file), IntDiskIndex<std::string, std::string>(
			*this, 0, 1, &pkgs, &tags, &pkgs, &tags),
	  pkgs(*this, 2), tags(*this, 3)
{
}

void BasicStringDiskIndexer::consumeItemUntagged(const string& item)
{
	cache.consume(item);
	pkgs.map(item.c_str());
}

void BasicStringDiskIndexer::consumeItem(const string& item, const OpSet<string>& tags)
{
	cache.consume(item, tags);
	pkgs.map(item.c_str());
	for (OpSet<string>::iterator i = tags.begin(); i != tags.end(); i++)
		this->tags.map(i->c_str());
}

void BasicStringDiskIndexer::write(const std::string& file)
{
	MasterMMapIndexer master(file);

	// Feed the mapped strings to the IntDiskIndexer
	IntDiskIndexer<string, string> idx(pkgs, tags);
	cache.output(idx);

	// Append package index and tag index
	master.append(idx.pkgIndexer());
	master.append(idx.tagIndexer());

	// Then append the two string indexes
	master.append(pkgs);
	master.append(tags);

	// Finally commit
	master.commit();
}


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>
#include <string>
#include <sys/stat.h>
#include <sys/types.h>

namespace tut {
using namespace tut_tagcoll;
using namespace std;

#include <iostream>

static void outts(const OpSet<string>& s)
{
	for (OpSet<string>::const_iterator i = s.begin(); i != s.end(); i++)
		if (i == s.begin())
			cerr << *i;
		else
			cerr << ", " << *i;
}

static const string fname = "test-basicdiskindex-collection";

struct tagcoll_basicstringdiskindex_shar {
	tagcoll_basicstringdiskindex_shar()
	{
		BasicStringDiskIndexer indexer;
		output_test_collection(indexer);
		indexer.write(fname);
	}
	~tagcoll_basicstringdiskindex_shar()
	{
		unlink(fname.c_str());
	}
};
TESTGRP(tagcoll_basicstringdiskindex);

template<> template<>
void to::test<1>()
{
	BasicStringDiskIndex idx(fname);

#if 0
	cerr << "Items: ";
	OpSet<string> s = idx.getTaggedItems();
	outts(s);
	cerr << endl;
	for (OpSet<string>::const_iterator i = s.begin(); i != s.end(); i++)
	{
		cerr << "  " << *i << ": ";
		outts(idx.getTags(*i));
		cerr << endl;
	}


	cerr << "Tags: ";
	s = idx.getAllTags();
	outts(s);
	cerr << endl;
	for (OpSet<string>::const_iterator i = s.begin(); i != s.end(); i++)
	{
		cerr << "  " << *i << ": ";
		outts(idx.getItems(*i));
		cerr << endl;
	}
#endif
	
	test_readonly_collection(idx);
}

}

#endif
// vim:set ts=4 sw=4:
