/***************************************************************************
 *   Copyright (C) 2004 by Ivan Forcada Atienza                            *
 *   ivan@forcada.info                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
 
#include "scanning.h"

void start_scan(char * iface, APList * list)
{
	int skfd;
	
	skfd=iw_sockets_open();
	print_scanning_info(skfd,iface,list);
	close(skfd);
}

//Modified to support APList class. Also uses sdtout to print the scan result
int print_scanning_info(int	skfd, char *	ifname, APList * listAPs)
{
  struct iwreq		wrq;
  unsigned char		buffer[IW_SCAN_MAX_DATA];	/* Results */
  struct timeval	tv;				/* Select timeout */
  int			timeout = 6000000;		/* 5s */

  /* Init timeout value -> 250ms*/
  tv.tv_sec = 0;
  tv.tv_usec = 250000;

  /*
   * Here we should look at the command line args and set the IW_SCAN_ flags
   * properly
   */
  wrq.u.param.flags = IW_SCAN_DEFAULT;
  wrq.u.param.value = 0;		/* Later */

  /* Initiate Scanning */
  if(iw_set_ext(skfd, ifname, SIOCSIWSCAN, &wrq) < 0)
    {
      if(errno != EPERM)
	{
	  fprintf(stderr, "%-8.8s  Interface doesn't support scanning : %s\n\n",
		  ifname, strerror(errno));
	  return(0);
	}
      /* If we don't have the permission to initiate the scan, we may
       * still have permission to read left-over results.
       * But, don't wait !!! */
#if 0
      /* Not cool, it display for non wireless interfaces... */
      fprintf(stderr, "%-8.8s  (Could not trigger scanning, just reading left-over results)\n", ifname);
#endif
      tv.tv_usec = 0;
    }
  timeout -= tv.tv_usec;

  /* Forever */
  while(1)
    {
      fd_set		rfds;		/* File descriptors for select */
      int		last_fd;	/* Last fd */
      int		ret;

      /* Guess what ? We must re-generate rfds each time */
      FD_ZERO(&rfds);
      last_fd = -1;

      /* In here, add the rtnetlink fd in the list */

      /* Wait until something happens */
      ret = select(last_fd + 1, &rfds, NULL, NULL, &tv);

      /* Check if there was an error */
      if(ret < 0)
	{
	  if(errno == EAGAIN || errno == EINTR)
	    continue;
	  fprintf(stderr, "Unhandled signal - exiting...\n");
	  return(0);
	}

      /* Check if there was a timeout */
      if(ret == 0)
	{
	  /* Try to read the results */
	  wrq.u.data.pointer = (char *)buffer;
	  wrq.u.data.flags = 0;
	  wrq.u.data.length = sizeof(buffer);
	  if(iw_get_ext(skfd, ifname, SIOCGIWSCAN, &wrq) < 0)
	    {
	      /* Check if results not available yet */
	      if(errno == EAGAIN)
		{
		  /* Restart timer for only 100ms*/
		  tv.tv_sec = 0;
		  tv.tv_usec = 100000;
		  timeout -= tv.tv_usec;
		  if(timeout > 0)
		    continue;	/* Try again later */
		}

	      /* Bad error */
	      fprintf(stderr, "%-8.8s  Failed to read scan data : %s\n\n",
		      ifname, strerror(errno));
	      return(1);
	    }
	  else
	    /* We have the results, go to process them */
	    break;
	}

      /* In here, check if event and event type
       * if scan event, read results. All errors bad & no reset timeout */
    }

  if(wrq.u.data.length)
    {
      struct iw_event		iwe;
      struct stream_descr	stream;
      int			ap_num = 1;
      int			ret;
      struct iw_range		range;
      int			has_range;
      APEntry* 			NewAP = NULL; 
      
      //int we_version = iw_get_kernel_we_version();
#if 0
      /* Debugging code. In theory useless, because it's debugged ;-) */
      int	i;
      printf("Scan result [%02X", buffer[0]);
      for(i = 1; i < wrq.u.data.length; i++)
	printf(":%02X", buffer[i]);
      printf("]\n");
#endif
      has_range = (iw_get_range_info(skfd, ifname, &range) >= 0);
      printf("%-8.8s  Scan completed :\n", ifname);
      iw_init_event_stream(&stream, (char *)buffer, wrq.u.data.length);
      do
	{
	  /* Extract an event and print it */
	  ret = iw_extract_event_stream(&stream, &iwe);
	  if(ret > 0)
	  {
	    	//ap_num = print_scanning_token(&iwe, ap_num, &range, has_range);
		  char buffer[128];	/* Temporary buffer */
		
		  /* Now, let's decode the event */
		  switch(iwe.cmd)
		    {
		    case SIOCGIWAP:
		      // printf("          Cell %02d - Address: %s\n", ap_num, iw_pr_ether(buffer, (const unsigned char *)iwe.u.ap_addr.sa_data));
		      printf("New AP found(%02d): %s\n", ap_num, iw_pr_ether(buffer, (const unsigned char *)iwe.u.ap_addr.sa_data));
		      // APEntry* NewAP;
		      NewAP = new APEntry;
		      listAPs->AddNew(NewAP);
		      NewAP->setMAC(iw_pr_ether(buffer, (const unsigned char *)iwe.u.ap_addr.sa_data)); ap_num++;
		      break;
		    case SIOCGIWNWID:
/*		      if(iwe.u.nwid.disabled)
			printf("                    NWID:off/any\n");
		      else
			printf("                    NWID:%X\n", iwe.u.nwid.value);
*/		      break;
		    case SIOCGIWFREQ:
		      {
			double		freq;			/* Frequency/channel */

			freq = iw_freq2float(&(iwe.u.freq));
			if (freq > 1000)
				NewAP->setChannel((unsigned int)iw_freq_to_channel(freq,&range));
			else
				NewAP->setChannel((unsigned int)freq);
			#if DEBUG_STDOUT
				iw_print_freq(buffer, freq);
				printf("                    %s\n", buffer);
				// printf("                    Freq(Chan):%f(%d)\n",freq, iw_freq_to_channel(freq,&range));
			#endif
		      }
		      break;
		    case SIOCGIWMODE:
			#if DEBUG_STDOUT
				printf("                    Mode:%s\n", iw_operation_mode[iwe.u.mode]);
			#endif
		      if (iwe.u.mode == 1) 		//ad-hoc
		      {
			      NewAP->setMode(1);
		      }
		      else if (iwe.u.mode == 3) 	//master
		      {
			      NewAP->setMode(0);
		      }
		      break;
		    case SIOCGIWNAME:
			#if DEBUG_STDOUT
				printf("                    Protocol:%-1.16s\n", iwe.u.name);
			#endif
		      break;
		    case SIOCGIWESSID:
		      {
			char essid[IW_ESSID_MAX_SIZE+1];
			if((iwe.u.essid.pointer) && (iwe.u.essid.length))
			  memcpy(essid, iwe.u.essid.pointer, iwe.u.essid.length);
			essid[iwe.u.essid.length] = '\0';
			if(iwe.u.essid.flags)
			  {
			    /* Does it have an ESSID index ? */
			    if((iwe.u.essid.flags & IW_ENCODE_INDEX) > 1)
			    {
				#if DEBUG_STDOUT
				      printf("                    ESSID:\"%s\" [%d]\n", essid,
										(iwe.u.essid.flags &IW_ENCODE_INDEX));
				#endif
			    }
			    else
			    	#if DEBUG_STDOUT
			      		printf("                    ESSID:\"%s\"\n", essid);
				#endif
			        NewAP->setEssid(essid);
			  }
			else
			{
			   #if DEBUG_STDOUT
			  	printf("                    ESSID:off/any\n");
			   #endif
			}
		      }
		      break;
		    case SIOCGIWENCODE:
		      {
			unsigned char	key[IW_ENCODING_TOKEN_MAX];
			if(iwe.u.data.pointer)
			  memcpy(key, iwe.u.essid.pointer, iwe.u.data.length);
			else
			  iwe.u.data.flags |= IW_ENCODE_NOKEY;
			#if DEBUG_STDOUT  
			  printf("                    Encryption key:");
			#endif
			if(iwe.u.data.flags & IW_ENCODE_DISABLED)
			{
			  #if DEBUG_STDOUT 
			    printf("off\n");
			  #endif
			  NewAP->setWEP(false);
		  	}
			else
			  {
			    NewAP->setWEP(true);
			    /* Display the key */
			    iw_print_key(buffer, key, iwe.u.data.length,
					 iwe.u.data.flags);
			    #if DEBUG_STDOUT
			    	printf("%s", buffer);
			    
			      /* Other info... */
			      if((iwe.u.data.flags & IW_ENCODE_INDEX) > 1)
			        printf(" [%d]", iwe.u.data.flags & IW_ENCODE_INDEX);
			      if(iwe.u.data.flags & IW_ENCODE_RESTRICTED)
			        printf("   Security mode:restricted");
			      if(iwe.u.data.flags & IW_ENCODE_OPEN)
			        printf("   Security mode:open");
			      printf("\n");
			    #endif
			  }
		      }
		      break;
		    case SIOCGIWRATE:
		    #if DEBUG_STDOUT
		      iw_print_bitrate(buffer, iwe.u.bitrate.value);
		      printf("                    Bit Rate:%s\n", buffer);
		    #endif
		      break;
		    case IWEVQUAL:
		      {
			iwe.u.qual.updated = 0x0;	/* Not that reliable, disable */
			iw_print_stats(buffer, &iwe.u.qual, &range, has_range);
			if(has_range && (iwe.u.qual.level != 0))
			{
			      /* If the statistics are in dBm */
				if(iwe.u.qual.level > range.max_qual.level)
				{
					NewAP->setSignal(iwe.u.qual.level - 0x100);
					NewAP->setNoise(iwe.u.qual.noise - 0x100);
				}
				else
				{
					NewAP->setSignal(iwe.u.qual.level);
					NewAP->setNoise(iwe.u.qual.noise);
				}
			}

			#if DEBUG_STDOUT
			  printf("                    %s\n", buffer);
			#endif
			break;
		      }
		#if WIRELESS_EXT > 14
		    case IWEVCUSTOM:
		      {
			char custom[IW_CUSTOM_MAX+1];
			if((iwe.u.data.pointer) && (iwe.u.data.length))
			  memcpy(custom, iwe.u.data.pointer, iwe.u.data.length);
			custom[iwe.u.data.length] = '\0';
			#if DEBUG_STDOUT
			  printf("                    Extra:%s\n", custom);
			#endif
			// For ipw driver (Intel Centrino)
			QString extra = QString(custom).stripWhiteSpace();
			#if DEBUG_STDOUT
				printf("                    (%d)(%d)\n",extra.find("Signal"),extra.find("RSSI"));
			#endif
			if ((extra.find("Signal") != -1) || (extra.find("RSSI") !=-1))
			{
				bool ok;
				int s;
				QStringList lst = lst.split(" ",extra,TRUE);
				QStringList::Iterator it = lst.begin();
				it++; // point to the next word so (*it) has the signal level
				s = QVariant(*it).toInt(&ok);
				#if DEBUG_STDOUT
				   if (ok)
				      printf("                    AP Signal found in 'Extra' event: %d dBm",s);
				   else
				      printf("                    Unable to parse 'Extra' signal event");
				#endif
				NewAP->setSignal(s);
				NewAP->setNoise(-256);	// I use this value since ipw driver doesn't support noise levels yet.
			}
				
		      }
		      break;
		#endif /* WIRELESS_EXT > 14 */
		    default:
		      printf("                    (Unknown Wireless Token 0x%04X)\n",
			     iwe.cmd);
		   }	/* switch(iwe.cmd) */
	  }	  
	}
      while(ret > 0);
      printf("\n");
    }
  else
    printf("%-8.8s  No scan results\n", ifname);

  return(1);
}

/*------------------------------------------------------------------*/
/*
 * Extract the interface name out of /proc/net/wireless or /proc/net/dev.
 */
/* NOTE: copied from iwlib.cpp. As it's an inline function, it's not accessible from other
files. So I copied it here not moved because I didn't want to alter iwlib.cpp. I could also
have used a declaration in header and definition, but I did it this way because this function
is only used here and once, so it fits quite well an inline function. */
static inline char *
iw_get_ifname(char *	name,	/* Where to store the name */
	      int	nsize,	/* Size of name buffer */
	      char *	buf)	/* Current position in buffer */
{
  char *	end;

  /* Skip leading spaces */
  while(isspace(*buf))
    buf++;

#ifndef IW_RESTRIC_ENUM
  /* Get name up to the last ':'. Aliases may contain ':' in them,
   * but the last one should be the separator */
  end = strrchr(buf, ':');
#else
  /* Get name up to ": "
   * Note : we compare to ": " to make sure to process aliased interfaces
   * properly. Doesn't work on /proc/net/dev, because it doesn't guarantee
   * a ' ' after the ':'*/
  end = strstr(buf, ": ");
#endif

  /* Not found ??? To big ??? */
  if((end == NULL) || (((end - buf) + 1) > nsize))
    return(NULL);

  /* Copy */
  memcpy(name, buf, (end - buf));
  name[end - buf] = '\0';

  return(end + 2);
}

void GetWirelessInterfacesNames_new (char * Names, int * Number)
{
  char		buff[1024];
  FILE *	fh;
	
#ifndef IW_RESTRIC_ENUM
  /* Check if /proc/net/wireless is available */
  fh = fopen(PROC_NET_DEV, "r");
#else
  /* Check if /proc/net/wireless is available */
  fh = fopen(PROC_NET_WIRELESS, "r");
#endif

  if(fh != NULL)
    {
      /* Success : use data from /proc/net/wireless */

      /* Eat 2 lines of header */
      fgets(buff, sizeof(buff), fh);
      fgets(buff, sizeof(buff), fh);

      /* Read each device line */
      while(fgets(buff, sizeof(buff), fh))
	{
	  char	name[IFNAMSIZ + 1];
	  char *s;

	  /* Extract interface name */
	  s = iw_get_ifname(name, sizeof(name), buff);

	  if(!s)
	    /* Failed to parse, complain and continue */
#ifndef IW_RESTRIC_ENUM
	    fprintf(stderr, "Cannot parse " PROC_NET_DEV "\n");
#else
	    fprintf(stderr, "Cannot parse " PROC_NET_WIRELESS "\n");
#endif
	  else
	  {
	    /* Got it, print info about this interface */
	    /* (*fn)(skfd, name, args, count); // don't use it now */
	    strcat(Names, name);
	    strcat(Names, "#");	 // I use '#' after, in split() function, as a separator
	    (*Number)++;
	  }
	}

      fclose(fh);
    }
  else
    {
      /* Get list of configured devices using "traditional" way (with the function below)*/
	GetWirelessInterfacesNames_old( Names, Number );
    }
}

void GetWirelessInterfacesNames_old(char* Names, int* Number)
{
        char buff[1024];
        struct ifconf ifc;
        struct ifreq *ifr;
        int i, skfd;

	skfd=iw_sockets_open();
        // Get list of active devices
        ifc.ifc_len = sizeof(buff);
        ifc.ifc_buf = buff;
	if(ioctl(skfd, SIOCGIFCONF, &ifc) < 0)
    	{
                fprintf(stderr, "SIOCGIFCONF: %s\n", strerror(errno));
                return;
    	}
        ifr = ifc.ifc_req;

        // Return them
        *Number= 0;
        Names[0]= '\0';

        for(i = ifc.ifc_len / sizeof(struct ifreq); --i >= 0; ifr++){
                struct iwreq wrq;
                // Get wireless name 
                strcpy(wrq.ifr_name, ifr->ifr_name);

                if(ioctl(skfd, SIOCGIWNAME, &wrq) < 0)
                        // If no wireless name : no wireless extensions
                        continue;
                strcat(Names, ifr->ifr_name);
                strcat(Names, "#");
                (*Number)++;
        }
	close(skfd);
}

// from iwconfig.c -->extracts DETAILED INFO
int iw_get_info(int skfd, char * ifname, struct wireless_info * info)
{
  struct iwreq		wrq;

  memset((char *) info, 0, sizeof(struct wireless_info));

  /* Get wireless name */
  if(iw_get_ext(skfd, ifname, SIOCGIWNAME, &wrq) < 0)
    {
      /* If no wireless name : no wireless extensions */
      /* But let's check if the interface exists at all */
      struct ifreq ifr;

      strcpy(ifr.ifr_name, ifname);
      if(ioctl(skfd, SIOCGIFFLAGS, &ifr) < 0)
	return(-ENODEV);
      else
	return(-ENOTSUP);
    }
  else
    {
      strncpy(info->name, wrq.u.name, IFNAMSIZ);
      info->name[IFNAMSIZ] = '\0';
    }

  /* Get ranges */
  if(iw_get_range_info(skfd, ifname, &(info->range)) >= 0)
    info->has_range = 1;

  /* Get network ID */
  if(iw_get_ext(skfd, ifname, SIOCGIWNWID, &wrq) >= 0)
    {
      info->has_nwid = 1;
      memcpy(&(info->nwid), &(wrq.u.nwid), sizeof(iwparam));
    }

  /* Get frequency / channel */
  if(iw_get_ext(skfd, ifname, SIOCGIWFREQ, &wrq) >= 0)
    {
      info->has_freq = 1;
      info->freq = iw_freq2float(&(wrq.u.freq));
    }

  /* Get sensitivity */
  if(iw_get_ext(skfd, ifname, SIOCGIWSENS, &wrq) >= 0)
    {
      info->has_sens = 1;
      memcpy(&(info->sens), &(wrq.u.sens), sizeof(iwparam));
    }

  /* Get encryption information */
  wrq.u.data.pointer = (caddr_t) info->key;
  wrq.u.data.length = IW_ENCODING_TOKEN_MAX;
  wrq.u.data.flags = 0;
  if(iw_get_ext(skfd, ifname, SIOCGIWENCODE, &wrq) >= 0)
    {
      info->has_key = 1;
      info->key_size = wrq.u.data.length;
      info->key_flags = wrq.u.data.flags;
    }

  /* Get ESSID */
  wrq.u.essid.pointer = (caddr_t) info->essid;
  wrq.u.essid.length = IW_ESSID_MAX_SIZE + 1;
  wrq.u.essid.flags = 0;
  if(iw_get_ext(skfd, ifname, SIOCGIWESSID, &wrq) >= 0)
    {
      info->has_essid = 1;
      info->essid_on = wrq.u.data.flags;
    }

  /* Get AP address */
  if(iw_get_ext(skfd, ifname, SIOCGIWAP, &wrq) >= 0)
    {
      info->has_ap_addr = 1;
      memcpy(&(info->ap_addr), &(wrq.u.ap_addr), sizeof (sockaddr));
    }

  /* Get NickName */
  wrq.u.essid.pointer = (caddr_t) info->nickname;
  wrq.u.essid.length = IW_ESSID_MAX_SIZE + 1;
  wrq.u.essid.flags = 0;
  if(iw_get_ext(skfd, ifname, SIOCGIWNICKN, &wrq) >= 0)
    if(wrq.u.data.length > 1)
      info->has_nickname = 1;

  /* Get bit rate */
  if(iw_get_ext(skfd, ifname, SIOCGIWRATE, &wrq) >= 0)
    {
      info->has_bitrate = 1;
      memcpy(&(info->bitrate), &(wrq.u.bitrate), sizeof(iwparam));
    }

  /* Get RTS threshold */
  if(iw_get_ext(skfd, ifname, SIOCGIWRTS, &wrq) >= 0)
    {
      info->has_rts = 1;
      memcpy(&(info->rts), &(wrq.u.rts), sizeof(iwparam));
    }

  /* Get fragmentation threshold */
  if(iw_get_ext(skfd, ifname, SIOCGIWFRAG, &wrq) >= 0)
    {
      info->has_frag = 1;
      memcpy(&(info->frag), &(wrq.u.frag), sizeof(iwparam));
    }

  /* Get operation mode */
  if(iw_get_ext(skfd, ifname, SIOCGIWMODE, &wrq) >= 0)
    {
      info->mode = wrq.u.mode;
      if((info->mode < IW_NUM_OPER_MODE) && (info->mode >= 0))
	info->has_mode = 1;
    }

  /* Get Power Management settings */
  wrq.u.power.flags = 0;
  if(iw_get_ext(skfd, ifname, SIOCGIWPOWER, &wrq) >= 0)
    {
      info->has_power = 1;
      memcpy(&(info->power), &(wrq.u.power), sizeof(iwparam));
    }

#if WIRELESS_EXT > 9
  /* Get Transmit Power */
  if(iw_get_ext(skfd, ifname, SIOCGIWTXPOW, &wrq) >= 0)
    {
      info->has_txpower = 1;
      memcpy(&(info->txpower), &(wrq.u.txpower), sizeof(iwparam));
    }
#endif

#if WIRELESS_EXT > 10
  /* Get retry limit/lifetime */
  if(iw_get_ext(skfd, ifname, SIOCGIWRETRY, &wrq) >= 0)
    {
      info->has_retry = 1;
      memcpy(&(info->retry), &(wrq.u.retry), sizeof(iwparam));
    }
#endif	/* WIRELESS_EXT > 10 */

  /* Get stats */
  if(iw_get_stats(skfd, ifname, &(info->stats)) >= 0)
    {
      info->has_stats = 1;
    }

  return(0);
}
