/*
   SwingWT
   Copyright(c)2003-2005, R. Rawson-Tetley

   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.

   Contact me by electronic mail: bobintetley@users.sourceforge.net

   $Log: JTabbedPane.java,v $
   Revision 1.37  2005/03/09 09:41:00  bobintetley
   (David Green) Various compatibility fixes

   Revision 1.36  2005/02/22 17:37:48  bobintetley
   JTabbedPane now supports add(index) methods. Few code errors fixed as a result
   of work on 15th.

   Revision 1.35  2005/01/05 09:22:31  bobintetley
   Updated copyright year on source

   Revision 1.34  2005/01/05 08:37:13  bobintetley
   Many compatibility fixes from David Barron

   Revision 1.33  2004/09/01 23:20:19  dannaab
   TextAction stubbed; fix VK_UP/DOWN/etc mappings to SWT

   Revision 1.32  2004/07/19 10:09:08  dannaab
   Call component.show() on tab focus change, for swing compatibility

   Revision 1.31  2004/06/08 09:24:23  dannaab
   Rename Component.getPeer() -> getSWTPeer().  added ComponentPeer and stubbed out support classes.

   Revision 1.30  2004/05/06 12:35:22  bobintetley
   Parity with Swing constants for Binary Compatibility + fixes to JDesktopPane

   Revision 1.29  2004/05/05 13:24:32  bobintetley
   Bugfixes and Laurent's patch for binary compatibility on Container.add()

   Revision 1.28  2004/04/30 16:52:17  bobintetley
   MenuListener support, JViewport support, TreeSelectionModel stubs, additional JTree methods

   Revision 1.27  2004/04/28 08:38:12  bobintetley
   Hierarchy fixes, code cleanup for base classes, additional javadocs and use of flag to identify JComponent descendants with peers

   Revision 1.26  2004/04/27 06:37:44  bobintetley
   Corrected hierarchy with JComponent descending Container

   Revision 1.25  2004/03/30 10:42:46  bobintetley
   Many minor bug fixes, event improvements by Dan Naab. Full swing.Icon support

   Revision 1.24  2004/01/26 15:45:47  bobintetley
   SwingSet fixes

   Revision 1.23  2004/01/26 10:57:45  bobintetley
   HTML handling (throws it away - SWT can't do anything with it)

   Revision 1.22  2004/01/26 10:03:37  bobintetley
   Fixed component painting, added overloaded TabbedPane add() methods

   Revision 1.21  2004/01/26 09:50:11  bobintetley
   Fix to tabbed pane cache

   Revision 1.20  2004/01/26 08:11:00  bobintetley
   Many bugfixes and addition of SwingSet

   Revision 1.19  2004/01/20 07:38:05  bobintetley
   Bug fixes and compatibility methods

   Revision 1.18  2004/01/16 15:53:32  bobintetley
   Many compatibility methods added to Container, Component, JInternalFrame,
      UIManager, SwingUtilities, JTabbedPane, JPasswordField, JCheckBox
      and JRadioButton.

   Revision 1.17  2004/01/02 11:30:53  bobintetley
   Change events are now supported

   Revision 1.16  2003/12/22 14:31:29  bobintetley
   Minor changes

   Revision 1.15  2003/12/22 14:17:30  bobintetley
   One-off timer method and better layout management for tabs

   Revision 1.14  2003/12/18 09:37:33  bobintetley
   Enabled support for JTabbedPane and JClosableTabbedPane (SWT doesn't
   support, so a bit of magic in here)

   Revision 1.13  2003/12/16 13:14:33  bobintetley
   Use of SwingWTUtils.isSWTControlAvailable instead of null test

   Revision 1.12  2003/12/16 09:19:02  bobintetley
   Various small fixes to match Swing more closely

   Revision 1.11  2003/12/15 18:29:57  bobintetley
   Changed setParent() method to setSwingWTParent() to avoid conflicts with applications

   Revision 1.10  2003/12/14 09:13:38  bobintetley
   Added CVS log to source headers

*/


package swingwtx.swing;

import org.eclipse.swt.widgets.*;
import org.eclipse.swt.*;

import swingwt.awt.Component;
import swingwtx.swing.event.*;

import java.util.*;

public class JTabbedPane extends JComponent implements SwingConstants {
    
    protected TabFolder ppeer = null;
    protected int pTabPlacement = TOP;
    protected Vector changeListeners = new Vector();
    
    /** Tab item cache until added to parent */
    protected Vector tabs = new Vector();
    
    /** Thread safe accessor */
    private int iRetVal = 0;
    private String sRetVal = "";
    private Object retval; 
    
    public JTabbedPane() {    }
    
    public void addChangeListener(ChangeListener l) {
        changeListeners.add(l);
    }
    
    public void removeChangeListener(ChangeListener l) {
        changeListeners.remove(l);
    }
    
    public void addTab(String title, swingwt.awt.Component component) {
        addTab(title, null, component);
    }
    
    public void addTab(String title, Icon icon, swingwt.awt.Component component) {
        addTab(title, icon, component, null, true);
    }
    
    public void addTab(final String title, final Icon icon, final swingwt.awt.Component component, final String tip) {
        addTab(title, icon, component, tip, true);
    }
    
    public void addTab(final String title, final Icon icon, final swingwt.awt.Component component, final String tip, final boolean enabled) {
        addTab(title, icon, component, tip, enabled, false);
    }
    
    public void add(swingwt.awt.Component component, Object modifiers) {
        addTab("", component);
    }
    public swingwt.awt.Component add(swingwt.awt.Component component) {
        addTab("", component);    
        return component;
    }
    public swingwt.awt.Component add(String title, swingwt.awt.Component component) {
        addTab(title, component);
        return component;
    }
    
    /** Warning: index is ignored. */
    public swingwt.awt.Component add(swingwt.awt.Component component, int index) {
        addTab("", component);
        return component;
    }
    
    /** Warning: index is ignored. */
    public void add(swingwt.awt.Component component, Object modifiers, int index) {
        addTab("", component);
    }
    
    public void addTab(String title, Icon icon, swingwt.awt.Component component, String tip, boolean enabled, boolean fromcache) {
        insertTab(title, icon, component, tip, tabs.size(), enabled, fromcache);
    }
    
    public void insertTab(String title, Icon icon, Component component, String tip, int index)
    {
        insertTab(title, icon, component, tip, index, true, false);
    }
    
    private void insertTab(final String title, final Icon icon, final Component component, final String tip, int index, final boolean enabled, boolean fromcache)
    {        
        // Create the cache entry
        if (!fromcache) {
            TabItemCache t = new TabItemCache();
            t.title = title;
            t.icon = icon;
            t.component = component;
            t.tip = tip;
            t.enabled = enabled;
            tabs.insertElementAt(t,index);
            t = null;
            return;
        }
        
        final JTabbedPane me = this;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (!SwingWTUtils.isSWTControlAvailable(ppeer)) return;

                TabItem t = new TabItem(ppeer, SWT.NONE);

                // Make sure the component contained by this
                // TabItem gets created now.
                try {
                    component.setSwingWTParent(me);
                }
                catch (Exception e) {
                    e.printStackTrace();
                }

                if (title != null) t.setText(SwingWTUtils.removeHTML(title));
                if (icon != null) t.setImage(SwingWTUtils.getSWTImageFromSwingIcon(me, icon));
                
                // Only display the component if it's enabled
                if (component != null && enabled) t.setControl(component.getSWTPeer());
                
                if (tip != null) t.setToolTipText(tip);
                t.setData("enabled", (enabled ? "true" : "false"));

                // Display the thing
                postponeLayout();
            }
        });
    }

    public Icon getIconAt(final int index) {
        if (!SwingWTUtils.isSWTControlAvailable(ppeer)) {
            TabItemCache t = (TabItemCache) tabs.get(index);
            return t.icon;
        }
        else {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    swingwt.awt.Image image = new swingwt.awt.Image();
                    image.image = ppeer.getItem(index).getImage();
                    retval = new ImageIcon(image);
                }
            });
            return (Icon) retval;
        }
    }
    public void setIconAt(final int index, final Icon icon) { 
        final JTabbedPane pthis = this;
        if (!SwingWTUtils.isSWTControlAvailable(ppeer)) {
            TabItemCache t = (TabItemCache) tabs.get(index);
            t.icon = icon;
        }
        else
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    ppeer.getItem(index).setImage(
                        SwingWTUtils.getSWTImageFromSwingIcon(pthis, icon)); 
                }
            });
    }
   
    public boolean isEnabledAt(final int index) {
	 if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
 	     SwingUtilities.invokeSync(new Runnable() {
		 public void run() {
		     retval = ppeer.getItem(index).getData("enabled");
		 }
	     });
	     return retval.equals("true");
	 }
	 else {
             TabItemCache t = (TabItemCache) tabs.get(index);
             return t.enabled;
	 }
    } 
    public void setEnabledAt(final int index, final boolean b) { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    ppeer.getItem(index).setData("enabled", (b ? "true" : "false"));
                    // If we're disabled - hide the tab's control. This is about all we
                    // can do with SWT (without writing our own tabbed interface of course).
                    if (!b)
                        ppeer.getItem(index).setControl(null);
                    else
                        ppeer.getItem(index).setControl( ((TabItemCache) tabs.get(index)).component.getSWTPeer() );
                }
            });
        }
        else {
            TabItemCache t = (TabItemCache) tabs.get(index);
            t.enabled = b;
        }
    }
    
    public int getTabCount() { 
        if (!SwingWTUtils.isSWTControlAvailable(ppeer)) 
            return tabs.size(); 
        else {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    iRetVal = ppeer.getItemCount(); 
                }
            });
            return iRetVal;
        }
    }
    public int getSelectedIndex() { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    iRetVal = ppeer.getSelectionIndex(); 
                }
            });
        }
        else
            iRetVal = -1;
        return iRetVal;
    }
    
    public swingwt.awt.Component getSelectedComponent() { 
        int i = getSelectedIndex();
        if (i != -1)
            return ((TabItemCache) tabs.get(i)).component;
        else
            return null;
    }
    
    public void setSelectedIndex(final int index) { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer))
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    ppeer.setSelection(index); 
                }
            });
    }
    
    public void removeTabAt(final int index) { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    tabs.removeElementAt(index);  
                    ppeer.getItem(index).dispose(); 
                }
            });
        }
        else
            tabs.removeElementAt(index);  
    }
    
    public String getTitleAt(final int index) { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    sRetVal = ppeer.getItem(index).getText(); 
                }
            });
            return sRetVal;
        }
        else {
            TabItemCache t = (TabItemCache) tabs.get(index);
            return t.title;
        }
    }
    
    public void setTitleAt(final int index, final String title) { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    ppeer.getItem(index).setText(SwingWTUtils.removeHTML(title)); 
                }
            });
        }
        else {
            TabItemCache t = (TabItemCache) tabs.get(index);
            t.title = title;
        }
    }
    
    public String getToolTipTextAt(final int index) { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    sRetVal = ppeer.getItem(index).getToolTipText(); 
                }
            });
            return sRetVal;
        }
        else {
            TabItemCache t = (TabItemCache) tabs.get(index);
            return t.tooltip;
        }
    }
    
    public void setToolTipTextAt(final int index, final String text) { 
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                    ppeer.getItem(index).setToolTipText(text); 
                }
            });
        }
        else {
            TabItemCache t = (TabItemCache) tabs.get(index);
            t.tooltip = text;
        }
    }
    
    public int indexOfComponent(final Component c) {
        final int[] ret = new int[1];
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
                    // Search items    
                    org.eclipse.swt.widgets.TabItem[] tabs = ppeer.getItems();
                    for (int i = 0; i < tabs.length; i++) {
                        if (tabs[i].getControl().equals(c.getSWTPeer())) {
                            ret[0] = i;
                            break;
                        }
                    }
                }
                else {
                    // Search cache
                    for (int i = 0; i < tabs.size(); i++) {
                        if (((TabItemCache) tabs.get(i)).component.equals(c)) {
                            ret[0] = i;
                            break;
                        }
                    }
                }
            }
        });
        return ret[0];
    }
    
    public int indexOfTab(final String name) {
        final int[] ret = new int[1];
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
                    // Search items    
                    org.eclipse.swt.widgets.TabItem[] tabs = ppeer.getItems();
                    for (int i = 0; i < tabs.length; i++) {
                        if (tabs[i].getText().equals(name)) {
                            ret[0] = i;
                            break;
                        }
                    }
                }
                else {
                    // Search cache
                    for (int i = 0; i < tabs.size(); i++) {
                        if (((TabItemCache) tabs.get(i)).title.equals(name)) {
                            ret[0] = i;
                            break;
                        }
                    }
                }
            }
        });
        return ret[0];   
    }
    
    public int indexOfTab(final Icon icon) {
        final int[] ret = new int[1];
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                // Search cache - we can only check the cache for Icons
                // as the objects are turned into something else for
                // use on the SWT peer.
                for (int i = 0; i < tabs.size(); i++) {
                    if (((TabItemCache) tabs.get(i)).icon.equals(icon)) {
                        ret[0] = i;
                        break;
                    }
                }
            }
        });
        return ret[0];   
    }
    
    public Component getComponentAt(int index) {
        return ((TabItemCache) tabs.get(index)).component;
    }
    
    public int getTabPlacement() { return pTabPlacement; }
    public void setTabPlacement(int place) { pTabPlacement = place; }
    
    public Control getSWTPeer() { return ppeer; }
    
    protected void postponeLayout() {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                 layoutTab();    
            }
        });
    }
    
    protected void layoutTab() {
        Control p = ppeer.getItem(ppeer.getSelectionIndex()).getControl();
        if (p instanceof Composite)
            ((Composite) p).layout(true);
        else
            parent.invalidate();
    }
    
    /**
     * Process change events - fired when the tab changes
     */
    protected void processChangeEvent(ChangeEvent e) {
        Iterator i = changeListeners.iterator();
        while (i.hasNext()) {
            ChangeListener l = (ChangeListener) i.next();
            l.stateChanged(e);
        }
    }
    
    public void setSwingWTParent(swingwt.awt.Container parent) throws Exception { 
        descendantHasPeer = true;
        ppeer = new TabFolder(parent.getComposite(), SwingWTUtils.translateSwingAlignmentConstant(pTabPlacement));
        peer = ppeer;
        composite = ppeer;
        this.parent = parent;
        
        // Add any cached items
        if (tabs.size() > 0) {
            Object[] tabso = tabs.toArray();
            for (int i = 0; i < tabso.length; i++) {
                TabItemCache t = (TabItemCache) tabso[i];
                addTab(t.title, t.icon, t.component, t.tip, t.enabled, true);
            }
        }
        
        // -- Tab is selected, may want to block it if we have it
        // flagged as disabled.
        final JTabbedPane me = this;
        ppeer.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
            public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
                // We don't send events if the tab is disabled
                if (checkTabEnabled(e)) {
                    postponeLayout(); // Update layout of tab if we went to a new one
                    
                    // Send change event
                    ChangeEvent ce = new ChangeEvent(me);
                    processChangeEvent(ce);
                    
                    // With Swing, components on a JTabbedPane receive a repaint
                    // event on focus gain.  For compatibility with code that relies on
                    // this, fire off a show() call here:
                    getSelectedComponent().show();
                }   
            }
            public void widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent e) {}
        });
        
        // Make sure first one gets displayed
        if (ppeer.getItemCount() > 0)
            postponeLayout();
        
    }
    
    /** Returns true if the tab is enabled according to the data set on it.
     *  This should be called from a selection event, determining whether
     *  the user is allowed to go to it or not.
     */
    protected boolean checkTabEnabled(org.eclipse.swt.events.SelectionEvent e) {
        e.doit = (e.item.getData("enabled").toString().equals("true"));
        return e.doit;
    }

    public Component getComponent(int i) {
            return ((TabItemCache)tabs.get(i)).component;
    }

    public Component[] getComponents() {
        
        TabItemCache[] comp = (TabItemCache[]) tabs.toArray(new TabItemCache[0]);
        Component[] cps = new Component[comp.length];
        for (int i = 0; i < comp.length; i++) {
            cps[i] = comp[i].component;
        }
        return cps;
    }

    public void setComponentAt(int index, Component panel)
    {
        if (index < tabs.size()) {
	        TabItemCache oldItem = (TabItemCache)tabs.get(index);
	        oldItem.component = panel;
        }
        else {
            insertTab("Tab "+(index+1), null, panel, null, index, true, false);
        }
    }

    public void setSelectedComponent(Component panel)
    {
        // TODO
    }
}
