/*
 * SwingWT Copyright(c)2003-2005, R. Rawson-Tetley
 *
 * For more information on distributing and using this program, please see the
 * accompanying "COPYING" file.
 *
 * Contact me by electronic mail: bobintetley@users.sourceforge.net
 *
 * $Log: JSplitPane.java,v $
 * Revision 1.21  2005/03/09 09:41:00  bobintetley
 * (David Green) Various compatibility fixes
 *
 * Revision 1.20  2005/02/22 17:37:47  bobintetley
 * JTabbedPane now supports add(index) methods. Few code errors fixed as a result
 * of work on 15th.
 *
 * Revision 1.19  2005/02/16 08:34:36  bobintetley
 * JFileChooser.showDialog support, JOptionPane instance support (no dialog creation at present), JSplitPane.getDividerLocation(), PixelGrabber.grabPixels(ms) throws InterruptedException, swingwtx.swing.MenuElement implemented, with JMenu* implementors, JPopupMenu.getSubElements(), JMenuBar.getSubElements(), JMenu.getSubElements()
 *
 * Revision 1.18  2005/01/05 09:22:31  bobintetley
 * Updated copyright year on source
 *
 * Revision 1.17  2004/10/30 20:11:57  bobintetley
 * Code cleanup
 *
 * Revision 1.16  2004/05/06 19:36:26  laurentmartelli
 * Removed debug trace
 *
 * Revision 1.15  2004/05/06 18:43:40  laurentmartelli
 * Fixed setDividerLocation(int) bug
 *
 * Revision 1.14  2004/05/05 12:43:21  bobintetley
 * Patches/new files from Laurent Martell
 *
 * Revision 1.13  2004/04/28 08:38:12  bobintetley
 * Hierarchy fixes, code cleanup for base classes, additional javadocs and use of flag to identify JComponent descendants with peers
 *
 * Revision 1.12  2004/04/27 06:37:44  bobintetley
 * Corrected hierarchy with JComponent descending Container
 *
 * Revision 1.11  2004/04/06 10:50:37  bobintetley
 * Thread safety work
 *
 * Revision 1.10  2004/03/26 14:52:46  bobintetley
 * Support for setDividerLocation() and thread safety
 *
 * Revision 1.9  2004/02/13 22:55:29  djspiewak
 * Improved the JToolBar class and added two new methods
 * Revision 1.8 2004/01/26 08:11:00 bobintetley Many
 * bugfixes and addition of SwingSet
 *
 * Revision 1.7 2003/12/15 18:29:57 bobintetley Changed setParent() method to
 * setSwingWTParent() to avoid conflicts with applications
 *
 * Revision 1.6 2003/12/14 09:13:38 bobintetley Added CVS log to source headers
 *
 */

package swingwtx.swing;

import org.eclipse.swt.custom.*;
import org.eclipse.swt.*;

import swingwt.awt.*;

/**
 * JSplitPane kindly contributed by Jack Park. Couple of changes here and there
 * by me to fit existing standards and things.
 *
 * 29/11/2003 - Thanks to Diane Trout for skeletons of additional methods
 * required.
 *
 * @author Jack Park
 * @author Diane Trout
 * @author Robin Rawson-Tetley
 */
public class JSplitPane extends JComponent {
    
    public final static int VERTICAL_SPLIT = 0;
    public final static int HORIZONTAL_SPLIT = 1;

    public final static String LEFT = "left";
    public final static String RIGHT = "right";
    
    protected int pOrientation = HORIZONTAL_SPLIT;
    protected double pResizeWeight = 0;
    
    protected swingwt.awt.Component leftComponent = null;
    protected swingwt.awt.Component rightComponent = null;
    
    private SashForm ppeer = null;
    
    public JSplitPane() {
    }
    public JSplitPane(int newOrientation) {
        this(newOrientation, false);
    }
    public JSplitPane(int newOrientation, boolean newContinuousLayout) {
        this(newOrientation, newContinuousLayout, null, null);
    }
    public JSplitPane(
    int newOrientation,
    swingwt.awt.Component newLeftComponent,
    swingwt.awt.Component newRightComponent) {
        this(newOrientation, false, newLeftComponent, newRightComponent);
    }
    
    public JSplitPane(
    int newOrientation,
    boolean newContinuousLayout,
    swingwt.awt.Component newLeftComponent,
    swingwt.awt.Component newRightComponent) {
        
        pOrientation = newOrientation;
        if (pOrientation != HORIZONTAL_SPLIT && pOrientation != VERTICAL_SPLIT)
            throw new IllegalArgumentException("Invalid orientation.");
        if (newLeftComponent != null)
            setLeftComponent(newLeftComponent);
        if (newRightComponent != null)
            setRightComponent(newRightComponent);
        
    }
    
    public void setLeftComponent(swingwt.awt.Component comp) {
        leftComponent = comp;
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            addLeft();
        }
    }
    
    public void setRightComponent(swingwt.awt.Component comp) {
        rightComponent = comp;
        if (SwingWTUtils.isSWTControlAvailable(ppeer)) {
            addRight();
        }
    }
   
    public void add(Component c, Object layoutModifier) {
        if (layoutModifier != null) {
	    if (layoutModifier.equals(LEFT)) {
		setLeftComponent(c);
		return;
	    }
	    else if (layoutModifier.equals(RIGHT)) {
		setRightComponent(c);
		return;
	    }
	}
	super.add(c, layoutModifier);
    }
    
    void addLeft() {
        final JSplitPane me = this;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                try {
                    leftComponent.setSwingWTParent(me);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }
    
    void addRight() {
        final JSplitPane me = this;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                try {
                    rightComponent.setSwingWTParent(me);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        });
    }
    
    public void setSwingWTParent(swingwt.awt.Container parent) throws Exception {
        descendantHasPeer = true;
        ppeer = new SashForm(parent.getComposite(), SWT.BORDER);
        peer = ppeer;
        composite = ppeer;
        this.parent = parent;
        ppeer.setOrientation( this.pOrientation == VERTICAL_SPLIT ? SWT.VERTICAL : SWT.HORIZONTAL );
        if (leftComponent != null) {
            addLeft();
            if (rightComponent != null)
                addRight();
            // No point adding right if we couldn't add left, since
            // SWT identifies left/right, top/bottom from parental order.
        }
        if (pResizeWeight != 0)
            setResizeWeight(pResizeWeight);
    }
    
    public swingwt.awt.Component getLeftComponent() {
        return leftComponent;
    }
    public void setTopComponent(swingwt.awt.Component comp) {
        setLeftComponent(comp);
    }
    public swingwt.awt.Component getTopComponent() {
        return leftComponent;
    }
    public swingwt.awt.Component getRightComponent() {
        return rightComponent;
    }
    public void setBottomComponent(swingwt.awt.Component comp) {
        setRightComponent(comp);
    }
    public swingwt.awt.Component getBottomComponent() {
        return rightComponent;
    }
    public int getOrientation() {
        return pOrientation;
    }
    public void setContinuousLayout(boolean b) {
    }
    public int getDividerSize() {
        return 2;
    }
    /** 
     * Unlike "real" Swing, we return the dividers actual
     * location since that's far more useful.
     * @return The divider's location
     */
    public int getDividerLocation() {
        if (leftComponent != null) {
            return leftComponent.getWidth();
        }
        return 0;
    }

    /**
     * Cannot set - it's fixed according to OS widgets.
     * This method doesn't do anything.
     */
    public void setDividerSize(int newSize) {
    }
    public void setOrientation(final int newOrientation) {
        if (pOrientation != HORIZONTAL_SPLIT && pOrientation != VERTICAL_SPLIT)
            throw new IllegalArgumentException("Invalid orientation.");
        pOrientation = newOrientation;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(ppeer))
                    ppeer.setOrientation(newOrientation == VERTICAL_SPLIT ? SWT.VERTICAL : SWT.HORIZONTAL);
            }
        });
    }
   
    /** Again, we deviate slightly from Swing here - the resize weight does set
     *  the initial weighting of the splitpane, however it seems to do something
     *  odd in Swing I can't fathom (it limits the resizability of the split)
     */
    public void setResizeWeight(final double value) {
        pResizeWeight = value;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(ppeer) && leftComponent != null && rightComponent != null) {
                    int leftWeight = (int) (value * 10);
                    int rightWeight = 10 - leftWeight;
                    ppeer.setWeights(new int[] { leftWeight, rightWeight });
                }
            }
        });
    }
    public double getResizeWeight() {
        return pResizeWeight;
    }
    
    /**
     * Enable one touch expand collapse of the splitter
     */
    public void setOneTouchExpandable(boolean newValue) {
    }
    
    /**
     * reset splitter to sizes based on child window sizes
     */
    public void resetToPreferredSizes() {
        if (SwingWTUtils.isSWTControlAvailable(ppeer) && leftComponent != null && rightComponent != null) {
            setPreferredSize(
            new Dimension(
            leftComponent.getPreferredSize().width
            + rightComponent.getPreferredSize().width,
            leftComponent.getPreferredSize().height
            + rightComponent.getPreferredSize().height));
        }
    }
    
    /**
     * Set the current location of the splitter proportional
     * to the width as a percentage.
     */
    public void setDividerLocation(double proportionalLocation) {
        if (SwingWTUtils.isSWTControlAvailable(ppeer) && leftComponent != null) {
            setResizeWeight(proportionalLocation);
        }
    }
    
    public void setDividerLocation(int location) {
        if (SwingWTUtils.isSWTControlAvailable(ppeer) && leftComponent != null) {
            setResizeWeight( ((double)location) / ((double)getWidth()) );
        }
    }
    
}
