/*
   SwingWT
   Copyright(c)2003-2005, R. Rawson-Tetley

   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.

   Contact me by electronic mail: bobintetley@users.sourceforge.net

   $Log: JButton.java,v $
   Revision 1.43  2005/01/05 09:22:30  bobintetley
   Updated copyright year on source

   Revision 1.42  2005/01/05 08:37:13  bobintetley
   Many compatibility fixes from David Barron

   Revision 1.41  2004/10/30 20:11:57  bobintetley
   Code cleanup

   Revision 1.40  2004/09/30 22:20:31  dannaab
   Kludgy workaround to allow setting of foreground colors on buttons in Windows

   Revision 1.39  2004/09/30 18:19:29  dannaab
   ButtonGroup, Popup listener fixes

   Revision 1.38  2004/09/01 23:20:19  dannaab
   TextAction stubbed; fix VK_UP/DOWN/etc mappings to SWT

   Revision 1.37  2004/05/25 01:04:12  dannaab
   Misc bugfixes, ActionMap.java added, added swt source to lib dir (for debugging
   purposes), misc import optimization

   Revision 1.36  2004/05/06 12:35:22  bobintetley
   Parity with Swing constants for Binary Compatibility + fixes to JDesktopPane

   Revision 1.35  2004/05/05 01:15:07  dannaab
   Remove duplicate setTooltipText() calls (Laurent Martelli)

   Revision 1.34  2004/04/30 23:18:26  dannaab
   List selection support, misc bug fixes

   Revision 1.33  2004/04/30 13:20:43  bobintetley
   Fix to unrealised peer preferred sizes, forwarding window events to
   content panes and fix for mouse drag events.

   Revision 1.32  2004/04/06 13:38:23  bobintetley
   (James Moger) patch to allow JButton.renderText() to support other fonts

   Revision 1.31  2004/04/06 12:51:39  bobintetley
   JButton text defaults to center alignment correctly again

   Revision 1.30  2004/03/30 10:42:46  bobintetley
   Many minor bug fixes, event improvements by Dan Naab. Full swing.Icon support

   Revision 1.29  2004/03/01 15:58:47  bobintetley
   Various little bug fixes

   Revision 1.28  2004/01/26 08:11:00  bobintetley
   Many bugfixes and addition of SwingSet

   Revision 1.27  2004/01/23 08:04:56  bobintetley
   JComboBox fixes and better Action implementation

   Revision 1.26  2004/01/20 07:38:05  bobintetley
   Bug fixes and compatibility methods

   Revision 1.25  2004/01/09 11:47:27  bobintetley
   Automatic JButton mapping!

   Revision 1.24  2004/01/09 10:33:57  bobintetley
   Changes for JToolBar to allow platform ToolBars, mixed with other components

   Revision 1.23  2004/01/06 15:38:30  bobintetley
   Adjusted render width calculations. Fixed horrible button border under Win32

   Revision 1.22  2004/01/02 10:50:50  bobintetley
   Button mnemonic/tooltip fixes

   Revision 1.21  2003/12/22 20:48:48  bobintetley
   Text and image support together for JButton

   Revision 1.20  2003/12/22 09:18:40  bobintetley
   Icon support for JToggleButton

   Revision 1.19  2003/12/17 16:30:37  bobintetley
   Flowlayout fix, vertical toolbar support and cleaned up text alignment
   hierarchy.

   Revision 1.18  2003/12/17 09:03:14  bobintetley
   Closely matches Swing behaviour + JScrollPane support for Panels

   Revision 1.17  2003/12/16 18:04:10  bobintetley
   Fixes to handling of mnemonics

   Revision 1.16  2003/12/16 17:46:17  bobintetley
   Additional thread safety methods

   Revision 1.15  2003/12/16 14:08:05  bobintetley
   Corrected event hierarchy for Button ActionEvents

   Revision 1.14  2003/12/16 13:51:45  bobintetley
   JButton fires ActionEvents when enter is pressed

   Revision 1.13  2003/12/16 13:14:33  bobintetley
   Use of SwingWTUtils.isSWTControlAvailable instead of null test

   Revision 1.12  2003/12/16 11:01:02  bobintetley
   Additional Swing compatibility

   Revision 1.11  2003/12/15 18:29:57  bobintetley
   Changed setParent() method to setSwingWTParent() to avoid conflicts with applications

   Revision 1.10  2003/12/15 15:53:06  bobintetley
   defaultCapable methods and alignment

   Revision 1.9  2003/12/14 09:13:38  bobintetley
   Added CVS log to source headers

*/


package swingwtx.swing;

import org.eclipse.swt.widgets.*;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.*;

import swingwt.awt.Color;

public class JButton extends AbstractButton implements SwingConstants, ButtonModel {

    protected Icon pImage = null;
    protected boolean pDefaultCapable = true;
    protected swingwt.awt.Container pDefaultButtonParent = null;
    /** If this JButton is representing an SWT tool button, a reference to it */
    protected ToolItem pSWTToolButton = null;
    
    public JButton() {this(""); }
    public JButton(String text) { pText = text; setModel(this); showMnemonic(); pHAlign = CENTER; }
    public JButton(Action a) { setAction(a); pHAlign = CENTER; }
    public JButton(Icon icon) { pImage = icon; pHAlign = CENTER; }
    public JButton(String text, Icon icon) { pImage = icon; pText = text; pHAlign = CENTER; setModel(this); showMnemonic(); }
    
    /** @see AbstractButton.setAction(Action, bool) */
    protected JButton(Action a, boolean addAsListener) { setAction(a, addAsListener); }
    
    /**
     * If this JButton is part of a JButtonMappedAction, update it
     */
    private void updateMappedAction() {
        if (pAction instanceof JButtonMappedAction) {
            pAction.putValue(Action.NAME, getText());
            pAction.putValue(Action.SHORT_DESCRIPTION, getToolTipText());
            pAction.putValue(Action.SMALL_ICON, getIcon());
            pAction.putValue(Action.DISABLED_ICON, getIcon());
            pAction.putValue(Action.MNEMONIC_KEY, new Integer(getMnemonic()));    
        }
    }
    
    /**
     * If this JButton is representing an SWT ToolItem, then update it's properties
     * from this
     */
    private void updateToolButton() {
        final JButton pthis = this;
        if (pSWTToolButton == null) return;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                pSWTToolButton.setText(getText());
                pSWTToolButton.setToolTipText(getToolTipText());
                pSWTToolButton.setEnabled(isEnabled());
                pSWTToolButton.setImage(SwingWTUtils.getSWTImageFromSwingIcon(pthis, getIcon()));
            }
        });  
    }
    
    public Icon getIcon() { return pImage; }
    
    public void setIcon(Icon icon) { 
        pImage = icon;
        final JButton pthis = this;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                if (SwingWTUtils.isSWTControlAvailable(ppeer)) ppeer.setImage(SwingWTUtils.getSWTImageFromSwingIcon(pthis, pImage));
                renderText();
                updateToolButton();
                updateMappedAction();
            }
        });
    }
    
    /** Overrides superclass so we can update any ToolItem */
    public void setToolTipText(String text) {
        super.setToolTipText(text);
        updateToolButton();
        updateMappedAction();
    }
    
    /** Overrides superclass as we need to read cached text for
     *  tool button purposes */
    public String getText() {
        return pText;    
    }
    
    /** Overrides superclass so we can render it where necessary */
    public void setText(String text) {
    	
        super.setText(text);
        
        // redraw the window version if we have a foreground color (see method for more info)
        if (pForeground != null) handleWindowsForegroundKludge();
        
        renderText();
        updateMappedAction();
        updateToolButton();
    }
    
    public void setEnabled(boolean b) {
        super.setEnabled(b);
        updateToolButton();
    }
    
    public boolean getDefaultCapable() { return pDefaultCapable; }
    public void setDefaultCapable(boolean b) { pDefaultCapable = b; }


 	protected void setColorTextImage(org.eclipse.swt.widgets.Button button, String text, Color color) {
 	
 		//if (button == null) return;
 		
 		button.setForeground(color.getSWTColor());
 		final org.eclipse.swt.graphics.Image img = createTextImage(button, text);
 		
 		button.setText(text);
 		button.setImage(img);
 		button.addDisposeListener(new DisposeListener() {
 			public void widgetDisposed(DisposeEvent e) {
 				if (!img.isDisposed()) {
 					img.dispose();
 				}
 			}
 		});
 	}

    private org.eclipse.swt.graphics.Image createTextImage(org.eclipse.swt.widgets.Button button, String text) {
 		GC buttonGC = new GC(button);
 		org.eclipse.swt.graphics.Point size = buttonGC.textExtent(text, SWT.DRAW_MNEMONIC);
 		org.eclipse.swt.graphics.Font font = buttonGC.getFont();
 		buttonGC.dispose();
 		
 		org.eclipse.swt.graphics.Image img = new org.eclipse.swt.graphics.Image(button.getDisplay(), size.x, size.y);
 		
 		GC imgGC = new GC(img);
 		imgGC.setFont(font);
 		imgGC.setBackground(button.getBackground());
 		imgGC.setForeground(button.getForeground());
 		imgGC.drawText(text, 0, 0, SWT.DRAW_MNEMONIC);
 		imgGC.dispose();
 		
 		return img;
    }
    
    /** This makes up for deficiencies in being able to set
      * images and text on a button. Call it after setting either the
     *  text or image.
     *
     *  @author Robin Rawson-Tetley
      */
     protected void renderText() {
         
        if (pText == null)
            return;
        if (!SwingWTUtils.isSWTControlAvailable(ppeer))
            return;
        if (pText.length() == 2 && pText.startsWith("&"))
            return;
        if (pText.equals("") || pImage == null)
            return;

        final JButton pthis = this;
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {
                org.eclipse.swt.graphics.GC gc = new org.eclipse.swt.graphics.GC(ppeer.getImage());
                gc.setFont(ppeer.getFont()); // Required to calculate correct string width

                // Calculate the text width
                org.eclipse.swt.graphics.Point p = gc.textExtent(pText);

                // Create a new image accounting for a two pixel margin and
                // the text width
                int width = ppeer.getImage().getBounds().width + 2;
                int height = ppeer.getImage().getBounds().height;
                org.eclipse.swt.graphics.Image im = new org.eclipse.swt.graphics.Image(SwingWTUtils.getDisplay(), width + p.x, height);

                // Draw the old image on it
                org.eclipse.swt.graphics.GC ngc = new org.eclipse.swt.graphics.GC(im);
                ngc.setFont(ppeer.getFont()); // Required to properly render font on image-text button

                 // Here, I set the background and foreground to the composite 
                 // background colour, so if your image is a PNG, you get
                 // the right alpha channel
                 ngc.setBackground(ppeer.getShell().getBackground());
                 ngc.setForeground(ppeer.getShell().getBackground());
                 ngc.fillRectangle(0, 0, width + p.x, height);

                 // Change foreground back to the regular colour for text rendering
                 ngc.setForeground(ppeer.getShell().getForeground());
                 ngc.drawImage(SwingWTUtils.getSWTImageFromSwingIcon(pthis, pImage), 0, 0);

                 // We have to take & out of the text, because I'm too lazy
                 // to support mnemonics (yes, I could draw a line, whatever - I'll
                 // revisit this)
                 String text = pText;
                 int mnPos = text.indexOf("&");
                 if (mnPos != -1)
                     text = text.substring(0, mnPos) + text.substring(mnPos + 1, text.length());

                 // Draw the text
                 ngc.drawText(text, width, (height - p.y) / 2, true);

                 // Drop the drawing contexts
                 ngc.dispose();
                 gc.dispose();

                 // Update the image
                 ppeer.setImage(im);
             }
         });
     }
     
     public void setForeground(Color foreground) {

     	super.setForeground(foreground);
     	handleWindowsForegroundKludge();
     }
     
     protected void handleWindowsForegroundKludge() {
     	
    	if (SwingWTUtils.isSWTControlAvailable(peer))
            SwingUtilities.invokeSync(new Runnable() {
                public void run() {
                	
                    /** Kludge alert!
			         * Buttons on Windows do no support setting of the foreground color - so we have to
			         * draw it ourself.
			         */ 
			        if (SwingWTUtils.isWindows() && pForeground != null && pText != null)
			        	setColorTextImage(ppeer, pText, pForeground);
                }
            });
     }
     
    /** Overriden to calculate non-realised
     *  preferred size.
     */
    protected swingwt.awt.Dimension calculatePreferredSize() {
        // Use the text height/width + 6 pixels for width
        // 10 for height button borders.
        swingwt.awt.Dimension size = new swingwt.awt.Dimension( 
            SwingWTUtils.getRenderStringWidth(pText) + 6, 
            SwingWTUtils.getRenderStringHeight(pText) + 12);
        setSize(size);
        return size;
    }
    
    protected int getSWTFlags() {
        return (parent instanceof JToolBar ? SWT.FLAT : SWT.PUSH);
    }
    
    /**
     * Once a parent component receives an "add" call for a child, this being
     * the child, this should be called to tell us to instantiate the peer
     * and load in any cached properties.
     */
    public void setSwingWTParent(swingwt.awt.Container parent) throws Exception {
        
        // If the Button's parent is a JToolbar, make it flat instead of
        // a push button
        ppeer = new Button(parent.getComposite(), getSWTFlags());
        peer = ppeer;

        // Refresh foreground color, if set
        if (pForeground != null) setForeground(pForeground);
        
        if (pText != null)
            ppeer.setText(pText);
        
        if (pMnemonic != ' ') showMnemonic();
        if (pImage != null) ppeer.setImage(SwingWTUtils.getSWTImageFromSwingIcon(this, pImage));
        if (pFont != null) ppeer.setFont(pFont.getSWTFont());
        ppeer.setAlignment(SwingWTUtils.translateSwingAlignmentConstant(pVAlign) | 
            SwingWTUtils.translateSwingAlignmentConstant(pHAlign));
 		
        renderText();
        
        if (pSize != null)
            ppeer.setSize(pSize.width, pSize.height);
        
        this.parent = parent;
        
        // If this button is the default for a given container,
        // make sure it knows about it
        if (pDefaultButtonParent != null) {
            
            // Is the peer a shell?
            if (pDefaultButtonParent.getComposite() instanceof Shell) {
                ((Shell) pDefaultButtonParent.getComposite()).setDefaultButton(ppeer);    
            }
            // Some other type of container
            else {
                pDefaultButtonParent.getComposite().getShell().setDefaultButton(ppeer);    
            }
        }
    }
    
    public void setDefaultButtonParent(swingwt.awt.Container window) {
        pDefaultButtonParent = window;
    }
    
    // TODO: This doesn't mesh with ButtonModels and ButtonGroups.
    // Use ButtonGroup.setSelected() to workaround, for now. (Can't here because the JButton
    // is by default it's own ButtonModel, so we'd end up in an infinite loop.
    public boolean isSelected() { return false; }
    public void setSelected(boolean b) {}
    
    public void addItemListener(swingwt.awt.event.ItemListener l) {
    }
    
    public Object[] getSelectedObjects() {
        return null;
    }
    
    public void removeItemListener(swingwt.awt.event.ItemListener l) {
    }
    
    public boolean isArmed() {
        return false;
    }
    
    public boolean isPressed() {
        return false;
    }
    
    public boolean isRollover() {
        return false;
    }
    
    public void setArmed(boolean b) {
    }
    
    public void setPressed(boolean b) {
    }
    
    public void setRollover(boolean b) {
    }

    public boolean isDefaultCapable() {
	return true;
    }
    
}
