/*
 * Sweep, a sound wave editor.
 *
 * Copyright (C) 2000 Conrad Parker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <sys/ioctl.h>
#include <pthread.h>
#include <errno.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <sweep/sweep_i18n.h>
#include <sweep/sweep_types.h>
#include <sweep/sweep_sample.h>

#include "driver.h"
#include "question_dialogs.h"
#include "preferences.h"

#ifdef DRIVER_OSS

#include <sys/soundcard.h>
#define DEV_DSP "/dev/dsp"
#define MONITOR_DEV_DSP "/dev/dsp1"

#define DEFAULT_LOG_FRAGS 6
#define LOG_FRAGS_MIN 1
#define LOG_FRAGS_MAX 10

#define DEFAULT_USE_MONITOR FALSE

#ifdef DEVEL_CODE
/*#define DEBUG*/
#endif

/* #define DEBUG_OFFSET */

typedef struct _oss_play_offset oss_play_offset;

struct _oss_play_offset {
  int framenr;
  sw_framecount_t offset;
};

extern GtkStyle * style_bw;

static oss_play_offset offsets[1<<LOG_FRAGS_MAX];

static gdouble configured_logfrags = DEFAULT_LOG_FRAGS;
static int nfrags = 0;

#define LOGFRAGS_TO_FRAGS(l) (1 << ((int)(floor((l)) - 1)))

static GtkWidget * dialog = NULL;

/*static char * configured_devicename = DEV_DSP;*/

static int oindex;
static int current_frame;
static int frame;

static GtkWidget * main_combo;
static GtkWidget * monitor_combo;
static GtkObject * adj;

#define DEV_KEY "OSS_Device"
#define MONITOR_DEV_KEY "OSS_MonitorDevice"
#define LOG_FRAGS_KEY "OSS_Logfrags"

#define USE_MONITOR_KEY "UseMonitor"

static char *
get_main_dev (void)
{
  char * main_dev;

  main_dev = prefs_get_string (DEV_KEY);

  if (main_dev == NULL) return DEV_DSP;
  
  return main_dev;
}

static char *
get_monitor_dev (void)
{
  char * monitor_dev;

  monitor_dev = prefs_get_string (MONITOR_DEV_KEY);

  if (monitor_dev == NULL) return MONITOR_DEV_DSP;
  
  return monitor_dev;
}

static gboolean
get_use_monitor (void)
{
  int * use_monitor;

  use_monitor = prefs_get_int (USE_MONITOR_KEY);

  if (use_monitor == NULL) return DEFAULT_USE_MONITOR;
  else return (*use_monitor != 0);
}

static int
get_log_frags (void)
{
  int * log_frags;

  log_frags = prefs_get_int (LOG_FRAGS_KEY);

  if (log_frags == NULL) return DEFAULT_LOG_FRAGS;
  else return (*log_frags);
}

static gboolean
monitor_checked (GtkWidget * dialog)
{
  GtkWidget * monitor_chb;

  monitor_chb =
    GTK_WIDGET(gtk_object_get_data (GTK_OBJECT(dialog), "monitor_chb"));

  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (monitor_chb));
}

static void
config_dev_dsp_dialog_ok_cb (GtkWidget * widget, gpointer data)
{
  GtkWidget * dialog = GTK_WIDGET (data);
  GtkAdjustment * adj;
  char * main_dev, * monitor_dev;

  adj = gtk_object_get_data (GTK_OBJECT(dialog), "buff_adj");

  configured_logfrags = adj->value;
  prefs_set_int (LOG_FRAGS_KEY, configured_logfrags);

  main_dev =
    gtk_entry_get_text (GTK_ENTRY(GTK_COMBO(main_combo)->entry));

  prefs_set_string (DEV_KEY, main_dev);

  if (monitor_checked (dialog)) {
    monitor_dev =
      gtk_entry_get_text (GTK_ENTRY(GTK_COMBO(monitor_combo)->entry)); 
    prefs_set_string (MONITOR_DEV_KEY, monitor_dev);

    prefs_set_int (USE_MONITOR_KEY, 1);
  } else {
    prefs_set_int (USE_MONITOR_KEY, 0);
  }

  gtk_widget_hide (dialog);
}

static void
config_dev_dsp_dialog_cancel_cb (GtkWidget * widget, gpointer data)
{
  GtkWidget * dialog;

  dialog = gtk_widget_get_toplevel (widget);
  gtk_widget_hide (dialog);
}

static void
update_ok_button (GtkWidget * widget, gpointer data)
{
  GtkWidget * dialog = GTK_WIDGET(data);
  GtkWidget * ok_button;
  gchar * main_devname, * monitor_devname;
  gboolean ok = FALSE;

  ok_button =
    GTK_WIDGET(gtk_object_get_data (GTK_OBJECT(dialog), "ok_button"));

  if (monitor_checked (dialog)) {
    main_devname =
      gtk_entry_get_text (GTK_ENTRY(GTK_COMBO(main_combo)->entry));
    monitor_devname =
      gtk_entry_get_text (GTK_ENTRY(GTK_COMBO(monitor_combo)->entry));

    ok = (strcmp (main_devname, monitor_devname) != 0);
  } else {
    ok = TRUE;
  }

  gtk_widget_set_sensitive (ok_button, ok);
}

static void
set_monitor_widgets (GtkWidget * dialog, gboolean use_monitor)
{
  GtkWidget * monitor_chb, * monitor_widget, * swap;

  monitor_chb =
    GTK_WIDGET(gtk_object_get_data (GTK_OBJECT(dialog), "monitor_chb"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(monitor_chb), use_monitor);

  monitor_widget = gtk_object_get_data (GTK_OBJECT(dialog), "monitor_widget");
  gtk_widget_set_sensitive (monitor_widget, use_monitor);

  swap = gtk_object_get_data (GTK_OBJECT(dialog), "swap");
  gtk_widget_set_sensitive (swap, use_monitor);

}

static void
set_buff_adj (GtkWidget * dialog, gint logfrags)
{
  GtkAdjustment * adj;

  adj = gtk_object_get_data (GTK_OBJECT(dialog), "buff_adj");
  gtk_adjustment_set_value (adj, logfrags);
}

static void
oss_devname_swap_cb (GtkWidget * widget, gpointer data)
{
  GtkWidget * dialog = GTK_WIDGET (data);
  char * main_dev, * monitor_dev;

  main_dev =
    g_strdup (gtk_entry_get_text (GTK_ENTRY(GTK_COMBO(main_combo)->entry)));
  monitor_dev = gtk_entry_get_text (GTK_ENTRY(GTK_COMBO(monitor_combo)->entry));

  gtk_entry_set_text (GTK_ENTRY(GTK_COMBO(main_combo)->entry), monitor_dev);
  gtk_entry_set_text (GTK_ENTRY(GTK_COMBO(monitor_combo)->entry), main_dev);

  g_free (main_dev);

  set_monitor_widgets (dialog, get_use_monitor());

  update_ok_button (widget, data);
}

static void
oss_devname_reset_cb (GtkWidget * widget, gpointer data)
{
  GtkWidget * dialog = GTK_WIDGET (data);
  char * main_dev, * monitor_dev;

  main_dev = get_main_dev ();
  monitor_dev = get_monitor_dev ();

  gtk_entry_set_text (GTK_ENTRY(GTK_COMBO(main_combo)->entry), main_dev);
  gtk_entry_set_text (GTK_ENTRY(GTK_COMBO(monitor_combo)->entry), monitor_dev);

  set_monitor_widgets (dialog, get_use_monitor());

  update_ok_button (widget, data);
}

static void
oss_devname_default_cb (GtkWidget * widget, gpointer data)
{
  GtkWidget * dialog = GTK_WIDGET (data);

  gtk_entry_set_text (GTK_ENTRY(GTK_COMBO(main_combo)->entry), DEV_DSP);
  gtk_entry_set_text (GTK_ENTRY(GTK_COMBO(monitor_combo)->entry), MONITOR_DEV_DSP);

  set_monitor_widgets (dialog, DEFAULT_USE_MONITOR);

  update_ok_button (widget, data);
}

static void
monitor_enable_cb (GtkWidget * widget, gpointer data)
{
  GtkWidget * dialog = GTK_WIDGET (data);
  gboolean active;

  active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(widget));

  set_monitor_widgets (dialog, active);
}

static void
oss_buffering_reset_cb (GtkWidget * widget, gpointer data)
{
  GtkWidget * dialog = GTK_WIDGET (data);

  set_buff_adj (dialog, get_log_frags());
}

static void
oss_buffering_default_cb (GtkWidget * widget, gpointer data)
{
  GtkWidget * dialog = GTK_WIDGET (data);

  set_buff_adj (dialog, DEFAULT_LOG_FRAGS);
}

static GtkWidget *
create_devices_combo (void)
{
  GtkWidget * combo;
  GList * cbitems = NULL;

  cbitems = NULL;
  cbitems = g_list_append (cbitems, "/dev/dsp");
  cbitems = g_list_append (cbitems, "/dev/dsp1");
  cbitems = g_list_append (cbitems, "/dev/sound/dsp");
  
  combo = gtk_combo_new ();
  
  gtk_combo_set_popdown_strings (GTK_COMBO(combo), cbitems);

  return combo;
}

static void
config_dev_dsp (void)
{
  GtkWidget * ebox;
  GtkWidget * notebook;
  GtkWidget * separator;
  GtkWidget * hbox, * hbox2;
  GtkWidget * vbox;
  GtkWidget * label;
  GtkWidget * checkbutton;
  GtkWidget * hscale;
  GtkWidget * ok_button;
  GtkWidget * button;

  GtkTooltips * tooltips;

  int * clf;

  if (dialog == NULL) {

    clf = prefs_get_int (LOG_FRAGS_KEY);
    if (clf != NULL) {
      configured_logfrags = (gdouble) *clf;
      free (clf);
    }

    dialog = gtk_dialog_new ();
    gtk_window_set_title (GTK_WINDOW(dialog), "Sweep: OSS audio device configuration");
    gtk_window_set_position (GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);

    /* OK */

    ok_button = gtk_button_new_with_label (_("OK"));
    GTK_WIDGET_SET_FLAGS (GTK_WIDGET (ok_button), GTK_CAN_DEFAULT);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), ok_button,
			TRUE, TRUE, 0);
    gtk_widget_show (ok_button);
    gtk_signal_connect (GTK_OBJECT(ok_button), "clicked",
			GTK_SIGNAL_FUNC (config_dev_dsp_dialog_ok_cb),
			dialog);

    gtk_object_set_data (GTK_OBJECT (dialog), "ok_button", ok_button);

    /* Cancel */

    button = gtk_button_new_with_label (_("Cancel"));
    GTK_WIDGET_SET_FLAGS (GTK_WIDGET (button), GTK_CAN_DEFAULT);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dialog)->action_area), button,
			TRUE, TRUE, 0);
    gtk_widget_show (button);
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC (config_dev_dsp_dialog_cancel_cb),
			NULL);

    gtk_widget_grab_default (ok_button);


    ebox = gtk_event_box_new ();
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), ebox,
			TRUE, TRUE, 0);
    gtk_widget_set_style (ebox, style_bw);
    gtk_widget_show (ebox);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(ebox), vbox);
    gtk_widget_show (vbox);

    /* Changes ... info */
    label = gtk_label_new (_("Changes to device settings will take effect on"
			     " next playback."));
    gtk_box_pack_start (GTK_BOX(vbox), label, TRUE, TRUE, 8);
    gtk_widget_show (label);


    /* Notebook */

    notebook = gtk_notebook_new ();
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook,
			TRUE, TRUE, 4);
    gtk_widget_show (notebook);

    /* Device name */
    label = gtk_label_new (_("Device name"));
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox, label);
    gtk_container_set_border_width (GTK_CONTAINER(vbox), 4);
    gtk_widget_show (vbox);

    /* Main output */ 
    hbox = gtk_hbox_new (FALSE, 8);
    gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, TRUE, 8);
    gtk_container_set_border_width (GTK_CONTAINER(hbox), 12);
    gtk_widget_show (hbox);
      
    label = gtk_label_new (_("Main output:"));
    gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);
    gtk_widget_show (label);
      
    main_combo = create_devices_combo ();
    gtk_box_pack_start (GTK_BOX(hbox), main_combo, TRUE, TRUE, 0);
    gtk_widget_show (main_combo);

    gtk_signal_connect (GTK_OBJECT(GTK_COMBO(main_combo)->entry), "changed",
			GTK_SIGNAL_FUNC(update_ok_button), dialog);

    gtk_object_set_data (GTK_OBJECT (dialog), "main_combo", main_combo);

#if 0
    button = gtk_button_new_with_label (_("Default"));
    gtk_box_pack_start (GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show (button);
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(default_devicename_cb), NULL);
#endif

    separator = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, FALSE, 8);
    gtk_widget_show (separator);

    /* Monitor */
    checkbutton =
      gtk_check_button_new_with_label (_("Use a different device for monitoring"));
    gtk_box_pack_start (GTK_BOX(vbox), checkbutton, FALSE, FALSE, 4);
    gtk_widget_show (checkbutton);

    gtk_signal_connect (GTK_OBJECT(checkbutton), "toggled",
			GTK_SIGNAL_FUNC(update_ok_button), dialog);

    hbox = gtk_hbox_new (FALSE, 8);
    gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, TRUE, 8);
    gtk_container_set_border_width (GTK_CONTAINER(hbox), 12);
    gtk_widget_show (hbox);
      
    label = gtk_label_new (_("Monitor output:"));
    gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 0);
    gtk_widget_show (label);

    monitor_combo = create_devices_combo ();
    gtk_box_pack_start (GTK_BOX(hbox), monitor_combo, TRUE, TRUE, 0);
    gtk_widget_show (monitor_combo);

    gtk_signal_connect (GTK_OBJECT(GTK_COMBO(monitor_combo)->entry), "changed",
			GTK_SIGNAL_FUNC(update_ok_button), dialog);

    gtk_signal_connect (GTK_OBJECT(checkbutton), "toggled",
			GTK_SIGNAL_FUNC(monitor_enable_cb), dialog);

    gtk_object_set_data (GTK_OBJECT (dialog), "monitor_chb", checkbutton);
    gtk_object_set_data (GTK_OBJECT (dialog), "monitor_widget", hbox);


    /* Swap / Remember / Reset device names*/

    hbox = gtk_hbox_new (FALSE, 4);
    gtk_box_pack_end (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER(hbox), 12);
    gtk_widget_show (hbox);

    button = gtk_button_new_with_label (_("Swap"));
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, TRUE, 4);
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(oss_devname_swap_cb), dialog);
    gtk_widget_show (button);

    tooltips = gtk_tooltips_new ();
    gtk_tooltips_set_tip (tooltips, button,
			  _("Swap main and monitor devices."),
			  NULL);

    gtk_object_set_data (GTK_OBJECT (dialog), "swap", button);

    hbox2 = gtk_hbox_new (TRUE, 4);
    gtk_box_pack_end (GTK_BOX (hbox), hbox2, FALSE, TRUE, 0);
    gtk_widget_show (hbox2);

    button = gtk_button_new_with_label (_("Reset"));
    gtk_box_pack_start (GTK_BOX (hbox2), button, FALSE, TRUE, 4);
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(oss_devname_reset_cb), dialog);
    gtk_widget_show (button);

    tooltips = gtk_tooltips_new ();
    gtk_tooltips_set_tip (tooltips, button,
			  _("Reset to the last remembered device names."),
			  NULL);

    /* Call the reset callback now to set remembered options */
    oss_devname_reset_cb (button, dialog);

    button = gtk_button_new_with_label (_("Defaults"));
    gtk_box_pack_start (GTK_BOX (hbox2), button, FALSE, TRUE, 4);
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(oss_devname_default_cb), dialog);
    gtk_widget_show (button);

    tooltips = gtk_tooltips_new ();
    gtk_tooltips_set_tip (tooltips, button,
			  _("Set to default device names."),
			  NULL);


    separator = gtk_hseparator_new ();
    gtk_box_pack_end (GTK_BOX (vbox), separator, FALSE, FALSE, 8);
    gtk_widget_show (separator);


    /* Buffering */

    label = gtk_label_new (_("Device buffering"));
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox, label);
    gtk_container_set_border_width (GTK_CONTAINER(vbox), 4);
    gtk_widget_show (vbox);

    hbox = gtk_hbox_new (FALSE, 8);
    gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 8);
    gtk_widget_show (hbox);

    label = gtk_label_new (_("Low latency /\nMore dropouts"));
    gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 8);
    gtk_widget_show (label);

    adj = gtk_adjustment_new (configured_logfrags, /* value */
			      LOG_FRAGS_MIN, /* lower */
			      LOG_FRAGS_MAX+1, /* upper */
			      1, /* step incr */
			      1, /* page incr */
			      1  /* page size */
			      );

    gtk_object_set_data (GTK_OBJECT(dialog), "buff_adj", adj);

    hscale = gtk_hscale_new (GTK_ADJUSTMENT(adj));
    gtk_box_pack_start (GTK_BOX(hbox), hscale, TRUE, TRUE, 4);
    gtk_scale_set_draw_value (GTK_SCALE(hscale), TRUE);
    gtk_scale_set_digits (GTK_SCALE(hscale), 0);
    gtk_range_set_update_policy (GTK_RANGE(hscale), GTK_UPDATE_CONTINUOUS);
    gtk_widget_set_usize (hscale, 160, -1);
    gtk_widget_show (hscale);

    label = gtk_label_new (_("High latency /\nFewer dropouts"));
    gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 8);
    gtk_widget_show (label);

    label = gtk_label_new (_("Varying this slider controls the lag between "
			     "cursor movements and playback. This is "
			     "particularly noticeable when \"scrubbing\" "
			     "during playback.\n\nLower values improve "
			     "responsiveness but may degrade audio quality "
			     "on heavily-loaded systems."));
    gtk_label_set_line_wrap (GTK_LABEL(label), TRUE);
    gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, FALSE, 8);
    gtk_widget_show (label);

    /* Remember / Reset device buffering */

    hbox = gtk_hbox_new (FALSE, 4);
    gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER(hbox), 12);
    gtk_widget_show (hbox);

#if 0
    checkbutton =
      gtk_check_button_new_with_label (_("Remember these options"));
    gtk_box_pack_start (GTK_BOX (hbox), checkbutton, TRUE, TRUE, 0);
    gtk_widget_show (checkbutton);

    gtk_object_set_data (GTK_OBJECT (dialog), "rem_options_chb", checkbutton);

    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(checkbutton), TRUE);
#endif

    hbox2 = gtk_hbox_new (TRUE, 4);
    gtk_box_pack_end (GTK_BOX (hbox), hbox2, FALSE, TRUE, 0);
    gtk_widget_show (hbox2);

    button = gtk_button_new_with_label (_("Reset"));
    gtk_box_pack_start (GTK_BOX (hbox2), button, FALSE, TRUE, 4);
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(oss_buffering_reset_cb), dialog);
    gtk_widget_show (button);

    tooltips = gtk_tooltips_new ();
    gtk_tooltips_set_tip (tooltips, button,
			  _("Reset to the last remembered device buffering."),
			  NULL);

    /* Call the reset callback now to set remembered options */
    oss_buffering_reset_cb (button, dialog);

    button = gtk_button_new_with_label (_("Default"));
    gtk_box_pack_start (GTK_BOX (hbox2), button, FALSE, TRUE, 4);
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(oss_buffering_default_cb), dialog);
    gtk_widget_show (button);

    tooltips = gtk_tooltips_new ();
    gtk_tooltips_set_tip (tooltips, button,
			  _("Set to default device buffering."),
			  NULL);
  }

  gtk_entry_set_text (GTK_ENTRY(GTK_COMBO(main_combo)->entry),
		      get_main_dev ());

  gtk_entry_set_text (GTK_ENTRY(GTK_COMBO(monitor_combo)->entry),
		      get_monitor_dev ());

  gtk_adjustment_set_value (GTK_ADJUSTMENT(adj), configured_logfrags);

  if (!GTK_WIDGET_VISIBLE(dialog)) {
    gtk_widget_show(dialog);
  } else {
    gdk_window_raise(dialog->window);
  }
}


/* driver functions */

static sw_handle *
open_dev_dsp (int monitoring, int flags)
{
  char * dev_name;
  int dev_dsp;
  sw_handle * handle;
  int i;

  if (monitoring) {
    if (get_use_monitor())
      dev_name = get_monitor_dev ();
    else
      return NULL;
  } else {
    dev_name = get_main_dev ();
  }

  flags &= O_RDONLY|O_WRONLY|O_RDWR; /* mask out direction flags */

  if((dev_dsp = open(dev_name, flags, 0)) == -1) {
    sweep_perror (errno, "Unable to open device %s", dev_name);
    return NULL;
  }

  handle = g_malloc0 (sizeof (sw_handle));
  handle->driver_fd = dev_dsp;

  oindex = 0;
  current_frame = 0;
  for (i = 0; i < (LOGFRAGS_TO_FRAGS(LOG_FRAGS_MAX)); i++) {
    offsets[i].framenr = 0;
    offsets[i].offset = -1;
  }
  frame = 0;

  return handle;
}

static void
setup_dev_dsp (sw_handle * handle, sw_format * format)
{
  int dev_dsp;
  /*  int mask, format, stereo, frequency;*/

  int stereo = 0;
  int bits;
  int i, want_channels, channels;
  int srate;
  int error;
  int fragsize, frag;
  int fmt;

  if (handle == NULL) {
#ifdef DEBUG
    g_print ("handle NULL in setup()\n");
#endif
    return;
  }

  dev_dsp = handle->driver_fd;

  if (ioctl (dev_dsp, SNDCTL_DSP_STEREO, &stereo) == -1) {
    /* Fatal error */
    perror("open_dsp_device 2 ") ;
    exit (1);
  } ;

  if (ioctl (dev_dsp, SNDCTL_DSP_RESET, 0)) {
    perror ("open_dsp_device 3 ") ;
    exit (1) ;
  } ;

  nfrags = LOGFRAGS_TO_FRAGS(configured_logfrags);
  fragsize = 8;
  frag = (nfrags << 16) | fragsize;
  if ((error = ioctl (dev_dsp, SNDCTL_DSP_SETFRAGMENT, &frag)) != 0) {
    perror ("OSS: error setting fragments");
  }

  fragsize = (frag & 0xffff);
  nfrags = (frag & 0x7fff000)>>16;
#ifdef DEBUG
  g_print ("Got %d frags of size 2^%d\n", nfrags, fragsize);
#endif
    
  bits = 16 ;
  if ((error = ioctl (dev_dsp, SOUND_PCM_WRITE_BITS, &bits)) != 0) {
    perror ("open_dsp_device 4 ");
    exit (1);
  }

  for (i=1; i <= format->channels; i *= 2) {
    channels = format->channels / i;
    want_channels = channels;

    if ((error = ioctl (dev_dsp, SOUND_PCM_WRITE_CHANNELS, &channels)) == -1) {
      perror ("open_dsp_device 5 ") ;
      exit (1) ;
    }

    if (channels == want_channels) break;
  }

  handle->driver_channels = channels;

  srate = format->rate;

  if ((error = ioctl (dev_dsp, SOUND_PCM_WRITE_RATE, &srate)) != 0) {
    perror ("open_dsp_device 6 ") ;
    exit (1) ;
  }

  handle->driver_rate = srate;

  if ((error = ioctl (dev_dsp, SNDCTL_DSP_SYNC, 0)) != 0) {
    perror ("open_dsp_device 7 ") ;
    exit (1) ;
  }

  fmt = AFMT_QUERY;
  if ((error = ioctl (dev_dsp, SOUND_PCM_SETFMT, &fmt)) != 0) {
    perror ("open_dsp_device 8") ;
    exit (1) ;
  }

  handle->custom_data = GINT_TO_POINTER(0);

#ifdef WORDS_BIGENDIAN
  if (fmt == AFMT_S16_LE || fmt == AFMT_U16_LE) {
    handle->custom_data = GINT_TO_POINTER(1);
  }
#else
  if (fmt == AFMT_S16_BE || fmt == AFMT_U16_BE) {
    handle->custom_data = GINT_TO_POINTER(1);
  }
#endif

#ifdef DEBUG
  {
    int caps;

    if (ioctl (dev_dsp, SNDCTL_DSP_GETCAPS, &caps) == -1) {
      sweep_perror (errno, "OSS: Unable to get device capabilities");
    }
    /* CAP_REALTIME tells whether or not this device can give exact
     * DMA pointers via GETOSPACE/GETISPACE. If this is true, then
     * the device reports with byte accuracy. If it is false it reports
     * to at least the nearest fragment bound, which is still pretty
     * good for small fragments, so it's not much of a problem if
     * this capability is not present.
     */
    g_print ("Realtime: %s\n", caps & DSP_CAP_REALTIME ? "YES" : "NO");
  }
#endif
}

#define RECORD_SCALE (SW_AUDIO_T_MAX / 32768.0)

static ssize_t
read_dev_dsp (sw_handle * handle, sw_audio_t * buf, size_t count)
{
  gint16 * bbuf;
  size_t byte_count;
  ssize_t bytes_read;
  int need_bswap;
  int i;

  byte_count = count * sizeof (gint16);
  bbuf = alloca (byte_count);
  bytes_read = read (handle->driver_fd, bbuf, byte_count);

  if (bytes_read == -1) {
    sweep_perror (errno, "Error reading from OSS audio device");
    return -1;
  }

  need_bswap = GPOINTER_TO_INT(handle->custom_data);

  if (need_bswap) {
    unsigned char * ucptr = (unsigned char *)bbuf;
    unsigned char temp;
    
    for (i = 0; i < count; i++) {
      temp = ucptr[2 * i];
      ucptr[2 * i] = ucptr [2 * i + 1];
      ucptr[2 * i + 1] = temp;
    }
  }

  for (i = 0; i < count; i++) {
    buf[i] = (sw_audio_t)(bbuf[i] * RECORD_SCALE);
  }

  return (bytes_read / sizeof (gint16));
}

#define PLAYBACK_SCALE (32768 / SW_AUDIO_T_MAX)

static ssize_t
write_dev_dsp (sw_handle * handle, sw_audio_t * buf, size_t count,
	       sw_framecount_t play_offset)
{
  gint16 * bbuf;
  size_t byte_count;
  ssize_t bytes_written;
  int need_bswap;
  int i;

  if (handle == NULL) {
#ifdef DEBUG
    g_print ("handle NULL in write()\n");
#endif
    return -1;
  }

  current_frame += count;
  offsets[oindex].framenr = current_frame;
  offsets[oindex].offset = play_offset;
  oindex++; oindex %= nfrags;

  byte_count = count * sizeof (gint16);
  bbuf = alloca (byte_count);

  for (i = 0; i < count; i++) {
    bbuf[i] = (gint16)(PLAYBACK_SCALE * buf[i]);
  }

  need_bswap = GPOINTER_TO_INT(handle->custom_data);

  if (need_bswap) {
    unsigned char * ucptr = (unsigned char *)bbuf;
    unsigned char temp;
    
    for (i = 0; i < count; i++) {
      temp = ucptr[2 * i];
      ucptr[2 * i] = ucptr [2 * i + 1];
      ucptr[2 * i + 1] = temp;
    }
  }

  bytes_written = write (handle->driver_fd, bbuf, byte_count);

  if (bytes_written == -1) {
    sweep_perror (errno, "Error writing to OSS audio device");
    return -1;
  } else {
    return (bytes_written / sizeof(gint16));
  }
}

static sw_framecount_t
offset_dev_dsp (sw_handle * handle)
{
  count_info info;
  int i, o;

  if (handle == NULL) {
#ifdef DEBUG
    g_print ("handle NULL in offset()\n");
#endif
    return -1;
  }

  if (ioctl (handle->driver_fd, SNDCTL_DSP_GETOPTR, &info) == -1) {
#ifdef DEBUG_OFFSET
    g_print ("error in GETOPTR\n");
#endif
    return -1;
  }

  frame = info.bytes;
#ifdef DEBUG_OFFSET
  g_print ("frame: %d\n", frame);
#endif

  o = oindex+1;
  for (i = 0; i < nfrags; i++) {
    o %= nfrags;
#ifdef DEBUG_OFFSET
    g_print ("\t(%d) Checking %d: %d\n", frame, o, offsets[o].framenr);
#endif
    if (offsets[o].framenr >= frame) {
      return offsets[o].offset;
    }
    o++;
  }

  return -1;
}

static void
reset_dev_dsp (sw_handle * handle)
{
  if (handle == NULL) {
#ifdef DEBUG
    g_print ("handle NULL in reset()\n");
#endif
    return;
  }

  if(ioctl (handle->driver_fd, SNDCTL_DSP_RESET, 0) == -1) {
    sweep_perror (errno, "Error resetting OSS audio device");
  }
}

static void
flush_dev_dsp (sw_handle * handle)
{
}

void
drain_dev_dsp (sw_handle * handle)
{
  if (handle == NULL) {
    g_print ("handle NULL in drain ()\n");
    return;
  }

  if(ioctl (handle->driver_fd, SNDCTL_DSP_POST, 0) == -1) {
    sweep_perror (errno, "POST error on OSS audio device");
  }

  if (ioctl (handle->driver_fd, SNDCTL_DSP_SYNC, 0) == -1) {
    sweep_perror (errno, "SYNC error on OSS audio device");
  }
}

static void
close_dev_dsp (sw_handle * handle)
{
  close (handle->driver_fd);
}

static sw_driver _driver_oss = {
  config_dev_dsp,
  open_dev_dsp,
  setup_dev_dsp,
  read_dev_dsp,
  write_dev_dsp,
  offset_dev_dsp,
  reset_dev_dsp,
  flush_dev_dsp,
  drain_dev_dsp,
  close_dev_dsp,
};

#else

static sw_driver _driver_oss = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
};

#endif

sw_driver * driver_oss = &_driver_oss;
