/*
 * Sweep, a sound wave editor.
 *
 * Copyright (C) 2000 Conrad Parker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * ALSA 0.6 support by Paul Davis
 * ALSA 0.9 updates by Zenaan Harkness
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <sys/ioctl.h>
#include <pthread.h>


#include <sweep/sweep_types.h>
#include <sweep/sweep_sample.h>

#include "driver.h"

#ifdef DRIVER_ALSA

#include <alsa/asoundlib.h>

#define ALSA_PCM_NAME "pcm.maestro3"

// taken from alsaplayer's alsa0.9 driver code (see alsa_setup below):
static int frag_count = 16;


static sw_handle *
alsa_open (int cueing, int flags)
{
  int err, count;
  char * alsa_pcm_name;
  snd_pcm_t * pcm_handle;
  sw_handle * handle;
  struct pollfd pfd;

  if (cueing) return NULL;

  printf ("sweep: alsa_open\n");

  if ((alsa_pcm_name = getenv ("SWEEP_ALSA_PCM")) == 0) {
    alsa_pcm_name = ALSA_PCM_NAME;
  }
  printf ("sweep: alsa_open 1\n");

  if ((err = snd_pcm_open(&pcm_handle, alsa_pcm_name,
			  SND_PCM_STREAM_PLAYBACK, SND_PCM_NONBLOCK)) < 0) {
    fprintf (stderr, "sweep: alsa_open: unable to open ALSA device %s (%s)\n",
	     alsa_pcm_name, snd_strerror (err));
    return NULL;
  }
  printf ("sweep: alsa_open 2\n");

  handle = g_malloc0 (sizeof (sw_handle));
  //handle->driver_fd = snd_pcm_poll_descriptor (pcm_handle);
  if ((count = snd_pcm_poll_descriptors (pcm_handle, &pfd, 1)) != 1) {
    fprintf (stderr,
	"sweep: alsa_open: could not get ALSA pcm poll descriptor (%s ?)\n",
	alsa_pcm_name);
    return NULL;
  }
  printf ("sweep: alsa_open 3\n");
  handle->driver_fd = pfd.fd;
  handle->custom_data = pcm_handle;

  printf ("sweep: alsa_open 4\n");
  return handle;
}

  // /src/alsa/alsaplayer-0.99.72/output/alsa-final/alsa.c
  // /src/alsa/alsa-lib-0.9.0rc3/test/pcm.c
static void
alsa_setup (sw_handle * handle, sw_format * format)
{
  int err;
  unsigned int val;
  snd_pcm_t * pcm_handle = (snd_pcm_t *)handle->custom_data;
  snd_pcm_hw_params_t * hwparams;
  snd_pcm_hw_params_alloca (&hwparams);

  printf ("sweep: alsa_setup \n");
  if ((err = snd_pcm_hw_params_any (pcm_handle, hwparams)) < 0) {
    fprintf(stderr,
	"sweep: alsa_setup: can't get PCM hw params (%s)\n",
	snd_strerror(err));
    return;
  }
  printf ("sweep: alsa_setup 1\n");
  if ((err = snd_pcm_hw_params_set_access (pcm_handle, hwparams,
      SND_PCM_ACCESS_RW_INTERLEAVED)) < 0) {
    fprintf(stderr,
	"sweep: alsa_setup: can't set interleaved access (%s)\n",
	snd_strerror(err));
    return;
  }
  printf ("sweep: alsa_setup 2\n");
  if ((err = snd_pcm_hw_params_set_format (pcm_handle, hwparams,
      SND_PCM_FORMAT_S16_LE)) < 0) {
    fprintf (stderr,
	"sweep: alsa_setup: audio interface does not support "
	"linear 16 bit little endian samples (%s)\n",
	snd_strerror(err));
    return;
  }
  printf ("sweep: alsa_setup 3\n");
  if ((err = snd_pcm_hw_params_set_rate (pcm_handle, hwparams,
      format->rate, 0)) < 0) {
    fprintf (stderr,
	"sweep: alsa_setup: audio interface does not support "
	"sample rate of %d (%s)\n",
	format->rate, snd_strerror (err));
    return;
  }
  printf ("sweep: alsa_setup 4\n");
  if ((err = snd_pcm_hw_params_set_channels (pcm_handle, hwparams,
      format->channels)) < 0) {
    fprintf (stderr,
	"sweep: alsa_setup: audio interface does not support "
	"%d channels (%s)\n",
	format->channels, snd_strerror (err));
    return;
  }
  printf ("sweep: alsa_setup 5\n");
  // this might have something to do with dropouts if they exist,
  if ((err = snd_pcm_hw_params_set_period_size (pcm_handle, hwparams,
      PBUF_SIZE/format->channels, 0)) < 0) {
    fprintf (stderr,
	"sweep: alsa_setup: audio interface does not support "
	"period size of %d (%s)\n",
	PBUF_SIZE/format->channels, snd_strerror (err));
    return;
  }
  printf ("sweep: alsa_setup 6\n");
  if ((val = snd_pcm_hw_params_get_periods_max (hwparams, 0))
      < frag_count) {
    fprintf (stderr,
	"sweep: alsa_setup: audio interface does not support "
	"period size of %d, so reducing to max of %d\n",
	frag_count, val);
    frag_count = val;
  }
  printf ("sweep: alsa_setup 7\n");
  // The above if makes it look like the following if will never fail ?
  // (code taken from alsaplayer's alsa_final output driver)
  if ((err = snd_pcm_hw_params_set_periods (pcm_handle, hwparams,
      frag_count, 0)) < 0) {
    fprintf (stderr,
	"sweep: alsa_setup: audio interface does not support "
	"period size of %d (%s) - suprising that we get this err!\n",
	frag_count, snd_strerror (err));
    return;
  }
  printf ("sweep: alsa_setup 8\n");
  // see alsa-lib html docs (may have to build them) for methods doco
  // The following is old alsa 0.6 code, which may need including somehow:
  //params.ready_mode = SND_PCM_READY_FRAGMENT;
  //params.start_mode = SND_PCM_START_DATA;
  //params.xrun_mode = SND_PCM_XRUN_FRAGMENT;
  //params.frag_size = PBUF_SIZE / params.format.channels;
  //params.avail_min = params.frag_size;
  // params.buffer_size = 3 * params.frag_size;

  if ((err = snd_pcm_hw_params (pcm_handle, hwparams)) < 0) {
    fprintf (stderr,
	"sweep: alsa_setup: audio interface could not be configured "
	"with specified parameters\n");
    return;
  }
  printf ("sweep: alsa_setup 9\n");

  //handle->driver_channels = params.format.channels;
  // TODO: K, is the following transform of above correct?:
  handle->driver_channels = snd_pcm_hw_params_get_channels_max (hwparams);

  if (snd_pcm_prepare (pcm_handle) < 0) {
    fprintf (stderr, "audio interface could not be prepared "
	     "for playback\n");
    return;
  }
  printf ("sweep: alsa_setup 10\n");
}

// alsaplayer's alsa_write method is very complex dealing with error,
// rather than just returning the error - it outputs xrun data.
// Someone might find that useful.
static ssize_t
alsa_write (sw_handle * handle, sw_audio_t * buf, size_t count,
    sw_framecount_t offset)
{
  snd_pcm_t * pcm_handle = (snd_pcm_t *)handle->custom_data;
  snd_pcm_uframes_t uframes; // unsigned long
  snd_pcm_sframes_t sframes; // signed long

  printf ("sweep: alsa_write \n");

  uframes = count / (2 * handle->driver_channels);
  printf ("sweep: alsa_write 1\n");

  sframes = snd_pcm_writei (pcm_handle, buf, uframes);

  printf ("sweep: alsa_write 2\n");
  return sframes;
}

static void
alsa_reset (sw_handle * handle)
{
}

static void
alsa_flush (sw_handle * handle)
{
}

static void
alsa_drain (sw_handle * handle)
{
  snd_pcm_t * pcm_handle = (snd_pcm_t *)handle->custom_data;

  if (snd_pcm_drop (pcm_handle) < 0) {
        fprintf (stderr, "audio interface could not be stopped\n");
        return;
  }
  if (snd_pcm_prepare (pcm_handle) < 0) {
        fprintf (stderr, "audio interface could not be re-prepared\n");
        return;
  }
}

static void
alsa_close (sw_handle * handle)
{
  snd_pcm_t * pcm_handle = (snd_pcm_t *)handle->custom_data;

  snd_pcm_close (pcm_handle);
}

static sw_driver _driver_alsa = {
  NULL, /* config */
  alsa_open,
  alsa_setup,
  NULL, /* read */
  alsa_write,
  alsa_reset,
  alsa_flush,
  alsa_drain,
  alsa_close
};

#else

static sw_driver _driver_alsa = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
};

#endif

sw_driver * driver_alsa = &_driver_alsa;
