/*
 * Sweep, a sound wave editor.
 *
 * Copyright (C) 2000 Conrad Parker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>
#include <sys/ioctl.h>
#include <pthread.h>

#include <sweep/sweep_types.h>
#include <sweep/sweep_sample.h>

#include "driver.h"

extern sw_driver * driver_alsa;
extern sw_driver * driver_oss;
extern sw_driver * driver_solaris;

extern GMutex * play_mutex;

/* preferred driver */
static sw_driver _driver_null = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
};

static sw_driver * pref = &_driver_null;

void
device_config (void)
{
  if (pref->config)
    pref->config ();
}

sw_handle *
device_open (int cueing, int flags)
{
  if (pref->open)
    return pref->open (cueing, flags);
  else
    return NULL;
}

void
device_setup (sw_handle * handle, sw_format * format)
{
  if (pref->setup)
    pref->setup (handle, format);
}

ssize_t
device_read (sw_handle * handle, sw_audio_t * buf, size_t count)
{
  if (pref->read)
    return pref->read (handle, buf, count);
  else
    return -1;
}

ssize_t
device_write (sw_handle * handle, sw_audio_t * buf, size_t count,
	      sw_framecount_t offset)
{
#ifdef DEBUG
  printf ("device_write: %d from %d\n", count, offset);
#endif

  if (pref->write)
    return pref->write (handle, buf, count, offset);
  else
    return -1;
}

sw_framecount_t
device_offset (sw_handle * handle)
{
  if (pref->offset)
    return pref->offset (handle);
  else
    return -1;
}

void
device_reset (sw_handle * handle)
{
  if (pref->reset)
    pref->reset (handle);
}

void
device_flush (sw_handle * handle)
{
  if (pref->flush)
    pref->flush (handle);
}

void
device_drain (sw_handle * handle)
{
  if (pref->drain)
    pref->drain (handle);
}

void
device_close (sw_handle * handle)
{
  if (pref->close)
    pref->close (handle);

  handle->driver_fd = -1;
}

void
init_devices (void)
{
#if defined(DRIVER_ALSA)
  pref = driver_alsa;
#elif defined(DRIVER_OSS)
  pref = driver_oss;
#elif defined(DRIVER_SOLARIS_AUDIO)
  pref = driver_solaris;
#endif

  play_mutex = g_mutex_new ();
}
