'''
 ====================================================================
 Copyright (c) 2003-2005 Barry A Scott.  All rights reserved.

 This software is licensed as described in the file LICENSE.txt,
 which you should have received as part of this distribution.

 ====================================================================

    wb_subversion_history.py

'''
import wx
import wb_images
import time
import pysvn
import wb_subversion_utils
import wb_read_file

id_view_cmd = wx.NewId()
id_diff_cmd = wx.NewId()

class HistoryFileFrame(wx.Frame):
    def __init__( self, app, project_info, filename, url, log_entries ):
        wx.Frame.__init__( self, None, -1, "History of %s" % filename, size=(700,400) )

        self.panel = HistoryFilePanel( self, app, project_info, filename, url, log_entries )

        # Set the application icon
        self.SetIcon( wb_images.getAppIconIcon() )

        wx.EVT_CLOSE( self, self.OnCloseWindow )

    def OnCloseWindow( self, event ):
        self.Destroy()


class HistoryDirFrame(wx.Frame):
    def __init__( self, app, filename, url, log_entries ):
        wx.Frame.__init__( self, None, -1, "History of %s" % filename, size=(500,400) )

        self.panel = HistoryDirPanel( self, app, filename, url, log_entries )

        # Set the application icon
        self.SetIcon( wb_images.getAppIconIcon() )

        wx.EVT_CLOSE( self, self.OnCloseWindow )

    def OnCloseWindow( self, event ):
        self.Destroy()


class HistoryCommonPanel(wx.Panel):
    col_revision = 0
    col_author = 1
    col_date = 2
    col_label = 3
    col_message = 4

    def __init__( self, parent, app, filename, url, log_entries ):
        wx.Panel.__init__( self, parent, -1 )

        self.app = app
        self.filename = filename
        self.url = url
        self.log_entries = log_entries

        # run from recent to old
        self.log_entries.sort( self.cmpLogEntries )

        self.selected_revisions = {}

        self.id_list = wx.NewId()

        self.v_sizer = wx.BoxSizer( wx.VERTICAL )

        self.list_ctrl = wx.ListCtrl( self, self.id_list, wx.DefaultPosition, wx.DefaultSize, wx.LC_REPORT|wx.NO_BORDER)

        self.list_ctrl.InsertColumn( self.col_revision, "Revision" )
        self.list_ctrl.InsertColumn( self.col_author, "Author" )
        self.list_ctrl.InsertColumn( self.col_date, "Date" )
        self.list_ctrl.InsertColumn( self.col_label, "Label" )
        self.list_ctrl.InsertColumn( self.col_message, "Message" )

        char_width = 9
        self.list_ctrl.SetColumnWidth( self.col_revision, 7*char_width )
        self.list_ctrl.SetColumnWidth( self.col_author, 14*char_width )
        self.list_ctrl.SetColumnWidth( self.col_date, 15*char_width )
        self.list_ctrl.SetColumnWidth( self.col_label, 12*char_width )
        self.list_ctrl.SetColumnWidth( self.col_message, 40*char_width )

        # don't make the ctrl readonly as that prevents copy as well as insert
        self.comment_ctrl = wx.TextCtrl( self, 0, '', size=wx.Size(400, 200), style=wx.TE_MULTILINE )
        self.comment_ctrl.SetInsertionPoint( 0 )

        self.initButtons( self.v_sizer )

        self.v_sizer.Add( self.list_ctrl, 2, wx.EXPAND|wx.ALL, 5 )
        self.v_sizer.Add( self.comment_ctrl, 1, wx.EXPAND|wx.ALL, 5 )

        wx.EVT_SIZE( self, self.OnSize )

        wx.EVT_LIST_ITEM_SELECTED( self, self.id_list, self.OnItemSelected)
        wx.EVT_LIST_ITEM_DESELECTED( self, self.id_list, self.OnItemDeselected)

        self.initList()

        self.SetAutoLayout( True )
        self.SetSizer( self.v_sizer )
        self.v_sizer.Fit( self )
        self.Layout()

    def cmpLogEntries( self, a, b ):
        return cmp( a[0], b[0] )

    def initButtons( self, sizer ):
        pass

    #---------- Event handlers ----------------------------------------------------------
    def OnItemSelected( self, event):
        pass

    def OnItemDeselected( self, event):
        pass

    #---------- Comment handlers ------------------------------------------------------------
    def updateComment( self, index ):
        if index >= 0:
            message = self.log_entries[ index ][4]
        else:
            message = ''

        self.comment_ctrl.SetValue( message )
        self.comment_ctrl.SetInsertionPoint( 0 )

    def OnSize( self, event):
        w,h = self.GetClientSizeTuple()
        self.v_sizer.SetDimension( 0, 0, w, h )

    def by_rev( self, a, b ):
        # highest rev first
        return -cmp( a[0], b[0] )

    def initList( self ):
        self.log_entries.sort( self.by_rev )

        for index, (rev,author,date,label,message) in enumerate( self.log_entries ):
            self.list_ctrl.InsertStringItem( index, str(rev) )
            self.list_ctrl.SetStringItem( index, self.col_author, author )
            self.list_ctrl.SetStringItem( index, self.col_date, wb_subversion_utils.fmtDateTime( date ) )
            self.list_ctrl.SetStringItem( index, self.col_label, label )
            self.list_ctrl.SetStringItem( index, self.col_message, message.replace( '\n', ' ' ) )

class HistoryDirPanel(HistoryCommonPanel):
    def __init__( self, parent, app, filename, url, log_entries ):
        HistoryCommonPanel.__init__( self, parent, app, filename, url, log_entries )

    #---------- Event handlers ----------------------------------------------------------
    def OnItemSelected( self, event):
        self.updateComment( event.m_itemIndex )

    def OnItemDeselected( self, event):
        self.updateComment( -1 )

class HistoryFilePanel(HistoryCommonPanel):
    def __init__( self, parent, app, project_info, filename, url, log_entries ):
        HistoryCommonPanel.__init__( self, parent, app, filename, url, log_entries )
        self.project_info = project_info

    def initButtons( self, sizer ):
        self.h_sizer = wx.BoxSizer( wx.HORIZONTAL )

        self.button_view = wx.Button( self, id_view_cmd, " View " )
        self.button_diff = wx.Button( self, id_diff_cmd, " Diff " )
        self.button_view.Enable( False )
        self.button_diff.Enable( False )

        self.h_sizer.Add( self.button_view, 0, wx.EXPAND|wx.LEFT, 5 )
        self.h_sizer.Add( self.button_diff, 0, wx.EXPAND|wx.LEFT, 5 )

        sizer.Add( self.h_sizer, 0, wx.EXPAND|wx.ALL, 5 )

        wx.EVT_BUTTON( self, id_view_cmd, self.OnViewCommand )
        wx.EVT_BUTTON( self, id_diff_cmd, self.OnDiffCommand )

    #---------- Command handlers ----------------------------------------------------------
    def OnViewCommand( self, event ):
        print 'History View not implemented'

    def OnDiffCommand( self, event ):
        indices = self.selected_revisions.keys()
        indices.sort()
        indices.reverse()

        if len( indices ) > 2:
            return

        revnum1 = None
        revnum2 = None
        rev1_lines = None
        rev2_lines = None

        if len( indices ) >= 1:
            index = indices[0]
            try:
                revnum1 = self.log_entries[ index ][0]
                rev1 = pysvn.Revision( pysvn.opt_revision_kind.number, revnum1 )

                rev1_text = self.project_info.client_bg.cat( self.filename, revision=rev1 )
                rev1_lines = wb_read_file.contentsAsUnicode( rev1_text ).split('\n')

            except pysvn.ClientError, e:
                self.app.log_client_error( e )
                return

        if len( indices ) == 2:
            index = indices[1]
            try:
                revnum2 = self.log_entries[ index ][0]
                rev2 = pysvn.Revision( pysvn.opt_revision_kind.number, revnum2 )

                rev2_text = self.project_info.client_bg.cat( self.filename, revision=rev2 )
                rev2_lines = wb_read_file.contentsAsUnicode( rev2_text ).split('\n')

            except pysvn.ClientError, e:
                self.app.log_client_error( e )
                return

        try:
            if len( indices ) == 1:
                self.app.Diff(    rev1_lines, '%s@%d' % (self.url, revnum1),
                        self.filename, self.filename )
            else:
                self.app.Diff(    rev1_lines, '%s@%d' % (self.url, revnum1),
                        rev2_lines, '%s@%d' % (self.url, revnum2) )

        except pysvn.ClientError, e:
            self.app.log_client_error( e )
            return


    #---------- Event handlers ----------------------------------------------------------
    def OnItemSelected( self, event):
        self.updateComment( event.m_itemIndex )
        self.selected_revisions[ event.m_itemIndex ] = None
        self.button_diff.Enable( len( self.selected_revisions ) in [1] )
        self.button_diff.Enable( len( self.selected_revisions ) in [1,2] )

    def OnItemDeselected( self, event):
        self.updateComment( -1 )
        if event.m_itemIndex in self.selected_revisions:
            del self.selected_revisions[ event.m_itemIndex ]

        self.button_diff.Enable( len( self.selected_revisions ) in [1] )
        self.button_diff.Enable( len( self.selected_revisions ) in [1,2] )
