#!/usr/bin/python

#
# This file is a test harness for svn-load
#
# Copyright (c) 2007, Hewlett-Packard Development Company, L.P. <dannf@hp.com>
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
#     * Redistributions of source code must retain the above
#       copyright notice, this list of conditions and the following
#       disclaimer.
#
#     * Redistributions in binary form must reproduce the above
#       copyright notice, this list of conditions and the following
#       disclaimer in the documentation and/or other materials
#       provided with the distribution.
#
#     * Neither the name of the Hewlett-Packard Co. nor the names
#       of its contributors may be used to endorse or promote
#       products derived from this software without specific prior
#       written permission.
#
# THIS SOFTWARE IS PROVIDED BY HEWLETT-PACKARD DEVELOPMENT COMPANY
# ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT
# NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
# FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
# HEWLETT-PACKARD DEVELOPMENT COMPANY BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

import unittest
import tempfile
import shutil
import os
import pysvn
import filecmp

## This case simply imports an empty directory into the toplevel of an
## empty repository. But, it provides the framework for other classes
## to inherit and do more complex things
class BaseLoadCase(unittest.TestCase):
    def setUp(self):
        self.tempdir = tempfile.mkdtemp()
        self.importDirSetUp()
        self.loadDirsSetUp()
        self.exportparent = os.path.join(self.tempdir, 'exportparent')
        os.mkdir(self.exportparent)
        self.repo = self.createRepository()
        self.url = 'file://' + self.repo
        ## idir is the relative path in repo where things get imported
        self.idir = ''
        self.extra_args = []

    def importDirSetUp(self):
        self.importdir = os.path.join(self.tempdir, 'importdir')
        os.mkdir(self.importdir)

    def loadDirsSetUp(self):
        self.loaddirs = [os.path.join(self.tempdir, 'loaddir0')]
        os.mkdir(self.loaddirs[0])
        
    def tearDown(self):
        shutil.rmtree(self.tempdir)

    def createRepository(self):
        svnadmin = '/usr/bin/svnadmin'
        repo = os.path.join(self.tempdir, 'repo')
        os.mkdir(repo)
        os.spawnl(os.P_WAIT, svnadmin, svnadmin, 'create', repo)
        return repo

    def exportRepository(self):
        svn = '/usr/bin/svn'
        ret = os.spawnl(os.P_WAIT, svn, svn, 'export', '--quiet', self.url,
                        os.path.join(self.exportparent, 'export'))
        if ret == 0:
            return True
        else:
            return False

    def loaddir(self, ldir):
        svnloaddirs = "./svn-load"
        ret = os.spawnl(os.P_WAIT, svnloaddirs, svnloaddirs,
                        self.url, self.idir, ldir, "--no-prompt",
                        *self.extra_args)
        if ret == 0:
            return True
        else:
            return False

    ## This should allow us to accept cases that dircmp considers "funny".
    ## The only known case so far is where both trees have the same identical
    ## symlink
    def funny_check(self, a, b, funny):
        haha = True
        strange = False
        for f in funny:
            funnyA = os.path.join(a, f)
            funnyB = os.path.join(b, f)

            if os.path.islink(funnyA) and os.path.islink(funnyB):
                if os.readlink(funnyA) == os.readlink(funnyB):
                    continue
            else:
                return strange
        return haha

    def compare_dirs(self, a, b):
        if os.path.islink(a) and not os.path.islink(b) or \
           not os.path.islink(a) and os.path.islink(b):
            return False

        c = filecmp.dircmp(a, b)
        funny_ok = self.funny_check(a, b, c.common_funny)
        if c.left_only or c.right_only or c.diff_files or not funny_ok:
            return False
        else:
            return c.common_dirs
    
    def compare_trees(self, a, b):
        subdirs = self.compare_dirs(a, b)
        if subdirs == False:
            return False
        else:
            for d in subdirs:
                if not self.compare_trees(os.path.join(a, d), os.path.join(b, d)):
                    return False
            return True
            
    def testLoad(self):
        client = pysvn.Client()
        client.import_(self.importdir, self.url, 'Initial import')
        for d in self.loaddirs:
            self.failUnless(self.loaddir(d))
            self.failUnless(self.exportRepository())
            self.failUnless(self.compare_trees(d, os.path.join(self.exportparent, 'export')))

class EmptyToSingleFile(BaseLoadCase):
    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        f = open(os.path.join(self.loaddirs[0], 'foo'), 'w')
        f.write("baz\n")
        f.close()

class EmptyToSingleFile3LevelsDown(EmptyToSingleFile):
    def loadDirsSetUp(self):
        EmptyToSingleFile.loadDirsSetUp(self)
        os.makedirs(os.path.join(self.loaddirs[0], '1/2/3'))
        f = open(os.path.join(self.loaddirs[0], '1/2/3/foo'), 'w')
        f.write("baz\n")
        f.close()

class EmptyToSymLink(BaseLoadCase):
    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        os.symlink('baz', os.path.join(self.loaddirs[0], 'foo'))

class EmptyToSingleFileAndSymLink(EmptyToSingleFile):
    def loadDirsSetUp(self):
        EmptyToSingleFile.loadDirsSetUp(self)
        os.symlink('foo', os.path.join(self.loaddirs[0], 'baz'))
        
class SingleFileContentChange(BaseLoadCase):
    def importDirSetUp(self):
        BaseLoadCase.importDirSetUp(self)
        f = open(os.path.join(self.importdir, 'foo'), 'w')
        f.write("bar\n")
        f.close()

    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        f = open(os.path.join(self.loaddirs[0], 'foo'), 'w')
        f.write("baz\n")
        f.close()

class BaseMoveCase(BaseLoadCase):
    # files_old: list of files in "old" side
    # files_new: list of files in "new" side, must be identical in length to
    #            files_old, and files_new[i] is the new path for files_old[i]
    files_old = []
    files_new = []

    def setUp(self):
        BaseLoadCase.setUp(self)
        self.movemap = tempfile.mkstemp()[1]
        self.makeMoveMap()
        self.extra_args.extend(['--move-map', self.movemap])
        self.filesmap_forward = dict(zip(self.files_old, self.files_new))
        self.filesmap_backwards = dict(zip(self.files_new, self.files_old))

    def tearDown(self):
        BaseLoadCase.tearDown(self)
        os.unlink(self.movemap)

    def makeMoveMap(self):
        # Override this method to make the move map
        f = open(self.movemap, 'w')
        f.write('\n')
        f.close()

    def makeFiles(dir, files):
        for filename in files:
            # strip off leading slashes, just in case
            filename = filename.lstrip('/')
            try:
                d = os.path.join(dir, os.path.dirname(filename))
                os.makedirs(d)
            except OSError:
                # already made
                pass
            f = open(os.path.join(dir, filename), 'w')
            f.write('foo\n')
            f.close()
    makeFiles = staticmethod(makeFiles)

    def importDirSetUp(self):
        BaseLoadCase.importDirSetUp(self)
        self.makeFiles(self.importdir, self.files_old)

    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        self.makeFiles(self.loaddirs[0], self.files_new)

    def testLoad(self):
        # superclass's testLoad method does some stuff that would conflict with
        # testFilesMoved
        pass

    def testFilesMoved(self):
        if not (self.files_old and self.files_new):
            # If empty (like on the BaseMoveCase), just ignore
            return

        client = pysvn.Client()
        client.import_(self.importdir, self.url, 'Initial import')
        
        self.failUnless(self.loaddir(self.loaddirs[0]))
        log = client.log(os.path.join(self.url, self.idir), limit=1,
                         discover_changed_paths=True)[0]
        changed_paths = log['changed_paths']
        for path_d in changed_paths:
            rel_path = path_d['path'][1 + len(self.idir):]
            if rel_path in self.files_old:
                # File was moved (changes in this class are only copies and
                # deletes, no actual content changes)
                self.assertEqual(path_d['action'], 'D')
            elif rel_path in self.files_new:
                old_path = os.path.join('/', self.idir,
                                        self.filesmap_backwards[rel_path])
                self.assertEqual(path_d['action'], 'A')
                self.assertEqual(path_d['copyfrom_path'], old_path)

class SingleFileMove(BaseMoveCase):

    files_old = ['foo']
    files_new = ['bar']

    def makeMoveMap(self):
        f = open(self.movemap, 'w')
        f.write("^foo$  'bar'\n")
        f.close()

class GraphicsFilesMove(BaseMoveCase):
    files_old = [
        'src/main.c',
        'src/foo.jpg',
        'src/bar.gif',
        'src/baz.png',
        'src/bang.png',
        'src/blast/boom.jpg',
    ]
    files_new = [
        'src/main.c',
        'graphics/foo.jpg',
        'graphics/bar.gif',
        'graphics/baz.png',
        'graphics/bang.png',
        'graphics/blast/boom.jpg',
    ]

    def makeMoveMap(self):
        f = open(self.movemap, 'w')
        f.write("^src/(?P<filename>.+\.(gif|jpg|png))$  "
                "lambda m: 'graphics/%s' % m.group('filename')\n")
        f.close()

class MultipleStringMoves(BaseMoveCase):
    files_old = [
        'foo',
        'bar',
        'baz',
    ]

    files_new = [
        'foo',
        'bang',
        'eek',
    ]

    def makeMoveMap(self):
        f = open(self.movemap, 'w')
        f.write("^bar$  'bang'\n")
        f.write("^baz$  'eek'\n")

class MoveMapWithDollarSign(BaseMoveCase):
    files_old = [
        'foo',
        'bar$baz',
    ]

    files_new = [
        'foo',
        'baz',
    ]

    def makeMoveMap(self):
        f = open(self.movemap, 'w')
        f.write(r"^bar\$baz$  'baz'")
        f.write("\n")


class SingleFileToBrokenSymLink(BaseLoadCase):
    def importDirSetUp(self):
        BaseLoadCase.importDirSetUp(self)
        f = open(os.path.join(self.importdir, 'foo'), 'w')
        f.write("bar\n")
        f.close()

    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        os.symlink('broken', os.path.join(self.loaddirs[0], 'foo'))

class SingleFileToSymLink(BaseLoadCase):
    def importDirSetUp(self):
        BaseLoadCase.importDirSetUp(self)
        f = open(os.path.join(self.importdir, 'foo'), 'w')
        f.close()
        f = open(os.path.join(self.importdir, 'bar'), 'w')
        f.close()

    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        f = open(os.path.join(self.loaddirs[0], 'foo'), 'w')
        f.close()
        os.symlink('foo', os.path.join(self.loaddirs[0], 'bar'))

class SingleFileToDirectorySymLink(BaseLoadCase):
    def importDirSetUp(self):
        BaseLoadCase.importDirSetUp(self)
        os.mkdir(os.path.join(self.importdir, 'foo'))
        open(os.path.join(self.importdir, 'bar'), 'w').close()

    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        os.mkdir(os.path.join(self.loaddirs[0], 'foo'))
        os.symlink('foo', os.path.join(self.loaddirs[0], 'bar'))

class BrokenSymLinkToFile(BaseLoadCase):
    def importDirSetUp(self):
        BaseLoadCase.importDirSetUp(self)
        os.symlink('broken', os.path.join(self.importdir, 'foo'))

    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        f = open(os.path.join(self.loaddirs[0], 'foo'), 'w')
        f.write("bar\n")
        f.close()

class SymLinkToFile(BaseLoadCase):
    def importDirSetUp(self):
        BaseLoadCase.importDirSetUp(self)
        open(os.path.join(self.importdir, 'foo'), 'w').close()
        os.symlink('foo', os.path.join(self.importdir, 'bar'))

    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        open(os.path.join(self.loaddirs[0], 'foo'), 'w').close()
        open(os.path.join(self.loaddirs[0], 'bar'), 'w').close()

class DirectorySymLinkToFile(BaseLoadCase):
    def importDirSetUp(self):
        BaseLoadCase.importDirSetUp(self)
        os.mkdir(os.path.join(self.importdir, 'foo'))
        os.symlink('foo', os.path.join(self.importdir, 'bar'))

    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        os.mkdir(os.path.join(self.loaddirs[0], 'foo'))
        open(os.path.join(self.loaddirs[0], 'bar'), 'w').close()


class ReimportSymlinkBase(BaseLoadCase):
    def _fillDirectory(self, directory):
        # overwrite in derived class
        pass

    def importDirSetUp(self):
        BaseLoadCase.importDirSetUp(self)
        self._fillDirectory(self.importdir)

    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        self._fillDirectory(self.loaddirs[0])
        # create files one alphabetically before and one after 'bar'
        open(os.path.join(self.loaddirs[0], 'a-file'), 'w').close()
        open(os.path.join(self.loaddirs[0], 'zoo-file'), 'w').close()
        # create a subdirectory hierarchy
        os.mkdir(os.path.join(self.loaddirs[0], 'zoo'))
        os.mkdir(os.path.join(self.loaddirs[0], 'zoo', 'subzoo'))
        os.mkdir(os.path.join(self.loaddirs[0], 'zoo', 'subzoo', 'subsubzoo'))
        open(os.path.join(self.loaddirs[0], 'zoo', 'subzoo', 'subsubzoo',
                          'subsubzoo-file'),
             'w').close()

class ReimportSymlink(ReimportSymlinkBase):
    def _fillDirectory(self, directory):
        open(os.path.join(directory, 'foo'), 'w').close()
        os.symlink('foo', os.path.join(directory, 'bar'))

class ReimportSymlinkToDirectory(ReimportSymlinkBase):
    """svn-load version 0.9 fails on this and _commits_ an
    _inconsistent_ content
    """
    def _fillDirectory(self, directory):
        os.mkdir(os.path.join(directory, 'foo'))
        os.symlink('foo', os.path.join(directory, 'bar'))        

class ReimportBrokenSymlink(ReimportSymlinkBase):
    def _fillDirectory(self, directory):
        os.symlink('foo', os.path.join(directory, 'bar'))

class SyblingDirSymlink(BaseLoadCase):
    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        os.mkdir(os.path.join(self.loaddirs[0], 'foo'))
        os.symlink('foo', os.path.join(self.loaddirs[0], 'bar'))

class GlobalIgnoreFile(BaseLoadCase):
    ## As of this writing, svn-load assumes that all files in a directory
    ## will be added when the subdirectory is added, so it doesn't bother
    ## descending down a tree it just added. However, some files are always
    ## ignored by svn, even if no svn:ignore property exists. This is
    ## controlled by the global-ignores parameter. svn-load blows up
    ## on these files during its check_permissions pass, as it assumes it
    ## can query their svn:executable property
    def loadDirsSetUp(self):
        BaseLoadCase.loadDirsSetUp(self)
        os.mkdir(os.path.join(self.loaddirs[0], 'foo'))
        f = open(os.path.join(self.loaddirs[0], 'foo', 'bar.o'), 'w')
        f.write("baz\n")
        f.close()

## This test creates two load dirs: foo and foo2, then attempts to load
## 'foo*'. On UNIX this should work because the shell expands the glob.
## On Windows it should work because svn-load expands the glob
class GlobHandling(BaseLoadCase):
    def loadDirsSetUp(self):
        self.loaddirparent = os.path.join(self.tempdir, 'loaddirparent')
        os.mkdir(self.loaddirparent)
        self.loaddirs = [os.path.join(self.loaddirparent, 'foo'),
                         os.path.join(self.loaddirparent, 'foo2')]
        for d in self.loaddirs:
            os.mkdir(d)
            f = open(os.path.join(d, 'test'), 'w')
            f.write(d + '\n')
            f.close()

    def testLoad(self):
        svnload = './svn-load'
        client = pysvn.Client()
        client.import_(self.importdir, self.url, 'Initial import')
        os.system('%s %s / %s*' %
                  (svnload, self.url, os.path.join(self.loaddirparent, "foo*")))
        self.failUnless(self.exportRepository())
        self.failUnless(self.compare_trees(self.loaddirs[-1], os.path.join(self.exportparent, 'export')))

if __name__ == '__main__':
    try:
        defaultTest = sys.argv[1]
    except:
        defaultTest = None
    unittest.main(defaultTest=defaultTest)
