/* Copyright (C) 2007-2010 Open Information Security Foundation
 *
 * You can copy, redistribute or modify this Program under the terms of
 * the GNU General Public License version 2 as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * version 2 along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

/**
 * \file
 *
 * \author Kirby Kuehl <kkuehl@gmail.com>
 * \author Anoop Saldanha <anoopsaldanha@gmail.com>
 *
 * \file DCE/RPC parser and decoder
 *
 * \todo Remove all the unnecessary per byte incremental loops with a full one
 *       time jump, i.e.
 *
 *       input[0], input_len
 *       for (i = 0; i < x; i++)
 *           input++;
 *
 *       with
 *
 *       input += x;
 *
 *       You'll be surprised at how many such cases we have here.  We also need
 *       to do the same for htp parser.  Should speed up the engine drastically.
 */

#include "suricata-common.h"
#include "suricata.h"

#include "debug.h"
#include "decode.h"
#include "threads.h"

#include "util-print.h"
#include "util-pool.h"
#include "util-debug.h"

#include "flow-util.h"

#include "detect-engine-state.h"

#include "stream-tcp-private.h"
#include "stream-tcp-reassemble.h"
#include "stream-tcp.h"
#include "stream.h"

#include "app-layer-protos.h"
#include "app-layer-parser.h"

#include "util-spm.h"
#include "util-unittest.h"

#include "app-layer-dcerpc.h"

enum {
    DCERPC_FIELD_NONE = 0,
    DCERPC_PARSE_DCERPC_HEADER,
    DCERPC_PARSE_DCERPC_BIND,
    DCERPC_PARSE_DCERPC_BIND_ACK,
    DCERPC_PARSE_DCERPC_REQUEST,
    /* must be last */
    DCERPC_FIELD_MAX,
};

/* \brief hexdump function from libdnet, used for debugging only */
void hexdump(/*Flow *f,*/ const void *buf, size_t len) {
    /* dumps len bytes of *buf to stdout. Looks like:
     * [0000] 75 6E 6B 6E 6F 77 6E 20
     *                  30 FF 00 00 00 00 39 00 unknown 0.....9.
     * (in a single line of course)
     */

    const unsigned char *p = buf;
    unsigned char c;
    size_t n;
    char bytestr[4] = {0};
    char addrstr[10] = {0};
    char hexstr[ 16*3 + 5] = {0};
    char charstr[16*1 + 5] = {0};
    for (n=1; n<=len; n++) {
        if (n%16 == 1) {
            /* store address for this line */
#if __WORDSIZE == 64
            snprintf(addrstr, sizeof(addrstr), "%.4lx",
            ((uint64_t)p-(uint64_t)buf) );
#else
            snprintf(addrstr, sizeof(addrstr), "%.4x",
            ((uint32_t)p-(uint32_t)buf) );
#endif
        }

        c = *p;
        if (isalnum(c) == 0) {
            c = '.';
        }

        /* store hex str (for left side) */
        snprintf(bytestr, sizeof(bytestr), "%02X ", *p);
        strlcat(hexstr, bytestr, sizeof(hexstr)-strlen(hexstr)-1);

        /* store char str (for right side) */
        snprintf(bytestr, sizeof(bytestr), "%c", c);
        strlcat(charstr, bytestr, sizeof(charstr)-strlen(charstr)-1);

        if(n%16 == 0) {
            /* line completed */
            printf("[%4.4s]   %-50.50s  %s\n", addrstr, hexstr, charstr);
            hexstr[0] = 0;
            charstr[0] = 0;
        } else if(n%8 == 0) {
            /* half line: add whitespaces */
            strlcat(hexstr, "  ", sizeof(hexstr)-strlen(hexstr)-1);
            strlcat(charstr, " ", sizeof(charstr)-strlen(charstr)-1);
        }
        p++; /* next byte */
    }

    if (strlen(hexstr) > 0) {
        /* print rest of buffer if not empty */
        printf("[%4.4s]   %-50.50s  %s\n", addrstr, hexstr, charstr);
    }
}

/**
 * \brief printUUID function used to print UUID, Major and Minor Version Number
 * and if it was Accepted or Rejected in the BIND_ACK.
 */
void printUUID(char *type, DCERPCUuidEntry *uuid) {
    uint8_t i = 0;
    if (uuid == NULL) {
        return;
    }
    printf("%s UUID [%2u] %s ", type, uuid->ctxid,
            (uuid->result == 0) ? "Accepted" : "Rejected");
    for (i = 0; i < 16; i++) {
        printf("%02x", uuid->uuid[i]);
    }
    printf(" Major Version 0x%04x Minor Version 0x%04x\n", uuid->version,
            uuid->versionminor);
}

/**
 * \brief DCERPCParseSecondaryAddr reads secondaryaddrlen bytes from the BIND_ACK
 * DCERPC call.
 */
static uint32_t DCERPCParseSecondaryAddr(DCERPC *dcerpc, uint8_t *input, uint32_t input_len) {
    SCEnter();
    uint8_t *p = input;
    while (dcerpc->dcerpcbindbindack.secondaryaddrlenleft-- && input_len--) {
        SCLogDebug("0x%02x ", *p);
        p++;
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

static uint32_t PaddingParser(DCERPC *dcerpc, uint8_t *input, uint32_t input_len) {
    SCEnter();
    uint8_t *p = input;
    while (dcerpc->padleft-- && input_len--) {
        SCLogDebug("0x%02x ", *p);
        p++;
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

static uint32_t DCERPCGetCTXItems(DCERPC *dcerpc, uint8_t *input, uint32_t input_len) {
    SCEnter();
    uint8_t *p = input;
    if (input_len) {
        switch (dcerpc->dcerpcbindbindack.ctxbytesprocessed) {
            case 0:
                if (input_len >= 4) {
                    dcerpc->dcerpcbindbindack.numctxitems = *p;
                    dcerpc->dcerpcbindbindack.numctxitemsleft = dcerpc->dcerpcbindbindack.numctxitems;
                    dcerpc->dcerpcbindbindack.ctxbytesprocessed += 4;
                    dcerpc->bytesprocessed += 4;
                    SCReturnUInt(4U);
                } else {
                    dcerpc->dcerpcbindbindack.numctxitems = *(p++);
                    dcerpc->dcerpcbindbindack.numctxitemsleft = dcerpc->dcerpcbindbindack.numctxitems;
                    if (!(--input_len))
                        break;
                }
            case 1:
                p++;
                if (!(--input_len))
                    break;
            case 2:
                p++;
                if (!(--input_len))
                    break;
            case 3:
                p++;
                input_len--;
                break;
        }
    }
    dcerpc->dcerpcbindbindack.ctxbytesprocessed += (p - input);
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

/**
 * \brief DCERPCParseBINDCTXItem is called for each CTXItem found the DCERPC BIND call.
 * each UUID is added to a TAILQ.
 */

static uint32_t DCERPCParseBINDCTXItem(DCERPC *dcerpc, uint8_t *input, uint32_t input_len) {
    SCEnter();
    uint8_t *p = input;

    if (input_len) {
        switch (dcerpc->dcerpcbindbindack.ctxbytesprocessed) {
            case 0:
                if (input_len >= 44) {
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpcbindbindack.ctxid = *(p);
                        dcerpc->dcerpcbindbindack.ctxid |= *(p + 1) << 8;
                        dcerpc->dcerpcbindbindack.uuid[3] = *(p + 4);
                        dcerpc->dcerpcbindbindack.uuid[2] = *(p + 5);
                        dcerpc->dcerpcbindbindack.uuid[1] = *(p + 6);
                        dcerpc->dcerpcbindbindack.uuid[0] = *(p + 7);
                        dcerpc->dcerpcbindbindack.uuid[5] = *(p + 8);
                        dcerpc->dcerpcbindbindack.uuid[4] = *(p + 9);
                        dcerpc->dcerpcbindbindack.uuid[7] = *(p + 10);
                        dcerpc->dcerpcbindbindack.uuid[6] = *(p + 11);
                        dcerpc->dcerpcbindbindack.uuid[8] = *(p + 12);
                        dcerpc->dcerpcbindbindack.uuid[9] = *(p + 13);
                        dcerpc->dcerpcbindbindack.uuid[10] = *(p + 14);
                        dcerpc->dcerpcbindbindack.uuid[11] = *(p + 15);
                        dcerpc->dcerpcbindbindack.uuid[12] = *(p + 16);
                        dcerpc->dcerpcbindbindack.uuid[13] = *(p + 17);
                        dcerpc->dcerpcbindbindack.uuid[14] = *(p + 18);
                        dcerpc->dcerpcbindbindack.uuid[15] = *(p + 19);
                        dcerpc->dcerpcbindbindack.version = *(p + 20);
                        dcerpc->dcerpcbindbindack.version |= *(p + 21) << 8;
                        dcerpc->dcerpcbindbindack.versionminor = *(p + 22);
                        dcerpc->dcerpcbindbindack.versionminor |= *(p + 23) << 8;
                    } else { /* Big Endian */
                        dcerpc->dcerpcbindbindack.ctxid = *(p) << 8;
                        dcerpc->dcerpcbindbindack.ctxid |= *(p + 1);
                        dcerpc->dcerpcbindbindack.uuid[0] = *(p + 4);
                        dcerpc->dcerpcbindbindack.uuid[1] = *(p + 5);
                        dcerpc->dcerpcbindbindack.uuid[2] = *(p + 6);
                        dcerpc->dcerpcbindbindack.uuid[3] = *(p + 7);
                        dcerpc->dcerpcbindbindack.uuid[4] = *(p + 8);
                        dcerpc->dcerpcbindbindack.uuid[5] = *(p + 9);
                        dcerpc->dcerpcbindbindack.uuid[6] = *(p + 10);
                        dcerpc->dcerpcbindbindack.uuid[7] = *(p + 11);
                        dcerpc->dcerpcbindbindack.uuid[8] = *(p + 12);
                        dcerpc->dcerpcbindbindack.uuid[9] = *(p + 13);
                        dcerpc->dcerpcbindbindack.uuid[10] = *(p + 14);
                        dcerpc->dcerpcbindbindack.uuid[11] = *(p + 15);
                        dcerpc->dcerpcbindbindack.uuid[12] = *(p + 16);
                        dcerpc->dcerpcbindbindack.uuid[13] = *(p + 17);
                        dcerpc->dcerpcbindbindack.uuid[14] = *(p + 18);
                        dcerpc->dcerpcbindbindack.uuid[15] = *(p + 19);
                        dcerpc->dcerpcbindbindack.version = *(p + 20) << 8;
                        dcerpc->dcerpcbindbindack.version |= *(p + 21);
                        dcerpc->dcerpcbindbindack.versionminor = *(p + 22) << 8;
                        dcerpc->dcerpcbindbindack.versionminor |= *(p + 23);
                    }
                    //if (dcerpc->dcerpcbindbindack.ctxid == dcerpc->dcerpcbindbindack.numctxitems
                    //        - dcerpc->dcerpcbindbindack.numctxitemsleft) {

                    dcerpc->dcerpcbindbindack.uuid_entry = (DCERPCUuidEntry *)SCCalloc(1, sizeof(DCERPCUuidEntry));
                    if (dcerpc->dcerpcbindbindack.uuid_entry == NULL) {
                        SCLogDebug("UUID Entry is NULL");
                        SCReturnUInt(0);
                    }

                    dcerpc->dcerpcbindbindack.uuid_entry->internal_id = dcerpc->dcerpcbindbindack.uuid_internal_id++;

                    memcpy(dcerpc->dcerpcbindbindack.uuid_entry->uuid,
                            dcerpc->dcerpcbindbindack.uuid,
                            sizeof(dcerpc->dcerpcbindbindack.uuid));

                    dcerpc->dcerpcbindbindack.uuid_entry->ctxid = dcerpc->dcerpcbindbindack.ctxid;
                    dcerpc->dcerpcbindbindack.uuid_entry->version = dcerpc->dcerpcbindbindack.version;
                    dcerpc->dcerpcbindbindack.uuid_entry->versionminor = dcerpc->dcerpcbindbindack.versionminor;

                    /* store the first frag flag in the uuid as pfc_flags will
                     * be overwritten by new packets. */
                    if (dcerpc->dcerpchdr.pfc_flags & PFC_FIRST_FRAG) {
                        dcerpc->dcerpcbindbindack.uuid_entry->flags |= DCERPC_UUID_ENTRY_FLAG_FF;
                    }

                    TAILQ_INSERT_HEAD(&dcerpc->dcerpcbindbindack.uuid_list,
                            dcerpc->dcerpcbindbindack.uuid_entry,
                            next);

#ifdef UNITTESTS
                    if (RunmodeIsUnittests()) {
                        printUUID("BIND", dcerpc->dcerpcbindbindack.uuid_entry);
                    }
#endif
                    dcerpc->dcerpcbindbindack.numctxitemsleft--;
                    dcerpc->bytesprocessed += (44);
                    dcerpc->dcerpcbindbindack.ctxbytesprocessed += (44);
                    SCReturnUInt(44U);

                    //} else {
                    //    SCLogDebug("ctxitem %u, expected %u\n", dcerpc->dcerpcbindbindack.ctxid,
                    //            dcerpc->dcerpcbindbindack.numctxitems - dcerpc->dcerpcbindbindack.numctxitemsleft);
                    //    SCReturnUInt(0);
                    //}
                } else {
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpcbindbindack.ctxid = *(p++);
                    } else {
                        dcerpc->dcerpcbindbindack.ctxid = *(p++) << 8;
                    }
                    if (!(--input_len))
                        break;
                }
            case 1:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.ctxid |= *(p++) << 8;
                } else {
                    dcerpc->dcerpcbindbindack.ctxid |= *(p++);
                }
                if (!(--input_len))
                    break;
            case 2:
                /* num transact items */
                p++;
                if (!(--input_len))
                    break;
            case 3:
                /* reserved */
                p++;
                if (!(--input_len))
                    break;
            case 4:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[3] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[0] = *(p++);
                }
                if (!(--input_len))
                    break;
            case 5:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[2] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[1] = *(p++);
                }
                if (!(--input_len))
                    break;
            case 6:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[1] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[2] = *(p++);
                }
                if (!(--input_len))
                    break;
            case 7:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[0] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[3] = *(p++);
                }
                if (!(--input_len))
                    break;
            case 8:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[5] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[4] = *(p++);
                }
                if (!(--input_len))
                    break;
            case 9:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[4] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[5] = *(p++);
                }
                if (!(--input_len))
                    break;
            case 10:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[7] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[6] = *(p++);
                }
                if (!(--input_len))
                    break;
            case 11:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.uuid[6] = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.uuid[7] = *(p++);
                }
                if (!(--input_len))
                    break;
            case 12:
                /* The following bytes are in the same order for both big and little endian */
                dcerpc->dcerpcbindbindack.uuid[8] = *(p++);
                if (!(--input_len))
                    break;
            case 13:
                dcerpc->dcerpcbindbindack.uuid[9] = *(p++);
                if (!(--input_len))
                    break;
            case 14:
                dcerpc->dcerpcbindbindack.uuid[10] = *(p++);
                if (!(--input_len))
                    break;
            case 15:
                dcerpc->dcerpcbindbindack.uuid[11] = *(p++);
                if (!(--input_len))
                    break;
            case 16:
                dcerpc->dcerpcbindbindack.uuid[12] = *(p++);
                if (!(--input_len))
                    break;
            case 17:
                dcerpc->dcerpcbindbindack.uuid[13] = *(p++);
                if (!(--input_len))
                    break;
            case 18:
                dcerpc->dcerpcbindbindack.uuid[14] = *(p++);
                if (!(--input_len))
                    break;
            case 19:
                dcerpc->dcerpcbindbindack.uuid[15] = *(p++);
                if (!(--input_len))
                    break;
            case 20:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.version = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.version = *(p++) << 8;
                }
                if (!(--input_len))
                    break;
            case 21:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.version |= *(p++) << 8;
                } else {
                    dcerpc->dcerpcbindbindack.version |= *(p++);
                }
                if (!(--input_len))
                    break;
            case 22:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.versionminor = *(p++);
                } else {
                    dcerpc->dcerpcbindbindack.versionminor = *(p++) << 8;
                }
                if (!(--input_len))
                    break;
            case 23:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.versionminor |= *(p++) << 8;
                } else {
                    dcerpc->dcerpcbindbindack.versionminor |= *(p++);
                }
                if (!(--input_len))
                    break;
            case 24:
                p++;
                if (!(--input_len))
                    break;
            case 25:
                p++;
                if (!(--input_len))
                    break;
            case 26:
                p++;
                if (!(--input_len))
                    break;
            case 27:
                p++;
                if (!(--input_len))
                    break;
            case 28:
                p++;
                if (!(--input_len))
                    break;
            case 29:
                p++;
                if (!(--input_len))
                    break;
            case 30:
                p++;
                if (!(--input_len))
                    break;
            case 31:
                p++;
                if (!(--input_len))
                    break;
            case 32:
                p++;
                if (!(--input_len))
                    break;
            case 33:
                p++;
                if (!(--input_len))
                    break;
            case 34:
                p++;
                if (!(--input_len))
                    break;
            case 35:
                p++;
                if (!(--input_len))
                    break;
            case 36:
                p++;
                if (!(--input_len))
                    break;
            case 37:
                p++;
                if (!(--input_len))
                    break;
            case 38:
                p++;
                if (!(--input_len))
                    break;
            case 39:
                p++;
                if (!(--input_len))
                    break;
            case 40:
                p++;
                if (!(--input_len))
                    break;
            case 41:
                p++;
                if (!(--input_len))
                    break;
            case 42:
                p++;
                if (!(--input_len))
                    break;
            case 43:
                p++;
                --input_len;
                //if (dcerpc->dcerpcbindbindack.ctxid == dcerpc->dcerpcbindbindack.numctxitems - dcerpc->dcerpcbindbindack.numctxitemsleft) {
                dcerpc->dcerpcbindbindack.uuid_entry = (DCERPCUuidEntry *)
                    SCCalloc(1, sizeof(DCERPCUuidEntry));
                if (dcerpc->dcerpcbindbindack.uuid_entry == NULL) {
                    SCLogDebug("UUID Entry is NULL\n");
                    SCReturnUInt(0);
                }

                dcerpc->dcerpcbindbindack.uuid_entry->internal_id =
                    dcerpc->dcerpcbindbindack.uuid_internal_id++;
                memcpy(dcerpc->dcerpcbindbindack.uuid_entry->uuid,
                        dcerpc->dcerpcbindbindack.uuid,
                        sizeof(dcerpc->dcerpcbindbindack.uuid));
                dcerpc->dcerpcbindbindack.uuid_entry->ctxid = dcerpc->dcerpcbindbindack.ctxid;
                dcerpc->dcerpcbindbindack.uuid_entry->version = dcerpc->dcerpcbindbindack.version;
                dcerpc->dcerpcbindbindack.uuid_entry->versionminor = dcerpc->dcerpcbindbindack.versionminor;

                /* store the first frag flag in the uuid as pfc_flags will
                 * be overwritten by new packets. */
                if (dcerpc->dcerpchdr.pfc_flags & PFC_FIRST_FRAG) {
                    dcerpc->dcerpcbindbindack.uuid_entry->flags |= DCERPC_UUID_ENTRY_FLAG_FF;
                }

                TAILQ_INSERT_HEAD(&dcerpc->dcerpcbindbindack.uuid_list,
                        dcerpc->dcerpcbindbindack.uuid_entry,
                        next);
#ifdef UNITTESTS
                if (RunmodeIsUnittests()) {
                    printUUID("BINDACK", dcerpc->dcerpcbindbindack.uuid_entry);
                }
#endif
                dcerpc->dcerpcbindbindack.numctxitemsleft--;
                dcerpc->bytesprocessed += (p - input);
                dcerpc->dcerpcbindbindack.ctxbytesprocessed += (p - input);
                SCReturnUInt((uint32_t)(p - input));

                //} else {
                //    SCLogDebug("ctxitem %u, expected %u\n", dcerpc->dcerpcbindbindack.ctxid,
                //            dcerpc->dcerpcbindbindack.numctxitems - dcerpc->dcerpcbindbindack.numctxitemsleft);
                //    SCReturnUInt(0);
                //}
                break;
        }
    }
    dcerpc->dcerpcbindbindack.ctxbytesprocessed += (p - input);
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

/**
 * \brief DCERPCParseBINDACKCTXItem is called for each CTXItem found in
 * the BIND_ACK call. The result (Accepted or Rejected) is added to the
 * correct UUID from the BIND call.
 */
static uint32_t DCERPCParseBINDACKCTXItem(DCERPC *dcerpc, uint8_t *input, uint32_t input_len) {
    SCEnter();
    uint8_t *p = input;
    DCERPCUuidEntry *uuid_entry;

    if (input_len) {
        switch (dcerpc->dcerpcbindbindack.ctxbytesprocessed) {
            case 0:
                if (input_len >= 24) {
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpcbindbindack.result = *p;
                        dcerpc->dcerpcbindbindack.result |= *(p + 1) << 8;
                    } else {
                        dcerpc->dcerpcbindbindack.result = *p << 8;
                        dcerpc->dcerpcbindbindack.result |= *(p + 1);
                    }
                    TAILQ_FOREACH(uuid_entry, &dcerpc->dcerpcbindbindack.uuid_list, next) {
                        if (uuid_entry->internal_id == dcerpc->dcerpcbindbindack.uuid_internal_id) {
                            uuid_entry->result = dcerpc->dcerpcbindbindack.result;
#ifdef UNITTESTS
                            if (RunmodeIsUnittests()) {
                                printUUID("BIND_ACK", uuid_entry);
                            }
#endif
                            if (uuid_entry->result != 0)
                                break;

                            dcerpc->dcerpcbindbindack.uuid_entry = (DCERPCUuidEntry *)
                                SCCalloc(1, sizeof(DCERPCUuidEntry));
                            if (dcerpc->dcerpcbindbindack.uuid_entry == NULL) {
                                SCLogError(SC_ERR_MEM_ALLOC,
                                           "Error allocating memory\n");
                                exit(EXIT_FAILURE);
                            }
                            memcpy(dcerpc->dcerpcbindbindack.uuid_entry,
                                   uuid_entry,
                                   sizeof(DCERPCUuidEntry));
                            TAILQ_INSERT_HEAD(&dcerpc->dcerpcbindbindack.accepted_uuid_list,
                                              dcerpc->dcerpcbindbindack.uuid_entry,
                                              next);
                            break;
                        }
                    }
                    dcerpc->dcerpcbindbindack.uuid_internal_id++;
                    dcerpc->dcerpcbindbindack.numctxitemsleft--;
                    dcerpc->bytesprocessed += (24);
                    dcerpc->dcerpcbindbindack.ctxbytesprocessed += (24);
                    SCReturnUInt(24U);
                } else {
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpcbindbindack.result = *(p++);
                    } else {
                        dcerpc->dcerpcbindbindack.result = *(p++) << 8;
                    }
                    if (!(--input_len))
                        break;
                }
            case 1:
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.result |= *(p++) << 8;
                } else {
                    dcerpc->dcerpcbindbindack.result |= *(p++);
                }
                if (!(--input_len))
                    break;
            case 2:
                /* num transact items */
                p++;
                if (!(--input_len))
                    break;
            case 3:
                /* reserved */
                p++;
                if (!(--input_len))
                    break;
            case 4:
                p++;
                if (!(--input_len))
                    break;
            case 5:
                p++;
                if (!(--input_len))
                    break;
            case 6:
                p++;
                if (!(--input_len))
                    break;
            case 7:
                p++;
                if (!(--input_len))
                    break;
            case 8:
                p++;
                if (!(--input_len))
                    break;
            case 9:
                p++;
                if (!(--input_len))
                    break;
            case 10:
                p++;
                if (!(--input_len))
                    break;
            case 11:
                p++;
                if (!(--input_len))
                    break;
            case 12:
                p++;
                if (!(--input_len))
                    break;
            case 13:
                p++;
                if (!(--input_len))
                    break;
            case 14:
                p++;
                if (!(--input_len))
                    break;
            case 15:
                p++;
                if (!(--input_len))
                    break;
            case 16:
                p++;
                if (!(--input_len))
                    break;
            case 17:
                p++;
                if (!(--input_len))
                    break;
            case 18:
                p++;
                if (!(--input_len))
                    break;
            case 19:
                p++;
                if (!(--input_len))
                    break;
            case 20:
                p++;
                if (!(--input_len))
                    break;
            case 21:
                p++;
                if (!(--input_len))
                    break;
            case 22:
                p++;
                if (!(--input_len))
                    break;
            case 23:
                TAILQ_FOREACH(uuid_entry, &dcerpc->dcerpcbindbindack.uuid_list, next) {
                    if (uuid_entry->internal_id == dcerpc->dcerpcbindbindack.uuid_internal_id) {
                        uuid_entry->result = dcerpc->dcerpcbindbindack.result;
#ifdef UNITTESTS
                        if (RunmodeIsUnittests()) {
                            printUUID("BIND_ACK", uuid_entry);
                        }
#endif
                        if (uuid_entry->result != 0)
                            break;

                        dcerpc->dcerpcbindbindack.uuid_entry = (DCERPCUuidEntry *)
                            SCCalloc(1, sizeof(DCERPCUuidEntry));
                        if (dcerpc->dcerpcbindbindack.uuid_entry == NULL) {
                            SCLogError(SC_ERR_MEM_ALLOC,
                                       "Error allocating memory\n");
                            exit(EXIT_FAILURE);
                        }
                        memcpy(dcerpc->dcerpcbindbindack.uuid_entry,
                               uuid_entry,
                               sizeof(DCERPCUuidEntry));
                        TAILQ_INSERT_HEAD(&dcerpc->dcerpcbindbindack.accepted_uuid_list,
                                          dcerpc->dcerpcbindbindack.uuid_entry,
                                          next);
                        break;
                    }
                }
                dcerpc->dcerpcbindbindack.uuid_internal_id++;
                dcerpc->dcerpcbindbindack.numctxitemsleft--;
                p++;
                --input_len;
                break;

        }
    }
    dcerpc->dcerpcbindbindack.ctxbytesprocessed += (p - input);
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

static uint32_t DCERPCParseBIND(DCERPC *dcerpc, uint8_t *input, uint32_t input_len) {
    SCEnter();
    DCERPCUuidEntry *item;
    uint8_t *p = input;
    if (input_len) {
        switch (dcerpc->bytesprocessed) {
            case 16:
                dcerpc->dcerpcbindbindack.numctxitems = 0;
                if (input_len >= 12) {
                    while ((item = TAILQ_FIRST(&dcerpc->dcerpcbindbindack.uuid_list))) {
                        TAILQ_REMOVE(&dcerpc->dcerpcbindbindack.uuid_list, item, next);
                        SCFree(item);
                    }
                    if (dcerpc->dcerpchdr.type == BIND) {
                        while ((item = TAILQ_FIRST(&dcerpc->dcerpcbindbindack.accepted_uuid_list))) {
                            TAILQ_REMOVE(&dcerpc->dcerpcbindbindack.accepted_uuid_list, item, next);
                            SCFree(item);
                        }
                        TAILQ_INIT(&dcerpc->dcerpcbindbindack.accepted_uuid_list);
                    }
                    dcerpc->dcerpcbindbindack.uuid_internal_id = 0;
                    dcerpc->dcerpcbindbindack.numctxitems = *(p + 8);
                    dcerpc->dcerpcbindbindack.numctxitemsleft = dcerpc->dcerpcbindbindack.numctxitems;
                    TAILQ_INIT(&dcerpc->dcerpcbindbindack.uuid_list);
                    dcerpc->bytesprocessed += 12;
                    SCReturnUInt(12U);
                } else {
                    /* max_xmit_frag */
                    p++;
                    if (!(--input_len))
                        break;
                }
            case 17:
                /* max_xmit_frag */
                p++;
                if (!(--input_len))
                    break;
            case 18:
                /* max_recv_frag */
                p++;
                if (!(--input_len))
                    break;
            case 19:
                /* max_recv_frag */
                p++;
                if (!(--input_len))
                    break;
            case 20:
                /* assoc_group_id */
                p++;
                if (!(--input_len))
                    break;
            case 21:
                /* assoc_group_id */
                p++;
                if (!(--input_len))
                    break;
            case 22:
                /* assoc_group_id */
                p++;
                if (!(--input_len))
                    break;
            case 23:
                /* assoc_group_id */
                p++;
                if (!(--input_len))
                    break;
            case 24:
                while ((item = TAILQ_FIRST(&dcerpc->dcerpcbindbindack.uuid_list))) {
                    TAILQ_REMOVE(&dcerpc->dcerpcbindbindack.uuid_list, item, next);
                    SCFree(item);
                }
                if (dcerpc->dcerpchdr.type == BIND) {
                    while ((item = TAILQ_FIRST(&dcerpc->dcerpcbindbindack.accepted_uuid_list))) {
                        TAILQ_REMOVE(&dcerpc->dcerpcbindbindack.accepted_uuid_list, item, next);
                        SCFree(item);
                    }
                    TAILQ_INIT(&dcerpc->dcerpcbindbindack.accepted_uuid_list);
                }
                dcerpc->dcerpcbindbindack.uuid_internal_id = 0;
                dcerpc->dcerpcbindbindack.numctxitems = *(p++);
                dcerpc->dcerpcbindbindack.numctxitemsleft = dcerpc->dcerpcbindbindack.numctxitems;
                TAILQ_INIT(&dcerpc->dcerpcbindbindack.uuid_list);
                if (!(--input_len))
                    break;
            case 25:
                /* pad byte 1 */
                p++;
                if (!(--input_len))
                    break;
            case 26:
                /* pad byte 2 */
                p++;
                if (!(--input_len))
                    break;
            case 27:
                /* pad byte 3 */
                p++;
                --input_len;
                break;
            default:
                dcerpc->bytesprocessed++;
                SCReturnUInt(1);
                break;
        }
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

static uint32_t DCERPCParseBINDACK(DCERPC *dcerpc, uint8_t *input, uint32_t input_len) {
    SCEnter();
    uint8_t *p = input;

    switch (dcerpc->bytesprocessed) {
        case 16:
            dcerpc->dcerpcbindbindack.uuid_internal_id = 0;
            dcerpc->dcerpcbindbindack.numctxitems = 0;
            if (input_len >= 10) {
                if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                    dcerpc->dcerpcbindbindack.secondaryaddrlen = *(p + 8);
                    dcerpc->dcerpcbindbindack.secondaryaddrlen |= *(p + 9) << 8;
                } else {
                    dcerpc->dcerpcbindbindack.secondaryaddrlen = *(p + 8) << 8;
                    dcerpc->dcerpcbindbindack.secondaryaddrlen |= *(p + 9);
                }
                dcerpc->dcerpcbindbindack.secondaryaddrlenleft = dcerpc->dcerpcbindbindack.secondaryaddrlen;
                dcerpc->bytesprocessed += 10;
                SCReturnUInt(10U);
            } else {
                /* max_xmit_frag */
                p++;
                if (!(--input_len))
                    break;
            }
        case 17:
            /* max_xmit_frag */
            p++;
            if (!(--input_len))
                break;
        case 18:
            /* max_recv_frag */
            p++;
            if (!(--input_len))
                break;
        case 19:
            /* max_recv_frag */
            p++;
            if (!(--input_len))
                break;
        case 20:
            /* assoc_group_id */
            p++;
            if (!(--input_len))
                break;
        case 21:
            /* assoc_group_id */
            p++;
            if (!(--input_len))
                break;
        case 22:
            /* assoc_group_id */
            p++;
            if (!(--input_len))
                break;
        case 23:
            /* assoc_group_id */
            p++;
            if (!(--input_len))
                break;
        case 24:
            dcerpc->dcerpcbindbindack.secondaryaddrlen = *(p++) << 8;
            if (!(--input_len))
                break;
        case 25:
            dcerpc->dcerpcbindbindack.secondaryaddrlen |= *(p++);
            if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                dcerpc->dcerpcbindbindack.secondaryaddrlen = SCByteSwap16(dcerpc->dcerpcbindbindack.secondaryaddrlen);
            }
            dcerpc->dcerpcbindbindack.secondaryaddrlenleft = dcerpc->dcerpcbindbindack.secondaryaddrlen;
            SCLogDebug("secondaryaddrlen %u 0x%04x\n", dcerpc->dcerpcbindbindack.secondaryaddrlen,
                    dcerpc->dcerpcbindbindack.secondaryaddrlen);
            --input_len;
            break;
        default:
            dcerpc->bytesprocessed++;
            SCReturnUInt(1);
            break;
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

static uint32_t DCERPCParseREQUEST(DCERPC *dcerpc, uint8_t *input, uint32_t input_len) {
    SCEnter();
    uint8_t *p = input;

    switch (dcerpc->bytesprocessed) {
        case 16:
            dcerpc->dcerpcbindbindack.numctxitems = 0;
            if (input_len >= 8) {
                if (dcerpc->dcerpchdr.type == REQUEST) {
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpcrequest.ctxid = *(p + 4);
                        dcerpc->dcerpcrequest.ctxid |= *(p + 5) << 8;
                        dcerpc->dcerpcrequest.opnum = *(p + 6);
                        dcerpc->dcerpcrequest.opnum |= *(p + 7) << 8;
                    } else {
                        dcerpc->dcerpcrequest.ctxid = *(p + 4) << 8;
                        dcerpc->dcerpcrequest.ctxid |= *(p + 5);
                        dcerpc->dcerpcrequest.opnum = *(p + 6) << 8;
                        dcerpc->dcerpcrequest.opnum |= *(p + 7);
                    }
                    dcerpc->dcerpcrequest.first_request_seen = 1;
                }
                dcerpc->bytesprocessed += 8;
                SCReturnUInt(8U);
            } else {
                /* alloc hint 1 */
                p++;
                if (!(--input_len))
                    break;
            }
        case 17:
            /* alloc hint 2 */
            p++;
            if (!(--input_len))
                break;
        case 18:
            /* alloc hint 3 */
            p++;
            if (!(--input_len))
                break;
        case 19:
            /* alloc hint 4 */
            p++;
            if (!(--input_len))
                break;
        case 20:
            /* context id 1 */
            dcerpc->dcerpcrequest.ctxid = *(p++);
            if (!(--input_len))
                break;
        case 21:
            /* context id 2 */
            dcerpc->dcerpcrequest.ctxid |= *(p++) << 8;
            if (!(dcerpc->dcerpchdr.packed_drep[0] & 0x10)) {
                dcerpc->dcerpcrequest.ctxid = SCByteSwap16(dcerpc->dcerpcrequest.ctxid);
            }
            dcerpc->dcerpcrequest.first_request_seen = 1;
            if (!(--input_len))
                break;
        case 22:
            if (dcerpc->dcerpchdr.type == REQUEST) {
                dcerpc->dcerpcrequest.opnum = *(p++);
            } else {
                p++;
            }
            if (!(--input_len))
                break;
        case 23:
            if (dcerpc->dcerpchdr.type == REQUEST) {
                dcerpc->dcerpcrequest.opnum |= *(p++) << 8;
                if (!(dcerpc->dcerpchdr.packed_drep[0] & 0x10)) {
                    dcerpc->dcerpcrequest.opnum = SCByteSwap16(dcerpc->dcerpcrequest.opnum);
                }
            } else {
                p++;
            }
            --input_len;
            break;
        default:
            dcerpc->bytesprocessed++;
            SCReturnUInt(1);
            break;
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnUInt((uint32_t)(p - input));
}

static uint32_t StubDataParser(DCERPC *dcerpc, uint8_t *input, uint32_t input_len) {
    SCEnter();
    uint8_t **stub_data_buffer = NULL;
    uint32_t *stub_data_buffer_len = NULL;
    uint8_t *stub_data_fresh = NULL;
    uint16_t stub_len = 0;

    /* request PDU.  Retrieve the request stub buffer */
    if (dcerpc->dcerpchdr.type == REQUEST) {
        stub_data_buffer = &dcerpc->dcerpcrequest.stub_data_buffer;
        stub_data_buffer_len = &dcerpc->dcerpcrequest.stub_data_buffer_len;
        stub_data_fresh = &dcerpc->dcerpcrequest.stub_data_fresh;

    /* response PDU.  Retrieve the response stub buffer */
    } else {
        stub_data_buffer = &dcerpc->dcerpcresponse.stub_data_buffer;
        stub_data_buffer_len = &dcerpc->dcerpcresponse.stub_data_buffer_len;
        stub_data_fresh = &dcerpc->dcerpcresponse.stub_data_fresh;
    }

    stub_len = (dcerpc->padleft < input_len) ? dcerpc->padleft : input_len;
    if (stub_len == 0) {
        SCLogError(SC_ERR_DCERPC, "stub_len is NULL.  We shouldn't be seeing "
                   "this.  In case you are, there is something gravely wrong "
                   "with the dcerpc parser");
        SCReturnInt(0);
    }

    /* To see what is in this stub fragment */
    //hexdump(input, stub_len);
    /* if the frag is the the first frag irrespective of it being a part of
     * a multi frag PDU or not, it indicates the previous PDU's stub would
     * have been buffered and processed and we can use the buffer to hold
     * frags from a fresh request/response.  Also if the state is in the
     * process of processing a fragmented pdu, we should append to the
     * existing stub and not reset the stub buffer */
    if ((dcerpc->dcerpchdr.pfc_flags & PFC_FIRST_FRAG) &&
        !dcerpc->pdu_fragged) {
        *stub_data_buffer_len = 0;
        /* just a hack to get thing working.  We shouldn't be setting
         * this var here.  The ideal thing would have been to use
         * an extra state var, to indicate that the stub parser has made a
         * fresh entry after reseting the buffer, but maintaing an extra var
         * would be a nuisance, while we can achieve the same thing with
         * little or no effort, with a simple set here, although semantically
         * it is a wrong thing to set it here, since we still can't conclude
         * if a pdu is fragmented or not at this point, if we are parsing a PDU
         * that has some stub data in the first segment, but it still doesn't
         * contain the entire PDU */
        dcerpc->pdu_fragged = 1;
    }

    *stub_data_buffer = SCRealloc(*stub_data_buffer, *stub_data_buffer_len + stub_len);
    if (*stub_data_buffer == NULL) {
        SCLogError(SC_ERR_MEM_ALLOC, "Error allocating memory");
        goto end;
    }
    memcpy(*stub_data_buffer + *stub_data_buffer_len, input, stub_len);

    *stub_data_fresh = 1;
    /* length of the buffered stub */
    *stub_data_buffer_len += stub_len;
    /* To see the total reassembled stubdata */
    //hexdump(*stub_data_buffer, *stub_data_buffer_len);

    dcerpc->padleft -= stub_len;
    dcerpc->bytesprocessed += stub_len;

#ifdef DEBUG
    if (SCLogDebugEnabled()) {
        int i = 0;
        for (i = 0; i < stub_len; i++) {
            SCLogDebug("0x%02x ", input[i]);
        }
    }
#endif

end:
    SCReturnUInt((uint32_t)stub_len);
}

/**
 * \brief DCERPCParseHeader parses the 16 byte DCERPC header
 * A fast path for normal decoding is used when there is enough bytes
 * present to parse the entire header. A slow path is used to parse
 * fragmented packets.
 * \retval -1 if DCERPC Header does not validate
 * \retval Number of bytes processed
 */
static int DCERPCParseHeader(DCERPC *dcerpc, uint8_t *input, uint32_t input_len) {
    SCEnter();
    uint8_t *p = input;

    if (input_len) {
        SCLogDebug("dcerpc->bytesprocessed %u", dcerpc->bytesprocessed);
        switch (dcerpc->bytesprocessed) {
            case 0:
                if (input_len >= DCERPC_HDR_LEN) {
                    dcerpc->dcerpchdr.rpc_vers = *p;
                    dcerpc->dcerpchdr.rpc_vers_minor = *(p + 1);
                    if ((dcerpc->dcerpchdr.rpc_vers != 5) ||
                       ((dcerpc->dcerpchdr.rpc_vers_minor != 0) &&
                       (dcerpc->dcerpchdr.rpc_vers_minor != 1))) {
							SCLogDebug("DCERPC Header did not validate");
							SCReturnInt(-1);
                    }
                    dcerpc->dcerpchdr.type = *(p + 2);
                    SCLogDebug("dcerpc->dcerpchdr.type %02x",
                            dcerpc->dcerpchdr.type);
                    dcerpc->dcerpchdr.pfc_flags = *(p + 3);
                    dcerpc->dcerpchdr.packed_drep[0] = *(p + 4);
                    dcerpc->dcerpchdr.packed_drep[1] = *(p + 5);
                    dcerpc->dcerpchdr.packed_drep[2] = *(p + 6);
                    dcerpc->dcerpchdr.packed_drep[3] = *(p + 7);
                    if (dcerpc->dcerpchdr.packed_drep[0] & 0x10) {
                        dcerpc->dcerpchdr.frag_length = *(p + 8);
                        dcerpc->dcerpchdr.frag_length |= *(p + 9) << 8;
                        dcerpc->dcerpchdr.auth_length = *(p + 10);
                        dcerpc->dcerpchdr.auth_length |= *(p + 11) << 8;
                        dcerpc->dcerpchdr.call_id = *(p + 12) << 24;
                        dcerpc->dcerpchdr.call_id |= *(p + 13) << 16;
                        dcerpc->dcerpchdr.call_id |= *(p + 14) << 8;
                        dcerpc->dcerpchdr.call_id |= *(p + 15);
                    } else {
                        dcerpc->dcerpchdr.frag_length = *(p + 8) << 8;
                        dcerpc->dcerpchdr.frag_length |= *(p + 9);
                        dcerpc->dcerpchdr.auth_length = *(p + 10) << 8;
                        dcerpc->dcerpchdr.auth_length |= *(p + 11);
                        dcerpc->dcerpchdr.call_id = *(p + 12);
                        dcerpc->dcerpchdr.call_id |= *(p + 13) << 8;
                        dcerpc->dcerpchdr.call_id |= *(p + 14) << 16;
                        dcerpc->dcerpchdr.call_id |= *(p + 15) << 24;
                    }
                    dcerpc->bytesprocessed = DCERPC_HDR_LEN;
                    SCReturnInt(16);
                    break;
                } else {
                    dcerpc->dcerpchdr.rpc_vers = *(p++);
                    if (!(--input_len))
                        break;
                }
            case 1:
                dcerpc->dcerpchdr.rpc_vers_minor = *(p++);
                if ((dcerpc->dcerpchdr.rpc_vers != 5) ||
                    ((dcerpc->dcerpchdr.rpc_vers_minor != 0) &&
                    (dcerpc->dcerpchdr.rpc_vers_minor != 1))) {
						SCLogDebug("DCERPC Header did not validate");
						SCReturnInt(-1);
                }
                if (!(--input_len))
                    break;
            case 2:
                dcerpc->dcerpchdr.type = *(p++);
                SCLogDebug("dcerpc->dcerpchdr.type %02x",
                        dcerpc->dcerpchdr.type);
                if (!(--input_len))
                    break;
            case 3:
                dcerpc->dcerpchdr.pfc_flags = *(p++);
                if (!(--input_len))
                    break;
            case 4:
                dcerpc->dcerpchdr.packed_drep[0] = *(p++);
                if (!(--input_len))
                    break;
            case 5:
                dcerpc->dcerpchdr.packed_drep[1] = *(p++);
                if (!(--input_len))
                    break;
            case 6:
                dcerpc->dcerpchdr.packed_drep[2] = *(p++);
                if (!(--input_len))
                    break;
            case 7:
                dcerpc->dcerpchdr.packed_drep[3] = *(p++);
                if (!(--input_len))
                    break;
            case 8:
                dcerpc->dcerpchdr.frag_length = *(p++);
                if (!(--input_len))
                    break;
            case 9:
                dcerpc->dcerpchdr.frag_length |= *(p++) << 8;
                if (!(--input_len))
                    break;
            case 10:
                dcerpc->dcerpchdr.auth_length = *(p++);
                if (!(--input_len))
                    break;
            case 11:
                dcerpc->dcerpchdr.auth_length |= *(p++) << 8;
                if (!(--input_len))
                    break;
            case 12:
                dcerpc->dcerpchdr.call_id = *(p++);
                if (!(--input_len))
                    break;
            case 13:
                dcerpc->dcerpchdr.call_id |= *(p++) << 8;
                if (!(--input_len))
                    break;
            case 14:
                dcerpc->dcerpchdr.call_id |= *(p++) << 16;
                if (!(--input_len))
                    break;
            case 15:
                dcerpc->dcerpchdr.call_id |= *(p++) << 24;
                if (!(dcerpc->dcerpchdr.packed_drep[0] & 0x10)) {
                    dcerpc->dcerpchdr.frag_length = SCByteSwap16(dcerpc->dcerpchdr.frag_length);
                    dcerpc->dcerpchdr.auth_length = SCByteSwap16(dcerpc->dcerpchdr.auth_length);
                    dcerpc->dcerpchdr.call_id = SCByteSwap32(dcerpc->dcerpchdr.call_id);
                }
                --input_len;
                break;
            default:
                dcerpc->bytesprocessed++;
                SCReturnInt(1);
        }
    }
    dcerpc->bytesprocessed += (p - input);
    SCReturnInt((p - input));
}

static inline void DCERPCResetParsingState(DCERPC *dcerpc) {
    dcerpc->bytesprocessed = 0;
    dcerpc->pdu_fragged = 0;
    dcerpc->dcerpcbindbindack.ctxbytesprocessed = 0;

    return;
}

static inline void DCERPCResetStub(DCERPC *dcerpc) {
    if (dcerpc->dcerpchdr.type == REQUEST)
        dcerpc->dcerpcrequest.stub_data_buffer_len = 0;
    else if (dcerpc->dcerpchdr.type == RESPONSE)
        dcerpc->dcerpcresponse.stub_data_buffer_len = 0;

    return;
}

static inline int DCERPCThrowOutExtraData(DCERPC *dcerpc, uint8_t *input,
                                           uint16_t input_len) {
    int parsed = 0;
    /* the function always assumes that
     * dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length */
    if (input_len > (dcerpc->dcerpchdr.frag_length - dcerpc->bytesprocessed)) {
        parsed = dcerpc->dcerpchdr.frag_length - dcerpc->bytesprocessed;
    } else {
        parsed = input_len;
    }
    dcerpc->bytesprocessed += parsed;

    return parsed;
}

/**
 * \todo - Currently the parser is very generic.  Enable target based
 *         reassembly.
 *       - Disable reiniting tailq for mid and last bind/alter_context pdus.
 *       - Use a PM to search for subsequent 05 00 when we see an inconsistent
 *         pdu.  This should be done for each platform based on how it handles
 *         a condition where it has receives a segment with 2 pdus, while the
 *         first pdu in the segment is corrupt.
 */
int32_t DCERPCParser(DCERPC *dcerpc, uint8_t *input, uint32_t input_len) {
    SCEnter();

    uint32_t retval = 0;
    uint32_t parsed = 0;
    int hdrretval = 0;

    dcerpc->dcerpcrequest.stub_data_fresh = 0;
    dcerpc->dcerpcresponse.stub_data_fresh = 0;

    /* temporary use.  we will get rid of this later, once we have ironed out
     * all the endless loops cases */
    int counter = 0;

    while (input_len) {
        /* in case we have any corner cases remainging, we have this */
        if (counter++ == 30) {
            SCLogDebug("Somehow seem to be stuck inside the dce "
                    "parser for quite sometime.  Let's get out of here.");
            DCERPCResetParsingState(dcerpc);
            SCReturnInt(0);
        }

        while (dcerpc->bytesprocessed < DCERPC_HDR_LEN && input_len) {
            hdrretval = DCERPCParseHeader(dcerpc, input + parsed, input_len);
            if (hdrretval == -1 || hdrretval > (int32_t)input_len) {
                SCLogDebug("Error parsing dce header.  Discarding "
                        "PDU and reseting parsing state to parse next PDU");
                /* error parsing pdu header.  Let's clear the dce state */
                DCERPCResetParsingState(dcerpc);
                SCReturnInt(0);
            } else {
                parsed += hdrretval;
                input_len -= hdrretval;
            }
        }
        SCLogDebug("Done with DCERPCParseHeader bytesprocessed %u/%u left %u",
                   dcerpc->bytesprocessed, dcerpc->dcerpchdr.frag_length, input_len);
#if 0
        printf("Done with DCERPCParseHeader bytesprocessed %u/%u input_len left %u\n",
               dcerpc->bytesprocessed, dcerpc->dcerpchdr.frag_length, input_len);
        printf("\nDCERPC Version:\t%u\n", dcerpc->dcerpchdr.rpc_vers);
        printf("DCERPC Version Minor:\t%u\n", dcerpc->dcerpchdr.rpc_vers_minor);
        printf("DCERPC Type:\t%u\n", dcerpc->dcerpchdr.type);
        printf("DCERPC Flags:\t0x%02x\n", dcerpc->dcerpchdr.pfc_flags);
        printf("DCERPC Packed Drep:\t%02x %02x %02x %02x\n",
               dcerpc->dcerpchdr.packed_drep[0], dcerpc->dcerpchdr.packed_drep[1],
               dcerpc->dcerpchdr.packed_drep[2], dcerpc->dcerpchdr.packed_drep[3]);
        printf("DCERPC Frag Length:\t0x%04x %u\n",
               dcerpc->dcerpchdr.frag_length, dcerpc->dcerpchdr.frag_length);
        printf("DCERPC Auth Length:\t0x%04x\n", dcerpc->dcerpchdr.auth_length);
        printf("DCERPC Call Id:\t0x%08x\n", dcerpc->dcerpchdr.call_id);
#endif

        /* check if we have parsed the entire input passed in the header parser.
         * If we have, time to leave */
        if (input_len == 0) {
            if (dcerpc->bytesprocessed < 10) {
                /* if the parser is known to be fragmented at this stage itself,
                 * we reset the stub buffer here itself */
                if (!dcerpc->pdu_fragged && (dcerpc->dcerpchdr.pfc_flags & PFC_FIRST_FRAG)) {
                    DCERPCResetStub(dcerpc);
                }
                dcerpc->pdu_fragged = 1;
            } else {
                if (dcerpc->bytesprocessed >= dcerpc->dcerpchdr.frag_length) {
                    SCLogDebug("Weird DCE PDU");
                    DCERPCResetParsingState(dcerpc);
                } else {
                    /* if the parser is known to be fragmented at this stage itself,
                     * we reset the stub buffer here itself */
                    if (!dcerpc->pdu_fragged && (dcerpc->dcerpchdr.pfc_flags & PFC_FIRST_FRAG)) {
                        DCERPCResetStub(dcerpc);
                    }
                    dcerpc->pdu_fragged = 1;
                }
            }
            SCReturnInt(parsed);
        }
        switch (dcerpc->dcerpchdr.type) {
            case BIND:
            case ALTER_CONTEXT:
                while (dcerpc->bytesprocessed < DCERPC_HDR_LEN + 12
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length
                       && input_len) {
                    retval = DCERPCParseBIND(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                    } else if (input_len) {
                        SCLogDebug("Error Parsing DCERPC %s PDU",
                                   (dcerpc->dcerpchdr.type == BIND) ?
                                   "BIND" : "ALTER_CONTEXT");
                        parsed = 0;
                        input_len = 0;
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }
                SCLogDebug("Done with DCERPCParseBIND bytesprocessed %u/%u numctxitems %u",
                           dcerpc->bytesprocessed, dcerpc->dcerpchdr.frag_length,
                           dcerpc->dcerpcbindbindack.numctxitems);
                while (dcerpc->dcerpcbindbindack.numctxitemsleft && dcerpc->bytesprocessed
                       < dcerpc->dcerpchdr.frag_length && input_len) {
                    retval = DCERPCParseBINDCTXItem(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        if (dcerpc->dcerpcbindbindack.ctxbytesprocessed == 44) {
                            dcerpc->dcerpcbindbindack.ctxbytesprocessed = 0;
                        }
                        parsed += retval;
                        input_len -= retval;
                        SCLogDebug("BIND processed %u/%u ctxitems %u/%u input_len left %u\n",
                                   dcerpc->bytesprocessed,
                                   dcerpc->dcerpchdr.frag_length,
                                   dcerpc->dcerpcbindbindack.numctxitemsleft,
                                   dcerpc->dcerpcbindbindack.numctxitems, input_len);
                    } else if (input_len) {
                        //parsed -= input_len;
                        SCLogDebug("Error Parsing CTX Item %u\n", parsed);
                        parsed = 0;
                        input_len = 0;
                        dcerpc->dcerpcbindbindack.numctxitemsleft = 0;
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }
                if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                    DCERPCResetParsingState(dcerpc);
                } else if (dcerpc->bytesprocessed > dcerpc->dcerpchdr.frag_length) {
                    DCERPCResetParsingState(dcerpc);
                    SCReturnInt(0);
                } else {
                    /* temporary fix */
                    if (input_len) {
                        retval = DCERPCThrowOutExtraData(dcerpc, input + parsed,
                                                         input_len);
                        if (retval && retval <= input_len) {
                            input_len -= retval;
                            parsed += retval;
                            if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                                DCERPCResetParsingState(dcerpc);
                            } else {
                                dcerpc->pdu_fragged = 1;
                            }
                        } else {
                            SCLogDebug("Error Parsing DCERPC");
                            parsed = 0;
                            input_len = 0;
                            DCERPCResetParsingState(dcerpc);
                            SCReturnInt(0);
                        }
                    } else {
                        dcerpc->pdu_fragged = 1;
                    }
                }
                break;

            case BIND_ACK:
            case ALTER_CONTEXT_RESP:
                while (dcerpc->bytesprocessed < DCERPC_HDR_LEN + 9
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length
                       && input_len) {
                    retval = DCERPCParseBINDACK(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                        SCLogDebug("DCERPCParseBINDACK processed %u/%u input_len left %u",
                                   dcerpc->bytesprocessed,
                                   dcerpc->dcerpchdr.frag_length, input_len);
                    } else if (input_len) {
                        SCLogDebug("Error parsing %s\n",
                                   (dcerpc->dcerpchdr.type == BIND_ACK) ?
                                   "BIND_ACK" : "ALTER_CONTEXT_RESP");
                        parsed = 0;
                        input_len = 0;
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                while (dcerpc->bytesprocessed < DCERPC_HDR_LEN + 10
                       + dcerpc->dcerpcbindbindack.secondaryaddrlen
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length && input_len) {
                    retval = DCERPCParseSecondaryAddr(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                        SCLogDebug("DCERPCParseSecondaryAddr %u/%u left %u secondaryaddr len(%u)",
                                   dcerpc->bytesprocessed, dcerpc->dcerpchdr.frag_length, input_len,
                                   dcerpc->dcerpcbindbindack.secondaryaddrlen);
                    } else if (input_len) {
                        SCLogDebug("Error parsing Secondary Address");
                        parsed = 0;
                        input_len = 0;
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                if (dcerpc->bytesprocessed == DCERPC_HDR_LEN + 10
                    + dcerpc->dcerpcbindbindack.secondaryaddrlen) {
                    if (dcerpc->bytesprocessed % 4) {
                        dcerpc->pad = (4 - dcerpc->bytesprocessed % 4);
                        dcerpc->padleft = dcerpc->pad;
                    }
                }

                while (dcerpc->bytesprocessed < DCERPC_HDR_LEN + 10
                       + dcerpc->dcerpcbindbindack.secondaryaddrlen + dcerpc->pad
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length && input_len) {
                    retval = PaddingParser(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                        SCLogDebug("PaddingParser %u/%u left %u pad(%u)",
                                   dcerpc->bytesprocessed, dcerpc->dcerpchdr.frag_length, input_len,
                                   dcerpc->pad);
                    } else if (input_len) {
                        SCLogDebug("Error parsing DCERPC Padding");
                        parsed = 0;
                        input_len = 0;
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                while (dcerpc->bytesprocessed >= DCERPC_HDR_LEN + 10 + dcerpc->pad
                       + dcerpc->dcerpcbindbindack.secondaryaddrlen && dcerpc->bytesprocessed
                       < DCERPC_HDR_LEN + 14 + dcerpc->pad + dcerpc->dcerpcbindbindack.secondaryaddrlen
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length && input_len) {
                    retval = DCERPCGetCTXItems(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                        SCLogDebug("DCERPCGetCTXItems %u/%u (%u)", dcerpc->bytesprocessed,
                                   dcerpc->dcerpchdr.frag_length, dcerpc->dcerpcbindbindack.numctxitems);
                    } else if (input_len) {
                        SCLogDebug("Error parsing CTX Items");
                        parsed = 0;
                        input_len = 0;
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                if (dcerpc->bytesprocessed == DCERPC_HDR_LEN + 14 + dcerpc->pad
                    + dcerpc->dcerpcbindbindack.secondaryaddrlen) {
                    dcerpc->dcerpcbindbindack.ctxbytesprocessed = 0;
                }

                while (dcerpc->dcerpcbindbindack.numctxitemsleft && dcerpc->bytesprocessed
                       < dcerpc->dcerpchdr.frag_length && input_len) {
                    retval = DCERPCParseBINDACKCTXItem(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        if (dcerpc->dcerpcbindbindack.ctxbytesprocessed == 24) {
                            dcerpc->dcerpcbindbindack.ctxbytesprocessed = 0;
                        }
                        parsed += retval;
                        input_len -= retval;
                    } else if (input_len) {
                        SCLogDebug("Error parsing CTX Items");
                        parsed = 0;
                        input_len = 0;
                        dcerpc->dcerpcbindbindack.numctxitemsleft = 0;
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }
                SCLogDebug("BINDACK processed %u/%u input_len left %u",
                           dcerpc->bytesprocessed,
                           dcerpc->dcerpchdr.frag_length, input_len);

                if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                    /* response and request done */
                    if (dcerpc->dcerpchdr.type == BIND_ACK) {
                        /* update transaction id */
                        dcerpc->transaction_id++;
                        SCLogDebug("transaction_id updated to %"PRIu16,
                                   dcerpc->transaction_id);
                    }
                    DCERPCResetParsingState(dcerpc);
                } else if (dcerpc->bytesprocessed > dcerpc->dcerpchdr.frag_length) {
                    DCERPCResetParsingState(dcerpc);
                    SCReturnInt(0);
                } else {
                    /* temporary fix */
                    if (input_len) {
                        retval = DCERPCThrowOutExtraData(dcerpc, input + parsed,
                                                             input_len);
                        if (retval && retval <= input_len) {
                            input_len -= retval;
                            parsed += retval;
                            if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                                DCERPCResetParsingState(dcerpc);
                            } else {
                                dcerpc->pdu_fragged = 1;
                            }
                        } else {
                            SCLogDebug("Error Parsing DCERPC");
                            parsed = 0;
                            input_len = 0;
                            DCERPCResetParsingState(dcerpc);
                            SCReturnInt(0);
                        }
                    } else {
                        dcerpc->pdu_fragged = 1;
                    }
                }
                break;

            case REQUEST:
            case RESPONSE:
                while (dcerpc->bytesprocessed < DCERPC_HDR_LEN + 8
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length
                       && input_len) {
                    retval = DCERPCParseREQUEST(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                        dcerpc->padleft = dcerpc->dcerpchdr.frag_length - dcerpc->bytesprocessed;
                    } else if (input_len) {
                        SCLogDebug("Error parsing DCERPC %s",
                                   (dcerpc->dcerpchdr.type == REQUEST) ? "REQUEST" : "RESPONSE");
                        parsed = 0;
                        dcerpc->padleft = 0;
                        input_len = 0;
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                while (dcerpc->bytesprocessed >= DCERPC_HDR_LEN + 8
                       && dcerpc->bytesprocessed < dcerpc->dcerpchdr.frag_length
                       && dcerpc->padleft && input_len) {
                    retval = StubDataParser(dcerpc, input + parsed, input_len);
                    if (retval && retval <= input_len) {
                        parsed += retval;
                        input_len -= retval;
                    } else if (input_len) {
                        SCLogDebug("Error parsing DCERPC Stub Data");
                        parsed = 0;
                        input_len = 0;
                        DCERPCResetParsingState(dcerpc);
                        SCReturnInt(0);
                    }
                }

                if (dcerpc->dcerpchdr.type == REQUEST) {
                    SCLogDebug("REQUEST processed %u frag length %u opnum %u input_len %u", dcerpc->bytesprocessed,
                               dcerpc->dcerpchdr.frag_length, dcerpc->dcerpcrequest.opnum, input_len);
                } else {
                    SCLogDebug("RESPONSE processed %u frag length %u opnum %u input_len %u", dcerpc->bytesprocessed,
                               dcerpc->dcerpchdr.frag_length, dcerpc->dcerpcrequest.opnum, input_len);
                }

                /* don't see how we could break the parser for request pdus, by
                 * pusing bytesprocessed beyond frag_length.  Let's have the
                 * check anyways */
                if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                    DCERPCResetParsingState(dcerpc);
                } else if (dcerpc->bytesprocessed > dcerpc->dcerpchdr.frag_length) {
                    DCERPCResetParsingState(dcerpc);
                    SCReturnInt(0);
                } else {
                    if (!dcerpc->pdu_fragged &&
                        (dcerpc->dcerpchdr.pfc_flags & PFC_FIRST_FRAG)) {
                        DCERPCResetStub(dcerpc);
                    }
                    /* temporary fix */
                    if (input_len) {
                        retval = DCERPCThrowOutExtraData(dcerpc, input + parsed,
                                                             input_len);
                        if (retval && retval <= input_len) {
                            input_len -= retval;
                            parsed += retval;
                            if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                                DCERPCResetParsingState(dcerpc);
                            } else {
                                dcerpc->pdu_fragged = 1;
                            }
                        } else {
                            SCLogDebug("Error Parsing DCERPC");
                            parsed = 0;
                            input_len = 0;
                            DCERPCResetParsingState(dcerpc);
                            SCReturnInt(0);
                        }
                    } else {
                        dcerpc->pdu_fragged = 1;
                    }
                }

                /* response and request done */
                if (dcerpc->dcerpchdr.type == RESPONSE) {
                    /* update transaction id */
                    dcerpc->transaction_id++;
                    SCLogDebug("transaction_id updated to %"PRIu16,
                               dcerpc->transaction_id);
                }
                break;

            default:
                SCLogDebug("DCERPC Type 0x%02x not implemented yet", dcerpc->dcerpchdr.type);
                retval = DCERPCThrowOutExtraData(dcerpc, input + parsed,
                                                 input_len);
                if (retval && retval <= input_len) {
                    input_len -= retval;
                    parsed += retval;
                    if (dcerpc->bytesprocessed == dcerpc->dcerpchdr.frag_length) {
                        DCERPCResetParsingState(dcerpc);
                    } else {
                        dcerpc->pdu_fragged = 1;
                    }
                } else {
                    SCLogDebug("Error Parsing DCERPC");
                    parsed = 0;
                    input_len = 0;
                    DCERPCResetParsingState(dcerpc);
                    SCReturnInt(0);
                }
                break;
        }
    }

    SCReturnInt(parsed);
}

static int DCERPCParse(Flow *f, void *dcerpc_state,
                       AppLayerParserState *pstate,
                       uint8_t *input, uint32_t input_len,
                       void *local_data, AppLayerParserResult *output) {
    SCEnter();

    int32_t retval = 0;
    DCERPCState *sstate = (DCERPCState *) dcerpc_state;

    retval = DCERPCParser(&sstate->dcerpc, input, input_len);
    if (retval == -1) {
        SCReturnInt(0);
    }

    if (pstate == NULL)
        SCReturnInt(-1);

    pstate->parse_field = 0;

    SCReturnInt(1);
}

static void *DCERPCStateAlloc(void) {
    SCEnter();

    DCERPCState *s = SCMalloc(sizeof(DCERPCState));
    if (unlikely(s == NULL)) {
        SCReturnPtr(NULL, "void");
    }
    memset(s, 0, sizeof(DCERPCState));

    s->dcerpc.transaction_id = 1;

    SCReturnPtr((void *)s, "void");
}

static void DCERPCStateFree(void *s) {
    DCERPCState *sstate = (DCERPCState *) s;

    DCERPCUuidEntry *item;

    while ((item = TAILQ_FIRST(&sstate->dcerpc.dcerpcbindbindack.uuid_list))) {
        //printUUID("Free", item);
        TAILQ_REMOVE(&sstate->dcerpc.dcerpcbindbindack.uuid_list, item, next);
        SCFree(item);
    }

    while ((item = TAILQ_FIRST(&sstate->dcerpc.dcerpcbindbindack.accepted_uuid_list))) {
        //printUUID("Free", item);
        TAILQ_REMOVE(&sstate->dcerpc.dcerpcbindbindack.accepted_uuid_list, item, next);
        SCFree(item);
    }

    if (sstate->dcerpc.dcerpcrequest.stub_data_buffer != NULL) {
        SCFree(sstate->dcerpc.dcerpcrequest.stub_data_buffer);
        sstate->dcerpc.dcerpcrequest.stub_data_buffer = NULL;
        sstate->dcerpc.dcerpcrequest.stub_data_buffer_len = 0;
    }
    if (sstate->dcerpc.dcerpcresponse.stub_data_buffer != NULL) {
        SCFree(sstate->dcerpc.dcerpcresponse.stub_data_buffer);
        sstate->dcerpc.dcerpcresponse.stub_data_buffer = NULL;
        sstate->dcerpc.dcerpcresponse.stub_data_buffer_len = 0;
    }

    if (s) {
        SCFree(s);
        s = NULL;
    }
}

/**
 *  \brief Update the transaction id based on the DCERPC state
 */
void DCERPCUpdateTransactionId(void *state, uint16_t *id) {
    SCEnter();

    DCERPCState *s = (DCERPCState *)state;
    SCLogDebug("original id %"PRIu16, *id);
    (*id) = s->dcerpc.transaction_id;
    SCLogDebug("updated id %"PRIu16, *id);

    SCReturn;
}

void RegisterDCERPCParsers(void) {
    char *proto_name = "dcerpc";

    /** DCERPC */
    AlpProtoAdd(&alp_proto_ctx, proto_name, IPPROTO_TCP, ALPROTO_DCERPC, "|05 00|", 2, 0, STREAM_TOSERVER);

    AppLayerRegisterProto(proto_name, ALPROTO_DCERPC, STREAM_TOSERVER,
            DCERPCParse);
    AppLayerRegisterProto(proto_name, ALPROTO_DCERPC, STREAM_TOCLIENT,
            DCERPCParse);
    AppLayerRegisterStateFuncs(ALPROTO_DCERPC, DCERPCStateAlloc,
            DCERPCStateFree);
    AppLayerRegisterTransactionIdFuncs(ALPROTO_DCERPC,
            DCERPCUpdateTransactionId, NULL);
}

/* UNITTESTS */
#ifdef UNITTESTS
/** \test DCERPC Header Parsing and  BIND / BIND_ACK multiple UUID handling
*/

/* set this to 1 to see problem */

int DCERPCParserTest01(void) {
    int result = 1;
    Flow f;
    uint8_t dcerpcbind[] = {
        0x05, 0x00,
        0x0b, 0x03, 0x10, 0x00, 0x00, 0x00, 0x3c, 0x04,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd0, 0x16,
        0xd0, 0x16, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x2c, 0xd0,
        0x28, 0xda, 0x76, 0x91, 0xf6, 0x6e, 0xcb, 0x0f,
        0xbf, 0x85, 0xcd, 0x9b, 0xf6, 0x39, 0x01, 0x00,
        0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x01, 0x00, 0x2c, 0x75, 0xce, 0x7e, 0x82, 0x3b,
        0x06, 0xac, 0x1b, 0xf0, 0xf5, 0xb7, 0xa7, 0xf7,
        0x28, 0xaf, 0x05, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x01, 0x00, 0xe3, 0xb2,
        0x10, 0xd1, 0xd0, 0x0c, 0xcc, 0x3d, 0x2f, 0x80,
        0x20, 0x7c, 0xef, 0xe7, 0x09, 0xe0, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x01, 0x00, 0xde, 0x85, 0x70, 0xc4, 0x02, 0x7c,
        0x60, 0x23, 0x67, 0x0c, 0x22, 0xbf, 0x18, 0x36,
        0x79, 0x17, 0x01, 0x00, 0x02, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x01, 0x00, 0x41, 0x65,
        0x29, 0x51, 0xaa, 0xe7, 0x7b, 0xa8, 0xf2, 0x37,
        0x0b, 0xd0, 0x3f, 0xb3, 0x36, 0xed, 0x05, 0x00,
        0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x05, 0x00,
        0x01, 0x00, 0x14, 0x96, 0x80, 0x01, 0x2e, 0x78,
        0xfb, 0x5d, 0xb4, 0x3c, 0x14, 0xb3, 0x3d, 0xaa,
        0x02, 0xfb, 0x06, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x06, 0x00, 0x01, 0x00, 0x3b, 0x04,
        0x68, 0x3e, 0x63, 0xfe, 0x9f, 0xd8, 0x64, 0x55,
        0xcd, 0xe7, 0x39, 0xaf, 0x98, 0x9f, 0x03, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x07, 0x00,
        0x01, 0x00, 0x16, 0x7a, 0x4f, 0x1b, 0xdb, 0x25,
        0x92, 0x55, 0xdd, 0xae, 0x9e, 0x5b, 0x3e, 0x93,
        0x66, 0x93, 0x04, 0x00, 0x01, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x01, 0x00, 0xe8, 0xa4,
        0x8a, 0xcf, 0x95, 0x6c, 0xc7, 0x8f, 0x14, 0xcc,
        0x56, 0xfc, 0x7b, 0x5f, 0x4f, 0xe8, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x01, 0x00, 0xd8, 0xda, 0xfb, 0xbc, 0xa2, 0x55,
        0x6f, 0x5d, 0xc0, 0x2d, 0x88, 0x6f, 0x00, 0x17,
        0x52, 0x8d, 0x06, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x0a, 0x00, 0x01, 0x00, 0x3f, 0x17,
        0x55, 0x0c, 0xf4, 0x23, 0x3c, 0xca, 0xe6, 0xa0,
        0xaa, 0xcc, 0xb5, 0xe3, 0xf9, 0xce, 0x04, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x0b, 0x00,
        0x01, 0x00, 0x6a, 0x28, 0x19, 0x39, 0x0c, 0xb1,
        0xd0, 0x11, 0x9b, 0xa8, 0x00, 0xc0, 0x4f, 0xd9,
        0x2e, 0xf5, 0x00, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x0c, 0x00, 0x01, 0x00, 0xc9, 0x9f,
        0x3e, 0x6e, 0x82, 0x0a, 0x2b, 0x28, 0x37, 0x78,
        0xe1, 0x13, 0x70, 0x05, 0x38, 0x4d, 0x01, 0x00,
        0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x0d, 0x00,
        0x01, 0x00, 0x11, 0xaa, 0x4b, 0x15, 0xdf, 0xa6,
        0x86, 0x3f, 0xfb, 0xe0, 0x09, 0xb7, 0xf8, 0x56,
        0xd2, 0x3f, 0x05, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x0e, 0x00, 0x01, 0x00, 0xee, 0x99,
        0xc4, 0x25, 0x11, 0xe4, 0x95, 0x62, 0x29, 0xfa,
        0xfd, 0x26, 0x57, 0x02, 0xf1, 0xce, 0x03, 0x00,
        0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x0f, 0x00,
        0x01, 0x00, 0xba, 0x81, 0x9e, 0x1a, 0xdf, 0x2b,
        0xba, 0xe4, 0xd3, 0x17, 0x41, 0x60, 0x6d, 0x2d,
        0x9e, 0x28, 0x03, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x10, 0x00, 0x01, 0x00, 0xa0, 0x24,
        0x03, 0x9a, 0xa9, 0x99, 0xfb, 0xbe, 0x49, 0x11,
        0xad, 0x77, 0x30, 0xaa, 0xbc, 0xb6, 0x02, 0x00,
        0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x11, 0x00,
        0x01, 0x00, 0x32, 0x04, 0x7e, 0xae, 0xec, 0x28,
        0xd1, 0x55, 0x83, 0x4e, 0xc3, 0x47, 0x5d, 0x1d,
        0xc6, 0x65, 0x02, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x12, 0x00, 0x01, 0x00, 0xc6, 0xa4,
        0x81, 0x48, 0x66, 0x2a, 0x74, 0x7d, 0x56, 0x6e,
        0xc5, 0x1d, 0x19, 0xf2, 0xb5, 0xb6, 0x03, 0x00,
        0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00,
        0x01, 0x00, 0xcb, 0xae, 0xb3, 0xc0, 0x0c, 0xf4,
        0xa4, 0x5e, 0x91, 0x72, 0xdd, 0x53, 0x24, 0x70,
        0x89, 0x02, 0x05, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x14, 0x00, 0x01, 0x00, 0xb8, 0xd0,
        0xa0, 0x1a, 0x5e, 0x7a, 0x2d, 0xfe, 0x35, 0xc6,
        0x7d, 0x08, 0x0d, 0x33, 0x73, 0x18, 0x02, 0x00,
        0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x15, 0x00,
        0x01, 0x00, 0x21, 0xd3, 0xaa, 0x09, 0x03, 0xa7,
        0x0b, 0xc2, 0x06, 0x45, 0xd9, 0x6c, 0x75, 0xc2,
        0x15, 0xa8, 0x01, 0x00, 0x03, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00, 0x16, 0x00, 0x01, 0x00, 0xe1, 0xbd,
        0x59, 0xfc, 0xbc, 0xa9, 0x95, 0xc2, 0x68, 0x79,
        0xf3, 0x75, 0xe0, 0xae, 0x6c, 0xe5, 0x04, 0x00,
        0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c,
        0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10,
        0x48, 0x60, 0x02, 0x00, 0x00, 0x00, 0x17, 0x00,
        0x01, 0x00, 0x06, 0x52, 0xb4, 0x71, 0x70, 0x15,
        0x4e, 0xf5, 0x7f, 0x08, 0x86, 0x14, 0xe6, 0x17,
        0xd5, 0x97, 0x04, 0x00, 0x00, 0x00, 0x04, 0x5d,
        0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8,
        0x08, 0x00, 0x2b, 0x10, 0x48, 0x60, 0x02, 0x00,
        0x00, 0x00};

    uint8_t dcerpcbindack[] = {
        0x05, 0x00, 0x0c, 0x03,
        0x10, 0x00, 0x00, 0x00, 0x6c, 0x02, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xb8, 0x10, 0xb8, 0x10,
        0xce, 0x47, 0x00, 0x00, 0x0c, 0x00, 0x5c, 0x50,
        0x49, 0x50, 0x45, 0x5c, 0x6c, 0x73, 0x61, 0x73,
        0x73, 0x00, 0xf6, 0x6e, 0x18, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

    uint8_t dcerpcrequest[] = {
        0x05, 0x00, 0x00, 0x00, 0x10,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00, 0x0b,
        0x00, 0x09, 0x00, 0x45, 0x00, 0x2c, 0x00, 0x4d,
        0x00, 0x73, 0x00, 0x53, 0x00, 0x59, 0x00, 0x2a,
        0x00, 0x4a, 0x00, 0x7a, 0x00, 0x3e, 0x00, 0x58,
        0x00, 0x21, 0x00, 0x4a, 0x00, 0x30, 0x00, 0x41,
        0x00, 0x4b, 0x00, 0x4b, 0x00, 0x3c, 0x00, 0x48,
        0x00, 0x24, 0x00, 0x38, 0x00, 0x54, 0x00, 0x60,
        0x00, 0x2d, 0x00, 0x29, 0x00, 0x64, 0x00, 0x5b,
        0x00, 0x77, 0x00, 0x3a, 0x00, 0x4c, 0x00, 0x24,
        0x00, 0x23, 0x00, 0x66, 0x00, 0x43, 0x00, 0x68,
        0x00, 0x22, 0x00, 0x55, 0x00, 0x29, 0x00, 0x2c,
        0x00, 0x4f, 0x00, 0x5a, 0x00, 0x50, 0x00, 0x61,
        0x00, 0x2a, 0x00, 0x6f, 0x00, 0x2f, 0x00, 0x4d,
        0x00, 0x68, 0x00, 0x3a, 0x00, 0x5c, 0x00, 0x67,
        0x00, 0x68, 0x00, 0x68, 0x00, 0x49, 0x00, 0x45,
        0x00, 0x4c, 0x00, 0x72, 0x00, 0x53, 0x00, 0x4c,
        0x00, 0x25, 0x00, 0x4d, 0x00, 0x67, 0x00, 0x2e,
        0x00, 0x4f, 0x00, 0x64, 0x00, 0x61, 0x00, 0x73,
        0x00, 0x24, 0x00, 0x46, 0x00, 0x35, 0x00, 0x2e,
        0x00, 0x45, 0x00, 0x6f, 0x00, 0x40, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x38, 0x00, 0x47, 0x00, 0x71,
        0x00, 0x5a, 0x00, 0x37, 0x00, 0x7a, 0x00, 0x35,
        0x00, 0x6b, 0x00, 0x3c, 0x00, 0x26, 0x00, 0x37,
        0x00, 0x69, 0x00, 0x75, 0x00, 0x36, 0x00, 0x37,
        0x00, 0x47, 0x00, 0x21, 0x00, 0x2d, 0x00, 0x69,
        0x00, 0x37, 0x00, 0x78, 0x00, 0x5f, 0x00, 0x72,
        0x00, 0x4b, 0x00, 0x5c, 0x00, 0x74, 0x00, 0x3e,
        0x00, 0x52, 0x00, 0x7a, 0x00, 0x49, 0x00, 0x31,
        0x00, 0x5a, 0x00, 0x7b, 0x00, 0x29, 0x00, 0x3b,
        0x00, 0x78, 0x00, 0x3b, 0x00, 0x55, 0x00, 0x3e,
        0x00, 0x35, 0x00, 0x2b, 0x00, 0x4e, 0x00, 0x4f,
        0x00, 0x59, 0x00, 0x38, 0x00, 0x2a, 0x00, 0x59,
        0x00, 0x6b, 0x00, 0x42, 0x00, 0x4c, 0x00, 0x3e,
        0x00, 0x6a, 0x00, 0x49, 0x00, 0x2c, 0x00, 0x79,
        0x00, 0x6e, 0x00, 0x35, 0x00, 0x4f, 0x00, 0x49,
        0x00, 0x55, 0x00, 0x35, 0x00, 0x61, 0x00, 0x72,
        0x00, 0x77, 0x00, 0x38, 0x00, 0x32, 0x00, 0x24,
        0x00, 0x46, 0x00, 0x32, 0x00, 0x32, 0x00, 0x27,
        0x00, 0x64, 0x00, 0x5a, 0x00, 0x77, 0x00, 0x2e,
        0x00, 0x37, 0x00, 0x77, 0x00, 0x2e, 0x00, 0x28,
        0x00, 0x63, 0x00, 0x4f, 0x00, 0x67, 0x00, 0x64,
        0x00, 0x39, 0x00, 0x37, 0x00, 0x31, 0x00, 0x30,
        0x00, 0x28, 0x00, 0x2e, 0x00, 0x6f, 0x00, 0x3e,
        0x00, 0x59, 0x00, 0x28, 0x00, 0x67, 0x00, 0x52,
        0x00, 0x35, 0x00, 0x5a, 0x00, 0x7c, 0x00, 0x56,
        0x00, 0x6a, 0x00, 0x5c, 0x00, 0x3c, 0x00, 0x30,
        0x00, 0x59, 0x00, 0x5c, 0x00, 0x5e, 0x00, 0x38,
        0x00, 0x54, 0x00, 0x5c, 0x00, 0x5b, 0x00, 0x42,
        0x00, 0x62, 0x00, 0x70, 0x00, 0x34, 0x00, 0x5c,
        0x00, 0x57, 0x00, 0x7a, 0x00, 0x4b, 0x00, 0x2f,
        0x00, 0x6b, 0x00, 0x6a, 0x00, 0x4f, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x52, 0x00, 0x36, 0x00, 0x27,
        0x00, 0x30, 0x00, 0x6d, 0x00, 0x4a, 0x00, 0x30,
        0x00, 0x78, 0x00, 0x46, 0x00, 0x65, 0x00, 0x4e,
        0x00, 0x29, 0x00, 0x66, 0x00, 0x3f, 0x00, 0x72,
        0x00, 0x71, 0x00, 0x75, 0x00, 0x4c, 0x00, 0x2b,
        0x00, 0x5c, 0x00, 0x46, 0x00, 0x52, 0x00, 0x7b,
        0x00, 0x5c, 0x00, 0x69, 0x00, 0x66, 0x00, 0x56,
        0x00, 0x31, 0x00, 0x2d, 0x00, 0x72, 0x00, 0x61,
        0x00, 0x68, 0x00, 0x28, 0x00, 0x7d, 0x00, 0x58,
        0x00, 0x2a, 0x00, 0x7b, 0x00, 0x28, 0x00, 0x5b,
        0x00, 0x54, 0x00, 0x3a, 0x00, 0x26, 0x00, 0x52,
        0x00, 0x44, 0x00, 0x60, 0x00, 0x50, 0x00, 0x65,
        0x00, 0x48, 0x00, 0x7d, 0x00, 0x2a, 0x00, 0x74,
        0x00, 0x49, 0x00, 0x7b, 0x00, 0x21, 0x00, 0x61,
        0x00, 0x52, 0x00, 0x43, 0x00, 0x5f, 0x00, 0x5a,
        0x00, 0x74, 0x00, 0x5c, 0x00, 0x62, 0x00, 0x68,
        0x00, 0x6c, 0x00, 0x6c, 0x00, 0x2b, 0x00, 0x6f,
        0x00, 0x7c, 0x00, 0x42, 0x00, 0x67, 0x00, 0x32,
        0x00, 0x58, 0x00, 0x35, 0x00, 0x30, 0x00, 0x2f,
        0x00, 0x2d, 0x00, 0x60, 0x00, 0x62, 0x00, 0x51,
        0x00, 0x2a, 0x00, 0x30, 0x00, 0x31, 0x00, 0x48,
        0x00, 0x5b, 0x00, 0x5b, 0x00, 0x5d, 0x00, 0x25,
        0x00, 0x58, 0x00, 0x4a, 0x00, 0x76, 0x00, 0x32,
        0x00, 0x62, 0x00, 0x27, 0x00, 0x42, 0x00, 0x40,
        0x00, 0x53, 0x00, 0x7c, 0x00, 0x7d, 0x00, 0x50,
        0x00, 0x3d, 0x00, 0x40, 0x00, 0x76, 0x00, 0x38,
        0x00, 0x58, 0x00, 0x39, 0x00, 0x63, 0x00, 0x3c,
        0x00, 0x5b, 0x00, 0x23, 0x00, 0x53, 0x00, 0x7a,
        0x00, 0x54, 0x00, 0x74, 0x00, 0x61, 0x00, 0x76,
        0x00, 0x4a, 0x00, 0x3e, 0x00, 0x33, 0x00, 0x75,
        0x00, 0x66, 0x00, 0x2d, 0x00, 0x48, 0x00, 0x33,
        0x00, 0x71, 0x00, 0x76, 0x00, 0x48, 0x00, 0x71,
        0x00, 0x41, 0x00, 0x6f, 0x00, 0x2a, 0x00, 0x67,
        0x00, 0x70, 0x00, 0x21, 0x00, 0x70, 0x00, 0x4b,
        0x00, 0x52, 0x00, 0x58, 0x00, 0x68, 0x00, 0x23,
        0x00, 0x39, 0x00, 0x46, 0x00, 0x4d, 0x00, 0x51,
        0x00, 0x57, 0x00, 0x3a, 0x00, 0x79, 0x00, 0x7b,
        0x00, 0x6c, 0x00, 0x55, 0x00, 0x33, 0x00, 0x65,
        0x00, 0x49, 0x00, 0x72, 0x00, 0x30, 0x00, 0x4f,
        0x00, 0x41, 0x00, 0x6e, 0x00, 0x31, 0x00, 0x4a,
        0x00, 0x60, 0x00, 0x79, 0x00, 0x70, 0x00, 0x4f,
        0x00, 0x58, 0x00, 0x75, 0x00, 0x44, 0x00, 0x59,
        0x00, 0x58, 0x00, 0x46, 0x00, 0x3d, 0x00, 0x46,
        0x00, 0x74, 0x00, 0x51, 0x00, 0x57, 0x00, 0x6e,
        0x00, 0x2d, 0x00, 0x47, 0x00, 0x23, 0x00, 0x45,
        0x00, 0x60, 0x00, 0x4c, 0x00, 0x72, 0x00, 0x4e,
        0x00, 0x74, 0x00, 0x40, 0x00, 0x76, 0x00, 0x75,
        0x00, 0x74, 0x00, 0x56, 0x00, 0x44, 0x00, 0x29,
        0x00, 0x62, 0x00, 0x58, 0x00, 0x31, 0x00, 0x78,
        0x00, 0x32, 0x00, 0x52, 0x00, 0x4a, 0x00, 0x6b,
        0x00, 0x55, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x6f,
        0x00, 0x4a, 0x00, 0x54, 0x00, 0x7d, 0x00, 0x68,
        0x00, 0x3f, 0x00, 0x28, 0x00, 0x21, 0x00, 0x53,
        0x00, 0x48, 0x00, 0x5a, 0x00, 0x34, 0x00, 0x36,
        0x00, 0x35, 0x00, 0x64, 0x00, 0x4e, 0x00, 0x75,
        0x00, 0x69, 0x00, 0x23, 0x00, 0x75, 0x00, 0x55,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x2f, 0x00, 0x73,
        0x00, 0x62, 0x00, 0x6f, 0x00, 0x37, 0x00, 0x4e,
        0x00, 0x25, 0x00, 0x25, 0x00, 0x21, 0x00, 0x3d,
        0x00, 0x3c, 0x00, 0x71, 0x00, 0x3e, 0x00, 0x3f,
        0x00, 0x30, 0x00, 0x36, 0x00, 0x62, 0x00, 0x63,
        0x00, 0x53, 0x00, 0x54, 0x00, 0x5d, 0x00, 0x61,
        0x00, 0x4c, 0x00, 0x28, 0x00, 0x2b, 0x00, 0x4c,
        0x00, 0x4e, 0x00, 0x66, 0x00, 0x5f, 0x00, 0x4b,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x45, 0x00, 0x37,
        0x00, 0x28, 0x00, 0x56, 0x00, 0x36, 0x00, 0x6a,
        0x00, 0x3e, 0x00, 0x64, 0x00, 0x34, 0x00, 0x6a,
        0x00, 0x7d, 0x00, 0x4a, 0x00, 0x66, 0x00, 0x7a,
        0x00, 0x3e, 0x00, 0x75, 0x00, 0x38, 0x00, 0x7b,
        0x00, 0x42, 0x00, 0x76, 0x00, 0x29, 0x00, 0x4c,
        0x00, 0x65, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x4b,
        0x00, 0x2b, 0x00, 0x51, 0x00, 0x47, 0x00, 0x22,
        0x00, 0x48, 0x00, 0x3d, 0x00, 0x49, 0x00, 0x44,
        0x00, 0x5d, 0x00, 0x59, 0x00, 0x63, 0x00, 0x5c,
        0x00, 0x24, 0x00, 0x35, 0x00, 0x34, 0x00, 0x70,
        0x00, 0x69, 0x00};
    uint32_t requestlen = sizeof(dcerpcrequest);

    uint32_t bindlen = sizeof(dcerpcbind);
    uint32_t bindacklen = sizeof(dcerpcbindack);
    TcpSession ssn;
    DCERPCUuidEntry *uuid_entry;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    int r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER|STREAM_START, dcerpcbind, bindlen);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.rpc_vers != 5) {
        printf("expected dcerpc version 0x05, got 0x%02x : ",
                dcerpc_state->dcerpc.dcerpchdr.rpc_vers);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.type != BIND) {
        printf("expected dcerpc type 0x%02x , got 0x%02x : ", BIND, dcerpc_state->dcerpc.dcerpchdr.type);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.frag_length != 1084) {
        printf("expected dcerpc frag_length 0x%02x , got 0x%02x : ", 1084, dcerpc_state->dcerpc.dcerpchdr.frag_length);
        result = 0;
        goto end;
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOCLIENT, dcerpcbindack, bindacklen);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    if (dcerpc_state->dcerpc.dcerpchdr.type != BIND_ACK) {
        printf("expected dcerpc type 0x%02x , got 0x%02x : ", BIND_ACK, dcerpc_state->dcerpc.dcerpchdr.type);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.frag_length != 620) {
        printf("expected dcerpc frag_length 0x%02x , got 0x%02x : ", 620, dcerpc_state->dcerpc.dcerpchdr.frag_length);
        result = 0;
        goto end;
    }
    TAILQ_FOREACH(uuid_entry, &dcerpc_state->dcerpc.dcerpcbindbindack.uuid_list, next) {
        printUUID("BIND_ACK", uuid_entry);
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER|STREAM_EOF, dcerpcrequest, requestlen);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }
    if (dcerpc_state->dcerpc.dcerpchdr.type != REQUEST) {
        printf("expected dcerpc type 0x%02x , got 0x%02x : ", REQUEST, dcerpc_state->dcerpc.dcerpchdr.type);
        result = 0;
        goto end;
    }
end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/** \test DCERPC Request decoding and opnum parsing.
*/
int DCERPCParserTest02(void) {
    int result = 1;
    Flow f;
    uint8_t dcerpcrequest[] = {
        0x05, 0x00, 0x00, 0x00, 0x10,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00, 0x0b,
        0x00, 0x09, 0x00, 0x45, 0x00, 0x2c, 0x00, 0x4d,
        0x00, 0x73, 0x00, 0x53, 0x00, 0x59, 0x00, 0x2a,
        0x00, 0x4a, 0x00, 0x7a, 0x00, 0x3e, 0x00, 0x58,
        0x00, 0x21, 0x00, 0x4a, 0x00, 0x30, 0x00, 0x41,
        0x00, 0x4b, 0x00, 0x4b, 0x00, 0x3c, 0x00, 0x48,
        0x00, 0x24, 0x00, 0x38, 0x00, 0x54, 0x00, 0x60,
        0x00, 0x2d, 0x00, 0x29, 0x00, 0x64, 0x00, 0x5b,
        0x00, 0x77, 0x00, 0x3a, 0x00, 0x4c, 0x00, 0x24,
        0x00, 0x23, 0x00, 0x66, 0x00, 0x43, 0x00, 0x68,
        0x00, 0x22, 0x00, 0x55, 0x00, 0x29, 0x00, 0x2c,
        0x00, 0x4f, 0x00, 0x5a, 0x00, 0x50, 0x00, 0x61,
        0x00, 0x2a, 0x00, 0x6f, 0x00, 0x2f, 0x00, 0x4d,
        0x00, 0x68, 0x00, 0x3a, 0x00, 0x5c, 0x00, 0x67,
        0x00, 0x68, 0x00, 0x68, 0x00, 0x49, 0x00, 0x45,
        0x00, 0x4c, 0x00, 0x72, 0x00, 0x53, 0x00, 0x4c,
        0x00, 0x25, 0x00, 0x4d, 0x00, 0x67, 0x00, 0x2e,
        0x00, 0x4f, 0x00, 0x64, 0x00, 0x61, 0x00, 0x73,
        0x00, 0x24, 0x00, 0x46, 0x00, 0x35, 0x00, 0x2e,
        0x00, 0x45, 0x00, 0x6f, 0x00, 0x40, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x38, 0x00, 0x47, 0x00, 0x71,
        0x00, 0x5a, 0x00, 0x37, 0x00, 0x7a, 0x00, 0x35,
        0x00, 0x6b, 0x00, 0x3c, 0x00, 0x26, 0x00, 0x37,
        0x00, 0x69, 0x00, 0x75, 0x00, 0x36, 0x00, 0x37,
        0x00, 0x47, 0x00, 0x21, 0x00, 0x2d, 0x00, 0x69,
        0x00, 0x37, 0x00, 0x78, 0x00, 0x5f, 0x00, 0x72,
        0x00, 0x4b, 0x00, 0x5c, 0x00, 0x74, 0x00, 0x3e,
        0x00, 0x52, 0x00, 0x7a, 0x00, 0x49, 0x00, 0x31,
        0x00, 0x5a, 0x00, 0x7b, 0x00, 0x29, 0x00, 0x3b,
        0x00, 0x78, 0x00, 0x3b, 0x00, 0x55, 0x00, 0x3e,
        0x00, 0x35, 0x00, 0x2b, 0x00, 0x4e, 0x00, 0x4f,
        0x00, 0x59, 0x00, 0x38, 0x00, 0x2a, 0x00, 0x59,
        0x00, 0x6b, 0x00, 0x42, 0x00, 0x4c, 0x00, 0x3e,
        0x00, 0x6a, 0x00, 0x49, 0x00, 0x2c, 0x00, 0x79,
        0x00, 0x6e, 0x00, 0x35, 0x00, 0x4f, 0x00, 0x49,
        0x00, 0x55, 0x00, 0x35, 0x00, 0x61, 0x00, 0x72,
        0x00, 0x77, 0x00, 0x38, 0x00, 0x32, 0x00, 0x24,
        0x00, 0x46, 0x00, 0x32, 0x00, 0x32, 0x00, 0x27,
        0x00, 0x64, 0x00, 0x5a, 0x00, 0x77, 0x00, 0x2e,
        0x00, 0x37, 0x00, 0x77, 0x00, 0x2e, 0x00, 0x28,
        0x00, 0x63, 0x00, 0x4f, 0x00, 0x67, 0x00, 0x64,
        0x00, 0x39, 0x00, 0x37, 0x00, 0x31, 0x00, 0x30,
        0x00, 0x28, 0x00, 0x2e, 0x00, 0x6f, 0x00, 0x3e,
        0x00, 0x59, 0x00, 0x28, 0x00, 0x67, 0x00, 0x52,
        0x00, 0x35, 0x00, 0x5a, 0x00, 0x7c, 0x00, 0x56,
        0x00, 0x6a, 0x00, 0x5c, 0x00, 0x3c, 0x00, 0x30,
        0x00, 0x59, 0x00, 0x5c, 0x00, 0x5e, 0x00, 0x38,
        0x00, 0x54, 0x00, 0x5c, 0x00, 0x5b, 0x00, 0x42,
        0x00, 0x62, 0x00, 0x70, 0x00, 0x34, 0x00, 0x5c,
        0x00, 0x57, 0x00, 0x7a, 0x00, 0x4b, 0x00, 0x2f,
        0x00, 0x6b, 0x00, 0x6a, 0x00, 0x4f, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x52, 0x00, 0x36, 0x00, 0x27,
        0x00, 0x30, 0x00, 0x6d, 0x00, 0x4a, 0x00, 0x30,
        0x00, 0x78, 0x00, 0x46, 0x00, 0x65, 0x00, 0x4e,
        0x00, 0x29, 0x00, 0x66, 0x00, 0x3f, 0x00, 0x72,
        0x00, 0x71, 0x00, 0x75, 0x00, 0x4c, 0x00, 0x2b,
        0x00, 0x5c, 0x00, 0x46, 0x00, 0x52, 0x00, 0x7b,
        0x00, 0x5c, 0x00, 0x69, 0x00, 0x66, 0x00, 0x56,
        0x00, 0x31, 0x00, 0x2d, 0x00, 0x72, 0x00, 0x61,
        0x00, 0x68, 0x00, 0x28, 0x00, 0x7d, 0x00, 0x58,
        0x00, 0x2a, 0x00, 0x7b, 0x00, 0x28, 0x00, 0x5b,
        0x00, 0x54, 0x00, 0x3a, 0x00, 0x26, 0x00, 0x52,
        0x00, 0x44, 0x00, 0x60, 0x00, 0x50, 0x00, 0x65,
        0x00, 0x48, 0x00, 0x7d, 0x00, 0x2a, 0x00, 0x74,
        0x00, 0x49, 0x00, 0x7b, 0x00, 0x21, 0x00, 0x61,
        0x00, 0x52, 0x00, 0x43, 0x00, 0x5f, 0x00, 0x5a,
        0x00, 0x74, 0x00, 0x5c, 0x00, 0x62, 0x00, 0x68,
        0x00, 0x6c, 0x00, 0x6c, 0x00, 0x2b, 0x00, 0x6f,
        0x00, 0x7c, 0x00, 0x42, 0x00, 0x67, 0x00, 0x32,
        0x00, 0x58, 0x00, 0x35, 0x00, 0x30, 0x00, 0x2f,
        0x00, 0x2d, 0x00, 0x60, 0x00, 0x62, 0x00, 0x51,
        0x00, 0x2a, 0x00, 0x30, 0x00, 0x31, 0x00, 0x48,
        0x00, 0x5b, 0x00, 0x5b, 0x00, 0x5d, 0x00, 0x25,
        0x00, 0x58, 0x00, 0x4a, 0x00, 0x76, 0x00, 0x32,
        0x00, 0x62, 0x00, 0x27, 0x00, 0x42, 0x00, 0x40,
        0x00, 0x53, 0x00, 0x7c, 0x00, 0x7d, 0x00, 0x50,
        0x00, 0x3d, 0x00, 0x40, 0x00, 0x76, 0x00, 0x38,
        0x00, 0x58, 0x00, 0x39, 0x00, 0x63, 0x00, 0x3c,
        0x00, 0x5b, 0x00, 0x23, 0x00, 0x53, 0x00, 0x7a,
        0x00, 0x54, 0x00, 0x74, 0x00, 0x61, 0x00, 0x76,
        0x00, 0x4a, 0x00, 0x3e, 0x00, 0x33, 0x00, 0x75,
        0x00, 0x66, 0x00, 0x2d, 0x00, 0x48, 0x00, 0x33,
        0x00, 0x71, 0x00, 0x76, 0x00, 0x48, 0x00, 0x71,
        0x00, 0x41, 0x00, 0x6f, 0x00, 0x2a, 0x00, 0x67,
        0x00, 0x70, 0x00, 0x21, 0x00, 0x70, 0x00, 0x4b,
        0x00, 0x52, 0x00, 0x58, 0x00, 0x68, 0x00, 0x23,
        0x00, 0x39, 0x00, 0x46, 0x00, 0x4d, 0x00, 0x51,
        0x00, 0x57, 0x00, 0x3a, 0x00, 0x79, 0x00, 0x7b,
        0x00, 0x6c, 0x00, 0x55, 0x00, 0x33, 0x00, 0x65,
        0x00, 0x49, 0x00, 0x72, 0x00, 0x30, 0x00, 0x4f,
        0x00, 0x41, 0x00, 0x6e, 0x00, 0x31, 0x00, 0x4a,
        0x00, 0x60, 0x00, 0x79, 0x00, 0x70, 0x00, 0x4f,
        0x00, 0x58, 0x00, 0x75, 0x00, 0x44, 0x00, 0x59,
        0x00, 0x58, 0x00, 0x46, 0x00, 0x3d, 0x00, 0x46,
        0x00, 0x74, 0x00, 0x51, 0x00, 0x57, 0x00, 0x6e,
        0x00, 0x2d, 0x00, 0x47, 0x00, 0x23, 0x00, 0x45,
        0x00, 0x60, 0x00, 0x4c, 0x00, 0x72, 0x00, 0x4e,
        0x00, 0x74, 0x00, 0x40, 0x00, 0x76, 0x00, 0x75,
        0x00, 0x74, 0x00, 0x56, 0x00, 0x44, 0x00, 0x29,
        0x00, 0x62, 0x00, 0x58, 0x00, 0x31, 0x00, 0x78,
        0x00, 0x32, 0x00, 0x52, 0x00, 0x4a, 0x00, 0x6b,
        0x00, 0x55, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x6f,
        0x00, 0x4a, 0x00, 0x54, 0x00, 0x7d, 0x00, 0x68,
        0x00, 0x3f, 0x00, 0x28, 0x00, 0x21, 0x00, 0x53,
        0x00, 0x48, 0x00, 0x5a, 0x00, 0x34, 0x00, 0x36,
        0x00, 0x35, 0x00, 0x64, 0x00, 0x4e, 0x00, 0x75,
        0x00, 0x69, 0x00, 0x23, 0x00, 0x75, 0x00, 0x55,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x2f, 0x00, 0x73,
        0x00, 0x62, 0x00, 0x6f, 0x00, 0x37, 0x00, 0x4e,
        0x00, 0x25, 0x00, 0x25, 0x00, 0x21, 0x00, 0x3d,
        0x00, 0x3c, 0x00, 0x71, 0x00, 0x3e, 0x00, 0x3f,
        0x00, 0x30, 0x00, 0x36, 0x00, 0x62, 0x00, 0x63,
        0x00, 0x53, 0x00, 0x54, 0x00, 0x5d, 0x00, 0x61,
        0x00, 0x4c, 0x00, 0x28, 0x00, 0x2b, 0x00, 0x4c,
        0x00, 0x4e, 0x00, 0x66, 0x00, 0x5f, 0x00, 0x4b,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x45, 0x00, 0x37,
        0x00, 0x28, 0x00, 0x56, 0x00, 0x36, 0x00, 0x6a,
        0x00, 0x3e, 0x00, 0x64, 0x00, 0x34, 0x00, 0x6a,
        0x00, 0x7d, 0x00, 0x4a, 0x00, 0x66, 0x00, 0x7a,
        0x00, 0x3e, 0x00, 0x75, 0x00, 0x38, 0x00, 0x7b,
        0x00, 0x42, 0x00, 0x76, 0x00, 0x29, 0x00, 0x4c,
        0x00, 0x65, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x4b,
        0x00, 0x2b, 0x00, 0x51, 0x00, 0x47, 0x00, 0x22,
        0x00, 0x48, 0x00, 0x3d, 0x00, 0x49, 0x00, 0x44,
        0x00, 0x5d, 0x00, 0x59, 0x00, 0x63, 0x00, 0x5c,
        0x00, 0x24, 0x00, 0x35, 0x00, 0x34, 0x00, 0x70,
        0x00, 0x69, 0x00};
    uint32_t requestlen = sizeof(dcerpcrequest);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    int r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER|STREAM_START, dcerpcrequest, requestlen);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.rpc_vers != 5) {
        printf("expected dcerpc version 0x05, got 0x%02x : ",
                dcerpc_state->dcerpc.dcerpchdr.rpc_vers);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.type != REQUEST) {
        printf("expected dcerpc type 0x%02x , got 0x%02x : ", REQUEST, dcerpc_state->dcerpc.dcerpchdr.type);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.frag_length != 1024) {
        printf("expected dcerpc frag_length 0x%02x , got 0x%02x : ", 1024, dcerpc_state->dcerpc.dcerpchdr.frag_length);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpcrequest.opnum != 9) {
        printf("expected dcerpc opnum 0x%02x , got 0x%02x : ", 9, dcerpc_state->dcerpc.dcerpcrequest.opnum);
        result = 0;
        goto end;
    }

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/** \test Test endianness handling
*/
int DCERPCParserTest03(void) {
    int result = 1;
    Flow f;
    uint8_t dcerpcrequest[] = {
        0x05, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0xe8, 0x03, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x09, 0x45, 0x00, 0x2c, 0x00, 0x4d,
        0x00, 0x73, 0x00, 0x53, 0x00, 0x59, 0x00, 0x2a,
        0x00, 0x4a, 0x00, 0x7a, 0x00, 0x3e, 0x00, 0x58,
        0x00, 0x21, 0x00, 0x4a, 0x00, 0x30, 0x00, 0x41,
        0x00, 0x4b, 0x00, 0x4b, 0x00, 0x3c, 0x00, 0x48,
        0x00, 0x24, 0x00, 0x38, 0x00, 0x54, 0x00, 0x60,
        0x00, 0x2d, 0x00, 0x29, 0x00, 0x64, 0x00, 0x5b,
        0x00, 0x77, 0x00, 0x3a, 0x00, 0x4c, 0x00, 0x24,
        0x00, 0x23, 0x00, 0x66, 0x00, 0x43, 0x00, 0x68,
        0x00, 0x22, 0x00, 0x55, 0x00, 0x29, 0x00, 0x2c,
        0x00, 0x4f, 0x00, 0x5a, 0x00, 0x50, 0x00, 0x61,
        0x00, 0x2a, 0x00, 0x6f, 0x00, 0x2f, 0x00, 0x4d,
        0x00, 0x68, 0x00, 0x3a, 0x00, 0x5c, 0x00, 0x67,
        0x00, 0x68, 0x00, 0x68, 0x00, 0x49, 0x00, 0x45,
        0x00, 0x4c, 0x00, 0x72, 0x00, 0x53, 0x00, 0x4c,
        0x00, 0x25, 0x00, 0x4d, 0x00, 0x67, 0x00, 0x2e,
        0x00, 0x4f, 0x00, 0x64, 0x00, 0x61, 0x00, 0x73,
        0x00, 0x24, 0x00, 0x46, 0x00, 0x35, 0x00, 0x2e,
        0x00, 0x45, 0x00, 0x6f, 0x00, 0x40, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x38, 0x00, 0x47, 0x00, 0x71,
        0x00, 0x5a, 0x00, 0x37, 0x00, 0x7a, 0x00, 0x35,
        0x00, 0x6b, 0x00, 0x3c, 0x00, 0x26, 0x00, 0x37,
        0x00, 0x69, 0x00, 0x75, 0x00, 0x36, 0x00, 0x37,
        0x00, 0x47, 0x00, 0x21, 0x00, 0x2d, 0x00, 0x69,
        0x00, 0x37, 0x00, 0x78, 0x00, 0x5f, 0x00, 0x72,
        0x00, 0x4b, 0x00, 0x5c, 0x00, 0x74, 0x00, 0x3e,
        0x00, 0x52, 0x00, 0x7a, 0x00, 0x49, 0x00, 0x31,
        0x00, 0x5a, 0x00, 0x7b, 0x00, 0x29, 0x00, 0x3b,
        0x00, 0x78, 0x00, 0x3b, 0x00, 0x55, 0x00, 0x3e,
        0x00, 0x35, 0x00, 0x2b, 0x00, 0x4e, 0x00, 0x4f,
        0x00, 0x59, 0x00, 0x38, 0x00, 0x2a, 0x00, 0x59,
        0x00, 0x6b, 0x00, 0x42, 0x00, 0x4c, 0x00, 0x3e,
        0x00, 0x6a, 0x00, 0x49, 0x00, 0x2c, 0x00, 0x79,
        0x00, 0x6e, 0x00, 0x35, 0x00, 0x4f, 0x00, 0x49,
        0x00, 0x55, 0x00, 0x35, 0x00, 0x61, 0x00, 0x72,
        0x00, 0x77, 0x00, 0x38, 0x00, 0x32, 0x00, 0x24,
        0x00, 0x46, 0x00, 0x32, 0x00, 0x32, 0x00, 0x27,
        0x00, 0x64, 0x00, 0x5a, 0x00, 0x77, 0x00, 0x2e,
        0x00, 0x37, 0x00, 0x77, 0x00, 0x2e, 0x00, 0x28,
        0x00, 0x63, 0x00, 0x4f, 0x00, 0x67, 0x00, 0x64,
        0x00, 0x39, 0x00, 0x37, 0x00, 0x31, 0x00, 0x30,
        0x00, 0x28, 0x00, 0x2e, 0x00, 0x6f, 0x00, 0x3e,
        0x00, 0x59, 0x00, 0x28, 0x00, 0x67, 0x00, 0x52,
        0x00, 0x35, 0x00, 0x5a, 0x00, 0x7c, 0x00, 0x56,
        0x00, 0x6a, 0x00, 0x5c, 0x00, 0x3c, 0x00, 0x30,
        0x00, 0x59, 0x00, 0x5c, 0x00, 0x5e, 0x00, 0x38,
        0x00, 0x54, 0x00, 0x5c, 0x00, 0x5b, 0x00, 0x42,
        0x00, 0x62, 0x00, 0x70, 0x00, 0x34, 0x00, 0x5c,
        0x00, 0x57, 0x00, 0x7a, 0x00, 0x4b, 0x00, 0x2f,
        0x00, 0x6b, 0x00, 0x6a, 0x00, 0x4f, 0x00, 0x41,
        0x00, 0x33, 0x00, 0x52, 0x00, 0x36, 0x00, 0x27,
        0x00, 0x30, 0x00, 0x6d, 0x00, 0x4a, 0x00, 0x30,
        0x00, 0x78, 0x00, 0x46, 0x00, 0x65, 0x00, 0x4e,
        0x00, 0x29, 0x00, 0x66, 0x00, 0x3f, 0x00, 0x72,
        0x00, 0x71, 0x00, 0x75, 0x00, 0x4c, 0x00, 0x2b,
        0x00, 0x5c, 0x00, 0x46, 0x00, 0x52, 0x00, 0x7b,
        0x00, 0x5c, 0x00, 0x69, 0x00, 0x66, 0x00, 0x56,
        0x00, 0x31, 0x00, 0x2d, 0x00, 0x72, 0x00, 0x61,
        0x00, 0x68, 0x00, 0x28, 0x00, 0x7d, 0x00, 0x58,
        0x00, 0x2a, 0x00, 0x7b, 0x00, 0x28, 0x00, 0x5b,
        0x00, 0x54, 0x00, 0x3a, 0x00, 0x26, 0x00, 0x52,
        0x00, 0x44, 0x00, 0x60, 0x00, 0x50, 0x00, 0x65,
        0x00, 0x48, 0x00, 0x7d, 0x00, 0x2a, 0x00, 0x74,
        0x00, 0x49, 0x00, 0x7b, 0x00, 0x21, 0x00, 0x61,
        0x00, 0x52, 0x00, 0x43, 0x00, 0x5f, 0x00, 0x5a,
        0x00, 0x74, 0x00, 0x5c, 0x00, 0x62, 0x00, 0x68,
        0x00, 0x6c, 0x00, 0x6c, 0x00, 0x2b, 0x00, 0x6f,
        0x00, 0x7c, 0x00, 0x42, 0x00, 0x67, 0x00, 0x32,
        0x00, 0x58, 0x00, 0x35, 0x00, 0x30, 0x00, 0x2f,
        0x00, 0x2d, 0x00, 0x60, 0x00, 0x62, 0x00, 0x51,
        0x00, 0x2a, 0x00, 0x30, 0x00, 0x31, 0x00, 0x48,
        0x00, 0x5b, 0x00, 0x5b, 0x00, 0x5d, 0x00, 0x25,
        0x00, 0x58, 0x00, 0x4a, 0x00, 0x76, 0x00, 0x32,
        0x00, 0x62, 0x00, 0x27, 0x00, 0x42, 0x00, 0x40,
        0x00, 0x53, 0x00, 0x7c, 0x00, 0x7d, 0x00, 0x50,
        0x00, 0x3d, 0x00, 0x40, 0x00, 0x76, 0x00, 0x38,
        0x00, 0x58, 0x00, 0x39, 0x00, 0x63, 0x00, 0x3c,
        0x00, 0x5b, 0x00, 0x23, 0x00, 0x53, 0x00, 0x7a,
        0x00, 0x54, 0x00, 0x74, 0x00, 0x61, 0x00, 0x76,
        0x00, 0x4a, 0x00, 0x3e, 0x00, 0x33, 0x00, 0x75,
        0x00, 0x66, 0x00, 0x2d, 0x00, 0x48, 0x00, 0x33,
        0x00, 0x71, 0x00, 0x76, 0x00, 0x48, 0x00, 0x71,
        0x00, 0x41, 0x00, 0x6f, 0x00, 0x2a, 0x00, 0x67,
        0x00, 0x70, 0x00, 0x21, 0x00, 0x70, 0x00, 0x4b,
        0x00, 0x52, 0x00, 0x58, 0x00, 0x68, 0x00, 0x23,
        0x00, 0x39, 0x00, 0x46, 0x00, 0x4d, 0x00, 0x51,
        0x00, 0x57, 0x00, 0x3a, 0x00, 0x79, 0x00, 0x7b,
        0x00, 0x6c, 0x00, 0x55, 0x00, 0x33, 0x00, 0x65,
        0x00, 0x49, 0x00, 0x72, 0x00, 0x30, 0x00, 0x4f,
        0x00, 0x41, 0x00, 0x6e, 0x00, 0x31, 0x00, 0x4a,
        0x00, 0x60, 0x00, 0x79, 0x00, 0x70, 0x00, 0x4f,
        0x00, 0x58, 0x00, 0x75, 0x00, 0x44, 0x00, 0x59,
        0x00, 0x58, 0x00, 0x46, 0x00, 0x3d, 0x00, 0x46,
        0x00, 0x74, 0x00, 0x51, 0x00, 0x57, 0x00, 0x6e,
        0x00, 0x2d, 0x00, 0x47, 0x00, 0x23, 0x00, 0x45,
        0x00, 0x60, 0x00, 0x4c, 0x00, 0x72, 0x00, 0x4e,
        0x00, 0x74, 0x00, 0x40, 0x00, 0x76, 0x00, 0x75,
        0x00, 0x74, 0x00, 0x56, 0x00, 0x44, 0x00, 0x29,
        0x00, 0x62, 0x00, 0x58, 0x00, 0x31, 0x00, 0x78,
        0x00, 0x32, 0x00, 0x52, 0x00, 0x4a, 0x00, 0x6b,
        0x00, 0x55, 0x00, 0x72, 0x00, 0x6f, 0x00, 0x6f,
        0x00, 0x4a, 0x00, 0x54, 0x00, 0x7d, 0x00, 0x68,
        0x00, 0x3f, 0x00, 0x28, 0x00, 0x21, 0x00, 0x53,
        0x00, 0x48, 0x00, 0x5a, 0x00, 0x34, 0x00, 0x36,
        0x00, 0x35, 0x00, 0x64, 0x00, 0x4e, 0x00, 0x75,
        0x00, 0x69, 0x00, 0x23, 0x00, 0x75, 0x00, 0x55,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x2f, 0x00, 0x73,
        0x00, 0x62, 0x00, 0x6f, 0x00, 0x37, 0x00, 0x4e,
        0x00, 0x25, 0x00, 0x25, 0x00, 0x21, 0x00, 0x3d,
        0x00, 0x3c, 0x00, 0x71, 0x00, 0x3e, 0x00, 0x3f,
        0x00, 0x30, 0x00, 0x36, 0x00, 0x62, 0x00, 0x63,
        0x00, 0x53, 0x00, 0x54, 0x00, 0x5d, 0x00, 0x61,
        0x00, 0x4c, 0x00, 0x28, 0x00, 0x2b, 0x00, 0x4c,
        0x00, 0x4e, 0x00, 0x66, 0x00, 0x5f, 0x00, 0x4b,
        0x00, 0x43, 0x00, 0x75, 0x00, 0x45, 0x00, 0x37,
        0x00, 0x28, 0x00, 0x56, 0x00, 0x36, 0x00, 0x6a,
        0x00, 0x3e, 0x00, 0x64, 0x00, 0x34, 0x00, 0x6a,
        0x00, 0x7d, 0x00, 0x4a, 0x00, 0x66, 0x00, 0x7a,
        0x00, 0x3e, 0x00, 0x75, 0x00, 0x38, 0x00, 0x7b,
        0x00, 0x42, 0x00, 0x76, 0x00, 0x29, 0x00, 0x4c,
        0x00, 0x65, 0x00, 0x2e, 0x00, 0x32, 0x00, 0x4b,
        0x00, 0x2b, 0x00, 0x51, 0x00, 0x47, 0x00, 0x22,
        0x00, 0x48, 0x00, 0x3d, 0x00, 0x49, 0x00, 0x44,
        0x00, 0x5d, 0x00, 0x59, 0x00, 0x63, 0x00, 0x5c,
        0x00, 0x24, 0x00, 0x35, 0x00, 0x34, 0x00, 0x70,
        0x00, 0x69, 0x00};
    uint32_t requestlen = sizeof(dcerpcrequest);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    int r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER|STREAM_START, dcerpcrequest, requestlen);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.packed_drep[0] != 0x01) {
        printf("expected dcerpc data representation 0x01, got 0x%02x : ",
                dcerpc_state->dcerpc.dcerpchdr.packed_drep[0]);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpchdr.frag_length != 1024) {
        printf("expected dcerpc frag_length 0x%02x , got 0x%02x : ", 1024, dcerpc_state->dcerpc.dcerpchdr.frag_length);
        result = 0;
        goto end;
    }

    if (dcerpc_state->dcerpc.dcerpcrequest.opnum != 9) {
        printf("expected dcerpc opnum 0x%02x , got 0x%02x : ", 9, dcerpc_state->dcerpc.dcerpcrequest.opnum);
        result = 0;
        goto end;
    }
end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \todo Needs to be rewritten
 */
int DCERPCParserTest04(void) {
    /* AWS - Disabled this test since clamav FPs on the payloads used.
     * We will have to rewrite this test with new payloads.  Will be done
     * as a part of dcerpc update/fixes */
#if 0
    int result = 1;
    Flow f;
    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x6a, 0x28, 0x19, 0x39, 0x0c, 0xb1, 0xd0, 0x11,
        0x9b, 0xa8, 0x00, 0xc0, 0x4f, 0xd9, 0x2e, 0xf5,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_len = sizeof(bind);

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x48, 0x1a, 0x00, 0x00,
        0x0c, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x6c, 0x73, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xad, 0x0d, 0x00, 0x00, 0x91, 0xfc, 0x27, 0x40,
        0x4a, 0x97, 0x4a, 0x98, 0x4b, 0x41, 0x3f, 0x48,
        0x99, 0x90, 0xf8, 0x27, 0xfd, 0x3f, 0x27, 0x37,
        0x40, 0xd6, 0x27, 0xfc, 0x3f, 0x9f, 0x4f, 0xfd,
        0x42, 0x47, 0x47, 0x49, 0x3f, 0xf9, 0x9b, 0xd6,
        0x48, 0x37, 0x27, 0x46, 0x93, 0x49, 0xfd, 0x93,
        0x91, 0xfd, 0x93, 0x90, 0x92, 0x96, 0xf5, 0x92,
        0x4e, 0x91, 0x98, 0x46, 0x4f, 0x4b, 0x46, 0xf5,
        0xf5, 0xfd, 0x40, 0xf9, 0x9b, 0x40, 0x9f, 0x93,
        0x4e, 0xf8, 0x40, 0x40, 0x4e, 0xf5, 0x4b, 0x98,
        0xf5, 0x91, 0xd6, 0x42, 0x99, 0x96, 0x27, 0x49,
        0x48, 0x47, 0x4f, 0x46, 0x99, 0x4b, 0x92, 0x92,
        0x90, 0x47, 0x46, 0x4e, 0x43, 0x9b, 0x43, 0x42,
        0x3f, 0x4b, 0x27, 0x97, 0x93, 0xf9, 0x42, 0x9b,
        0x46, 0x9b, 0x4b, 0x98, 0x41, 0x98, 0x37, 0x41,
        0x9f, 0x98, 0x4e, 0x93, 0x48, 0x46, 0x46, 0x9f,
        0x97, 0x9b, 0x42, 0x37, 0x90, 0x46, 0xf9, 0x97,
        0x91, 0xf5, 0x4e, 0x97, 0x4e, 0x99, 0xf8, 0x99,
        0x41, 0xf5, 0x41, 0x9f, 0x49, 0xfd, 0x92, 0x96,
        0x3f, 0x3f, 0x42, 0x27, 0x27, 0x93, 0x47, 0x49,
        0x91, 0x27, 0x27, 0x40, 0x42, 0x99, 0x9f, 0xfc,
        0x97, 0x47, 0x99, 0x4a, 0xf9, 0x3f, 0x48, 0x91,
        0x47, 0x97, 0x91, 0x42, 0x4b, 0x9b, 0x4a, 0x48,
        0x9f, 0x43, 0x43, 0x40, 0x99, 0xf9, 0x48, 0x4e,
        0x92, 0x93, 0x92, 0x41, 0x46, 0x4b, 0x4a, 0x4a,
        0x49, 0x96, 0x4a, 0x4f, 0xf5, 0x42, 0x47, 0x98,
        0x9b, 0xf5, 0x91, 0xf9, 0xd6, 0x9b, 0x48, 0x4e,
        0x9f, 0x91, 0xd6, 0x93, 0x4b, 0x37, 0x3f, 0x43,
        0xf5, 0x41, 0x41, 0xf5, 0x37, 0x4f, 0x43, 0x92,
        0x97, 0x27, 0x93, 0x92, 0x46, 0x47, 0x4b, 0x96,
        0x41, 0x90, 0x90, 0x3f, 0x96, 0x27, 0x41, 0xd6,
        0xd6, 0xd6, 0xf9, 0xf8, 0x47, 0x27, 0x46, 0x37,
        0x41, 0x90, 0x91, 0xfc, 0x46, 0x41, 0x43, 0x97,
        0x9f, 0x4a, 0x49, 0x92, 0x41, 0x91, 0x41, 0x92,
        0x42, 0x4a, 0x3f, 0x93, 0x99, 0x9b, 0x9f, 0x4e,
        0x47, 0x93, 0xd6, 0x37, 0x37, 0x40, 0x98, 0xfd,
        0x41, 0x42, 0x97, 0x4e, 0x4e, 0x98, 0x9f, 0x4e,
        0x48, 0x3f, 0x48, 0x42, 0x96, 0x9f, 0x99, 0x4f,
        0x4e, 0x42, 0x97, 0xf9, 0x3f, 0x37, 0x27, 0x46,
        0x41, 0xf9, 0x92, 0x96, 0x41, 0x93, 0x91, 0x4b,
        0x96, 0x4f, 0x43, 0xfd, 0xf5, 0x9f, 0x43, 0x27,
        0x99, 0xd6, 0xf5, 0x4e, 0xfd, 0x97, 0x4b, 0x47,
        0x47, 0x92, 0x98, 0x4f, 0x47, 0x49, 0x37, 0x97,
        0x3f, 0x4e, 0x40, 0x46, 0x4e, 0x9f, 0x4e, 0x4e,
        0xfc, 0x41, 0x47, 0xf8, 0x37, 0x9b, 0x41, 0x4e,
        0x96, 0x99, 0x46, 0x99, 0x46, 0xf9, 0x4e, 0x4f,
        0x48, 0x97, 0x97, 0x93, 0xd6, 0x9b, 0x41, 0x40,
        0x97, 0x97, 0x4f, 0x92, 0x91, 0xd6, 0x96, 0x40,
        0x4f, 0x4b, 0x91, 0x46, 0x27, 0x92, 0x3f, 0xf5,
        0xfc, 0x3f, 0x91, 0x97, 0xf8, 0x43, 0x4e, 0xfd,
        0x9b, 0x27, 0xfd, 0x9b, 0xf5, 0x27, 0x47, 0x42,
        0x46, 0x93, 0x37, 0x93, 0x91, 0x91, 0x91, 0xf8,
        0x4f, 0x92, 0x4f, 0xf8, 0x93, 0xf5, 0x49, 0x91,
        0x4b, 0x3f, 0xfc, 0x37, 0x4f, 0x46, 0x98, 0x97,
        0x9f, 0x40, 0xfd, 0x9f, 0x98, 0xfd, 0x4e, 0x97,
        0x4f, 0x47, 0x91, 0x27, 0x4a, 0x90, 0x96, 0x40,
        0x98, 0x97, 0x41, 0x3f, 0xd6, 0xfd, 0x41, 0xfd,
        0x42, 0x97, 0x4b, 0x9b, 0x46, 0x4e, 0xfc, 0x96,
        0xf9, 0x37, 0x4b, 0x96, 0x9f, 0x9b, 0x42, 0x9f,
        0x93, 0x40, 0x42, 0x43, 0xf5, 0x93, 0x48, 0x3f,
        0x4b, 0xfd, 0x9f, 0x4b, 0x41, 0x4a, 0x90, 0x9b,
        0x46, 0x97, 0x98, 0x96, 0x9b, 0x98, 0x92, 0xd6,
        0x4e, 0x4a, 0x27, 0x90, 0x96, 0x99, 0x91, 0x46,
        0x49, 0x41, 0x4b, 0x90, 0x43, 0x91, 0xd6, 0x48,
        0x42, 0x90, 0x4f, 0x96, 0x43, 0x9b, 0xf9, 0x9b,
        0x9f, 0x9f, 0x27, 0x47, 0x4b, 0xf5, 0x43, 0x99,
        0x99, 0x91, 0x4e, 0x41, 0x42, 0x46, 0x97, 0x46,
        0x47, 0xf9, 0xf5, 0x48, 0x4a, 0xf8, 0x4e, 0xd6,
        0x43, 0x4a, 0x27, 0x9b, 0x42, 0x90, 0x46, 0x46,
        0x3f, 0x99, 0x96, 0x9b, 0x91, 0x9f, 0xf5, 0x48,
        0x43, 0x9f, 0x4a, 0x99, 0x96, 0xfd, 0x92, 0x49,
        0x46, 0x91, 0x40, 0xfd, 0x4a, 0x48, 0x4f, 0x90,
        0x91, 0x98, 0x48, 0x4b, 0x9f, 0x42, 0x27, 0x93,
        0x47, 0xf8, 0x4f, 0x48, 0x3f, 0x90, 0x47, 0x41,
        0xf5, 0xfc, 0x27, 0xf8, 0x97, 0x4a, 0x49, 0x37,
        0x40, 0x4f, 0x40, 0x37, 0x41, 0x27, 0x96, 0x37,
        0xfc, 0x42, 0xd6, 0x4b, 0x48, 0x37, 0x42, 0xf5,
        0x27, 0xf9, 0xd6, 0x48, 0x9b, 0xfd, 0x40, 0x96,
        0x4e, 0x43, 0xf8, 0x90, 0x40, 0x40, 0x49, 0x3f,
        0xfc, 0x4a, 0x42, 0x47, 0xf8, 0x49, 0x42, 0x97,
        0x4f, 0x91, 0xfd, 0x4b, 0x46, 0x4b, 0xfc, 0x48,
        0x49, 0x96, 0x4b, 0x96, 0x43, 0x9f, 0x90, 0x37,
        0xd6, 0x4a, 0xd6, 0x3f, 0xd6, 0x90, 0x49, 0x27,
        0x4e, 0x96, 0x96, 0xf8, 0x49, 0x96, 0xf8, 0x37,
        0x90, 0x4e, 0x4b, 0x4f, 0x99, 0xf8, 0x6a, 0x52,
        0x59, 0xd9, 0xee, 0xd9, 0x74, 0x24, 0xf4, 0x5b,
        0x81, 0x73, 0x13, 0x30, 0x50, 0xf0, 0x82, 0x83,
        0xeb, 0xfc, 0xe2, 0xf4, 0xb1, 0x94, 0x0f, 0x6d,
        0xcf, 0xaf, 0xb4, 0x7e, 0x5a, 0xbb, 0xbf, 0x6a,
        0xc9, 0xaf, 0x0f, 0x7d, 0x50, 0xdb, 0x9c, 0xa6,
        0x14, 0xdb, 0xb5, 0xbe, 0xbb, 0x2c, 0xf5, 0xfa,
        0x31, 0xbf, 0x7b, 0xcd, 0x28, 0xdb, 0xaf, 0xa2,
        0x31, 0xbb, 0x13, 0xb2, 0x79, 0xdb, 0xc4, 0x09,
        0x31, 0xbe, 0xc1, 0x42, 0xa9, 0xfc, 0x74, 0x42,
        0x44, 0x57, 0x31, 0x48, 0x3d, 0x51, 0x32, 0x69,
        0xc4, 0x6b, 0xa4, 0xa6, 0x18, 0x25, 0x13, 0x09,
        0x6f, 0x74, 0xf1, 0x69, 0x56, 0xdb, 0xfc, 0xc9,
        0xbb, 0x0f, 0xec, 0x83, 0xdb, 0x53, 0xdc, 0x09,
        0xb9, 0x3c, 0xd4, 0x9e, 0x51, 0x93, 0xc1, 0x42,
        0x54, 0xdb, 0xb0, 0xb2, 0xbb, 0x10, 0xfc, 0x09,
        0x40, 0x4c, 0x5d, 0x09, 0x70, 0x58, 0xae, 0xea,
        0xbe, 0x1e, 0xfe, 0x6e, 0x60, 0xaf, 0x26, 0xb3,
        0xeb, 0x36, 0xa3, 0xe4, 0x58, 0x63, 0xc2, 0xea,
        0x47, 0x23, 0xc2, 0xdd, 0x64, 0xaf, 0x20, 0xea,
        0xfb, 0xbd, 0x0c, 0xb9, 0x60, 0xaf, 0x26, 0xdd,
        0xb9, 0xb5, 0x96, 0x03, 0xdd, 0x58, 0xf2, 0xd7,
        0x5a, 0x52, 0x0f, 0x52, 0x58, 0x89, 0xf9, 0x77,
        0x9d, 0x07, 0x0f, 0x54, 0x63, 0x03, 0xa3, 0xd1,
        0x63, 0x13, 0xa3, 0xc1, 0x63, 0xaf, 0x20, 0xe4,
        0x58, 0x41, 0xac, 0xe4, 0x63, 0xd9, 0x11, 0x17,
        0x58, 0xf4, 0xea, 0xf2, 0xf7, 0x07, 0x0f, 0x54,
        0x5a, 0x40, 0xa1, 0xd7, 0xcf, 0x80, 0x98, 0x26,
        0x9d, 0x7e, 0x19, 0xd5, 0xcf, 0x86, 0xa3, 0xd7,
        0xcf, 0x80, 0x98, 0x67, 0x79, 0xd6, 0xb9, 0xd5,
        0xcf, 0x86, 0xa0, 0xd6, 0x64, 0x05, 0x0f, 0x52,
        0xa3, 0x38, 0x17, 0xfb, 0xf6, 0x29, 0xa7, 0x7d,
        0xe6, 0x05, 0x0f, 0x52, 0x56, 0x3a, 0x94, 0xe4,
        0x58, 0x33, 0x9d, 0x0b, 0xd5, 0x3a, 0xa0, 0xdb,
        0x19, 0x9c, 0x79, 0x65, 0x5a, 0x14, 0x79, 0x60,
        0x01, 0x90, 0x03, 0x28, 0xce, 0x12, 0xdd, 0x7c,
        0x72, 0x7c, 0x63, 0x0f, 0x4a, 0x68, 0x5b, 0x29,
        0x9b, 0x38, 0x82, 0x7c, 0x83, 0x46, 0x0f, 0xf7,
        0x74, 0xaf, 0x26, 0xd9, 0x67, 0x02, 0xa1, 0xd3,
        0x61, 0x3a, 0xf1, 0xd3, 0x61, 0x05, 0xa1, 0x7d,
        0xe0, 0x38, 0x5d, 0x5b, 0x35, 0x9e, 0xa3, 0x7d,
        0xe6, 0x3a, 0x0f, 0x7d, 0x07, 0xaf, 0x20, 0x09,
        0x67, 0xac, 0x73, 0x46, 0x54, 0xaf, 0x26, 0xd0
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xcf, 0x80, 0x98, 0x6d, 0xfe, 0xb0, 0x90, 0xd1,
        0xcf, 0x86, 0x0f, 0x52, 0x2c, 0x23, 0x66, 0x28,
        0x27, 0x30, 0x48, 0x55, 0x42, 0x6a, 0x48, 0x4b,
        0x68, 0x22, 0x2e, 0x23, 0x64, 0x33, 0x2c, 0x2d,
        0x5c, 0x51, 0x48, 0x55, 0x24, 0x67, 0x6c, 0x4c,
        0x45, 0x71, 0x35, 0x72, 0x5a, 0x48, 0x5e, 0x35,
        0x61, 0x78, 0x35, 0x42, 0x2c, 0x7a, 0x75, 0x61,
        0x5b, 0x4e, 0x76, 0x30, 0x26, 0x2f, 0x2a, 0x34,
        0x48, 0x29, 0x25, 0x6e, 0x5c, 0x3a, 0x6c, 0x3e,
        0x79, 0x4e, 0x2a, 0x21, 0x6f, 0x6f, 0x34, 0x46,
        0x43, 0x26, 0x5b, 0x35, 0x78, 0x27, 0x69, 0x23,
        0x72, 0x21, 0x69, 0x56, 0x6a, 0x7d, 0x4b, 0x5e,
        0x65, 0x37, 0x60, 0x44, 0x7c, 0x5d, 0x5b, 0x72,
        0x7d, 0x73, 0x7b, 0x47, 0x57, 0x21, 0x41, 0x38,
        0x76, 0x38, 0x76, 0x5c, 0x58, 0x32, 0x4a, 0x37,
        0x2f, 0x40, 0x4b, 0x4c, 0x3d, 0x41, 0x33, 0x56,
        0x73, 0x38, 0x61, 0x71, 0x24, 0x49, 0x4c, 0x4a,
        0x44, 0x2e, 0x3a, 0x3f, 0x74, 0x54, 0x4c, 0x65,
        0x54, 0x2d, 0x3b, 0x28, 0x41, 0x45, 0x49, 0x2c,
        0x6e, 0x48, 0x44, 0x43, 0x37, 0x3d, 0x7b, 0x6d,
        0x2b, 0x4b, 0x32, 0x5a, 0x31, 0x61, 0x6e, 0x2b,
        0x27, 0x50, 0x6b, 0x66, 0x76, 0x4e, 0x55, 0x35,
        0x2b, 0x72, 0x2d, 0x5e, 0x42, 0x3e, 0x5a, 0x5d,
        0x36, 0x45, 0x32, 0x3a, 0x58, 0x78, 0x78, 0x3e,
        0x60, 0x6c, 0x5d, 0x63, 0x41, 0x7c, 0x52, 0x21,
        0x75, 0x6a, 0x5a, 0x70, 0x55, 0x45, 0x76, 0x58,
        0x33, 0x40, 0x38, 0x39, 0x21, 0x37, 0x7d, 0x77,
        0x21, 0x70, 0x2b, 0x72, 0x29, 0x6a, 0x31, 0x5f,
        0x38, 0x4a, 0x66, 0x65, 0x62, 0x2c, 0x39, 0x52,
        0x5f, 0x2a, 0x2b, 0x63, 0x4f, 0x76, 0x43, 0x25,
        0x6a, 0x50, 0x37, 0x52, 0x5e, 0x23, 0x3c, 0x42,
        0x28, 0x75, 0x75, 0x42, 0x25, 0x23, 0x28, 0x56,
        0x6c, 0x46, 0x5c, 0x5e, 0x6b, 0x7d, 0x48, 0x24,
        0x77, 0x6c, 0x70, 0x62, 0x2e, 0x28, 0x7d, 0x6b,
        0x69, 0x4a, 0x75, 0x3d, 0x5d, 0x56, 0x21, 0x49,
        0x56, 0x47, 0x64, 0x2b, 0x4c, 0x52, 0x43, 0x60,
        0x77, 0x49, 0x46, 0x46, 0x33, 0x2c, 0x4b, 0x4b,
        0x3d, 0x63, 0x5d, 0x33, 0x78, 0x76, 0x51, 0x56,
        0x77, 0x3c, 0x72, 0x74, 0x52, 0x27, 0x40, 0x6c,
        0x42, 0x79, 0x49, 0x24, 0x62, 0x5e, 0x26, 0x31,
        0x5c, 0x22, 0x2b, 0x4c, 0x64, 0x49, 0x52, 0x45,
        0x47, 0x49, 0x3a, 0x2a, 0x51, 0x71, 0x22, 0x22,
        0x70, 0x24, 0x34, 0x67, 0x4b, 0x6d, 0x58, 0x29,
        0x63, 0x26, 0x7b, 0x6f, 0x38, 0x78, 0x25, 0x62,
        0x4d, 0x3a, 0x7d, 0x40, 0x23, 0x57, 0x67, 0x33,
        0x38, 0x31, 0x4e, 0x54, 0x3c, 0x4b, 0x48, 0x69,
        0x3c, 0x39, 0x31, 0x2b, 0x26, 0x70, 0x44, 0x66,
        0x4a, 0x37, 0x2b, 0x75, 0x36, 0x45, 0x59, 0x34,
        0x3e, 0x3e, 0x29, 0x70, 0x71, 0x5a, 0x55, 0x49,
        0x3e, 0x4b, 0x68, 0x4e, 0x75, 0x70, 0x3c, 0x5c,
        0x50, 0x58, 0x28, 0x75, 0x3c, 0x2a, 0x41, 0x70,
        0x2f, 0x2b, 0x37, 0x26, 0x75, 0x71, 0x55, 0x22,
        0x3a, 0x44, 0x30, 0x48, 0x5d, 0x2f, 0x6c, 0x44,
        0x28, 0x4b, 0x34, 0x45, 0x21, 0x60, 0x44, 0x36,
        0x7b, 0x32, 0x39, 0x5f, 0x6d, 0x3f, 0x68, 0x73,
        0x25, 0x45, 0x56, 0x7c, 0x78, 0x7a, 0x49, 0x6a,
        0x46, 0x3d, 0x2d, 0x33, 0x6c, 0x6f, 0x23, 0x77,
        0x38, 0x33, 0x36, 0x74, 0x7b, 0x57, 0x4b, 0x6d,
        0x27, 0x75, 0x24, 0x6e, 0x43, 0x61, 0x4d, 0x44,
        0x6d, 0x27, 0x48, 0x58, 0x5e, 0x7b, 0x26, 0x6a,
        0x50, 0x7c, 0x51, 0x23, 0x3c, 0x4f, 0x37, 0x4c,
        0x47, 0x3e, 0x45, 0x56, 0x22, 0x33, 0x7c, 0x66,
        0x35, 0x54, 0x7a, 0x6e, 0x5a, 0x24, 0x70, 0x62,
        0x29, 0x3f, 0x69, 0x79, 0x24, 0x43, 0x41, 0x24,
        0x65, 0x25, 0x62, 0x4f, 0x73, 0x3e, 0x2b, 0x36,
        0x46, 0x69, 0x27, 0x55, 0x2a, 0x6e, 0x24, 0x6c,
        0x7d, 0x64, 0x7c, 0x61, 0x26, 0x67, 0x2a, 0x53,
        0x73, 0x60, 0x28, 0x2d, 0x6b, 0x44, 0x54, 0x61,
        0x34, 0x53, 0x22, 0x59, 0x6d, 0x73, 0x56, 0x55,
        0x25, 0x2c, 0x38, 0x4a, 0x3b, 0x4e, 0x78, 0x46,
        0x54, 0x6e, 0x6d, 0x4f, 0x47, 0x4f, 0x4f, 0x5a,
        0x67, 0x77, 0x39, 0x66, 0x28, 0x29, 0x4e, 0x43,
        0x55, 0x6e, 0x60, 0x59, 0x28, 0x3b, 0x65, 0x62,
        0x61, 0x5a, 0x29, 0x6e, 0x79, 0x60, 0x41, 0x53,
        0x2f, 0x5d, 0x44, 0x36, 0x7b, 0x3e, 0x7c, 0x2b,
        0x77, 0x36, 0x70, 0x3f, 0x40, 0x55, 0x48, 0x67,
        0x4b, 0x4d, 0x5d, 0x51, 0x79, 0x76, 0x48, 0x4a,
        0x2d, 0x21, 0x60, 0x40, 0x46, 0x55, 0x7a, 0x60,
        0x22, 0x25, 0x3f, 0x4b, 0x54, 0x6a, 0x6a, 0x3c,
        0x77, 0x22, 0x5b, 0x43, 0x67, 0x58, 0x71, 0x22,
        0x79, 0x4b, 0x32, 0x61, 0x44, 0x4d, 0x6f, 0x42,
        0x33, 0x2d, 0x53, 0x35, 0x3d, 0x6f, 0x57, 0x48,
        0x33, 0x3b, 0x5a, 0x53, 0x3f, 0x4e, 0x3f, 0x6b,
        0x4c, 0x27, 0x26, 0x3b, 0x73, 0x49, 0x22, 0x55,
        0x79, 0x2f, 0x47, 0x2f, 0x55, 0x5a, 0x7a, 0x71,
        0x6c, 0x31, 0x43, 0x40, 0x56, 0x7b, 0x21, 0x7a,
        0x6d, 0x4c, 0x43, 0x5e, 0x38, 0x47, 0x29, 0x38,
        0x62, 0x49, 0x45, 0x78, 0x70, 0x2b, 0x2e, 0x65,
        0x47, 0x71, 0x58, 0x79, 0x39, 0x67, 0x7d, 0x6d,
        0x6a, 0x67, 0x4a, 0x71, 0x27, 0x35, 0x2a, 0x4c,
        0x3e, 0x58, 0x55, 0x30, 0x4d, 0x75, 0x77, 0x48,
        0x5f, 0x4b, 0x59, 0x34, 0x65, 0x68, 0x57, 0x59,
        0x63, 0x23, 0x47, 0x38, 0x47, 0x5e, 0x56, 0x28,
        0x79, 0x58, 0x3e, 0x39, 0x66, 0x77, 0x67, 0x33,
        0x29, 0x61, 0x24, 0x7d, 0x37, 0x44, 0x37, 0x67,
        0x3a, 0x58, 0x76, 0x21, 0x51, 0x59, 0x61, 0x73,
        0x66, 0x75, 0x71, 0x53, 0x4d, 0x24, 0x2d, 0x4b,
        0x29, 0x30, 0x32, 0x26, 0x59, 0x64, 0x27, 0x55,
        0x2c, 0x5a, 0x4c, 0x3c, 0x6c, 0x53, 0x56, 0x4b,
        0x3e, 0x55, 0x2e, 0x44, 0x38, 0x6b, 0x47, 0x76,
        0x2d, 0x2c, 0x3f, 0x4d, 0x22, 0x7b, 0x6d, 0x61,
        0x34, 0x6b, 0x50, 0x73, 0x28, 0x6d, 0x41, 0x71,
        0x21, 0x76, 0x52, 0x2a, 0x6d, 0x53, 0x2a, 0x74,
        0x28, 0x27, 0x62, 0x2a, 0x66, 0x25, 0x6e, 0x5e,
        0x37, 0x4f, 0x27, 0x72, 0x28, 0x47, 0x63, 0x6e,
        0x5a, 0x6a, 0x41, 0x35, 0x3a, 0x42, 0x3f, 0x27,
        0x75, 0x3e, 0x26, 0x3e, 0x6b, 0x55, 0x59, 0x60,
        0x24, 0x70, 0x49, 0x3c, 0x4e, 0x2c, 0x39, 0x7a,
        0x36, 0x6c, 0x27, 0x3e, 0x6a, 0x4a, 0x59, 0x5a,
        0x3e, 0x21, 0x73, 0x4e, 0x59, 0x6e, 0x3d, 0x32,
        0x27, 0x45, 0x49, 0x58, 0x7d, 0x37, 0x39, 0x77,
        0x28, 0x51, 0x79, 0x54, 0x2b, 0x78, 0x46, 0x5a,
        0x21, 0x75, 0x33, 0x21, 0x63, 0x5a, 0x7b, 0x3e,
        0x33, 0x4f, 0x67, 0x75, 0x3a, 0x50, 0x48, 0x60,
        0x26, 0x64, 0x76, 0x5c, 0x42, 0x5c, 0x72, 0x38,
        0x6c, 0x52, 0x21, 0x2b, 0x25, 0x6b, 0x7c, 0x6b,
        0x2d, 0x5e, 0x63, 0x2a, 0x4c, 0x26, 0x5b, 0x4c,
        0x58, 0x52, 0x51, 0x55, 0x31, 0x79, 0x6c, 0x53,
        0x62, 0x3a, 0x36, 0x46, 0x7a, 0x29, 0x27, 0x78,
        0x1a, 0xbf, 0x49, 0x74, 0x68, 0x24, 0x51, 0x44,
        0x5b, 0x3e, 0x34, 0x44, 0x29, 0x5e, 0x4f, 0x2a,
        0xe9, 0x3f, 0xf8, 0xff, 0xff, 0x52, 0x7d, 0x47,
        0x67, 0x40, 0x27, 0x5e, 0x47, 0x46, 0x6d, 0x72,
        0x5d, 0x49, 0x26, 0x45, 0x33, 0x6b, 0x4d, 0x4a,
        0x6f, 0x62, 0x60, 0x45, 0x62, 0x27, 0x27, 0x7d,
        0x6a, 0x41, 0x2c, 0x6c, 0x5b, 0x2a, 0x2b, 0x36,
        0x29, 0x58, 0x7a, 0x4c, 0x6e, 0x2d, 0x74, 0x5c,
        0x38, 0x22, 0x5f, 0x49, 0x63, 0x43, 0x5b, 0x67
    };
    uint32_t request2_len = sizeof(request2);

    uint8_t request3[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x26, 0x65, 0x3c, 0x6e, 0x6d, 0x64, 0x24, 0x39,
        0x56, 0x43, 0x3e, 0x61, 0x5c, 0x54, 0x42, 0x23,
        0x75, 0x6b, 0x71, 0x27, 0x66, 0x2e, 0x6e, 0x3d,
        0x58, 0x23, 0x54, 0x77, 0x3b, 0x52, 0x6b, 0x50,
        0x3b, 0x74, 0x2c, 0x54, 0x25, 0x5c, 0x51, 0x7c,
        0x29, 0x7c, 0x5f, 0x4a, 0x35, 0x5c, 0x3d, 0x3f,
        0x33, 0x55, 0x3b, 0x5a, 0x57, 0x31, 0x59, 0x4f,
        0x6d, 0x6d, 0x7b, 0x3e, 0x38, 0x4d, 0x68, 0x75,
        0x64, 0x21, 0x50, 0x63, 0x47, 0x42, 0x56, 0x39,
        0x6c, 0x6f, 0x61, 0x53, 0x32, 0x56, 0x43, 0x52,
        0x43, 0x67, 0x26, 0x45, 0x28, 0x6b, 0x77, 0x28,
        0x7c, 0x64, 0x61, 0x24, 0x38, 0x6b, 0x59, 0x2a,
        0x4f, 0x6e, 0x5b, 0x57, 0x24, 0x54, 0x33, 0x37,
        0x47, 0x58, 0x4b, 0x58, 0x3d, 0x21, 0x38, 0x7c,
        0x2c, 0x24, 0x5f, 0x67, 0x3a, 0x41, 0x3e, 0x2a,
        0x72, 0x66, 0x2d, 0x6b, 0x66, 0x7b, 0x2b, 0x75,
        0x78, 0x2f, 0x4d, 0x4c, 0x51, 0x70, 0x5d, 0x55,
        0x54, 0x3c, 0x63, 0x46, 0x6b, 0x64, 0x4d, 0x25,
        0x45, 0x21, 0x34, 0x65, 0x48, 0x32, 0x58, 0x4c,
        0x70, 0x4c, 0x4c, 0x75, 0x5c, 0x77, 0x68, 0x78,
        0x34, 0x5c, 0x2d, 0x39, 0x58, 0x3b, 0x40, 0x71,
        0x77, 0x47, 0x32, 0x2e, 0x3c, 0x61, 0x6f, 0x6d,
        0x5f, 0x43, 0x74, 0x36, 0x4f, 0x21, 0x44, 0x66,
        0x36, 0x62, 0x30, 0x29, 0x5a, 0x34, 0x66, 0x4e,
        0x51, 0x23, 0x4e, 0x38, 0x51, 0x78, 0x74, 0x58,
        0x2e, 0x6d, 0x51, 0x49, 0x55, 0x73, 0x2a, 0x71,
        0x3c, 0x74, 0x38, 0x6f, 0x5d, 0x4b, 0x74, 0x68,
        0x65, 0x4a, 0x58, 0x41, 0x55, 0x29, 0x42, 0x69,
        0x55, 0x3b, 0x2b, 0x47, 0x64, 0x3b, 0x77, 0x72,
        0x74, 0x38, 0x53, 0x5c, 0x69, 0x49, 0x49, 0x5b,
        0x31, 0x41, 0x6a, 0x4e, 0x2c, 0x6a, 0x63, 0x3f,
        0x58, 0x4e, 0x25, 0x3e, 0x57, 0x41, 0x61, 0x26,
        0x5e, 0x24, 0x69, 0x7a, 0x38, 0x60, 0x73, 0x70,
        0x7d, 0x63, 0x34, 0x78, 0x4d, 0x50, 0x35, 0x69,
        0x49, 0x22, 0x45, 0x44, 0x3f, 0x6e, 0x75, 0x64,
        0x57, 0x3a, 0x61, 0x60, 0x34, 0x21, 0x61, 0x21,
        0x2a, 0x78, 0x7b, 0x52, 0x43, 0x50, 0x5b, 0x76,
        0x5f, 0x4b, 0x6a, 0x5d, 0x23, 0x5b, 0x57, 0x40,
        0x53, 0x51, 0x33, 0x21, 0x35, 0x7d, 0x31, 0x46,
        0x65, 0x52, 0x28, 0x25, 0x30, 0x5a, 0x37, 0x7c,
        0x2c, 0x3d, 0x2a, 0x48, 0x24, 0x5a, 0x2f, 0x47,
        0x64, 0x73, 0x64, 0x3d, 0x7a, 0x5b, 0x34, 0x5e,
        0x42, 0x22, 0x32, 0x47, 0x6e, 0x58, 0x3b, 0x3e,
        0x25, 0x2f, 0x58, 0x78, 0x42, 0x66, 0x71, 0x56,
        0x2a, 0x66, 0x66, 0x5b, 0x55, 0x35, 0x7a, 0x41,
        0x7c, 0x7c, 0x6a, 0x2d, 0x59, 0x25, 0x22, 0x34,
        0x5a, 0x61, 0x37, 0x48, 0x39, 0x31, 0x4a, 0x55,
        0x6a, 0x68, 0x40, 0x2f, 0x45, 0x69, 0x46, 0x25,
        0x51, 0x7d, 0x4f, 0x71, 0x21, 0x33, 0x55, 0x50,
        0x56, 0x5f, 0x75, 0x27, 0x64, 0x36, 0x7a, 0x39,
        0x40, 0x6a, 0x77, 0x38, 0x5d, 0x39, 0x30, 0x5e,
        0x74, 0x54, 0x24, 0x3f, 0x3d, 0x79, 0x3b, 0x27,
        0x7d, 0x68, 0x7d, 0x40, 0x71, 0x7a, 0x65, 0x54,
        0x50, 0x66, 0x33, 0x3c, 0x42, 0x69, 0x6e, 0x3c,
        0x63, 0x63, 0x69, 0x7a, 0x5e, 0x7b, 0x76, 0x26,
        0x71, 0x6f, 0x4a, 0x6d, 0x70, 0x73, 0x66, 0x3b,
        0x26, 0x70, 0x43, 0x5b, 0x52, 0x4c, 0x6d, 0x51,
        0x2a, 0x66, 0x6c, 0x3e, 0x68, 0x6a, 0x31, 0x41,
        0x79, 0x72, 0x37, 0x47, 0x7d, 0x2b, 0x3c, 0x40,
        0x6b, 0x75, 0x56, 0x70, 0x7b, 0x2d, 0x5f, 0x33,
        0x30, 0x30, 0x21, 0x35, 0x7a, 0x7a, 0x67, 0x48,
        0x5e, 0x3b, 0x73, 0x50, 0x54, 0x47, 0x23, 0x2b,
        0x4c, 0x4e, 0x2f, 0x24, 0x44, 0x34, 0x23, 0x5d,
        0x76, 0x51, 0x5a, 0x73, 0x72, 0x3e, 0x47, 0x77,
        0x40, 0x28, 0x65, 0x2e, 0x2a, 0x75, 0x3c, 0x2a,
        0x27, 0x4a, 0x3f, 0x3c, 0x66, 0x2d, 0x21, 0x79,
        0x2d, 0x2b, 0x78, 0x7c, 0x5a, 0x73, 0x46, 0x6b,
        0x39, 0x65, 0x5e, 0x3d, 0x38, 0x40, 0x32, 0x3e,
        0x21, 0x62, 0x34, 0x41, 0x58, 0x53, 0x67, 0x34,
        0x58, 0x56, 0x61, 0x5b, 0x3e, 0x4e, 0x2c, 0x5b,
        0x73, 0x35, 0x34, 0x35, 0x21, 0x3a, 0x61, 0x5f,
        0x6e, 0x45, 0x78, 0x44, 0x28, 0x23, 0x48, 0x65,
        0x53, 0x47, 0x6e, 0x2c, 0x38, 0x5e, 0x2c, 0x57,
        0x58, 0x30, 0x7a, 0x3b, 0x4b, 0x4a, 0x74, 0x7d,
        0x3e, 0x4d, 0x30, 0x24, 0x76, 0x66, 0x6d, 0x2e,
        0x74, 0x75, 0x28, 0x48, 0x5c, 0x23, 0x6c, 0x46,
        0x27, 0x46, 0x6e, 0x34, 0x63, 0x21, 0x58, 0x54,
        0x50, 0x2f, 0x40, 0x47, 0x40, 0x32, 0x36, 0x48,
        0x5f, 0x7d, 0x4a, 0x41, 0x6e, 0x60, 0x2c, 0x4a,
        0x6a, 0x67, 0x6c, 0x41, 0x27, 0x23, 0x30, 0x48,
        0x6a, 0x49, 0x73, 0x26, 0x77, 0x75, 0x4d, 0x65,
        0x5b, 0x34, 0x79, 0x67, 0x61, 0x5b, 0x5c, 0x2b,
        0x71, 0x3f, 0x62, 0x51, 0x3a, 0x53, 0x42, 0x26,
        0x6f, 0x36, 0x57, 0x3f, 0x2b, 0x34, 0x24, 0x30,
        0x60, 0x55, 0x70, 0x65, 0x70, 0x57, 0x5d, 0x68,
        0x36, 0x52, 0x5d, 0x3f, 0x6a, 0x3a, 0x33, 0x31,
        0x6c, 0x4e, 0x57, 0x79, 0x49, 0x79, 0x69, 0x71,
        0x6f, 0x70, 0x6a, 0x76, 0x4b, 0x2f, 0x33, 0x51,
        0x68, 0x30, 0x2e, 0x77, 0x78, 0x55, 0x2f, 0x53,
        0x52, 0x5e, 0x57, 0x60, 0x3b, 0x6f, 0x69, 0x61,
        0x6c, 0x60, 0x5a, 0x34, 0x5a, 0x35, 0x4b, 0x28,
        0x54, 0x32, 0x6a, 0x35, 0x36, 0x6d, 0x68, 0x47,
        0x5c, 0x74, 0x2e, 0x5f, 0x6c, 0x6d, 0x55, 0x42,
        0x77, 0x64, 0x7d, 0x53, 0x4d, 0x39, 0x2c, 0x41,
        0x42, 0x23, 0x3a, 0x73, 0x40, 0x60, 0x5d, 0x38,
        0x6d, 0x36, 0x56, 0x57, 0x2a, 0x28, 0x3d, 0x3b,
        0x5c, 0x75, 0x35, 0x2d, 0x69, 0x2d, 0x44, 0x51,
        0x27, 0x63, 0x66, 0x33, 0x46, 0x42, 0x2e, 0x36,
        0x6b, 0x7b, 0x2c, 0x23, 0x3b, 0x5a, 0x50, 0x2a,
        0x65, 0x28, 0x3b, 0x3c, 0x51, 0x3f, 0x4d, 0x63,
        0x38, 0x25, 0x74, 0x2e, 0x51, 0x22, 0x31, 0x74,
        0x35, 0x33, 0x23, 0x2d, 0x3f, 0x77, 0x26, 0x2c,
        0x55, 0x6d, 0x27, 0x39, 0x79, 0x76, 0x63, 0x4b,
        0x43, 0x4a, 0x3a, 0x6b, 0x59, 0x55, 0x65, 0x26,
        0x2f, 0x3f, 0x56, 0x67, 0x5a, 0x77, 0x71, 0x22,
        0x51, 0x2b, 0x6d, 0x4c, 0x2c, 0x57, 0x66, 0x76,
        0x37, 0x70, 0x5f, 0x52, 0x29, 0x44, 0x52, 0x22,
        0x57, 0x37, 0x27, 0x79, 0x29, 0x5c, 0x57, 0x3b,
        0x54, 0x3c, 0x3f, 0x53, 0x35, 0x27, 0x5e, 0x7c,
        0x49, 0x77, 0x57, 0x5a, 0x22, 0x76, 0x7c, 0x5b,
        0x2f, 0x53, 0x5e, 0x55, 0x6d, 0x64, 0x67, 0x34,
        0x41, 0x23, 0x76, 0x67, 0x23, 0x78, 0x6a, 0x63,
        0x27, 0x68, 0x43, 0x7d, 0x58, 0x49, 0x2d, 0x79,
        0x2e, 0x75, 0x60, 0x6b, 0x34, 0x48, 0x6f, 0x4a,
        0x6c, 0x48, 0x40, 0x68, 0x5f, 0x35, 0x25, 0x6c,
        0x38, 0x5c, 0x30, 0x32, 0x4c, 0x36, 0x31, 0x29,
        0x74, 0x4a, 0x55, 0x56, 0x6d, 0x4e, 0x23, 0x54,
        0x2e, 0x69, 0x78, 0x61, 0x76, 0x66, 0x22, 0x44,
        0x73, 0x25, 0x44, 0x29, 0x2a, 0x28, 0x3b, 0x67,
        0x48, 0x58, 0x37, 0x4a, 0x76, 0x76, 0x51, 0x4a,
        0x61, 0x70, 0x51, 0x74, 0x40, 0x23, 0x29, 0x63,
        0x69, 0x4a, 0x29, 0x23, 0x34, 0x6a, 0x3b, 0x25,
        0x28, 0x54, 0x45, 0x33, 0x28, 0x44, 0x30, 0x61,
        0x5b, 0x60, 0x51, 0x3f, 0x68, 0x50, 0x70, 0x3d,
        0x58, 0x2e, 0x6e, 0x59, 0x5a, 0x62, 0x66, 0x4d,
        0x7a, 0x2e, 0x37, 0x37, 0x3d, 0x7b, 0x74, 0x79,
        0x48, 0x45, 0x77, 0x56, 0x33, 0x76, 0x71, 0x60,
        0x74, 0x3f, 0x61, 0x22, 0x52, 0x51, 0x71, 0x69
    };
    uint32_t request3_len = sizeof(request3);

    uint8_t request4[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x75, 0x3e, 0x76, 0x3e, 0x66, 0x6b, 0x6b, 0x3e,
        0x6d, 0x59, 0x38, 0x2b, 0x63, 0x4d, 0x2c, 0x73,
        0x54, 0x57, 0x34, 0x25, 0x5b, 0x42, 0x7d, 0x5d,
        0x37, 0x34, 0x2c, 0x79, 0x24, 0x4b, 0x74, 0x73,
        0x25, 0x36, 0x73, 0x3a, 0x2c, 0x55, 0x69, 0x3c,
        0x58, 0x67, 0x33, 0x53, 0x67, 0x5c, 0x61, 0x7b,
        0x44, 0x2e, 0x42, 0x2d, 0x6b, 0x50, 0x55, 0x24,
        0x70, 0x58, 0x60, 0x38, 0x42, 0x45, 0x70, 0x6d,
        0x2f, 0x27, 0x27, 0x2c, 0x21, 0x6d, 0x57, 0x6e,
        0x43, 0x3c, 0x5b, 0x27, 0x7a, 0x34, 0x49, 0x5a,
        0x69, 0x30, 0x3f, 0x6f, 0x77, 0x70, 0x39, 0x2d,
        0x51, 0x74, 0x4b, 0x25, 0x70, 0x51, 0x64, 0x4d,
        0x75, 0x52, 0x5e, 0x3e, 0x37, 0x30, 0x5d, 0x3b,
        0x2c, 0x72, 0x25, 0x6c, 0x6f, 0x79, 0x69, 0x3c,
        0x5b, 0x73, 0x3d, 0x41, 0x28, 0x28, 0x64, 0x60,
        0x4b, 0x7a, 0x2c, 0x4a, 0x6b, 0x3d, 0x2e, 0x6c,
        0x7a, 0x54, 0x70, 0x61, 0x6f, 0x4b, 0x40, 0x28,
        0x59, 0x31, 0x25, 0x21, 0x57, 0x79, 0x4b, 0x31,
        0x6f, 0x4e, 0x71, 0x2b, 0x3c, 0x24, 0x30, 0x28,
        0x3c, 0x61, 0x28, 0x4b, 0x35, 0x61, 0x4d, 0x55,
        0x5e, 0x66, 0x34, 0x5f, 0x61, 0x70, 0x7b, 0x67,
        0x51, 0x55, 0x68, 0x78, 0x26, 0x3a, 0x27, 0x4e,
        0x71, 0x79, 0x4f, 0x67, 0x2c, 0x5a, 0x79, 0x75,
        0x59, 0x3a, 0x33, 0x4a, 0x36, 0x71, 0x72, 0x6d,
        0x49, 0x3e, 0x53, 0x59, 0x2b, 0x2b, 0x27, 0x4e,
        0x50, 0x5d, 0x21, 0x55, 0x64, 0x4b, 0x72, 0x73,
        0x25, 0x55, 0x26, 0x4f, 0x3a, 0x21, 0x54, 0x29,
        0x4f, 0x64, 0x51, 0x59, 0x60, 0x7b, 0x7c, 0x6f,
        0x3e, 0x65, 0x74, 0x6a, 0x5b, 0x52, 0x2c, 0x56,
        0x4e, 0x45, 0x53, 0x4b, 0x7c, 0x38, 0x49, 0x4b,
        0x4e, 0x4f, 0x4a, 0x47, 0x5e, 0x7c, 0x46, 0x3b,
        0x67, 0x2e, 0x43, 0x79, 0x35, 0x55, 0x59, 0x6d,
        0x38, 0x70, 0x2f, 0x59, 0x4f, 0x27, 0x63, 0x40,
        0x66, 0x2d, 0x39, 0x4f, 0x3d, 0x2e, 0x4c, 0x67,
        0x71, 0x7d, 0x34, 0x22, 0x52, 0x4e, 0x36, 0x7b,
        0x2c, 0x39, 0x4d, 0x42, 0x60, 0x75, 0x74, 0x72,
        0x4f, 0x72, 0x68, 0x3a, 0x51, 0x31, 0x2d, 0x21,
        0x4a, 0x35, 0x47, 0x6d, 0x69, 0x3c, 0x50, 0x4c,
        0x59, 0x66, 0x4c, 0x71, 0x24, 0x3a, 0x36, 0x67,
        0x24, 0x5a, 0x59, 0x28, 0x7c, 0x21, 0x5e, 0x77,
        0x68, 0x5e, 0x7b, 0x6e, 0x56, 0x62, 0x36, 0x29,
        0x6f, 0x4f, 0x5d, 0x57, 0x56, 0x2b, 0x75, 0x2a,
        0x2c, 0x69, 0x63, 0x51, 0x74, 0x6e, 0x5e, 0x46,
        0x50, 0x28, 0x2c, 0x3b, 0x32, 0x53, 0x28, 0x78,
        0x59, 0x72, 0x39, 0x5e, 0x44, 0x5c, 0x77, 0x60,
        0x72, 0x44, 0x3b, 0x75, 0x68, 0x39, 0x55, 0x3e,
        0x44, 0x50, 0x76, 0x3c, 0x48, 0x46, 0x43, 0x22,
        0x56, 0x27, 0x21, 0x31, 0x33, 0x4a, 0x5a, 0x74,
        0x41, 0x58, 0x3f, 0x39, 0x29, 0x71, 0x73, 0x30,
        0x57, 0x70, 0x33, 0x62, 0x7b, 0x4a, 0x75, 0x3e,
        0x4d, 0x4c, 0x4e, 0x55, 0x63, 0x38, 0x66, 0x7d,
        0x68, 0x7d, 0x6f, 0x23, 0x55, 0x50, 0x3d, 0x34,
        0x46, 0x5e, 0x2f, 0x55, 0x27, 0x62, 0x68, 0x7c,
        0x6c, 0x21, 0x2b, 0x63, 0x4b, 0x47, 0x6b, 0x6a,
        0x5b, 0x7b, 0x5c, 0x71, 0x37, 0x7c, 0x52, 0x2b,
        0x2f, 0x4a, 0x47, 0x70, 0x78, 0x50, 0x2f, 0x75,
        0x28, 0x4c, 0x60, 0x4c, 0x4c, 0x54, 0x6b, 0x68,
        0x63, 0x4f, 0x47, 0x39, 0x2a, 0x70, 0x51, 0x7d,
        0x28, 0x59, 0x52, 0x46, 0x4b, 0x38, 0x27, 0x49,
        0x50, 0x5d, 0x25, 0x22, 0x5f, 0x48, 0x2c, 0x2f,
        0x67, 0x59, 0x5d, 0x7d, 0x21, 0x3d, 0x72, 0x4f,
        0x5c, 0x5b, 0x41, 0x47, 0x5f, 0x56, 0x69, 0x42,
        0x55, 0x60, 0x68, 0x4b, 0x77, 0x44, 0x4c, 0x3b,
        0x7d, 0x5a, 0x58, 0x43, 0x7a, 0x33, 0x22, 0x58,
        0x58, 0x6f, 0x74, 0x53, 0x57, 0x6d, 0x6e, 0x29,
        0x6b, 0x33, 0x71, 0x68, 0x29, 0x48, 0x67, 0x35,
        0x52, 0x41, 0x6b, 0x36, 0x4f, 0x46, 0x31, 0x24,
        0x73, 0x56, 0x40, 0x48, 0x37, 0x51, 0x24, 0x2a,
        0x59, 0x21, 0x74, 0x76, 0x25, 0x2e, 0x4a, 0x74,
        0x32, 0x29, 0x5f, 0x57, 0x7c, 0x58, 0x30, 0x2c,
        0x7b, 0x70, 0x5b, 0x51, 0x73, 0x27, 0x4a, 0x28,
        0x77, 0x2a, 0x43, 0x28, 0x2e, 0x32, 0x3d, 0x38,
        0x36, 0x2e, 0x6b, 0x40, 0x6c, 0x76, 0x54, 0x66,
        0x4a, 0x5c, 0x25, 0x62, 0x2e, 0x61, 0x48, 0x30,
        0x28, 0x41, 0x40, 0x69, 0x3c, 0x39, 0x36, 0x4b,
        0x64, 0x50, 0x76, 0x3d, 0x52, 0x50, 0x77, 0x33,
        0x3b, 0x65, 0x59, 0x31, 0x5c, 0x48, 0x6a, 0x74,
        0x78, 0x5b, 0x74, 0x60, 0x47, 0x27, 0x60, 0x22,
        0x4a, 0x72, 0x25, 0x34, 0x5d, 0x3a, 0x21, 0x66,
        0x61, 0x7b, 0x34, 0x41, 0x3b, 0x3a, 0x27, 0x44,
        0x48, 0x7c, 0x7a, 0x74, 0x3a, 0x68, 0x59, 0x48,
        0x61, 0x32, 0x49, 0x61, 0x40, 0x22, 0x33, 0x75,
        0x29, 0x76, 0x5b, 0x24, 0x5b, 0x5c, 0x76, 0x5c,
        0x28, 0x75, 0x36, 0x26, 0x2c, 0x65, 0x5e, 0x51,
        0x7b, 0x3a, 0x7d, 0x4f, 0x35, 0x73, 0x6b, 0x5b,
        0x5c, 0x37, 0x35, 0x6b, 0x41, 0x35, 0x40, 0x3a,
        0x22, 0x28, 0x6c, 0x71, 0x46, 0x68, 0x7b, 0x66,
        0x56, 0x24, 0x7c, 0x54, 0x28, 0x30, 0x22, 0x4e,
        0x3c, 0x65, 0x69, 0x36, 0x44, 0x53, 0x3d, 0x6c,
        0x5f, 0x73, 0x6c, 0x6f, 0x5e, 0x27, 0x23, 0x4e,
        0x60, 0x45, 0x2f, 0x3d, 0x37, 0x28, 0x51, 0x29,
        0x77, 0x6a, 0x6b, 0x2a, 0x2a, 0x51, 0x26, 0x4c,
        0x4e, 0x71, 0x77, 0x73, 0x71, 0x2d, 0x5a, 0x2c,
        0x23, 0x3d, 0x5f, 0x62, 0x63, 0x2e, 0x72, 0x2a,
        0x75, 0x66, 0x43, 0x56, 0x5f, 0x21, 0x64, 0x66,
        0x35, 0x3b, 0x7a, 0x45, 0x3f, 0x4f, 0x57, 0x22,
        0x5a, 0x45, 0x65, 0x37, 0x58, 0x5b, 0x43, 0x66,
        0x4f, 0x5d, 0x6e, 0x41, 0x41, 0x62, 0x5e, 0x39,
        0x65, 0x6f, 0x43, 0x4b, 0x5e, 0x51, 0x42, 0x3f,
        0x2d, 0x68, 0x4b, 0x6e, 0x46, 0x6f, 0x21, 0x44,
        0x3c, 0x22, 0x46, 0x31, 0x31, 0x2e, 0x56, 0x2e,
        0x77, 0x48, 0x68, 0x23, 0x4a, 0x36, 0x52, 0x5d,
        0x61, 0x47, 0x71, 0x2e, 0x3a, 0x4a, 0x5b, 0x56,
        0x6b, 0x52, 0x2a, 0x4c, 0x4f, 0x24, 0x34, 0x60,
        0x70, 0x58, 0x7a, 0x76, 0x4b, 0x68, 0x24, 0x5f,
        0x51, 0x6d, 0x75, 0x45, 0x48, 0x21, 0x53, 0x4d,
        0x27, 0x75, 0x5f, 0x50, 0x3e, 0x40, 0x3f, 0x5e,
        0x64, 0x41, 0x5f, 0x68, 0x48, 0x30, 0x71, 0x4b,
        0x66, 0x2c, 0x2f, 0x76, 0x4b, 0x23, 0x46, 0x34,
        0x50, 0x58, 0x52, 0x69, 0x2b, 0x6e, 0x7a, 0x33,
        0x53, 0x43, 0x43, 0x35, 0x54, 0x30, 0x73, 0x63,
        0x3b, 0x43, 0x52, 0x29, 0x45, 0x37, 0x71, 0x79,
        0x5a, 0x26, 0x24, 0x72, 0x73, 0x4e, 0x44, 0x38,
        0x5b, 0x71, 0x36, 0x3a, 0x4f, 0x5b, 0x71, 0x28,
        0x71, 0x79, 0x72, 0x40, 0x6e, 0x51, 0x72, 0x29,
        0x3d, 0x4f, 0x33, 0x22, 0x73, 0x5a, 0x30, 0x71,
        0x58, 0x54, 0x59, 0x48, 0x29, 0x2b, 0x5c, 0x73,
        0x6f, 0x4e, 0x60, 0x2a, 0x72, 0x39, 0x50, 0x59,
        0x6f, 0x48, 0x3e, 0x62, 0x6c, 0x62, 0x49, 0x6c,
        0x2c, 0x3f, 0x43, 0x3f, 0x32, 0x7c, 0x6f, 0x6c,
        0x39, 0x26, 0x26, 0x7b, 0x5d, 0x65, 0x6f, 0x41,
        0x7c, 0x42, 0x2b, 0x65, 0x6f, 0x3e, 0x7b, 0x69,
        0x46, 0x4d, 0x68, 0x68, 0x5a, 0x33, 0x25, 0x5d,
        0x6f, 0x48, 0x7c, 0x77, 0x7d, 0x3f, 0x4e, 0x30,
        0x69, 0x65, 0x28, 0x2e, 0x34, 0x34, 0x41, 0x43,
        0x5e, 0x30, 0x23, 0x3b, 0x60, 0x79, 0x5b, 0x26,
        0x7c, 0x77, 0x3e, 0x43, 0x24, 0x31, 0x3a, 0x56,
        0x24, 0x3c, 0x60, 0x3f, 0x60, 0x55, 0x6a, 0x68
    };
    uint32_t request4_len = sizeof(request4);

    uint8_t request5[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x69, 0x3e, 0x72, 0x44, 0x31, 0x6b, 0x28, 0x2f,
        0x79, 0x37, 0x58, 0x5d, 0x5f, 0x68, 0x71, 0x47,
        0x7a, 0x68, 0x7c, 0x6c, 0x65, 0x3c, 0x74, 0x67,
        0x59, 0x5c, 0x3d, 0x28, 0x65, 0x28, 0x58, 0x74,
        0x44, 0x62, 0x2e, 0x36, 0x54, 0x2f, 0x24, 0x34,
        0x4b, 0x6d, 0x3a, 0x7b, 0x60, 0x71, 0x5a, 0x77,
        0x4a, 0x27, 0x25, 0x70, 0x75, 0x56, 0x78, 0x73,
        0x2e, 0x38, 0x6c, 0x70, 0x66, 0x7b, 0x7b, 0x2d,
        0x78, 0x27, 0x65, 0x63, 0x58, 0x4f, 0x7d, 0x5c,
        0x31, 0x3e, 0x36, 0x6e, 0x65, 0x61, 0x2e, 0x4e,
        0x26, 0x68, 0x2b, 0x33, 0x7d, 0x54, 0x2c, 0x28,
        0x47, 0x3a, 0x31, 0x47, 0x56, 0x32, 0x74, 0x51,
        0x79, 0x65, 0x42, 0x45, 0x60, 0x55, 0x6f, 0x48,
        0x61, 0x23, 0x72, 0x62, 0x74, 0x3a, 0x5a, 0x26,
        0x2d, 0x41, 0x58, 0x62, 0x75, 0x4b, 0x37, 0x2e,
        0x3f, 0x2a, 0x6e, 0x2e, 0x2c, 0x43, 0x6f, 0x53,
        0x5f, 0x48, 0x7a, 0x53, 0x7b, 0x54, 0x28, 0x30,
        0x2b, 0x7a, 0x34, 0x33, 0x28, 0x2b, 0x23, 0x23,
        0x72, 0x38, 0x25, 0x30, 0x35, 0x66, 0x76, 0x46,
        0x2a, 0x57, 0x7a, 0x60, 0x38, 0x5a, 0x26, 0x4f,
        0x78, 0x43, 0x2c, 0x7d, 0x3d, 0x76, 0x7d, 0x66,
        0x48, 0x7d, 0x3e, 0x59, 0x31, 0x58, 0x6b, 0x30,
        0x76, 0x45, 0x6e, 0x70, 0x72, 0x5f, 0x3c, 0x70,
        0x6d, 0x77, 0x42, 0x75, 0x42, 0x73, 0x68, 0x5e,
        0x5f, 0x72, 0x2b, 0x2a, 0x70, 0x38, 0x7a, 0x4c,
        0x58, 0x2e, 0x5e, 0x2d, 0x2d, 0x78, 0x67, 0x5a,
        0x77, 0x34, 0x5a, 0x50, 0x76, 0x2d, 0x2b, 0x77,
        0x37, 0x6e, 0x38, 0x2d, 0x7b, 0x44, 0x78, 0x67,
        0x52, 0x57, 0x79, 0x43, 0x7d, 0x6d, 0x4d, 0x32,
        0x23, 0x37, 0x51, 0x4b, 0x41, 0x60, 0x6e, 0x53,
        0x4e, 0x78, 0x37, 0x37, 0x60, 0x56, 0x64, 0x52,
        0x25, 0x46, 0x53, 0x5f, 0x2b, 0x56, 0x2b, 0x3b,
        0x40, 0x37, 0x33, 0x37, 0x23, 0x43, 0x36, 0x6b,
        0x6b, 0x5d, 0x35, 0x28, 0x7d, 0x6a, 0x2c, 0x68,
        0x28, 0x4b, 0x4a, 0x6c, 0x27, 0x35, 0x51, 0x66,
        0x30, 0x39, 0x28, 0x4d, 0x61, 0x2f, 0x64, 0x36,
        0x59, 0x39, 0x68, 0x4b, 0x24, 0x51, 0x7b, 0x6e,
        0x38, 0x49, 0x55, 0x72, 0x5f, 0x33, 0x5c, 0x26,
        0x45, 0x2f, 0x71, 0x66, 0x33, 0x3d, 0x36, 0x68,
        0x65, 0x48, 0x42, 0x40, 0x58, 0x61, 0x4f, 0x50,
        0x70, 0x5e, 0x3c, 0x5d, 0x56, 0x43, 0x4c, 0x41,
        0x45, 0x54, 0x76, 0x4b, 0x21, 0x25, 0x45, 0x4c,
        0x5e, 0x58, 0x23, 0x7d, 0x34, 0x61, 0x5c, 0x53,
        0x2a, 0x47, 0x37, 0x22, 0x6d, 0x31, 0x42, 0x6e,
        0x22, 0x72, 0x62, 0x55, 0x59, 0x66, 0x28, 0x73,
        0x55, 0x50, 0x5c, 0x6f, 0x52, 0x40, 0x3e, 0x3b,
        0x44, 0x2a, 0x51, 0x3d, 0x4d, 0x47, 0x3a, 0x57,
        0x3e, 0x29, 0x29, 0x7d, 0x40, 0x36, 0x41, 0x3f,
        0x58, 0x77, 0x3b, 0x41, 0x2d, 0x64, 0x5a, 0x72,
        0x7c, 0x7d, 0x30, 0x68, 0x54, 0x34, 0x40, 0x21,
        0x7d, 0x2b, 0x2d, 0x2b, 0x6d, 0x5f, 0x49, 0x57,
        0x68, 0x65, 0x79, 0x2c, 0x21, 0x41, 0x31, 0x55,
        0x27, 0x4d, 0x78, 0x55, 0x2f, 0x61, 0x62, 0x78,
        0x58, 0x25, 0x3a, 0x4b, 0x3e, 0x67, 0x44, 0x7c,
        0x7d, 0x52, 0x3d, 0x3e, 0x3b, 0x62, 0x2d, 0x28,
        0x48, 0x70, 0x2c, 0x79, 0x31, 0x5a, 0x5e, 0x3f,
        0x6a, 0x30, 0x78, 0x41, 0x44, 0x60, 0x4e, 0x63,
        0x63, 0x2e, 0x31, 0x79, 0x2b, 0x47, 0x57, 0x26,
        0x22, 0x6a, 0x46, 0x43, 0x70, 0x30, 0x51, 0x7d,
        0x21, 0x3c, 0x68, 0x74, 0x40, 0x5a, 0x6e, 0x71,
        0x3f, 0x76, 0x73, 0x2e, 0x29, 0x3f, 0x6a, 0x55,
        0x21, 0x72, 0x65, 0x75, 0x5e, 0x6b, 0x39, 0x6e,
        0x3e, 0x76, 0x42, 0x41, 0x65, 0x3f, 0x2b, 0x37,
        0x70, 0x7a, 0x7a, 0x29, 0x50, 0x66, 0x21, 0x67,
        0x3f, 0x54, 0x32, 0x5f, 0x73, 0x27, 0x59, 0x6f,
        0x39, 0x4b, 0x4e, 0x23, 0x54, 0x3b, 0x39, 0x21,
        0x38, 0x41, 0x33, 0x44, 0x57, 0x6b, 0x51, 0x30,
        0x6a, 0x76, 0x62, 0x2c, 0x5c, 0x5e, 0x49, 0x3e,
        0x59, 0x38, 0x5e, 0x4a, 0x59, 0x77, 0x34, 0x25,
        0x4f, 0x76, 0x6a, 0x68, 0x6f, 0x73, 0x7c, 0x3d,
        0x2d, 0x64, 0x6c, 0x7a, 0x3d, 0x2c, 0x26, 0x28,
        0x58, 0x2b, 0x4b, 0x45, 0x68, 0x38, 0x74, 0x63,
        0x7b, 0x4a, 0x63, 0x52, 0x26, 0x54, 0x3c, 0x46,
        0x77, 0x2d, 0x6b, 0x78, 0x63, 0x7b, 0x6a, 0x50,
        0x26, 0x42, 0x62, 0x63, 0x65, 0x6b, 0x63, 0x54,
        0x4d, 0x47, 0x59, 0x48, 0x2e, 0x60, 0x7c, 0x4d,
        0x33, 0x4d, 0x61, 0x72, 0x76, 0x72, 0x21, 0x4d,
        0x2b, 0x43, 0x58, 0x47, 0x4a, 0x36, 0x2d, 0x7b,
        0x32, 0x72, 0x21, 0x78, 0x22, 0x38, 0x2c, 0x7a,
        0x34, 0x44, 0x45, 0x66, 0x31, 0x7b, 0x37, 0x68,
        0x62, 0x65, 0x62, 0x6d, 0x4e, 0x7c, 0x75, 0x38,
        0x2a, 0x73, 0x27, 0x64, 0x33, 0x4f, 0x21, 0x41,
        0x7c, 0x41, 0x3f, 0x60, 0x68, 0x34, 0x72, 0x5b,
        0x38, 0x33, 0x6f, 0x65, 0x3e, 0x5a, 0x7d, 0x25,
        0x49, 0x50, 0x60, 0x36, 0x59, 0x5e, 0x6b, 0x25,
        0x66, 0x7a, 0x7d, 0x71, 0x40, 0x6c, 0x2c, 0x6e,
        0x6a, 0x5a, 0x24, 0x5a, 0x76, 0x21, 0x67, 0x39,
        0x4b, 0x4a, 0x31, 0x24, 0x66, 0x66, 0x2e, 0x58,
        0x43, 0x46, 0x75, 0x6c, 0x47, 0x28, 0x4f, 0x21,
        0x75, 0x77, 0x6f, 0x71, 0x48, 0x3f, 0x4d, 0x4c,
        0x51, 0x37, 0x3b, 0x41, 0x4d, 0x41, 0x48, 0x28,
        0x71, 0x24, 0x2f, 0x7a, 0x22, 0x49, 0x4a, 0x39,
        0x44, 0x43, 0x68, 0x21, 0x3a, 0x34, 0x4e, 0x52,
        0x7a, 0x60, 0x71, 0x61, 0x6d, 0x51, 0x58, 0x2a,
        0x59, 0x4c, 0x4a, 0x59, 0x6b, 0x77, 0x78, 0x2e,
        0x27, 0x78, 0x76, 0x48, 0x4f, 0x46, 0x79, 0x2c,
        0x54, 0x42, 0x7b, 0x2c, 0x52, 0x41, 0x54, 0x2b,
        0x2c, 0x33, 0x6b, 0x70, 0x77, 0x2e, 0x2e, 0x41,
        0x25, 0x7a, 0x48, 0x6e, 0x71, 0x55, 0x6a, 0x43,
        0x5a, 0x2c, 0x6c, 0x76, 0x6d, 0x71, 0x72, 0x4d,
        0x76, 0x5b, 0x7b, 0x22, 0x4b, 0x45, 0x31, 0x30,
        0x26, 0x53, 0x75, 0x3f, 0x26, 0x59, 0x36, 0x2f,
        0x68, 0x4f, 0x34, 0x5e, 0x2b, 0x30, 0x63, 0x68,
        0x7b, 0x32, 0x5e, 0x77, 0x7d, 0x7b, 0x53, 0x5f,
        0x63, 0x53, 0x77, 0x7a, 0x7d, 0x35, 0x28, 0x3e,
        0x41, 0x6f, 0x5b, 0x31, 0x78, 0x7b, 0x2b, 0x51,
        0x23, 0x43, 0x46, 0x6a, 0x32, 0x32, 0x25, 0x45,
        0x57, 0x43, 0x22, 0x50, 0x60, 0x32, 0x70, 0x2e,
        0x79, 0x2e, 0x6b, 0x33, 0x67, 0x6c, 0x43, 0x5b,
        0x3b, 0x68, 0x53, 0x53, 0x6a, 0x48, 0x59, 0x5f,
        0x30, 0x72, 0x7d, 0x6b, 0x37, 0x24, 0x75, 0x52,
        0x50, 0x2b, 0x75, 0x35, 0x24, 0x3b, 0x6e, 0x53,
        0x56, 0x34, 0x23, 0x54, 0x65, 0x4f, 0x78, 0x3e,
        0x46, 0x7d, 0x25, 0x3f, 0x2f, 0x49, 0x6b, 0x49,
        0x47, 0x45, 0x24, 0x38, 0x3b, 0x68, 0x6c, 0x4f,
        0x29, 0x21, 0x50, 0x32, 0x67, 0x47, 0x5a, 0x72,
        0x76, 0x21, 0x39, 0x67, 0x3c, 0x72, 0x47, 0x43,
        0x4a, 0x2e, 0x31, 0x32, 0x34, 0x3c, 0x53, 0x2d,
        0x22, 0x5b, 0x5b, 0x6a, 0x77, 0x75, 0x31, 0x68,
        0x30, 0x45, 0x43, 0x5f, 0x60, 0x5d, 0x56, 0x67,
        0x66, 0x55, 0x6a, 0x72, 0x77, 0x7b, 0x44, 0x61,
        0x22, 0x64, 0x36, 0x39, 0x6e, 0x44, 0x37, 0x54,
        0x45, 0x46, 0x6f, 0x58, 0x35, 0x51, 0x3c, 0x62,
        0x49, 0x3a, 0x50, 0x58, 0x56, 0x5d, 0x77, 0x6f,
        0x56, 0x64, 0x7b, 0x49, 0x39, 0x21, 0x31, 0x2d,
        0x5f, 0x56, 0x56, 0x33, 0x31, 0x69, 0x4a, 0x52,
        0x62, 0x5b, 0x6e, 0x65, 0x7c, 0x3d, 0x31, 0x55,
        0x3d, 0x75, 0x25, 0x61, 0x50, 0x71, 0x45, 0x29
    };
    uint32_t request5_len = sizeof(request5);

    uint8_t request6[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x5b, 0x56, 0x3d, 0x5a, 0x6b, 0x43, 0x73, 0x26,
        0x65, 0x3b, 0x38, 0x79, 0x26, 0x5e, 0x60, 0x59,
        0x40, 0x71, 0x7c, 0x72, 0x28, 0x29, 0x69, 0x32,
        0x72, 0x5a, 0x6c, 0x55, 0x43, 0x65, 0x3f, 0x4a,
        0x21, 0x66, 0x59, 0x30, 0x76, 0x39, 0x21, 0x69,
        0x4b, 0x25, 0x5d, 0x6e, 0x5f, 0x24, 0x2b, 0x38,
        0x70, 0x78, 0x35, 0x7d, 0x39, 0x36, 0x31, 0x72,
        0x44, 0x49, 0x45, 0x3d, 0x25, 0x50, 0x24, 0x3b,
        0x52, 0x27, 0x66, 0x46, 0x5d, 0x4f, 0x34, 0x50,
        0x26, 0x5a, 0x25, 0x3e, 0x3f, 0x34, 0x4b, 0x35,
        0x77, 0x3a, 0x3f, 0x3e, 0x23, 0x4e, 0x30, 0x23,
        0x70, 0x72, 0x33, 0x34, 0x60, 0x2a, 0x4a, 0x32,
        0x6e, 0x29, 0x54, 0x73, 0x5f, 0x26, 0x71, 0x3a,
        0x78, 0x5d, 0x3f, 0x31, 0x48, 0x59, 0x61, 0x44,
        0x5c, 0x38, 0x4f, 0x41, 0x73, 0x67, 0x62, 0x73,
        0x33, 0x52, 0x77, 0x73, 0x57, 0x49, 0x7a, 0x59,
        0x26, 0x21, 0x34, 0x38, 0x2b, 0x5f, 0x5f, 0x37,
        0x74, 0x28, 0x46, 0x3d, 0x43, 0x42, 0x26, 0x66,
        0x63, 0x37, 0x6d, 0x2a, 0x65, 0x3f, 0x71, 0x2d,
        0x4c, 0x72, 0x29, 0x4b, 0x3a, 0x77, 0x64, 0x6a,
        0x6b, 0x42, 0x70, 0x5c, 0x51, 0x38, 0x71, 0x25,
        0x4c, 0x7c, 0x6f, 0x74, 0x71, 0x39, 0x71, 0x25,
        0x3f, 0x62, 0x23, 0x45, 0x5f, 0x77, 0x59, 0x56,
        0x56, 0x67, 0x78, 0x3a, 0x2e, 0x4e, 0x27, 0x59,
        0x65, 0x2f, 0x64, 0x3c, 0x62, 0x40, 0x69, 0x52,
        0x36, 0x49, 0x3e, 0x3b, 0x2c, 0x47, 0x4f, 0x3e,
        0x61, 0x78, 0x2d, 0x45, 0x71, 0x3f, 0x7b, 0x55,
        0x34, 0x36, 0x47, 0x5e, 0x36, 0x51, 0x3d, 0x5a,
        0x4b, 0x75, 0x44, 0x72, 0x61, 0x44, 0x71, 0x4e,
        0x42, 0x6a, 0x2c, 0x34, 0x40, 0x3b, 0x40, 0x31,
        0x31, 0x75, 0x4b, 0x32, 0x71, 0x69, 0x3a, 0x5d,
        0x31, 0x25, 0x53, 0x2a, 0x61, 0x54, 0x68, 0x2a,
        0x76, 0x71, 0x57, 0x67, 0x56, 0x23, 0x7d, 0x70,
        0x7d, 0x28, 0x57, 0x5f, 0x2f, 0x4c, 0x71, 0x2e,
        0x40, 0x63, 0x49, 0x5b, 0x7c, 0x7b, 0x56, 0x76,
        0x77, 0x46, 0x69, 0x56, 0x3d, 0x75, 0x31, 0x3b,
        0x35, 0x40, 0x37, 0x2c, 0x51, 0x37, 0x49, 0x6a,
        0x79, 0x68, 0x53, 0x31, 0x4c, 0x6f, 0x57, 0x4c,
        0x48, 0x31, 0x6a, 0x30, 0x2b, 0x69, 0x30, 0x56,
        0x58, 0x4b, 0x76, 0x3b, 0x60, 0x6d, 0x35, 0x4d,
        0x74, 0x2f, 0x74, 0x2c, 0x54, 0x4f, 0x6e, 0x3f,
        0x38, 0x56, 0x5c, 0x67, 0x2b, 0x4a, 0x35, 0x30,
        0x67, 0x7d, 0x58, 0x24, 0x59, 0x54, 0x48, 0x2e,
        0x28, 0x7d, 0x6e, 0x51, 0x55, 0x68, 0x56, 0x54,
        0x59, 0x31, 0x4a, 0x65, 0x5a, 0x5e, 0x27, 0x76,
        0x76, 0x65, 0x6d, 0x2f, 0x75, 0x63, 0x67, 0x52,
        0x5e, 0x29, 0x58, 0x3d, 0x5c, 0x3f, 0x54, 0x7c,
        0x67, 0x21, 0x6e, 0x75, 0x67, 0x35, 0x77, 0x31,
        0x3d, 0x26, 0x3f, 0x60, 0x45, 0x2d, 0x2b, 0x45,
        0x5d, 0x3f, 0x55, 0x73, 0x59, 0x4c, 0x5e, 0x6c,
        0x30, 0x4a, 0x4e, 0x47, 0x55, 0x42, 0x6a, 0x4b,
        0x32, 0x3c, 0x75, 0x6e, 0x36, 0x51, 0x5f, 0x4c,
        0x68, 0x72, 0x72, 0x27, 0x3b, 0x51, 0x59, 0x7b,
        0x68, 0x7b, 0x3b, 0x54, 0x35, 0x37, 0x7c, 0x44,
        0x43, 0x36, 0x4c, 0x4f, 0x67, 0x62, 0x4e, 0x39,
        0x4b, 0x7a, 0x49, 0x36, 0x68, 0x38, 0x4c, 0x4a,
        0x64, 0x33, 0x35, 0x2f, 0x3e, 0x5c, 0x58, 0x61,
        0x23, 0x5b, 0x50, 0x6e, 0x34, 0x44, 0x60, 0x28,
        0x54, 0x41, 0x5c, 0x31, 0x53, 0x2d, 0x58, 0x58,
        0x54, 0x28, 0x77, 0x51, 0x6f, 0x64, 0x4c, 0x68,
        0x34, 0x79, 0x45, 0x66, 0x2c, 0x26, 0x77, 0x64,
        0x5f, 0x6c, 0x3b, 0x71, 0x28, 0x4d, 0x68, 0x2a,
        0x6b, 0x37, 0x6a, 0x34, 0x51, 0x27, 0x2a, 0x46,
        0x3a, 0x2e, 0x35, 0x21, 0x21, 0x79, 0x51, 0x44,
        0x58, 0x5d, 0x6f, 0x65, 0x6b, 0x76, 0x68, 0x3a,
        0x43, 0x70, 0x36, 0x41, 0x6b, 0x56, 0x64, 0x75,
        0x5b, 0x37, 0x24, 0x56, 0x7c, 0x6e, 0x6c, 0x41,
        0x3a, 0x60, 0x56, 0x38, 0x55, 0x63, 0x77, 0x4d,
        0x6e, 0x50, 0x3c, 0x3d, 0x7a, 0x44, 0x71, 0x42,
        0x4b, 0x55, 0x75, 0x72, 0x61, 0x60, 0x65, 0x6f,
        0x7a, 0x26, 0x64, 0x46, 0x67, 0x74, 0x29, 0x2a,
        0x5b, 0x62, 0x41, 0x28, 0x62, 0x30, 0x34, 0x33,
        0x40, 0x79, 0x7a, 0x38, 0x56, 0x38, 0x73, 0x22,
        0x7a, 0x7d, 0x73, 0x2a, 0x2a, 0x28, 0x2b, 0x63,
        0x27, 0x6f, 0x3d, 0x3e, 0x2c, 0x56, 0x23, 0x32,
        0x4b, 0x3b, 0x58, 0x4d, 0x72, 0x4c, 0x49, 0x6f,
        0x30, 0x76, 0x23, 0x21, 0x21, 0x3c, 0x49, 0x56,
        0x7a, 0x56, 0x79, 0x2f, 0x50, 0x7a, 0x5b, 0x21,
        0x21, 0x4a, 0x48, 0x61, 0x33, 0x52, 0x49, 0x2e,
        0x30, 0x7d, 0x2c, 0x2d, 0x67, 0x23, 0x55, 0x62,
        0x66, 0x52, 0x5a, 0x61, 0x75, 0x63, 0x3c, 0x39,
        0x69, 0x41, 0x31, 0x6b, 0x4e, 0x6f, 0x25, 0x34,
        0x74, 0x30, 0x21, 0x3a, 0x40, 0x72, 0x44, 0x40,
        0x60, 0x4c, 0x53, 0x74, 0x42, 0x64, 0x44, 0x49,
        0x76, 0x67, 0x21, 0x79, 0x36, 0x3c, 0x37, 0x70,
        0x4f, 0x58, 0x29, 0x71, 0x2a, 0x3a, 0x4d, 0x5d,
        0x67, 0x68, 0x52, 0x63, 0x23, 0x24, 0x4b, 0x21,
        0x3f, 0x68, 0x69, 0x6c, 0x66, 0x66, 0x42, 0x28,
        0x59, 0x35, 0x34, 0x6f, 0x2d, 0x6a, 0x25, 0x66,
        0x34, 0x54, 0x5d, 0x50, 0x26, 0x41, 0x22, 0x4f,
        0x34, 0x79, 0x3c, 0x50, 0x68, 0x2d, 0x5f, 0x7b,
        0x63, 0x7d, 0x58, 0x2e, 0x73, 0x46, 0x2f, 0x54,
        0x61, 0x27, 0x74, 0x45, 0x23, 0x72, 0x31, 0x7d,
        0x63, 0x4b, 0x43, 0x5e, 0x44, 0x54, 0x2c, 0x38,
        0x58, 0x24, 0x75, 0x6c, 0x50, 0x3c, 0x23, 0x5f,
        0x35, 0x57, 0x4f, 0x7b, 0x2f, 0x57, 0x29, 0x73,
        0x58, 0x2a, 0x66, 0x3e, 0x49, 0x42, 0x5a, 0x6b,
        0x75, 0x6a, 0x38, 0x3f, 0x73, 0x44, 0x42, 0x46,
        0x2d, 0x39, 0x66, 0x5b, 0x28, 0x3e, 0x63, 0x62,
        0x53, 0x75, 0x65, 0x64, 0x79, 0x32, 0x35, 0x71,
        0x22, 0x6a, 0x7b, 0x41, 0x2b, 0x26, 0x43, 0x79,
        0x58, 0x6f, 0x71, 0x25, 0x24, 0x34, 0x72, 0x5b,
        0x4a, 0x2c, 0x5c, 0x77, 0x23, 0x42, 0x27, 0x6a,
        0x67, 0x51, 0x5f, 0x3c, 0x75, 0x2c, 0x3f, 0x43,
        0x45, 0x5b, 0x48, 0x65, 0x6f, 0x6c, 0x27, 0x65,
        0x21, 0x3e, 0x33, 0x37, 0x5f, 0x2b, 0x2e, 0x24,
        0x22, 0x47, 0x4e, 0x33, 0x5b, 0x7b, 0x21, 0x3c,
        0x53, 0x69, 0x2e, 0x31, 0x3d, 0x48, 0x57, 0x3a,
        0x56, 0x48, 0x6b, 0x47, 0x5d, 0x33, 0x41, 0x6c,
        0x66, 0x4c, 0x61, 0x67, 0x32, 0x69, 0x53, 0x2c,
        0x2f, 0x3e, 0x36, 0x68, 0x37, 0x28, 0x40, 0x21,
        0x76, 0x27, 0x44, 0x26, 0x24, 0x6a, 0x30, 0x75,
        0x2a, 0x73, 0x48, 0x36, 0x52, 0x4a, 0x3b, 0x51,
        0x4e, 0x2f, 0x23, 0x36, 0x4b, 0x49, 0x33, 0x5a,
        0x70, 0x2c, 0x54, 0x5b, 0x67, 0x48, 0x53, 0x5d,
        0x21, 0x3e, 0x6b, 0x52, 0x6a, 0x3c, 0x48, 0x29,
        0x68, 0x27, 0x32, 0x75, 0x61, 0x7c, 0x51, 0x2e,
        0x7b, 0x49, 0x2f, 0x5b, 0x3d, 0x74, 0x5a, 0x28,
        0x26, 0x29, 0x2c, 0x30, 0x54, 0x74, 0x45, 0x55,
        0x4a, 0x3d, 0x39, 0x35, 0x66, 0x56, 0x28, 0x6d,
        0x6e, 0x38, 0x7b, 0x2b, 0x40, 0x31, 0x56, 0x61,
        0x74, 0x2b, 0x79, 0x5f, 0x63, 0x51, 0x53, 0x52,
        0x7d, 0x73, 0x4e, 0x2e, 0x45, 0x3b, 0x22, 0x28,
        0x6c, 0x2b, 0x47, 0x21, 0x50, 0x2a, 0x7c, 0x45,
        0x48, 0x57, 0x3e, 0x2f, 0x6d, 0x66, 0x6c, 0x51,
        0x23, 0x6c, 0x37, 0x4d, 0x4b, 0x4b, 0x66, 0x55,
        0x69, 0x2e, 0x4a, 0x69, 0x71, 0x7c, 0x71, 0x30,
        0x5c, 0x43, 0x46, 0x63, 0x5a, 0x23, 0x75, 0x40
    };
    uint32_t request6_len = sizeof(request6);

    uint8_t request7[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x5d, 0x32, 0x55, 0x71, 0x51, 0x45, 0x4e, 0x54,
        0x34, 0x21, 0x46, 0x77, 0x5e, 0x5b, 0x75, 0x62,
        0x2b, 0x5c, 0x34, 0x26, 0x72, 0x2b, 0x2c, 0x64,
        0x4b, 0x65, 0x56, 0x72, 0x31, 0x7d, 0x6a, 0x5f,
        0x70, 0x26, 0x32, 0x29, 0x7d, 0x21, 0x5b, 0x3e,
        0x5e, 0x53, 0x3d, 0x48, 0x5e, 0x2a, 0x4c, 0x37,
        0x3d, 0x59, 0x79, 0x21, 0x4f, 0x56, 0x79, 0x2a,
        0x4e, 0x28, 0x61, 0x7d, 0x2c, 0x58, 0x2f, 0x78,
        0x5c, 0x3f, 0x5c, 0x42, 0x6d, 0x2f, 0x71, 0x54,
        0x25, 0x31, 0x73, 0x38, 0x6c, 0x31, 0x5a, 0x2e,
        0x42, 0x5b, 0x2d, 0x41, 0x24, 0x4c, 0x37, 0x40,
        0x39, 0x7d, 0x2a, 0x67, 0x60, 0x6a, 0x7a, 0x62,
        0x24, 0x4e, 0x3f, 0x2e, 0x69, 0x35, 0x28, 0x65,
        0x77, 0x53, 0x23, 0x44, 0x59, 0x71, 0x31, 0x5c,
        0x40, 0x5d, 0x3a, 0x27, 0x46, 0x55, 0x30, 0x56,
        0x21, 0x74, 0x3e, 0x73, 0x41, 0x22, 0x52, 0x68,
        0x40, 0x6c, 0x37, 0x3e, 0x62, 0x5a, 0x2e, 0x21,
        0x23, 0x33, 0x27, 0x73, 0x68, 0x26, 0x60, 0x67,
        0x70, 0x58, 0x50, 0x42, 0x58, 0x27, 0x3a, 0x35,
        0x6f, 0x51, 0x62, 0x78, 0x25, 0x2c, 0x7b, 0x66,
        0x34, 0x6a, 0x5a, 0x39, 0x60, 0x70, 0x41, 0x2d,
        0x65, 0x26, 0x5a, 0x67, 0x58, 0x2d, 0x3e, 0x56,
        0x6d, 0x30, 0x4b, 0x4d, 0x5d, 0x45, 0x41, 0x3d,
        0x6e, 0x27, 0x4e, 0x5a, 0x7d, 0x2e, 0x62, 0x4d,
        0x42, 0x70, 0x31, 0x24, 0x73, 0x5c, 0x78, 0x77,
        0x50, 0x73, 0x27, 0x48, 0x3d, 0x35, 0x2c, 0x4b,
        0x40, 0x2d, 0x25, 0x77, 0x5d, 0x3d, 0x6b, 0x50,
        0x6f, 0x57, 0x73, 0x2f, 0x4f, 0x6e, 0x4c, 0x6e,
        0x56, 0x7b, 0x55, 0x3c, 0x6d, 0x60, 0x47, 0x53,
        0x56, 0x39, 0x3b, 0x51, 0x61, 0x71, 0x75, 0x73,
        0x6b, 0x70, 0x58, 0x5f, 0x2c, 0x27, 0x74, 0x49,
        0x2c, 0x2b, 0x53, 0x2d, 0x5b, 0x79, 0x43, 0x34,
        0x39, 0x5a, 0x38, 0x3e, 0x2d, 0x66, 0x70, 0x3d,
        0x49, 0x51, 0x29, 0x4d, 0x5d, 0x4c, 0x57, 0x4a,
        0x2f, 0x41, 0x69, 0x56, 0x57, 0x77, 0x49, 0x58,
        0x75, 0x28, 0x29, 0x4a, 0x6d, 0x54, 0x4f, 0x4f,
        0x3f, 0x58, 0x5f, 0x58, 0x6f, 0x39, 0x22, 0x4d,
        0x5d, 0x31, 0x75, 0x43, 0x2f, 0x7d, 0x31, 0x3d,
        0x4c, 0x4d, 0x76, 0x74, 0x4d, 0x57, 0x3b, 0x56,
        0x57, 0x48, 0x2b, 0x5d, 0x32, 0x67, 0x51, 0x6e,
        0x60, 0x39, 0x6f, 0x64, 0x38, 0x37, 0x52, 0x4b,
        0x52, 0x42, 0x32, 0x4f, 0x24, 0x53, 0x31, 0x6e,
        0x4a, 0x68, 0x2f, 0x28, 0x2e, 0x27, 0x49, 0x75,
        0x77, 0x75, 0x26, 0x47, 0x7c, 0x5d, 0x72, 0x5a,
        0x77, 0x50, 0x2e, 0x6c, 0x27, 0x68, 0x6b, 0x7b,
        0x27, 0x63, 0x21, 0x3d, 0x30, 0x2d, 0x5c, 0x67,
        0x4d, 0x41, 0x79, 0x47, 0x42, 0x50, 0x6d, 0x32,
        0x74, 0x39, 0x62, 0x4d, 0x5f, 0x65, 0x78, 0x4f,
        0x67, 0x3a, 0x60, 0x26, 0x45, 0x61, 0x7c, 0x61,
        0x63, 0x40, 0x46, 0x79, 0x52, 0x47, 0x57, 0x49,
        0x53, 0x4c, 0x48, 0x36, 0x67, 0x47, 0x5c, 0x71,
        0x50, 0x4d, 0x4f, 0x58, 0x26, 0x40, 0x6d, 0x54,
        0x55, 0x67, 0x66, 0x23, 0x70, 0x23, 0x68, 0x70,
        0x4d, 0x2c, 0x7a, 0x3d, 0x60, 0x51, 0x35, 0x64,
        0x56, 0x2f, 0x26, 0x6d, 0x72, 0x6a, 0x59, 0x34,
        0x3a, 0x73, 0x4b, 0x27, 0x33, 0x61, 0x26, 0x45,
        0x61, 0x28, 0x74, 0x22, 0x54, 0x50, 0x2e, 0x39,
        0x6a, 0x2c, 0x27, 0x59, 0x26, 0x73, 0x44, 0x71,
        0x67, 0x4c, 0x37, 0x74, 0x2c, 0x63, 0x52, 0x2a,
        0x60, 0x4f, 0x7b, 0x32, 0x39, 0x21, 0x79, 0x54,
        0x79, 0x6d, 0x28, 0x27, 0x3a, 0x6a, 0x7d, 0x40,
        0x6a, 0x4f, 0x4b, 0x46, 0x61, 0x36, 0x6a, 0x22,
        0x3f, 0x77, 0x2d, 0x6a, 0x3b, 0x73, 0x71, 0x72,
        0x3c, 0x21, 0x2e, 0x3f, 0x33, 0x25, 0x76, 0x64,
        0x64, 0x70, 0x43, 0x32, 0x44, 0x73, 0x61, 0x51,
        0x3c, 0x3b, 0x45, 0x3a, 0x68, 0x46, 0x5b, 0x6e,
        0x36, 0x47, 0x4d, 0x38, 0x26, 0x4f, 0x5c, 0x7d,
        0x73, 0x29, 0x24, 0x78, 0x44, 0x75, 0x40, 0x42,
        0x41, 0x2a, 0x73, 0x2b, 0x24, 0x38, 0x51, 0x67,
        0x36, 0x67, 0x2f, 0x70, 0x58, 0x54, 0x6e, 0x5d,
        0x3b, 0x41, 0x59, 0x76, 0x7d, 0x2d, 0x40, 0x70,
        0x29, 0x4a, 0x4a, 0x31, 0x79, 0x2c, 0x4e, 0x22,
        0x31, 0x59, 0x31, 0x3c, 0x2f, 0x21, 0x29, 0x3f,
        0x65, 0x6c, 0x38, 0x55, 0x4f, 0x27, 0x66, 0x66,
        0x34, 0x45, 0x49, 0x41, 0x56, 0x24, 0x2e, 0x40,
        0x36, 0x23, 0x5a, 0x46, 0x40, 0x23, 0x7b, 0x2d,
        0x69, 0x54, 0x6c, 0x51, 0x58, 0x73, 0x56, 0x60,
        0x5f, 0x60, 0x63, 0x5f, 0x77, 0x6a, 0x4c, 0x2c,
        0x35, 0x39, 0x60, 0x73, 0x63, 0x3e, 0x2d, 0x55,
        0x5a, 0x26, 0x4b, 0x43, 0x3b, 0x56, 0x33, 0x58,
        0x74, 0x51, 0x4f, 0x5c, 0x2a, 0x44, 0x78, 0x66,
        0x78, 0x71, 0x40, 0x29, 0x5e, 0x26, 0x57, 0x51,
        0x49, 0x30, 0x29, 0x73, 0x38, 0x56, 0x6c, 0x41,
        0x78, 0x3d, 0x61, 0x3d, 0x2c, 0x33, 0x46, 0x57,
        0x54, 0x63, 0x3e, 0x79, 0x55, 0x4a, 0x7d, 0x2e,
        0x2a, 0x3c, 0x77, 0x47, 0x35, 0x29, 0x5a, 0x6d,
        0x69, 0x48, 0x6b, 0x73, 0x7d, 0x4f, 0x5f, 0x6f,
        0x3a, 0x7a, 0x4e, 0x54, 0x59, 0x38, 0x62, 0x44,
        0x72, 0x51, 0x57, 0x6a, 0x74, 0x54, 0x4f, 0x77,
        0x6b, 0x66, 0x4a, 0x6b, 0x39, 0x29, 0x69, 0x60,
        0x71, 0x52, 0x6a, 0x32, 0x66, 0x6c, 0x25, 0x76,
        0x27, 0x7a, 0x2c, 0x38, 0x72, 0x4e, 0x5f, 0x40,
        0x26, 0x74, 0x6a, 0x5e, 0x42, 0x38, 0x78, 0x34,
        0x4f, 0x4f, 0x35, 0x27, 0x39, 0x62, 0x52, 0x61,
        0x37, 0x54, 0x47, 0x38, 0x70, 0x31, 0x7a, 0x66,
        0x69, 0x72, 0x24, 0x52, 0x2a, 0x2a, 0x78, 0x72,
        0x2b, 0x2e, 0x2a, 0x57, 0x4a, 0x21, 0x52, 0x3c,
        0x2a, 0x2f, 0x24, 0x58, 0x34, 0x3c, 0x42, 0x5c,
        0x5b, 0x78, 0x27, 0x55, 0x63, 0x58, 0x3e, 0x26,
        0x50, 0x2c, 0x72, 0x60, 0x36, 0x6c, 0x46, 0x58,
        0x63, 0x59, 0x23, 0x2a, 0x2d, 0x63, 0x6a, 0x68,
        0x69, 0x74, 0x3f, 0x49, 0x4f, 0x48, 0x4a, 0x3b,
        0x59, 0x56, 0x77, 0x43, 0x6d, 0x57, 0x28, 0x5f,
        0x39, 0x73, 0x28, 0x74, 0x3c, 0x4f, 0x43, 0x48,
        0x6a, 0x57, 0x5d, 0x41, 0x73, 0x3f, 0x41, 0x7c,
        0x65, 0x5e, 0x2d, 0x38, 0x72, 0x3a, 0x53, 0x3e,
        0x33, 0x47, 0x69, 0x6a, 0x6e, 0x78, 0x67, 0x5d,
        0x35, 0x3b, 0x3f, 0x23, 0x7c, 0x71, 0x3d, 0x7c,
        0x3a, 0x3c, 0x75, 0x6e, 0x00, 0x00, 0x00, 0x00,
        0x50, 0x6a, 0x40, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x50, 0x6a, 0x40, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x50, 0x6a, 0x40, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x50, 0x6a, 0x40, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x50, 0x80, 0x23, 0x00, 0xdf, 0xaf, 0xff, 0x33,
        0x9b, 0x78, 0x70, 0x43, 0xc5, 0x0a, 0x4d, 0x98,
        0x96, 0x02, 0x64, 0x92, 0xc1, 0xee, 0x70, 0x32
    };
    uint32_t request7_len = sizeof(request7);

    uint8_t request8[] = {
        0x05, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x18, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0x65, 0xc1, 0xef, 0x7b, 0xd6, 0xaa, 0xd6, 0x09,
        0x21, 0xf6, 0xe7, 0xd1, 0x4c, 0xdf, 0x6a, 0x2d,
        0x0a, 0xfb, 0x43, 0xea, 0xda, 0x07, 0x24, 0x84,
        0x88, 0x52, 0x9e, 0xa8, 0xa1, 0x7f, 0x4b, 0x60,
        0xec, 0x94, 0x57, 0x33, 0x06, 0x93, 0x92, 0x25,
        0xd6, 0xac, 0xdc, 0x89, 0x68, 0x5e, 0xbb, 0x32,
        0x2b, 0x17, 0x68, 0xf2, 0x06, 0xb7, 0x86, 0xac,
        0x81, 0xfe, 0x52, 0x27, 0xf5, 0x80, 0x11, 0x0d,
        0x4e, 0x2e, 0x1b, 0xa3, 0x44, 0x8a, 0x58, 0xed,
        0xf3, 0x9c, 0xe9, 0x31, 0x01, 0x72, 0xa6, 0xab,
        0xfa, 0xa8, 0x05, 0x00, 0x37, 0x60, 0x6b, 0x81,
        0xef, 0xf4, 0x96, 0x9a, 0xf7, 0x67, 0x95, 0x27,
        0x7a, 0x25, 0xef, 0x6f, 0x0e, 0xff, 0x2d, 0x15,
        0x7f, 0x23, 0x1c, 0xa7, 0x56, 0x94, 0x4a, 0x18,
        0x98, 0xc6, 0xd8, 0xd2, 0x29, 0x5b, 0x57, 0xb8,
        0x5d, 0x3a, 0x93, 0x58, 0x45, 0x77, 0x36, 0xe3,
        0xd1, 0x36, 0x87, 0xff, 0xe3, 0x94, 0x0f, 0x00,
        0xe6, 0x7c, 0x1a, 0x92, 0xc1, 0x5f, 0x40, 0xc3,
        0xa3, 0x25, 0xce, 0xd4, 0xaf, 0x39, 0xeb, 0x17,
        0xcf, 0x22, 0x43, 0xd9, 0x0c, 0xce, 0x37, 0x86,
        0x46, 0x54, 0xd6, 0xce, 0x00, 0x30, 0x36, 0xae,
        0xf9, 0xb5, 0x2b, 0x11, 0xa0, 0xfe, 0xa3, 0x4b,
        0x2e, 0x05, 0xbe, 0x54, 0xa9, 0xd8, 0xa5, 0x76,
        0x83, 0x5b, 0x63, 0x01, 0x1c, 0xd4, 0x56, 0x72,
        0xcd, 0xdc, 0x4a, 0x1d, 0x77, 0xda, 0x8a, 0x9e,
        0xba, 0xcb, 0x6c, 0xe8, 0x19, 0x5d, 0x68, 0xef,
        0x8e, 0xbc, 0x6a, 0x05, 0x53, 0x0b, 0xc7, 0xc5,
        0x96, 0x84, 0x04, 0xd9, 0xda, 0x4c, 0x42, 0x31,
        0xd9, 0xbd, 0x99, 0x06, 0xf7, 0xa3, 0x0a, 0x19,
        0x49, 0x07, 0x77, 0xf0, 0xdb, 0x7c, 0x43, 0xfa,
        0xb2, 0xad, 0xb0, 0xfa, 0x87, 0x52, 0xba, 0xc9,
        0x94, 0x61, 0xdc, 0xcf, 0x16, 0xac, 0x0f, 0x4a,
        0xa3, 0x6b, 0x5b, 0x6e, 0x27, 0x86, 0x1f, 0xfe,
        0x4d, 0x28, 0x3a, 0xa5, 0x10, 0x54, 0x6d, 0xed,
        0x53, 0xf9, 0x73, 0xc6, 0x6e, 0xa8, 0xc0, 0x97,
        0xcf, 0x56, 0x3b, 0x61, 0xdf, 0xab, 0x83, 0x18,
        0xe8, 0x09, 0xee, 0x6a, 0xb7, 0xf5, 0xc9, 0x62,
        0x55, 0x2d, 0xc7, 0x0c, 0x0d, 0xa0, 0x22, 0xd8,
        0xd4, 0xd6, 0xb2, 0x12, 0x21, 0xd7, 0x73, 0x3e,
        0x41, 0xb0, 0x5c, 0xd4, 0xcf, 0x98, 0xf3, 0x70,
        0xe6, 0x08, 0xe6, 0x2a, 0x4f, 0x24, 0x85, 0xe8,
        0x74, 0xa8, 0x41, 0x5f, 0x0e, 0xfd, 0xf1, 0xf3,
        0xbe, 0x9b, 0x14, 0xfd, 0xc0, 0x73, 0x11, 0xff,
        0xa5, 0x5b, 0x06, 0x34, 0xc3, 0x6c, 0x28, 0x42,
        0x07, 0xfe, 0x8a, 0xa5, 0xbe, 0x72, 0x7a, 0xf7,
        0xfa, 0x25, 0xec, 0x35, 0x5e, 0x98, 0x71, 0x50,
        0x60, 0x35, 0x76, 0x53, 0x40, 0x1a, 0x34, 0xa5,
        0x99, 0x09, 0xa2, 0xc6, 0xca, 0xa5, 0xce, 0x08,
        0x50, 0x45, 0xab, 0x8d, 0xfb, 0xe3, 0xb8, 0xe4,
        0x8a, 0x61, 0x48, 0x14, 0x6e, 0xf7, 0x58, 0x71,
        0xe5, 0x2e, 0xbc, 0x12, 0xd1, 0x25, 0xe9, 0x65,
        0x7a, 0xa1, 0x27, 0xbe, 0x3b, 0x8b, 0xe8, 0xe7,
        0xbc, 0xe1, 0x05, 0xe7, 0x92, 0xeb, 0xb9, 0xdf,
        0x5d, 0x53, 0x74, 0xc0, 0x63, 0x97, 0x80, 0xb8,
        0x3c, 0xae, 0xf3, 0xf2, 0x09, 0x12, 0x81, 0x6c,
        0x69, 0x10, 0x6f, 0xf6, 0xbe, 0x03, 0x7b, 0x88,
        0xcf, 0x26, 0x6b, 0x51, 0x06, 0x23, 0x68, 0x03,
        0xa1, 0xb7, 0xd3, 0x0c, 0xca, 0xbf, 0x29, 0x01,
        0xa9, 0x61, 0x34, 0x75, 0x98, 0x1e, 0x05, 0x59,
        0xb3, 0x46, 0x44, 0xff, 0x2b, 0x98, 0x04, 0x88,
        0x89, 0xfd, 0x7f, 0xd5, 0x19, 0x8a, 0xa6, 0xf3,
        0xd9, 0x44, 0xd5, 0xf9, 0x3a, 0x3c, 0xec, 0xd9,
        0x9b, 0x8c, 0x93, 0x93, 0x2b, 0x44, 0x86, 0x8b,
        0x80, 0x83, 0x23, 0x00, 0xdf, 0xaf, 0xff, 0x33,
        0x9b, 0x78, 0x70, 0x43, 0xf1, 0x55, 0x87, 0xb1,
        0xa1, 0xb3, 0x8e, 0x79, 0x02, 0x70, 0x82, 0x6c,
        0x0b, 0xc1, 0xef, 0x96, 0xf1, 0xef, 0xdd, 0xa2,
        0x69, 0x86, 0xc7, 0x85, 0x09, 0x7e, 0xf0, 0x2f,
        0x8e, 0xa0, 0x5f, 0xea, 0x39, 0x2e, 0x24, 0xf0,
        0x82, 0x30, 0x26, 0xa8, 0xa1, 0x4f, 0xc6, 0x5c,
        0xec, 0x94, 0x87, 0x52, 0x9b, 0x93, 0x92, 0xf3,
        0xa3, 0x1b, 0xc7, 0x8f, 0x9e, 0xb3, 0xbb, 0x32,
        0x2b, 0x17, 0x54, 0xf2, 0x06, 0x0c, 0x86, 0x92,
        0x0f, 0xb8, 0xe0, 0x27, 0x50, 0xaa, 0xeb, 0xf5,
        0x4e, 0x2b, 0x1b, 0xb2, 0x44, 0xe6, 0x58, 0x02,
        0xd7, 0x65, 0xdc, 0x31, 0x01, 0xec, 0xa6, 0xab,
        0xfa, 0xa8, 0x05, 0x00, 0x37, 0x60, 0x4f, 0xa1,
        0x3c, 0x4f, 0x7a, 0x9a, 0x10, 0x67, 0x95, 0xc2,
        0x5b, 0x25, 0xef, 0x76, 0x0e, 0xff, 0x2d, 0x15,
        0x7f, 0x23, 0x1c, 0x77, 0x56, 0x94, 0x4a, 0x18,
        0x98, 0xc6, 0xd8, 0xd2, 0x29, 0x44, 0x57, 0xb8,
        0x40, 0x3a, 0x93, 0x58, 0x45, 0x77, 0x36, 0x36,
        0x07, 0x35, 0x2a, 0xff, 0x00, 0x94, 0x5c, 0x80,
        0xe6, 0x7c, 0x1a, 0x92, 0xc1, 0x5f, 0x40, 0xc3,
        0xbc, 0xf8, 0xce, 0x05, 0x77, 0x39, 0x40, 0x17,
        0xcf, 0x63, 0x43, 0x77, 0x27, 0xce, 0x37, 0x86,
        0x46, 0x54, 0xd6, 0xce, 0x00, 0x30, 0x36, 0xae,
        0x9f, 0x24, 0x2b, 0x5a, 0xa0, 0xfe, 0xa3, 0x4b,
        0x2e, 0x7e, 0xf7, 0x54, 0xa9, 0xd8, 0xa5, 0x76,
        0x83, 0x7b, 0x63, 0x01, 0x1c, 0xd4, 0x56, 0x17,
        0x02, 0xdc, 0x4a, 0x89, 0x77, 0xda, 0x8f, 0x9e,
        0xba, 0xcb, 0x37, 0xe8, 0x19, 0x5d, 0x68, 0x38,
        0x8e, 0xbc, 0x6a, 0x05, 0x53, 0x0b, 0xc7, 0xc5,
        0x96, 0x84, 0x5a, 0xd9, 0x6d, 0x4c, 0x42, 0x31,
        0xd9, 0xf2, 0x99, 0x06, 0xf7, 0x0c, 0x99, 0xbe,
        0x49, 0x07, 0x77, 0xf0, 0x8b, 0x7c, 0x43, 0xfa,
        0xb2, 0xad, 0xb0, 0xfa, 0x87, 0x52, 0xba, 0xc9,
        0x94, 0x61, 0xdc, 0xcf, 0x16, 0xac, 0x0f, 0x4a,
        0xa3, 0x6b, 0x5b, 0x6e, 0x27, 0x86, 0x1f, 0xfe,
        0x4d, 0x28, 0x3a, 0xa5, 0x10, 0x98, 0x6d, 0xed,
        0x53, 0xf9, 0x73, 0xc6, 0xa5, 0xa8, 0xf7, 0x66,
        0xcf, 0x56, 0x3b, 0x61, 0xdf, 0xab, 0x83, 0x18,
        0xe8, 0x09, 0xee, 0x6a, 0xb7, 0xf5, 0xc9, 0x62,
        0x55, 0x2d, 0xc7, 0x0c, 0x0d, 0xa0, 0x22, 0xd8,
        0xd4, 0xd6, 0xb2, 0x12, 0x21, 0xd7, 0x73, 0x3e,
        0x41, 0xb0, 0x5c, 0xd4, 0xcf, 0x98, 0xf3, 0x70,
        0xe6, 0x08, 0xe6, 0x2a, 0x4f, 0x92, 0x85, 0xe8,
        0x74, 0xa8, 0x41, 0x5f, 0x0e, 0xfd, 0xf1, 0xf3,
        0xbe, 0x9b, 0x14, 0xfd, 0xc0, 0x73, 0x11, 0xff,
        0xa5, 0x5b, 0x06, 0x34, 0xc3, 0x5d, 0x28, 0x42,
        0x34, 0xfe, 0x8a, 0xa5, 0xbe, 0x72, 0x7a, 0xf7,
        0xfa, 0x25, 0x2b, 0x35, 0x5e, 0x98, 0x71, 0x50,
        0x2c, 0x35, 0x76, 0x53, 0x4e, 0x1a, 0x34, 0xa5,
        0x99, 0x09, 0xa2, 0xc6, 0xca, 0xa5, 0xce, 0x08,
        0x50, 0x45, 0xab, 0x8d, 0xfb, 0xe3, 0xb8, 0xe4,
        0x8a, 0x61, 0x48, 0x14, 0x6e, 0xf7, 0x58, 0x71,
        0xe5, 0x2e, 0xbc, 0x12, 0xd1, 0x25, 0xe9, 0x65,
        0x7a, 0xa1, 0x27, 0xbe, 0x3b, 0x8b, 0xe8, 0xe7,
        0xbc, 0x77, 0x05, 0xe7, 0x92, 0xeb, 0xb9, 0xdf,
        0x5d, 0x53, 0x74, 0xc0, 0x63, 0x97, 0x80, 0xb8,
        0x3c, 0xae, 0xf3, 0xf2, 0x09, 0x12, 0x81, 0x6c,
        0x69, 0x10, 0x6f, 0xf6, 0xbe, 0x03, 0x7b, 0x88,
        0xcf, 0x26, 0x6b, 0x51, 0x06, 0x23, 0x68, 0x03,
        0xa1, 0xb7, 0xd3, 0x0c, 0xca, 0xbf, 0x29, 0x01,
        0xa9, 0x61, 0x34, 0x75, 0x98, 0x1e, 0x6f, 0x59,
        0xb3, 0x46, 0x44, 0xff, 0x2b, 0x98, 0x04, 0x88,
        0x89, 0xfd, 0x1c, 0xd5, 0x19, 0x8a, 0xa6, 0xf3,
        0xd9, 0x44, 0xd5, 0xf9, 0x79, 0x26, 0x46, 0xf7
    };
    uint32_t request8_len = sizeof(request8);

    uint8_t request9[] = {
        0x05, 0x00, 0x00, 0x02, 0x10, 0x00, 0x00, 0x00,
        0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00,
        0xbf, 0xa1, 0x12, 0x73, 0x23, 0x44, 0x86, 0x8b,
        0x50, 0x6a, 0x40, 0x00
    };
    uint32_t request9_len = sizeof(request9);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    /* bind */
    int r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER|STREAM_START,
                          bind, bind_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh == 0) &&
                (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) );
    if (result == 0)
        goto end;

    /* bind_ack */
    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOCLIENT,
                          bind_ack, bind_ack_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh == 0) &&
                (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) );
    if (result == 0)
        goto end;

    /* request1 */
    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request1, request1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 1024 &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh == 1) &&
                (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) );
    if (result == 0)
        goto end;
    dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh = 0;

    /* request2 */
    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request2, request2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 2048 &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh == 1) &&
                (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) );
    if (result == 0)
        goto end;
    dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh = 0;

    /* request3 */
    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request3, request3_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL  &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 3072 &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh == 1) &&
                (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) );
    if (result == 0)
        goto end;
    dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh = 0;

    /* request4 */
    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request4, request4_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 4096 &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh == 1) &&
                (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) );
    if (result == 0)
        goto end;
    dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh = 0;

    /* request5 */
    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request5, request5_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 5120) &&
                dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL);
    if (result == 0)
        goto end;

    /* request6 */
    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request6, request6_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 6144 &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh == 1) &&
                (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) );
    if (result == 0)
        goto end;
    dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh = 0;

    /* request7 */
    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request7, request7_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 7168 &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh == 1) &&
                (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) );
    if (result == 0)
        goto end;
    dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh = 0;

    /* request8 */
    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request8, request8_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 8192 &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh == 1) &&
                (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) );
    if (result == 0)
        goto end;
    dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh = 0;

    /* request9 */
    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request9, request9_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 8204 &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh == 1) &&
                (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) );
    if (result == 0)
        goto end;
    dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh = 0;

    /* request1 again */
    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request1, request1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= ( (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 1024 &&
                 dcerpc_state->dcerpc.dcerpcrequest.stub_data_fresh == 1) &&
                (dcerpc_state->dcerpc.dcerpcresponse.stub_data_buffer == NULL &&
                 dcerpc_state->dcerpc.dcerpcresponse.stub_data_fresh == 0) );
    if (result == 0)
        goto end;

end:
    StreamTcpFreeConfig(TRUE);
    return result;
#endif
    return 1;
}

/**
 * \test General test.
 */
int DCERPCParserTest05(void) {
    int result = 1;
    Flow f;
    int r = 0;
    uint8_t bind1[] = {
        0x05, 0x00, 0x0b, 0x01, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0xb8, 0x4a, 0x9f, 0x4d, 0x1c, 0x7d, 0xcf, 0x11,
        0x86, 0x1e, 0x00, 0x20, 0xaf, 0x6e, 0x7c, 0x57,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind1_len = sizeof(bind1);

    uint8_t bind2[] = {
        0x05, 0x00, 0x0b, 0x02, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
        0xb8, 0x4a, 0x9f, 0x4d, 0x1c, 0x7d, 0xcf, 0x11,
        0x86, 0x1e, 0x00, 0x20, 0xaf, 0x6e, 0x7c, 0x67,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind2_len = sizeof(bind2);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER | STREAM_START,
                      bind1, bind1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    DCERPCUuidEntry *item = NULL;
    int m = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.uuid_list, next) {
        printf("%d ", m);
        printUUID("BIND",item);
        m++;
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      bind2, bind2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    item = NULL;
    m = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.uuid_list, next) {
        printf("%d ", m);
        printUUID("BIND",item);
        m++;
    }

    /* we will need this test later for fragged bind pdus.  keep it */
    result = 1;

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented bind PDU(one PDU which is frag'ed)
 */
int DCERPCParserTest06(void) {
    int result = 1;
    Flow f;
    int r = 0;
    uint8_t bind1[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0xdc, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0xc7, 0x70, 0x0d, 0x3e, 0x71, 0x37, 0x39, 0x0d,
        0x3a, 0x4f, 0xd3, 0xdc, 0xca, 0x49, 0xe8, 0xa3,
        0x05, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x01, 0x00, 0x84, 0xb6, 0x55, 0x75,
        0xdb, 0x9e, 0xba, 0x54, 0x56, 0xd3, 0x45, 0x10,
        0xb7, 0x7a, 0x2a, 0xe2, 0x04, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x6e, 0x39, 0x21, 0x24, 0x70, 0x6f, 0x41, 0x57,
        0x54, 0x70, 0xb8, 0xc3, 0x5e, 0x89, 0x3b, 0x43,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x01, 0x00, 0x39, 0x6a, 0x86, 0x5d,
        0x24, 0x0f, 0xd2, 0xf7, 0xb6, 0xce, 0x95, 0x9c,
        0x54, 0x1d, 0x3a, 0xdb, 0x02, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0x12, 0xa5, 0xdd, 0xc5, 0x55, 0xce, 0xc3, 0x46,
        0xbd, 0xa0, 0x94, 0x39, 0x3c, 0x0d, 0x9b, 0x5b,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x01, 0x00, 0x87, 0x1c, 0x8b, 0x6e,
        0x11, 0xa8, 0x67, 0x98, 0xd4, 0x5d, 0xf6, 0x8a,
        0x2f, 0x33, 0x24, 0x7b, 0x05, 0x00, 0x03, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x06, 0x00, 0x01, 0x00,
        0x9b, 0x82, 0x13, 0xd1, 0x28, 0xe0, 0x63, 0xf3,
        0x62, 0xee, 0x76, 0x73, 0xf9, 0xac, 0x3d, 0x2e,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x01, 0x00, 0xa9, 0xd4, 0x73, 0xf2,
        0xed, 0xad, 0xe8, 0x82, 0xf8, 0xcf, 0x9d, 0x9f,
        0x66, 0xe6, 0x43, 0x37, 0x02, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x01, 0x00,
        0x06, 0x2b, 0x85, 0x38, 0x4f, 0x73, 0x96, 0xb1,
        0x73, 0xe1, 0x59, 0xbe, 0x9d, 0xe2, 0x6c, 0x07,
        0x05, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60};
    uint32_t bind1_len = sizeof(bind1);

    uint8_t bind2[] = {
        0x02, 0x00, 0x00, 0x00, 0x09, 0x00, 0x01, 0x00,
        0xbf, 0xfa, 0xbb, 0xa4, 0x9e, 0x5c, 0x80, 0x61,
        0xb5, 0x8b, 0x79, 0x69, 0xa6, 0x32, 0x88, 0x77,
        0x01, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0a, 0x00, 0x01, 0x00, 0x39, 0xa8, 0x2c, 0x39,
        0x73, 0x50, 0x06, 0x8d, 0xf2, 0x37, 0x1e, 0x1e,
        0xa8, 0x8f, 0x46, 0x98, 0x02, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x01, 0x00,
        0x91, 0x13, 0xd0, 0xa7, 0xef, 0xc4, 0xa7, 0x96,
        0x0c, 0x4a, 0x0d, 0x29, 0x80, 0xd3, 0xfe, 0xbf,
        0x00, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x01, 0x00, 0xcc, 0x2b, 0x55, 0x1d,
        0xd4, 0xa4, 0x0d, 0xfb, 0xcb, 0x6f, 0x86, 0x36,
        0xa6, 0x57, 0xc3, 0x21, 0x02, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x01, 0x00,
        0x43, 0x7b, 0x07, 0xee, 0x85, 0xa8, 0xb9, 0x3a,
        0x0f, 0xf9, 0x83, 0x70, 0xe6, 0x0b, 0x4f, 0x33,
        0x02, 0x00, 0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0e, 0x00, 0x01, 0x00, 0x9c, 0x6a, 0x15, 0x8c,
        0xd6, 0x9c, 0xa6, 0xc3, 0xb2, 0x9e, 0x62, 0x9f,
        0x3d, 0x8e, 0x47, 0x73, 0x02, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x01, 0x00,
        0xc8, 0x4f, 0x32, 0x4b, 0x70, 0x16, 0xd3, 0x01,
        0x12, 0x78, 0x5a, 0x47, 0xbf, 0x6e, 0xe1, 0x88,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind2_len = sizeof(bind2);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER|STREAM_START,
                      bind1, bind1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 420);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.ctxbytesprocessed == 40);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.numctxitems == 16);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.numctxitemsleft == 8);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      bind2, bind2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.ctxbytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.numctxitems == 16);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.numctxitemsleft == 0);

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented bind PDU(one PDU which is frag'ed).
 */
int DCERPCParserTest07(void) {
    int result = 1;
    Flow f;
    int r = 0;
    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x2C, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x09, 0x0A, 0x0B, 0x0C
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x0D, 0x0E
    };
    uint32_t request2_len = sizeof(request2);

    uint8_t request3[] = {
        0x0F, 0x10, 0x11, 0x12, 0x13, 0x14
    };
    uint32_t request3_len = sizeof(request3);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER|STREAM_START,
                      request1, request1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 36);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len ==  12);
    result &= (dcerpc_state->dcerpc.pdu_fragged = 1);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request2, request2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 38);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 14);
    result &= (dcerpc_state->dcerpc.pdu_fragged = 1);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request3, request3_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 20);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 0);

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented bind PDU(one PDU which is frag'ed).
 */
int DCERPCParserTest08(void) {
    int result = 1;
    Flow f;
    int r = 0;
    uint8_t request[] = {
        0x05, 0x02, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x2C, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x09, 0x0A, 0x0B, 0x0C,
    };
    uint32_t request_len = sizeof(request);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER | STREAM_START,
                      request, request_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer == NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 0);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 0);

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented bind PDU(one PDU which is frag'ed).
 */
int DCERPCParserTest09(void) {
    int result = 1;
    Flow f;
    int r = 0;
    uint8_t request[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x2C, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x09, 0x0A, 0x0B, 0x0C,
    };
    uint32_t request_len = sizeof(request);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER | STREAM_START,
                      request, request_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 36);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len ==  12);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 1);

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented PDU.
 */
int DCERPCParserTest10(void) {
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t fault[] = {
        0x05, 0x00, 0x03, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0xf7, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };
    uint32_t fault_len = sizeof(fault);

    uint8_t request1[] = {
        0x05, 0x00
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0x24, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x02,
        0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A,
        0x0B, 0x0C
    };
    uint32_t request2_len = sizeof(request2);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER|STREAM_START,
                      fault, fault_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request1, request1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 2);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer == NULL);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 1);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request2, request2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len == 12);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 0);

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented PDU.
 */
int DCERPCParserTest11(void) {
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x09, 0x0A, 0x0B, 0x0C
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x05, 0x00
    };
    uint32_t request2_len = sizeof(request2);

    uint8_t request3[] = {
        0x00, 0x03, 0x10, 0x00, 0x00, 0x00, 0x26, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x02,
        0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A,
        0x0B, 0x0C, 0xFF, 0xFF
    };
    uint32_t request3_len = sizeof(request3);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request1, request1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len ==  12);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 0);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request2, request2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 2);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 1);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request3, request3_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len ==  14);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 0);

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented PDU.
 */
int DCERPCParserTest12(void) {
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bind_ack1[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x48, 0x1a, 0x00, 0x00,
    };
    uint32_t bind_ack1_len = sizeof(bind_ack1);

    uint8_t bind_ack2[] = {
        0x0c, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x6c, 0x73, 0x61, 0x73, 0x73, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack2_len = sizeof(bind_ack2);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      bind_ack1, bind_ack1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 24);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 1);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOCLIENT,
                      bind_ack2, bind_ack2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 0);

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test Check if the parser accepts bind pdus that have context ids starting
 *       from a non-zero value.
 */
int DCERPCParserTest13(void) {
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x7f, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
        0xa0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_len = sizeof(bind);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      bind, bind_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 0);
    result &= (dcerpc_state->dcerpc.dcerpcbindbindack.numctxitems == 1);
    if (result == 0)
        goto end;

    result = 0;
    uint8_t ctx_uuid_from_pcap[16] = {
        0x00, 0x00, 0x01, 0xa0, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46};
    DCERPCUuidEntry *item = NULL;
    int internal_id = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (ctx_uuid_from_pcap[i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        result = 1;
        result &= (item->internal_id == internal_id++);
    }

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test Check for another endless loop with bind pdus.
 */
int DCERPCParserTest14(void) {
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x4A, 0x00, 0x00, 0x00, 0x7f, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
        0xa0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
        0x01, 0x02, 0x03, 0x04, 0xFF /* ka boom - endless loop */
    };
    uint32_t bind_len = sizeof(bind);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      bind, bind_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test Check for another endless loop for bind_ack pdus.
 */
int DCERPCParserTest15(void) {
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x3e, 0x00, 0x00, 0x00, 0x7f, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0xfd, 0x04, 0x01, 0x00,
        0x04, 0x00, 0x31, 0x33, 0x35, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x01, 0x02, 0x03, 0x04, 0xFF
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOCLIENT,
                      bind_ack, bind_ack_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test Check for correct internal ids for bind_acks.
 */
int DCERPCParserTest16(void) {
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bind1[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x58, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x50, 0x08, 0x43, 0x95, 0x43, 0x5a, 0x8b, 0xb2,
        0xf4, 0xc5, 0xb9, 0xee, 0x67, 0x55, 0x7c, 0x19,
        0x00, 0x00, 0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x01, 0x00, 0xda, 0xc2, 0xbc, 0x9b,
        0x35, 0x2e, 0xd4, 0xc9, 0x1f, 0x85, 0x01, 0xe6,
        0x4e, 0x5a, 0x5e, 0xd4, 0x04, 0x00, 0x03, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0xb2, 0x97, 0xcc, 0x14, 0x6f, 0x70, 0x0d, 0xa5,
        0x33, 0xd7, 0xf4, 0xe3, 0x8e, 0xb2, 0x2a, 0x1e,
        0x05, 0x00, 0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x01, 0x00, 0x96, 0x4e, 0xa6, 0xf6,
        0xb2, 0x4b, 0xae, 0xb3, 0x21, 0xf4, 0x97, 0x7c,
        0xcd, 0xa7, 0x08, 0xb0, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0xbc, 0xc0, 0xf7, 0x71, 0x3f, 0x71, 0x54, 0x44,
        0x22, 0xa8, 0x55, 0x0f, 0x98, 0x83, 0x1f, 0xfe,
        0x04, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x01, 0x00, 0xbe, 0x52, 0xf2, 0x58,
        0x4a, 0xc3, 0xb5, 0xd0, 0xba, 0xac, 0xda, 0xf0,
        0x12, 0x99, 0x38, 0x6e, 0x04, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x06, 0x00, 0x01, 0x00,
        0xdb, 0xfa, 0x73, 0x01, 0xb3, 0x81, 0x01, 0xd4,
        0x7f, 0xa0, 0x36, 0xb1, 0x97, 0xae, 0x29, 0x7f,
        0x01, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x01, 0x00, 0x89, 0xbe, 0x41, 0x1d,
        0x38, 0x75, 0xf5, 0xb5, 0xad, 0x27, 0x73, 0xf1,
        0xb0, 0x7a, 0x28, 0x82, 0x05, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x01, 0x00,
        0xf6, 0x87, 0x09, 0x93, 0xb8, 0xa8, 0x20, 0xc4,
        0xb8, 0x63, 0xe6, 0x95, 0xed, 0x59, 0xee, 0x3f,
        0x05, 0x00, 0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x09, 0x00, 0x01, 0x00, 0x92, 0x77, 0x92, 0x68,
        0x3e, 0xa4, 0xbc, 0x3f, 0x44, 0x33, 0x0e, 0xb8,
        0x33, 0x0a, 0x2f, 0xdf, 0x01, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x01, 0x00,
        0xa1, 0x03, 0xd2, 0xa9, 0xd2, 0x16, 0xc9, 0x89,
        0x67, 0x18, 0x3e, 0xb1, 0xee, 0x6b, 0xf9, 0x18,
        0x02, 0x00, 0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x01, 0x00, 0x2f, 0x09, 0x5e, 0x74,
        0xec, 0xa0, 0xbb, 0xc1, 0x60, 0x18, 0xf1, 0x93,
        0x04, 0x17, 0x11, 0xf9, 0x01, 0x00, 0x03, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x01, 0x00,
        0xc8, 0x4f, 0x32, 0x4b, 0x70, 0x16, 0xd3, 0x01,
        0x12, 0x78, 0x5a, 0x47, 0xbf, 0x6e, 0xe1, 0x88,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind1_len = sizeof(bind1);

    uint8_t bind_ack1[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x64, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0xc1, 0x2b, 0x00, 0x00,
        0x0e, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x62, 0x72, 0x6f, 0x77, 0x73, 0x65, 0x72, 0x00,
        0x0d, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack1_len = sizeof(bind_ack1);

    uint8_t bind2[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0xdc, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0xc7, 0x70, 0x0d, 0x3e, 0x71, 0x37, 0x39, 0x0d,
        0x3a, 0x4f, 0xd3, 0xdc, 0xca, 0x49, 0xe8, 0xa3,
        0x05, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x01, 0x00, 0x84, 0xb6, 0x55, 0x75,
        0xdb, 0x9e, 0xba, 0x54, 0x56, 0xd3, 0x45, 0x10,
        0xb7, 0x7a, 0x2a, 0xe2, 0x04, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x6e, 0x39, 0x21, 0x24, 0x70, 0x6f, 0x41, 0x57,
        0x54, 0x70, 0xb8, 0xc3, 0x5e, 0x89, 0x3b, 0x43,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x01, 0x00, 0x39, 0x6a, 0x86, 0x5d,
        0x24, 0x0f, 0xd2, 0xf7, 0xb6, 0xce, 0x95, 0x9c,
        0x54, 0x1d, 0x3a, 0xdb, 0x02, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0x12, 0xa5, 0xdd, 0xc5, 0x55, 0xce, 0xc3, 0x46,
        0xbd, 0xa0, 0x94, 0x39, 0x3c, 0x0d, 0x9b, 0x5b,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x01, 0x00, 0x87, 0x1c, 0x8b, 0x6e,
        0x11, 0xa8, 0x67, 0x98, 0xd4, 0x5d, 0xf6, 0x8a,
        0x2f, 0x33, 0x24, 0x7b, 0x05, 0x00, 0x03, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x06, 0x00, 0x01, 0x00,
        0x9b, 0x82, 0x13, 0xd1, 0x28, 0xe0, 0x63, 0xf3,
        0x62, 0xee, 0x76, 0x73, 0xf9, 0xac, 0x3d, 0x2e,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x01, 0x00, 0xa9, 0xd4, 0x73, 0xf2,
        0xed, 0xad, 0xe8, 0x82, 0xf8, 0xcf, 0x9d, 0x9f,
        0x66, 0xe6, 0x43, 0x37, 0x02, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x01, 0x00,
        0x06, 0x2b, 0x85, 0x38, 0x4f, 0x73, 0x96, 0xb1,
        0x73, 0xe1, 0x59, 0xbe, 0x9d, 0xe2, 0x6c, 0x07,
        0x05, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x09, 0x00, 0x01, 0x00, 0xbf, 0xfa, 0xbb, 0xa4,
        0x9e, 0x5c, 0x80, 0x61, 0xb5, 0x8b, 0x79, 0x69,
        0xa6, 0x32, 0x88, 0x77, 0x01, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x01, 0x00,
        0x39, 0xa8, 0x2c, 0x39, 0x73, 0x50, 0x06, 0x8d,
        0xf2, 0x37, 0x1e, 0x1e, 0xa8, 0x8f, 0x46, 0x98,
        0x02, 0x00, 0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x01, 0x00, 0x91, 0x13, 0xd0, 0xa7,
        0xef, 0xc4, 0xa7, 0x96, 0x0c, 0x4a, 0x0d, 0x29,
        0x80, 0xd3, 0xfe, 0xbf, 0x00, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x01, 0x00,
        0xcc, 0x2b, 0x55, 0x1d, 0xd4, 0xa4, 0x0d, 0xfb,
        0xcb, 0x6f, 0x86, 0x36, 0xa6, 0x57, 0xc3, 0x21,
        0x02, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0d, 0x00, 0x01, 0x00, 0x43, 0x7b, 0x07, 0xee,
        0x85, 0xa8, 0xb9, 0x3a, 0x0f, 0xf9, 0x83, 0x70,
        0xe6, 0x0b, 0x4f, 0x33, 0x02, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x01, 0x00,
        0x9c, 0x6a, 0x15, 0x8c, 0xd6, 0x9c, 0xa6, 0xc3,
        0xb2, 0x9e, 0x62, 0x9f, 0x3d, 0x8e, 0x47, 0x73,
        0x02, 0x00, 0x02, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0f, 0x00, 0x01, 0x00, 0xc8, 0x4f, 0x32, 0x4b,
        0x70, 0x16, 0xd3, 0x01, 0x12, 0x78, 0x5a, 0x47,
        0xbf, 0x6e, 0xe1, 0x88, 0x03, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind2_len = sizeof(bind2);

    uint8_t bind_ack2[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0xac, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0xc2, 0x2b, 0x00, 0x00,
        0x0e, 0x00, 0x5c, 0x50, 0x49, 0x50, 0x45, 0x5c,
        0x62, 0x72, 0x6f, 0x77, 0x73, 0x65, 0x72, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack2_len = sizeof(bind_ack2);

    uint8_t bind3[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x2c, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0xa4, 0x7f, 0x8e, 0xc6, 0xef, 0x56, 0x9b, 0x63,
        0x92, 0xfa, 0x08, 0xb3, 0x35, 0xe2, 0xa5, 0x81,
        0x00, 0x00, 0x03, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x01, 0x00, 0x9f, 0xfc, 0x78, 0xd2,
        0x5f, 0x16, 0x0b, 0xbc, 0xc6, 0xdb, 0x5d, 0xef,
        0xde, 0x54, 0xa2, 0x6f, 0x04, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x78, 0xb8, 0x96, 0xc7, 0x2f, 0xda, 0x11, 0x6b,
        0xd1, 0x28, 0x68, 0xe1, 0xd6, 0x71, 0xac, 0x9d,
        0x03, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x03, 0x00, 0x01, 0x00, 0xcf, 0xf4, 0xd7, 0x37,
        0x03, 0xda, 0xcc, 0xe3, 0x3e, 0x34, 0x7f, 0x67,
        0x99, 0x91, 0x41, 0x3d, 0x01, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x01, 0x00,
        0x48, 0xeb, 0x32, 0xf0, 0x27, 0xd5, 0x9d, 0xd0,
        0x1e, 0xc6, 0x48, 0x46, 0x97, 0xe9, 0xdb, 0x09,
        0x05, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x05, 0x00, 0x01, 0x00, 0x82, 0xec, 0x0d, 0x08,
        0xf2, 0x8f, 0x22, 0x57, 0x42, 0x9b, 0xce, 0xa8,
        0x74, 0x16, 0xc6, 0xec, 0x00, 0x00, 0x01, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x06, 0x00, 0x01, 0x00,
        0x2e, 0x00, 0x70, 0x44, 0xee, 0xc9, 0x30, 0x6b,
        0xf4, 0x34, 0x1e, 0x3d, 0x35, 0x0f, 0xf7, 0xf7,
        0x00, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x01, 0x00, 0x59, 0x04, 0x39, 0x3f,
        0x59, 0x87, 0x14, 0x0e, 0x76, 0x8d, 0x17, 0xc2,
        0x47, 0xfa, 0x67, 0x7f, 0x04, 0x00, 0x02, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x01, 0x00,
        0x30, 0xd6, 0xed, 0x2e, 0x57, 0xfa, 0xf4, 0x72,
        0x6c, 0x10, 0x0d, 0xe5, 0x51, 0x7f, 0xd0, 0x39,
        0x02, 0x00, 0x01, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x09, 0x00, 0x01, 0x00, 0xea, 0x8b, 0x84, 0x4d,
        0x44, 0x43, 0xc1, 0x94, 0x75, 0xe2, 0x81, 0x48,
        0xd8, 0x77, 0xd9, 0xce, 0x05, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x01, 0x00,
        0x89, 0x4f, 0xe7, 0x95, 0xa3, 0xc1, 0x62, 0x36,
        0x26, 0x9e, 0x67, 0xdb, 0x2c, 0x52, 0x89, 0xd3,
        0x01, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00,
        0x0b, 0x00, 0x01, 0x00, 0x78, 0x56, 0x34, 0x12,
        0x34, 0x12, 0xcd, 0xab, 0xef, 0x00, 0x01, 0x23,
        0x45, 0x67, 0x89, 0xab, 0x01, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind3_len = sizeof(bind3);

    uint8_t bind_ack3[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x4c, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x1a, 0x33, 0x00, 0x00,
        0x0e, 0x00, 0x5c, 0x70, 0x69, 0x70, 0x65, 0x5c,
        0x73, 0x70, 0x6f, 0x6f, 0x6c, 0x73, 0x73, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack3_len = sizeof(bind_ack3);

    TcpSession ssn;
    DCERPCUuidEntry *item = NULL;
    int count = 0;

    uint8_t accepted_uuids[3][16] = {
        {0x4b, 0x32, 0x4f, 0xc8, 0x16, 0x70, 0x01, 0xd3,
         0x12, 0x78, 0x5a, 0x47, 0xbf, 0x6e, 0xe1, 0x88},
        {0x4b, 0x32, 0x4f, 0xc8, 0x16, 0x70, 0x01, 0xd3,
         0x12, 0x78, 0x5a, 0x47, 0xbf, 0x6e, 0xe1, 0x88},
        {0x12, 0x34, 0x56, 0x78, 0x12, 0x34, 0xab, 0xcd,
         0xef, 0x00, 0x01, 0x23, 0x45, 0x67, 0x89, 0xab}
    };

    uint16_t accepted_ctxids[3] = {12, 15, 11};

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      bind1, bind1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOCLIENT,
                      bind_ack1, bind_ack1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (accepted_uuids[0][i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        if (accepted_ctxids[0] != item->ctxid) {
            result = 0;
            goto end;
        }
        count++;
    }
    if (count != 1) {
        result = 0;
        goto end;
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      bind2, bind2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        count++;
    }
    if (count != 0) {
        result = 0;
        goto end;
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOCLIENT,
                      bind_ack2, bind_ack2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (accepted_uuids[1][i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        if (accepted_ctxids[1] != item->ctxid) {
            result = 0;
            goto end;
        }
        count++;
    }
    if (count != 1) {
        result = 0;
        goto end;
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      bind3, bind3_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        count++;
    }
    if (count != 0) {
        result = 0;
        goto end;
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOCLIENT,
                      bind_ack3, bind_ack3_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (accepted_uuids[2][i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        if (accepted_ctxids[2] != item->ctxid) {
            result = 0;
            goto end;
        }
        count++;
    }
    if (count != 1) {
        result = 0;
        goto end;
    }

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}


/**
 * \test Check for correct internal ids for bind_acks + alter_contexts
 */
int DCERPCParserTest17(void) {
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t bind[] = {
        0x05, 0x00, 0x0b, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x40, 0xfd, 0x2c, 0x34, 0x6c, 0x3c, 0xce, 0x11,
        0xa8, 0x93, 0x08, 0x00, 0x2b, 0x2e, 0x9c, 0x6d,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_len = sizeof(bind);

    uint8_t bind_ack[] = {
        0x05, 0x00, 0x0c, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x44, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x7d, 0xd8, 0x00, 0x00,
        0x0d, 0x00, 0x5c, 0x70, 0x69, 0x70, 0x65, 0x5c,
        0x6c, 0x6c, 0x73, 0x72, 0x70, 0x63, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x04, 0x5d, 0x88, 0x8a, 0xeb, 0x1c, 0xc9, 0x11,
        0x9f, 0xe8, 0x08, 0x00, 0x2b, 0x10, 0x48, 0x60,
        0x02, 0x00, 0x00, 0x00
    };
    uint32_t bind_ack_len = sizeof(bind_ack);

    uint8_t alter_context[] = {
        0x05, 0x00, 0x0e, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0xd0, 0x16, 0xd0, 0x16, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
        0xd0, 0x4c, 0x67, 0x57, 0x00, 0x52, 0xce, 0x11,
        0xa8, 0x97, 0x08, 0x00, 0x2b, 0x2e, 0x9c, 0x6d,
        0x01, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t alter_context_len = sizeof(alter_context);

    uint8_t alter_context_resp[] = {
        0x05, 0x00, 0x0f, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0xb8, 0x10, 0xb8, 0x10, 0x7d, 0xd8, 0x00, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x5d, 0x88, 0x8a,
        0xeb, 0x1c, 0xc9, 0x11, 0x9f, 0xe8, 0x08, 0x00,
        0x2b, 0x10, 0x48, 0x60, 0x02, 0x00, 0x00, 0x00
    };
    uint32_t alter_context_resp_len = sizeof(alter_context_resp);


    TcpSession ssn;
    DCERPCUuidEntry *item = NULL;
    int count = 0;

    uint8_t accepted_uuids[2][16] = {
        {0x57, 0x67, 0x4c, 0xd0, 0x52, 0x00, 0x11, 0xce,
         0xa8, 0x97, 0x08, 0x00, 0x2b, 0x2e, 0x9c, 0x6d},
        {0x34, 0x2c, 0xfd, 0x40, 0x3c, 0x6c, 0x11, 0xce,
         0xa8, 0x93, 0x08, 0x00, 0x2b, 0x2e, 0x9c, 0x6d},
    };

    uint16_t accepted_ctxids[2] = {1, 0};

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      bind, bind_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOCLIENT,
                      bind_ack, bind_ack_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (accepted_uuids[1][i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        if (accepted_ctxids[1] != item->ctxid) {
            result = 0;
            goto end;
        }
        count++;
    }
    if (count != 1) {
        result = 0;
        goto end;
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      alter_context, alter_context_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        count++;
    }
    if (count != 1) {
        result = 0;
        goto end;
    }

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOCLIENT,
                      alter_context_resp, alter_context_resp_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    count = 0;
    TAILQ_FOREACH(item, &dcerpc_state->dcerpc.dcerpcbindbindack.accepted_uuid_list, next) {
        int i = 0;
        /* check the interface uuid */
        for (i = 0; i < 16; i++) {
            if (accepted_uuids[count][i] != item->uuid[i]) {
                result = 0;
                goto end;
            }
        }
        if (accepted_ctxids[count] != item->ctxid) {
            result = 0;
            goto end;
        }
        count++;
    }
    if (count != 2) {
        result = 0;
        goto end;
    }

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

/**
 * \test DCERPC fragmented PDU.
 */
int DCERPCParserTest18(void) {
    int result = 1;
    Flow f;
    int r = 0;

    uint8_t request1[] = {
        0x05, 0x00, 0x00, 0x03, 0x10, 0x00, 0x00, 0x00,
        0x26, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0c, 0x00,
    };
    uint32_t request1_len = sizeof(request1);

    uint8_t request2[] = {
        0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x01, 0x02,
        0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A,
        0x0B, 0x0C, 0xFF, 0xFF
    };
    uint32_t request2_len = sizeof(request2);

    TcpSession ssn;

    memset(&f, 0, sizeof(f));
    memset(&ssn, 0, sizeof(ssn));

    FLOW_INITIALIZE(&f);
    f.protoctx = (void *)&ssn;

    StreamTcpInitConfig(TRUE);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request1, request1_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    DCERPCState *dcerpc_state = f.alstate;
    if (dcerpc_state == NULL) {
        printf("no dcerpc state: ");
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 18);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer == NULL);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 1);

    r = AppLayerParse(NULL, &f, ALPROTO_DCERPC, STREAM_TOSERVER,
                      request2, request2_len);
    if (r != 0) {
        printf("dcerpc header check returned %" PRId32 ", expected 0: ", r);
        result = 0;
        goto end;
    }

    result &= (dcerpc_state->dcerpc.bytesprocessed == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer != NULL &&
               dcerpc_state->dcerpc.dcerpcrequest.stub_data_buffer_len ==  14);
    result &= (dcerpc_state->dcerpc.pdu_fragged == 0);
    result &= (dcerpc_state->dcerpc.dcerpcrequest.opnum == 2);

end:
    StreamTcpFreeConfig(TRUE);
    FLOW_DESTROY(&f);
    return result;
}

#endif /* UNITTESTS */

void DCERPCParserRegisterTests(void) {
#ifdef UNITTESTS
    UtRegisterTest("DCERPCParserTest01", DCERPCParserTest01, 1);
    UtRegisterTest("DCERPCParserTest02", DCERPCParserTest02, 1);
    UtRegisterTest("DCERPCParserTest03", DCERPCParserTest03, 1);
    UtRegisterTest("DCERPCParserTest04", DCERPCParserTest04, 1);
    UtRegisterTest("DCERPCParserTest05", DCERPCParserTest05, 1);
    UtRegisterTest("DCERPCParserTest06", DCERPCParserTest06, 1);
    UtRegisterTest("DCERPCParserTest07", DCERPCParserTest07, 1);
    UtRegisterTest("DCERPCParserTest08", DCERPCParserTest08, 1);
    UtRegisterTest("DCERPCParserTest09", DCERPCParserTest09, 1);
    UtRegisterTest("DCERPCParserTest10", DCERPCParserTest10, 1);
    UtRegisterTest("DCERPCParserTest11", DCERPCParserTest11, 1);
    UtRegisterTest("DCERPCParserTest12", DCERPCParserTest12, 1);
    UtRegisterTest("DCERPCParserTest13", DCERPCParserTest13, 1);
    UtRegisterTest("DCERPCParserTest14", DCERPCParserTest14, 1);
    UtRegisterTest("DCERPCParserTest15", DCERPCParserTest15, 1);
    UtRegisterTest("DCERPCParserTest16", DCERPCParserTest16, 1);
    UtRegisterTest("DCERPCParserTest17", DCERPCParserTest17, 1);
    UtRegisterTest("DCERPCParserTest18", DCERPCParserTest18, 1);
#endif /* UNITTESTS */

    return;
}
