
/*

    This file is part of the Samsung Uproar Linux Utility (sulu).
 
    Sulu is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
 
    Sulu is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
 
    You should have received a copy of the GNU General Public License
    along with sulu.  See the file COPYING.  If you haven't received
    a copy of the GNU General Public License, write to:
 
        Free Software Foundation, Inc., 
        59 Temple Place, Suite 330, Boston, MA  
        02111-1307  USA
 
  kal@users.sourceforge.net
 
*/


#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <usb.h>

#include <popt.h>

#include "main.h"
#include "system_pane.h"
#include "device_pane.h"
#include "lib_uproar.h"

gint Delete_event(GtkWidget * widget, GdkEvent * event, gpointer data)
{
    gtk_main_quit();
    return (FALSE);
}

int gtkRun(int argc, char **argv)
{
    GtkWidget *window;
    GtkWidget *vertical_layout;
    GtkWidget *hpaned;
    GtkWidget *frame1;
    GtkWidget *frame2;
    GtkWidget *system_pane;
    GtkWidget *device_pane;
    GdkColormap *cmap;
    GdkColor col1 = { 0, 0x1000, 0x7000, 0x1000 };
    GdkColor col2 = { 0, 0x7000, 0x1000, 0x1000 };

    /* initialize the gtk */
    gtk_init(&argc, &argv);

    /* setup the default style */
    gtk_rc_parse(".sulurc");

    /* get some colors */
    cmap = gdk_colormap_get_system();
    if (gdk_colormap_alloc_color(cmap, &col1, FALSE, TRUE) &&
	gdk_colormap_alloc_color(cmap, &col2, FALSE, TRUE)) {
	FgFile = &col1;
	FgDir = &col2;
    } else {
	fprintf(stderr, "\nNO COLORS");
	FgDir = 0;
	FgFile = 0;
    }

    /* create the main window */
    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_name(window, "main_window");
    gtk_window_set_title(GTK_WINDOW(window),
			 "Samsung Uproar Linux Utility (sulu) Version 0.16");
    gtk_signal_connect(GTK_OBJECT(window), "destroy",
		       GTK_SIGNAL_FUNC(Delete_event), NULL);
    gtk_container_set_border_width(GTK_CONTAINER(window), 0);
    gtk_widget_set_usize(GTK_WIDGET(window), STARTUP_X_SIZE,
			 STARTUP_Y_SIZE);

    /* create a vertical box layout */
    vertical_layout = gtk_vbox_new(FALSE, 5);
    gtk_container_add(GTK_CONTAINER(window), vertical_layout);
    gtk_widget_show(vertical_layout);

    /* create the side by side pane */
    hpaned = gtk_hpaned_new();
    gtk_paned_set_gutter_size(GTK_PANED(hpaned), GUTTER_SIZE);
    gtk_widget_set_usize(hpaned, 2 * MIN_PANE_WIDTH + GUTTER_SIZE, -1);
    gtk_container_add(GTK_CONTAINER(vertical_layout), hpaned);
    gtk_widget_show(hpaned);

    /* create the system pane, left side */
    system_pane = Create_system_pane();
    gtk_widget_set_name(system_pane, "system_pane");
    frame1 = gtk_frame_new(NULL);
    gtk_paned_pack1(GTK_PANED(hpaned), frame1, TRUE, FALSE);
    gtk_widget_set_usize(frame1, MIN_PANE_WIDTH, -1);
    gtk_widget_show(frame1);
    gtk_container_add(GTK_CONTAINER(frame1), system_pane);
    gtk_widget_show(system_pane);

    /* create the device pane, right side */
    device_pane = Create_device_pane();
    gtk_widget_set_name(device_pane, "device_pane");
    frame2 = gtk_frame_new(NULL);
    gtk_paned_pack2(GTK_PANED(hpaned), frame2, TRUE, FALSE);
    gtk_widget_set_usize(frame2, MIN_PANE_WIDTH, -1);
    gtk_widget_show(frame2);
    gtk_container_add(GTK_CONTAINER(frame2), device_pane);
    gtk_widget_show(device_pane);

    // gtk_paned_add2(GTK_PANED(hpaned), device_pane);
    // gtk_widget_show(device_pane);

    /* show the window and start gtk processing */
    gtk_widget_show(window);
    gtk_main();

    return 0;
}

void printTrackInfo(int trackNumber)
{
    char *retString;
    int retInt;

    retString = Uproar_get_track_info(trackNumber, &retInt);
    printf("Track #%i: %s\t%i\n", trackNumber, retString, retInt);
}

int poptRun(int argc, char **argv)
{
    int c, i;

    static int track_info = 0;
    static int del_track = 0;

    static char *transfer = NULL;

    unsigned char *retString;
    int retInt;

    poptContext context;

    struct poptOption options[] = {
	{
	 "get-dev-info", 'd', POPT_ARG_NONE,
	 NULL, GET_DEV_INFO_FLAG,
	 "Get Uproar Device Info", NULL},
	{
	 "print-playlist", 'p', POPT_ARG_NONE,
	 NULL, PRINT_PLAYLIST_FLAG,
	 "Print Uproar Playlist", NULL},
	{
	 "get-track-info", 'i', POPT_ARG_INT,
	 &track_info, TRACK_INFO_FLAG,
	 "Get Track Info", "TRACK NUMBER"},
	{
	 "get-capacity", 'c', POPT_ARG_NONE,
	 NULL, GET_CAPACITY_FLAG,
	 "Get Uproar Capacity", NULL},
	{
	 "get-available", 'a', POPT_ARG_NONE,
	 NULL, GET_AVAILABLE_FLAG,
	 "Get Available Uproar Capacity", NULL},
	{
	 "get-num-tracks", 'n', POPT_ARG_NONE,
	 NULL, GET_NUM_TRACKS_FLAG,
	 "Get Number of Tracks", NULL},
	{
	 "del-track", 'D', POPT_ARG_INT,
	 &del_track, DEL_TRACK_FLAG,
	 "Delete track", "TRACK NUMBER"},
	{
	 "reformat-memory", 'r', POPT_ARG_NONE,
	 NULL, REFORMAT_MEMORY_FLAG,
	 "Print Uproar Playlist", NULL},
	{
	 "transfer", 't', POPT_ARG_STRING,
	 &transfer, TRANSFER_FLAG,
	 "Transfer file to Uproar", "FILENAME"},
	{
	 "usage", 'u', POPT_ARG_NONE,
	 NULL, USAGE_FLAG,
	 "Get usage information", NULL},
	{
	 "help", '?', POPT_ARG_NONE,
	 NULL, HELP_FLAG,
	 "Get this help", NULL},
	{
	 "gtk-mode", 'g', POPT_ARG_NONE,
	 NULL, GTK_MODE_FLAG,
	 "Run GTK interface", NULL},
	{NULL, 0, 0, NULL, 0}
    };

    context = poptGetContext(NULL, argc, (const char **) argv, options, 0);

    retInt = Uproar_load_playlist();
    if (retInt != UPROAR_SUCCESS) {
	fprintf(stderr,
		"%s: Error loading uproar playlist.  Error code: %i\n",
		argv[0], retInt);
	return -1;
    }

    while ((c = poptGetNextOpt(context)) >= 0) {
	switch (c) {
	case GET_DEV_INFO_FLAG:
	    retString = Uproar_get_device_info();
	    printf("%s\n", retString);
	    break;
	case PRINT_PLAYLIST_FLAG:
	    retInt = Uproar_num_tracks();
	    for (i = 0; i < retInt; i++)
		printTrackInfo(i);
	    break;
	case TRACK_INFO_FLAG:
	    printTrackInfo(track_info);
	    break;
	case GET_CAPACITY_FLAG:
	    retInt = Uproar_get_capacity();
	    printf("Uproar capacity: %i\n", retInt);
	    break;
	case GET_AVAILABLE_FLAG:
	    retInt = Uproar_get_available();
	    printf("Uproar available: %i\n", retInt);
	    break;
	case GET_NUM_TRACKS_FLAG:
	    retInt = Uproar_num_tracks();
	    printf("Uproar tracks: %i\n", retInt);
	    break;
	case DEL_TRACK_FLAG:
	    printf("Deleteing Uproar track: %i. . .", del_track);
	    fflush(stdout);
	    retInt = Uproar_delete_track(del_track);
	    if (retInt != UPROAR_SUCCESS) {
		printf("FAILED %i\n", retInt);
		return -1;
	    } else {
		printf("succeeded\n");
	    }
	    break;
	case REFORMAT_MEMORY_FLAG:
	    printf("Reformating Uproar memory. . .");
	    fflush(stdout);
	    retInt = Uproar_reformat_memory();
	    if (retInt != UPROAR_SUCCESS) {
		printf("FAILED %i\n", retInt);
		return -1;
	    } else {
		printf("succeeded\n");
	    }
	    break;
	case TRANSFER_FLAG:
	    printf("Transfering file \"%s\" to Uproar . . .", transfer);
	    fflush(stdout);
	    retInt = Uproar_download(transfer);
	    if (retInt != UPROAR_SUCCESS) {
		printf("FAILED %i\n", retInt);
		return -1;
	    } else {
		printf("succeeded\n");
	    }
	    break;
	case USAGE_FLAG:
	    poptPrintUsage(context, stderr, 0);
	    break;
	case HELP_FLAG:
	    poptPrintHelp(context, stderr, 0);
	    break;
	case GTK_MODE_FLAG:
	    return gtkRun(argc, argv);
	    break;
	}
    }

    if (c < -1) {
	/* an error occurred during option processing */
	fprintf(stderr, "%s: %s\n",
		poptBadOption(context, 0), poptStrerror(c));
	printf("error\n");
	return 1;
    }

    return 0;
}

int main(int argc, char **argv)
{
    int retval;
    char *devstr;

    retval = Uproar_initialize();
    if (retval < 0) {
	fprintf(stderr, "\n\nNO MP3 DEVICE FOUND.\n");
	exit(-1);
    }
    devstr = Uproar_get_device_info();
    if (devstr == 0) {
	fprintf(stderr, "\n\nNO MP3 DEVICE STRING.\n");
	exit(-1);
    }

    if (argc > 1)
	return poptRun(argc, argv);
    else
	return gtkRun(argc, argv);
}
