#ifndef _GstLaunch_h
#define _GstLaunch_h

/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include <gst/gst.h>
#include <glibmm.h>
#include "VideoInfo.h"

class GstLaunch
{
public:

	enum Error
	{
		PARSE_ERROR=0,
		FAILED,
		UNKNOWN
	};

	/*
	 *	constructeur
	 *	timeout permet de regler le temps en msecs
	 *	pour l'envoi d'information ( Glib::signal_timeout() )
	 *	utiliser pendant la lecture
	 */
	GstLaunch(guint timeout=0);

	/*
	 *	destructeur
	 *	disconnect signal_timeout et init le pipeline (GST_STATE_NULL)
	 */
	virtual ~GstLaunch();

	/*
	 *	s'utilise comme gst-launch
	 *	l'element principale doit s'appeler "src"
	 *	exemple : playbin name=src ...
	 */
	bool parse(const Glib::ustring &pipe) throw (Error);

	/*
	 *	gestion des messages gstreamer :
	 *	GST_MESSAGE_ERROR, GST_MESSAGE_WARNING, 
	 *	GST_MESSAGE_EOS, GST_MESSAGE_STATE_CHANGED
	 *
	 *	cette fonction est important 
	 *	il faut l'appeler si on fait deriver cette fonction
	 */
	virtual bool bus_message(GstBus *bus, GstMessage *message) throw(Error);

	/*
	 *	le media joue t'il ?
	 */
	bool is_playing();

	/*
	 *	init pipeline a GST_STATE_NULL
	 *	(clear)
	 */
	bool null();

	/*
	 *	init le pipeline a GST_STATE_READY
	 */
	bool ready();

	/*
	 *	init le pipeline a GST_STATE_PLAYING
	 */
	bool play();

	/*
	 *	init le pipeline a GST_STATE_PAUSE
	 */
	bool pause();

	/*
	 *	retourne la longueur du media (GST_FORMAT_TIME)
	 */
	gint64 get_duration();

	/*
	 *	retourne la position (lecture) dans le media
	 */
	gint64 get_position();

	/*
	 *	retourne la position en %
	 *	TODO verifier la fonction
	 */
	gint64 get_percent();

	/*
	 *	ce deplacer a pos
	 */
	bool seek(gint64 pos);

	/*
	 *	ce deplacer a start et lire jusqu'a end
	 *	ne declanche pas play
	 */
	bool seek(gint64 start, gint64 end);

	/*
	 *	fonction de rappel
	 *	seulement pendant la lecture du media
	 *	configure la frequence dans le constructeur "timeout"
	 */
	sigc::signal<void>& get_signal_timeout();

	/*
	 *	emit si on arrive a la fin du fichier
	 */
	virtual void on_eos();
	
	/*
	 *	emit en cas d'erreur
	 */
	virtual void on_error();


	/*
	 *
	 */
	const VideoInfo& get_info();

	/*
	 *
	 */
	virtual void info_update();
protected:
	
	/*
	 *	fonction callback pour Glib::timeout()
	 */
	bool	on_timeout();

	/*
	 *
	 */
	static void static_parse_stream_info(GObject *obj, GParamSpec *pspec, GstLaunch *launch);
	static void static_caps_set(GObject *obj, GParamSpec *pspec, GstLaunch *launch);

	void parse_stream_info();
protected:
	GstElement*		m_pipeline;
	GstElement*		m_src;
	GstBus*				m_bus;

	VideoInfo			m_video_info;

	
	guint m_timeout_t;
	Glib::RefPtr<Glib::TimeoutSource>	m_timeout;
	sigc::connection m_connection;

	sigc::signal<void>	m_signal_timeout;

};

namespace Gst
{
	Glib::ustring time_to_string (gint64 time);
}

#endif//_GstLaunch_h
