#include "SubtitleSystem.h"
#include <regex.h>
#include <fstream>
#include <iostream>
#include "SubtitleSubRip.h"
#include "SubtitleMicroDVD.h"
#include "SubtitleTTXT.h"
#include "SubtitleMPL2.h"
#include "SubtitleASS.h"
#include "SubtitleSSA.h"
#include "SubtitleSubViewer2.h"
#include "SubtitleMPsub.h"
#include "utility.h"

/*
 *
 */
class IFactory
{
public:
	virtual ~IFactory() { }
	virtual SubtitleFormat* create(Document *doc) = 0;
	virtual Glib::ustring get_name() = 0;
	virtual Glib::ustring get_extension() = 0;
	virtual bool check(const std::string &line) = 0;
};

/*
 *
 */
template<class Sub>
class Factory : public IFactory
{
public:
	Factory()
	{
	}
	
	virtual SubtitleFormat* create(Document *doc)
	{
		return new Sub(doc);
	}

	virtual Glib::ustring get_name()
	{
		return Sub::get_name();
	}
	
	virtual bool check(const std::string &line)
	{
		return Sub::check(line);
	}

	virtual Glib::ustring get_extension()
	{
		return Sub::get_extension();
	}
};

/*
 *
 */
#define REGISTER_SUBTITLE(class) \
	{ static Factory<class> factory; m_static_subtitle_formats.push_back(&factory); }

/*
 *
 */
static std::list<IFactory*> m_static_subtitle_formats;


/*
 *
 */
SubtitleSystem& SubtitleSystem::getInstance()
{
	if(m_static_subtitle_formats.empty())
	{
		REGISTER_SUBTITLE(SubtitleASS);
		REGISTER_SUBTITLE(SubtitleSSA);
		REGISTER_SUBTITLE(SubtitleSubRip);
		REGISTER_SUBTITLE(SubtitleMicroDVD);
		REGISTER_SUBTITLE(SubtitleMPL2);
		REGISTER_SUBTITLE(SubtitleSubViewer2);
		REGISTER_SUBTITLE(SubtitleMPsub);
#ifdef ENABLE_TTXT
		REGISTER_SUBTITLE(SubtitleTTXT);
#endif
	}
	static SubtitleSystem system;
	return system;
}


/*
 *	retourne la list des formats supporter
 */
std::list<Glib::ustring> SubtitleSystem::get_formats()
{
	static std::list<Glib::ustring> list;
	if(list.empty())
	{
		for(std::list<IFactory*>::const_iterator it=m_static_subtitle_formats.begin();
				it!=m_static_subtitle_formats.end(); ++it)
		{
			list.push_back((*it)->get_name());
		}
	}
	return list;
}


/*
 *	determine quel est le format du sous-titre
 */
Glib::ustring SubtitleSystem::find_subtitle_format(const Glib::ustring &filename)
{
	std::ifstream file(filename.c_str());

	if(!file)
	{
		std::cerr << "SubtitleSystem::find_subtitle_format open <" << filename << "> failed." << std::endl;
		return "";
	}

	std::string line;

	unsigned int count = 0;	

	while(!file.eof() && std::getline(file, line))
	{
		Glib::ustring fmt = find_format(line.c_str());
		
		if(!fmt.empty())
		{
			file.close();
			std::cout << "format detected: " << fmt << std::endl;
			return fmt;		
		}

		if(++count > 100)
			break;
	}

	file.close();

	throw SubtitleException("SubtitleSystem", _("I can't find what is this format or it's not supported."));
}

/*
 *	crée la class du format
 */
SubtitleFormat* SubtitleSystem::create_subtitle_format(const Glib::ustring &name, Document *doc)
{
	g_return_val_if_fail(doc, NULL);

	for(std::list<IFactory*>::const_iterator it=m_static_subtitle_formats.begin();
				it!=m_static_subtitle_formats.end(); ++it)
	{
		if((*it)->get_name() == name)
			return (*it)->create(doc);
	}
	return NULL;
}

/*
 *	cherche le format de la line str
 */
Glib::ustring	SubtitleSystem::find_format(const char *str)
{
	//std::cout << "SubtitleSystem::find_format > " << str << std::endl;

	for(std::list<IFactory*>::const_iterator it=m_static_subtitle_formats.begin();
			it!=m_static_subtitle_formats.end(); ++it)
	{
		if((*it)->check(str))
			return (*it)->get_name();
	}
	return "";
}


/*
 *	retourne l'extension utiliser par le format
 *	ex: "ass", "ssa", "srt", ...
 */
Glib::ustring SubtitleSystem::get_extension(const Glib::ustring &format)
{
	for(std::list<IFactory*>::const_iterator it=m_static_subtitle_formats.begin();
				it!=m_static_subtitle_formats.end(); ++it)
	{
		if((*it)->get_name() == format)
			return (*it)->get_extension();
	}
	return "";
}

