/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include "SubtitleFormat.h"
#include "Encodings.h"
#include <iostream>
#include "Config.h"

/*
 *
 */
SubtitleFormat::SubtitleFormat()
{
	m_use_auto_deteced = false;

	Config::getInstance().get_value_string_list("encodings", "encodings", m_list_encodings);
}

/*
 *
 */
SubtitleFormat::~SubtitleFormat()
{
}


/*
 *
 */
void SubtitleFormat::set_charset(const std::string &charset)
{
	m_charset = charset;

	m_use_auto_deteced = charset.empty();
}

/*
 *
 */
std::string SubtitleFormat::get_charset()
{
	return m_charset;
}

	
/*
 *
 */
bool SubtitleFormat::open(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	set_charset(encoding);
	
	debug_msg(message("open > %s (%s)", filename.c_str(), encoding.c_str()));

	return false;
}
	
/*
 *
 */
bool SubtitleFormat::save(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	set_charset(encoding);

	debug_msg(message("save > %s (%s)", filename.c_str(), encoding.c_str()));

	return false;
}

/*
 *
 */
std::string convert_to_utf8_from_charset(const std::string &content, const std::string &charset)
{
	// seulement si c'est de l'UTF-8 to UTF-8 on ne passe pas par une convertion
	// juste une verification
	if(charset == "UTF-8")
	{
		if(Glib::ustring(content).validate())
			return content;
		else
		{
			throw Glib::ConvertError(Glib::ConvertError::ILLEGAL_SEQUENCE, 
					_("It's not valid UTF-8.\nPlease use another character encoding."));
		}
	}

	// on convertion en utf-8 a partir de charset
	std::string utf8_content = Glib::convert(content, "UTF-8", charset);

	// ok ?
	if(Glib::ustring(utf8_content).validate() == false)
	{
		throw Glib::ConvertError(Glib::ConvertError::ILLEGAL_SEQUENCE, _("Please use another character encoding."));
	}

	return utf8_content;
}

/*
 *
 */
std::string convert_to_utf8(const std::list<Glib::ustring> &m_list_encodings, const std::string &content, std::string& encoding)
{
	if(encoding.empty() == false) // has encoding
	{
		return convert_to_utf8_from_charset(content, encoding);
	}
	else
	{
		// Automatically detect the encoding used
		
		// has user preference
		if(m_list_encodings.size() > 0)
		{
			std::list<Glib::ustring>::const_iterator it;
			for(it = m_list_encodings.begin(); it != m_list_encodings.end(); ++it)
			{
				std::string utf8_content;
			
				try
				{
					utf8_content = convert_to_utf8_from_charset(content,(*it));
				
					if(utf8_content.empty() == false)
					{
						encoding = (*it);
						return utf8_content;
					}
				}
				catch(Glib::ConvertError &ex)
				{
					// invalid, next
				}
			}
		}
		else
		{
			for(unsigned int i=0; encodings_info[i].name!= NULL; ++i)
			{
				std::string charset = encodings_info[i].charset;
				std::string utf8_content;
			
				try
				{
					utf8_content = convert_to_utf8_from_charset(content,charset);
				
					if(utf8_content.empty() == false)
					{
						encoding = charset;
						return utf8_content;
					}
				}
				catch(Glib::ConvertError &ex)
				{
					// invalid, next
				}
			}
		}
	}


	throw Glib::ConvertError(
			Glib::ConvertError::FAILED, 
			"SubtitleEditor has not been able to automatically determine the encoding of the file you want to open.");
	
	return "Invalid Encoding";
}

/*
 *	convertir text en utf8
 *	le charset du text est defini par m_charset
 */
Glib::ustring SubtitleFormat::charset_to_utf8(const std::string &content)
{
	if(content.empty())
		return "";

	return convert_to_utf8(m_list_encodings, content, m_charset);
}

/*
 *	convertir le text (utf-8) en codage defini par m_charset
 */
std::string SubtitleFormat::utf8_to_charset(const Glib::ustring &content)
{
	try
	{
		if(m_charset.empty())
		{
			std::cerr << "utf8_to_charset charset is empty" << std::endl;
			return content;
		}
		
		std::string res = Glib::convert(content, m_charset, "UTF-8");
		return res;
	}
	catch(Glib::ConvertError& ex)
	{
		std::cerr << content << std::endl;
		std::cerr << "utf8_to_charset[" << m_charset << "] " << ex.what() << std::endl;
		std::cerr << std::endl;
	}

	return "";
}
