/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include <gtkmm/filechooserdialog.h>
#include <gtkmm/entry.h>
#include <gtkmm/accelmap.h>
#include <gtkmm/checkbutton.h>
#include <gtkmm/main.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/stock.h>
#include <gtkmm/toggleaction.h>

#include <gdk/gdkkeysyms.h>

#include "SubtitleEditor.h"
#include "SpellDialog.h"

#include <config.h>
#include "utility.h"
#include "SubtitleText.h"

#include "Config.h"
#include "MenuBar.h"
#include "DialogSplit.h"
#include "DialogPreferences.h"
#include "DialogFind.h"

#include "SubtitleSystem.h"

namespace SE
{
ISubtitleEditor* getInstance()
{
	return SubtitleEditor::getInstance();
}

}

SubtitleEditor* SubtitleEditor::m_static_subtitleEditor = NULL;

SubtitleEditor* SubtitleEditor::getInstance()
{
	//debug_msg("getInstance()");

	if(m_static_subtitleEditor == NULL)
	{
		debug_msg("getInstance > new SubtitleEditor");
		Glib::RefPtr<Gnome::Glade::Xml> refXml = Gnome::Glade::Xml::create(get_share_dir("subtitleeditor.glade"));

		refXml->get_widget_derived("window-subtitleeditor", m_static_subtitleEditor);
		
	//m_static_subtitleEditor = new SubtitleEditor();
	}

	return m_static_subtitleEditor;
}



/*
 *
 */
SubtitleEditor::SubtitleEditor(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade)
:Gtk::Window(cobject), m_treeviewSubtitle(NULL)
{
	m_static_subtitleEditor = this;

	debug_msg("SubtitleEditor::SubtitleEditor > start");
	
	m_dialogFindAndReplace = NULL;
	m_dialogScale = NULL;
	m_dialogChangeFPS = NULL;



	Gtk::VBox* vbox = NULL;
	refGlade->get_widget("vbox-all", vbox);
	refGlade->get_widget("statusbar", m_statusbar);

	refGlade->get_widget_derived("frame-timing-system", m_timingSystem);
	refGlade->get_widget_derived("treeview-subtitle", m_treeviewSubtitle);

	MenuBar* menubar = manage(new MenuBar(*this));
	vbox->pack_start(*menubar, false,false);
	vbox->reorder_child(*menubar, 0);

	debug_msg("SubtitleEditor::SubtitleEditor > load AccelMap");

	// on va chercher la configuration clavier
	Glib::ustring path_se_accelmap = Glib::build_filename(Glib::build_filename(Glib::get_home_dir(), ".subtitleeditor"), "accelmap");
	
	Gtk::AccelMap::load(path_se_accelmap);


	debug_msg("SubtitleEditor::SubtitleEditor > loadCfg");

	// config
	loadCfg();
	//
	debug_msg("SubtitleEditor::SubtitleEditor > finish");

}

/*
 *
 */
SubtitleEditor::~SubtitleEditor()
{
	debug_msg("SubtitleEditor::~SubtitleEditor > start");

	debug_msg("SubtitleEditor::~SubtitleEditor > saveCfg");
	
	saveCfg();

	debug_msg("SubtitleEditor::~SubtitleEditor > save .subtitleeditor");

	// on sauvegarde la configuration clavier
	Glib::ustring path_se = Glib::build_filename(Glib::get_home_dir(), ".subtitleeditor");
	Glib::ustring path_se_accelmap = Glib::build_filename(path_se, "accelmap");

	Gtk::AccelMap::save(path_se_accelmap);

	debug_msg("SubtitleEditor::~SubtitleEditor > delete dialog");

	delete m_dialogFindAndReplace;
	delete m_dialogScale;
	delete m_dialogChangeFPS;

	debug_msg("SubtitleEditor::~SubtitleEditor > finish");
}

/*
 *
 */
void SubtitleEditor::open_subtitle(const Glib::ustring &file)
{
	if(getDocument()->open(file, ""))
		setStatusbar(_("Loaded file \"%s\" (%s, %s)"), 
				file.c_str(), getDocument()->getFormat().c_str(), getDocument()->getCharset().c_str());
	else
		setStatusbar(_("Error loading file \"%s\""), file.c_str());
}

/*
 *
 */
void SubtitleEditor::open_movie(const Glib::ustring &uri)
{
	getDocument()->m_uriMovie = uri;
}

/*
 *
 */
void SubtitleEditor::loadCfg()
{
	Config &cfg = Config::getInstance();

	bool boolean = false;
	//int integer = 0;
	int width=0, height=0;
		
	if(cfg.get_value_int("general", "width", width) && cfg.get_value_int("general", "height", height))
		set_default_size(width, height);
	else
		set_default_size(600,400);

	if(cfg.get_value_bool("general", "maximize", boolean))
	{
		if(boolean)
			maximize();
	}
	else
		cfg.set_value_bool("general", "maximize", false);
}

/*
 *
 */
void SubtitleEditor::saveCfg()
{
	Config &cfg = Config::getInstance();

	cfg.set_value_int("general", "width", get_width());
	cfg.set_value_int("general", "height", get_height());
	//cfg.set_value_bool("general", "maximize", );
}

/*
 *
 */
SubtitleView* SubtitleEditor::getSubtitleView()
{
	return m_treeviewSubtitle;
}
	
/*
 *
 */
Document* SubtitleEditor::getDocument()
{
	return &m_refDocument;
}

/*
 * 
 */
void SubtitleEditor::setStatusbar(const char *format, ...)
{
	int context = m_statusbar->get_context_id("txt");
	//m_statusbar->remove_message(0, context);
//	m_statusbar->push(txt, context);
	va_list args;
	char *formatted = NULL;

	va_start(args, format);
	formatted = g_strdup_vprintf(format, args);
	va_end(args);

	m_statusbar->push(formatted, context);
	g_free(formatted);
}

/*
 *
 */
void SubtitleEditor::createMenuBar()
{
	//MenuBar *menubar= manage(new MenuBar(*this));
	//vbox.pack_start(*menubar, false,false);
}



/*
 *
 */
void SubtitleEditor::on_new_subtitle()
{
	debug_msg("SubtitleEditor::on_new_subtitle");

	getDocument()->clear();
	setStatusbar(_("New Document"));
}

/*
 *
 */
void SubtitleEditor::on_quit()
{
	debug_msg("SubtitleEditor::on_quit");

	//Gtk::MessageDialog dialog(*this, "Are you sure ?", false, Gtk::MESSAGE_QUESTION , Gtk::BUTTONS_YES_NO ,true);
	//if(dialog.run() == Gtk::RESPONSE_YES)
		Gtk::Main::quit();
}

/*
 *	on utilise l'extention du fichier pour ouvrir selon le type
 */
void SubtitleEditor::on_open_subtitle()
{
	debug_msg("SubtitleEditor::on_open_subtitle");

	Glib::ustring file ;
	m_dialogOpenSubtitle.show();
	
	if(m_dialogOpenSubtitle.run() == Gtk::RESPONSE_OK)
	{
		file = m_dialogOpenSubtitle.get_filename();
		
		if(getDocument()->open(file, m_dialogOpenSubtitle.getEncoding()))
			setStatusbar(_("Loaded file \"%s\" (%s, %s)."), 
					file.c_str(),
					getDocument()->getFormat().c_str(),
					getDocument()->getCharset().c_str());
		else
			setStatusbar(_("Error loading file \"%s\"."), file.c_str());
	}
	m_dialogOpenSubtitle.hide();
}

/*
 *
 */
void SubtitleEditor::on_open_media_for_timing()
{
	debug_msg("SubtitleEditor::on_open_media_for_timing");

	DialogOpenMedia	dialog;
	if(dialog.run() == Gtk::RESPONSE_OK)
	{
		Glib::ustring uri = dialog.get_uri();

		dialog.hide();

		if(m_timingSystem->open_media(uri))
		{
			Signal::getInstance().setup_view("timing");
			//
			if(m_timingSystem->get_waveform() != NULL)
				getDocument()->m_uriMovie = m_timingSystem->get_waveform()->uri;
		}
	}
}

/*
 *
 */
void SubtitleEditor::on_save_waveform()
{
	DialogSaveWaveform	dialog;
	if(dialog.run() == Gtk::RESPONSE_OK)
	{
		Glib::ustring uri = dialog.get_uri();

		dialog.hide();

		if(uri.substr(uri.size()-3, 3) != ".wf")
			uri+=".wf";
		
		if(m_timingSystem->save_waveform(uri))
			setStatusbar(_("Save waveform in \"%s\""), uri.c_str());
		else
			setStatusbar(_("Save Waveform failed"));
	}
}


/*
 *
 */
void SubtitleEditor::on_timing_play_previous_subtitle()
{
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
		iter = getDocument()->get_subtitle_model()->getFirst();
	else
		iter = getDocument()->get_subtitle_model()->find_previous(iter);

	if(iter)
	{
		m_timingSystem->play_subtitle(iter);
	}
}

/*
 *
 */
void SubtitleEditor::on_timing_play_current_subtitle()
{
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	
	if(!iter)
		iter = getDocument()->get_subtitle_model()->getFirst();

	if(iter)
		m_timingSystem->play_subtitle(iter);
}

/*
 *
 */
void SubtitleEditor::on_timing_play_next_subtitle()
{
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
		iter = getDocument()->get_subtitle_model()->getFirst();
	else
		iter = getDocument()->get_subtitle_model()->find_next(iter);

	if(iter)
	{
		m_timingSystem->play_subtitle(iter);
	}
}

/*
 *
 */
void SubtitleEditor::on_timing_stop()
{
	m_timingSystem->stop();
}

/*
 *
 */
void SubtitleEditor::on_timing_play_previous_second()
{
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
		return;

	SubtitleTime time((*iter)[m_column.start]);
	
	m_timingSystem->play_subtitle(time - SubtitleTime(0,0,1,0), time);
}

/*
 *
 */
void SubtitleEditor::on_timing_play_first_second()
{
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
		return;

	SubtitleTime time((*iter)[m_column.start]);
	
	m_timingSystem->play_subtitle(time, time + SubtitleTime(0,0,1,0));
}

/*
 *
 */
void SubtitleEditor::on_timing_play_last_second()
{
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
		return;

	SubtitleTime time((*iter)[m_column.end]);
	
	m_timingSystem->play_subtitle(time - SubtitleTime(0,0,1,0), time);
}

/*
 *
 */
void SubtitleEditor::on_timing_play_next_second()
{
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
		return;
	
	SubtitleTime time((*iter)[m_column.end]);
	
	m_timingSystem->play_subtitle(time, time + SubtitleTime(0,0,1,0));
}



/*
 *
 */
void SubtitleEditor::on_open_movie()
{
	debug_msg("SubtitleEditor::on_open_movie");

	m_dialogOpenMovie.show();
	if(m_dialogOpenMovie.run() == Gtk::RESPONSE_OK)
	{
		Glib::ustring uri = m_dialogOpenMovie.get_uri();

		getDocument()->m_uriMovie = uri;
	}
	m_dialogOpenMovie.hide();
}

/*
 *
 */
void SubtitleEditor::on_video_properties()
{
	DialogVideoProperties *m_dialogVideoProperties = NULL;
	create_dialog("dialog-video-properties", &m_dialogVideoProperties);

	m_dialogVideoProperties->execute();

	delete m_dialogVideoProperties;
}

/*
 *
 */
void SubtitleEditor::on_save_subtitle()
{
	debug_msg("SubtitleEditor::on_save_subtitle");

	if(!getDocument()->filenameDocument.empty())
	{
		Glib::ustring fmt = getDocument()->getFormat();
		Glib::ustring filename = getDocument()->filenameDocument;
		Glib::ustring charset = getDocument()->getCharset();

		if(getDocument()->save(filename, fmt))
			setStatusbar(_("Saving file \"%s\" (%s, %s)."), filename.c_str(), fmt.c_str(), charset.c_str());
		else
			setStatusbar(_("The file \"%s\" (%s, %s) has not been saved."), filename.c_str(), fmt.c_str(), charset.c_str());
	}
	else
	{
		on_save_as_subtitle();
	}
}

/*
 *
 */
void SubtitleEditor::on_save_as_subtitle()
{
	debug_msg("SubtitleEditor::on_save_as_subtitle");

	m_dialogSaveSubtitle.show();
	if(m_dialogSaveSubtitle.run() == Gtk::RESPONSE_OK)
	{
		Glib::ustring file = m_dialogSaveSubtitle.get_filename();
		Glib::ustring encoding = m_dialogSaveSubtitle.getEncoding();
		Glib::ustring fmt = m_dialogSaveSubtitle.getFormat();

		if(getDocument()->save(file, fmt, encoding))
			setStatusbar(_("Saving file \"%s\" (%s, %s)."), file.c_str(), fmt.c_str(), encoding.c_str());
		else
			setStatusbar(_("The file \"%s\" (%s, %s) has not been saved."), file.c_str(), fmt.c_str(), encoding.c_str());
	}
	m_dialogSaveSubtitle.hide();
}

/*
 *
 */
void SubtitleEditor::on_import_text()
{
	debug_msg("SubtitleEditor::on_import_text");

	m_dialogImportText.show();
	if(m_dialogImportText.run() == Gtk::RESPONSE_OK)
	{
		Glib::ustring filename = m_dialogImportText.get_filename();
	
		getDocument()->clear();
		SubtitleText text(getDocument());
		if(text.open(filename, m_dialogImportText.getEncoding()))
			setStatusbar(_("Import text \"%s\"."), filename.c_str());
		else
			setStatusbar(_("Import Text \"%s\" failed."), filename.c_str());
	}
	m_dialogImportText.hide();

}

/*
 *
 */
void SubtitleEditor::on_export_text()
{
	debug_msg("SubtitleEditor::on_export_text");

	m_dialogExportText.show();
	if(m_dialogExportText.run() == Gtk::RESPONSE_OK)
	{
		Glib::ustring filename = m_dialogExportText.get_filename();
		Glib::ustring encoding = m_dialogExportText.getEncoding();
		
		SubtitleText text(getDocument());
		
		if(text.save(filename, encoding))
			setStatusbar(_("Export text \"%s\" (%s)."), filename.c_str(), encoding.c_str());
		else
			setStatusbar(_("Export Text \"%s\" (%s) failed."), filename.c_str(), encoding.c_str());
	}
	m_dialogExportText.hide();	
	
}

/*
 *
 */
void SubtitleEditor::on_properties()
{
	debug_msg("SubtitleEditor::on_properties");

	DialogScriptProperties * dialog = NULL;
	
	create_dialog("dialog-script-properties", &dialog);

	dialog->execute();

	delete dialog;
}

/*
 *
 */
void SubtitleEditor::on_play_movie()
{
	debug_msg("SubtitleEditor::on_play_movie > start");

	Document *doc = getDocument();

	if(doc->filenameDocument.empty())
	{
		debug_msg("SubtitleEditor::on_play_movie > filenameDocument == NULL");
		setStatusbar(_("I can't play movie"));
		return;
	}

	if(doc->m_uriMovie.empty())
	{
		debug_msg("SubtitleEditor::on_play_movie > m_uriMovie == NULL");
		setStatusbar(_("I can't play movie. Please select a movie"));
		
		on_open_movie();
		return;
	}

	
	debug_msg("SubtitleEditor::on_play_movie > save tmp subtitle");

	Glib::ustring subtitle_name = "subtitle." + SubtitleSystem::getInstance().get_extension(doc->getFormat());
	// on ecrit un fichier tmp pour les sous titres..
	Glib::ustring subtitle_file = Glib::build_filename(Glib::get_tmp_dir(), subtitle_name);

	getDocument()->save(subtitle_file, doc->getFormat(), "", false);
	
	
	debug_msg("SubtitleEditor::on_play_movie > select subtitle for start movie");

	long position = 0;

	// si il y a un subtitle de select 
	// alors on commence a partir de ça position moins 4 sec
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(iter)
	{
		SubtitleColumnRecorder m_column;
		SubtitleTime time = SubtitleTime((*iter)[m_column.start]) - SubtitleTime(0,0,4,0);
		
		position = time.hours*3600 + time.mins*60 + time.secs;
	}

	debug_msg("SubtitleEditor::on_play_movie > create cmd line");

	/*
	 *	#subtitle_file
	 *	#subtitle_uri
	 *	#video_file
	 *	#video_uri
	 *	#seconds
	 */

	Config &cfg = Config::getInstance();

	Glib::ustring player;
	if(cfg.get_value_string("video-player", "default", player))
	{
		Glib::ustring cmd;
		if(cfg.get_value_string("video-player", player, cmd))
		{
			Glib::ustring video_uri = getDocument()->m_uriMovie;
			Glib::ustring video_file = Glib::filename_from_uri(video_uri);

			Glib::ustring seconds = to_string(position);
			
			Glib::ustring subtitle_uri = Glib::filename_to_uri(subtitle_file);
			
			find_and_replace(cmd, "#video_file", video_file);
			find_and_replace(cmd, "#video_uri", video_uri);
			find_and_replace(cmd, "#subtitle_file", subtitle_file);
			find_and_replace(cmd, "#subtitle_uri", subtitle_uri);
			find_and_replace(cmd, "#seconds", seconds);

			std::cout << std::endl << cmd << std::endl << std::endl;

			Glib::spawn_command_line_async(cmd);
		}
	}
	else
	{
		// error
	}
	debug_msg("SubtitleEditor::on_play_movie > finish");
}

/*
 *
 */
void SubtitleEditor::on_move_subtitle()
{
	debug_msg("SubtitleEditor::on_move_subtitle");

	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
		return;

	SubtitleColumnRecorder m_column;
	
	SubtitleTime start((*iter)[m_column.start]);

	m_dialogMoveSubtitle.set_value(start);
	m_dialogMoveSubtitle.show();

	if(m_dialogMoveSubtitle.run() == Gtk::RESPONSE_OK)
	{
		SubtitleTime diff = m_dialogMoveSubtitle.get_value() - start;
		getDocument()->m_subtitleModel->move_all((*iter)[m_column.num], diff.totalmsecs);
	}
	m_dialogMoveSubtitle.hide();
}

void SubtitleEditor::on_move_subtitle_start_plus()
{
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
		return;

	SubtitleColumnRecorder m_column;
	
	SubtitleTime start((*iter)[m_column.start]);
	
	start = start + SubtitleTime(0,0,0,100);
	
	(*iter)[m_column.start] = start.str();

	Signal::getInstance().subtitle_time_changed(iter);
}

void SubtitleEditor::on_move_subtitle_start_minus()
{
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
		return;

	SubtitleColumnRecorder m_column;
	
	SubtitleTime start((*iter)[m_column.start]);
	
	start = start - SubtitleTime(0,0,0,100);
	
	(*iter)[m_column.start] = start.str();

	Signal::getInstance().subtitle_time_changed(iter);
}

void SubtitleEditor::on_move_subtitle_end_plus()
{
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
		return;

	SubtitleColumnRecorder m_column;
	
	SubtitleTime end((*iter)[m_column.end]);
	
	end = end + SubtitleTime(0,0,0,100);
	
	(*iter)[m_column.end] = end.str();

	Signal::getInstance().subtitle_time_changed(iter);
}

void SubtitleEditor::on_move_subtitle_end_minus()
{
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
		return;

	SubtitleColumnRecorder m_column;
	
	SubtitleTime end((*iter)[m_column.end]);
	
	end = end - SubtitleTime(0,0,0,100);
	
	(*iter)[m_column.end] = end.str();

	Signal::getInstance().subtitle_time_changed(iter);
}


/*
 * edit le subtitle select
 */
void SubtitleEditor::on_edit_subtitle()
{
#warning "fixme > Subtitle"
/*	
	Gtk::TreeIter iter = m_treeviewSubtitle->getSelected();
	if(!iter)
	{
		// si le model est vide alors on cre un par defaut
		if(getDocument()->m_subtitleModel->getSize() == 0)
		{
			Gtk::TreeIter it = getDocument()->m_subtitleModel->append();
			Subtitle sub;
			getDocument()->m_subtitleModel->refresh(it, sub);
		}
		return;
	}

	SubtitleColumnRecorder m_column;
	Subtitle select = (*iter)[m_column.subtitle];

	m_dialogEditSubtitle.set_value(select);
	
	m_dialogEditSubtitle.show();
	
	if(m_dialogEditSubtitle.run() == Gtk::RESPONSE_OK)
	{
		select = m_dialogEditSubtitle.get_value();
		(*iter)[m_column.subtitle] = select;
		getDocument()->m_subtitleModel->refresh(iter, select);
	}

	m_dialogEditSubtitle.hide();
*/
}

void SubtitleEditor::on_select_previous_subtitle()
{
	Gtk::TreeIter it = getSubtitleView()->getSelected();
	if(it)
	{
		Gtk::TreeIter previous = getDocument()->get_subtitle_model()->find_previous(it);
		if(previous)
			getSubtitleView()->select_and_set_cursor(previous);
	}
	else
	{
		it = getDocument()->get_subtitle_model()->getFirst();
		if(it)
			getSubtitleView()->select_and_set_cursor(it);
	}
}

void SubtitleEditor::on_select_next_subtitle()
{
	Gtk::TreeIter it = getSubtitleView()->getSelected();
	if(it)
	{
		Gtk::TreeIter next = getDocument()->get_subtitle_model()->find_next(it);
		if(next)
			getSubtitleView()->select_and_set_cursor(next);
	}
	else
	{
		it = getDocument()->get_subtitle_model()->getFirst();
		if(it)
			getSubtitleView()->select_and_set_cursor(it);
	}
}

/*
 *
 */
void SubtitleEditor::on_insert_before_subtitle()
{
	debug_msg("SubtitleEditor::on_insert_before_subtitle");

	Gtk::TreeIter select = m_treeviewSubtitle->getSelected();
	if(!select)
	{
		// si le model est vide alors on cre un par defaut
		if(getDocument()->m_subtitleModel->getSize() == 0)
		{
			Gtk::TreeIter it = getDocument()->m_subtitleModel->append();
			m_treeviewSubtitle->select_and_set_cursor(it);
		}
		return;
	}

	SubtitleColumnRecorder m_column;

	SubtitleTime time_start;
	Gtk::TreeIter prev = getDocument()->m_subtitleModel->find_previous(select);
	
	if(prev)
	{
		time_start = SubtitleTime((*prev)[m_column.end]);
	}

	// on recupere le temps de depart pour s'en servir comme temps de fin
	SubtitleTime time_end = SubtitleTime((*select)[m_column.start]);
	
	
	Gtk::TreeIter new_iter = getDocument()->m_subtitleModel->insertBefore(select);
	m_treeviewSubtitle->select_and_set_cursor(new_iter);

	(*new_iter)[m_column.start]	= time_start.str();
	(*new_iter)[m_column.end]		= time_end.str();
}

/*
 *
 */
void SubtitleEditor::on_insert_after_subtitle()
{
	debug_msg("SubtitleEditor::on_insert_after_subtitle");

	Gtk::TreeIter iter_select = m_treeviewSubtitle->getSelected();
	if(!iter_select)
	{
		// si le model est vide alors on cre un par defaut
		if(getDocument()->m_subtitleModel->getSize() == 0)
		{
			Gtk::TreeIter it = getDocument()->m_subtitleModel->append();
			m_treeviewSubtitle->select_and_set_cursor(it);
		}
		return;
	}

	SubtitleColumnRecorder m_column;
	
	SubtitleTime time_start, time_end;
	
	time_start = SubtitleTime((*iter_select)[m_column.end]);
	
	Gtk::TreeIter next = getDocument()->m_subtitleModel->find_next(iter_select);
	if(next)
	{
		time_end = SubtitleTime((*next)[m_column.start]);
	}
	else	// s'il n'y a pas de suivant par defaut on ajoute 2 secs
		time_end = time_start + SubtitleTime(0,0,2,0);
	
	Gtk::TreeIter new_iter = getDocument()->m_subtitleModel->insertAfter(iter_select);
	m_treeviewSubtitle->select_and_set_cursor(new_iter);
	
	(*new_iter)[m_column.start]	= time_start.str();
	(*new_iter)[m_column.end]		= time_end.str();
}

/*
 *
 */
void SubtitleEditor::on_delete_subtitle()
{
	debug_msg("SubtitleEditor::on_delete_subtitle");

	Gtk::TreeIter iter_select = m_treeviewSubtitle->getSelected();
	if(!iter_select)
		return;
	
	Gtk::TreeIter next = getDocument()->m_subtitleModel->find_next(iter_select);
	Gtk::TreeIter prev = getDocument()->m_subtitleModel->find_previous(iter_select);


	getDocument()->m_subtitleModel->remove(iter_select);

	// select next line
	if(next)
	{
		m_treeviewSubtitle->select_and_set_cursor(next);
	}
	else if(prev)
	{
		m_treeviewSubtitle->select_and_set_cursor(prev);
	}
}

/*
 *
 */
void SubtitleEditor::on_change_fps()
{
	debug_msg("SubtitleEditor::on_change_fps");

	create_dialog("dialog-change-fps", &m_dialogChangeFPS);

	m_dialogChangeFPS->execute();
}

/*
 *
 */
void SubtitleEditor::on_scale()
{
	debug_msg("SubtitleEditor::on_scale");

	create_dialog("dialog-scale", &m_dialogScale);
	
	m_dialogScale->execute();
}



/*
 * tools
 */
void SubtitleEditor::on_find()
{
	debug_msg("SubtitleEditor::on_find");

	create_dialog("dialog-find-and-replace", &m_dialogFindAndReplace);
	
	m_dialogFindAndReplace->show_replace(false);
	m_dialogFindAndReplace->execute();
}

/*
 *
 */
void SubtitleEditor::on_find_next()
{
	debug_msg("SubtitleEditor::on_find_next");

	if(m_dialogFindAndReplace == NULL)
	{
		on_find();
		return;
	}
	create_dialog("dialog-find-and-replace", &m_dialogFindAndReplace);

	m_dialogFindAndReplace->findNext();
}

/*
 *	TODO : utility.h
 */
bool find_and_replace2(Glib::ustring &source, 
		const Glib::ustring &str_find, 
		const Glib::ustring &str_replace)
{
	//g_return_if_fail(source.size());
	//g_return_if_fail(str_find.size());
	if(source.size()==0) return false;
	if(str_find.size()==0) return false;

	// on regarde si le txt rechercher ce trouve dans la source
	Glib::ustring::size_type pos = source.find(str_find, 0);

	// on regarde si il l'a trouver au moins une fois pour retrouner true
	if(pos == Glib::ustring::npos)
		return false;
	
	// tant qu'il est dans la source on le recherche
	while(pos != Glib::ustring::npos)
	{
		source.replace(pos, str_find.size(), str_replace);
		pos = source.find(str_find, pos);
	}

	return true;
}


/*
 *
 */
void SubtitleEditor::on_find_and_replace()
{
	debug_msg("SubtitleEditor::on_find_and_replace");
	
	create_dialog("dialog-find-and-replace", &m_dialogFindAndReplace);

	m_dialogFindAndReplace->show_replace(true);
	m_dialogFindAndReplace->execute();
}


/*
 *
 */
void SubtitleEditor::on_remove_line_nul()
{
	debug_msg("SubtitleEditor::on_remove_line_nul");

	SubtitleColumnRecorder m_column;
	Glib::ustring text;
	
	Gtk::TreeIter iter = getDocument()->m_subtitleModel->getFirst();
	for(; iter; )
	{
		text = (*iter)[m_column.text];
		
		if(text.size() == 0 || text == " ")
		{
			iter = getDocument()->m_subtitleModel->erase(iter);
		}
		else
			++iter;
	}
}


/*
 *	on verifie la validiter du temps
 */
void SubtitleEditor::on_check_time()
{
	debug_msg("SubtitleEditor::on_check_time");

	SubtitleColumnRecorder m_column;

	Gtk::TreeIter iter = getDocument()->m_subtitleModel->getFirst();
/*
	Gtk::TreeIter iter;
	iter = m_treeviewSubtitle->getSelected();
	//++iter;
	if(!iter || iter == getDocument()->m_subtitleModel->getLast())
		iter = getDocument()->m_subtitleModel->getFirst();
*/
	SubtitleTime start, end;
	SubtitleTime prev_start, prev_end;
	
	for(; iter; ++iter)
	{
		start = SubtitleTime((*iter)[m_column.start]);
		end = SubtitleTime((*iter)[m_column.end]);
		
		// on verifie que end et bien > a start
		if(end <= start)
		{
			m_treeviewSubtitle->select_and_set_cursor(iter);
			setStatusbar(_("Check Time Error : start > end"));
			return;
		}

		// on verifie qu'il est bien > au precedant
		if(start < prev_start)
		{
			m_treeviewSubtitle->select_and_set_cursor(iter);
			//setStatusbar(_("Check Time : This subtitle starts before the starts of the preceding subtitle"));
			setStatusbar(_("Check Time : Overlapping with previous subtitle"));
			return;
		}

		if(start < prev_end)
		{
			m_treeviewSubtitle->select_and_set_cursor(iter);
			//setStatusbar(_("Check Time : This subtitle starts before the end of the preceding subtitle"));
			setStatusbar(_("Check Time : Overlapping with previous subtitle"));
			return;
		}

		prev_start	= start;
		prev_end	= end;
	}

	setStatusbar(_("Check Time OK!"));
}

/*
 *
 */
void SubtitleEditor::on_spell_check()
{
#ifdef ENABLE_ASPELL

	debug_msg("SubtitleEditor::on_spell_check");

	DialogSpellCheck *spell = NULL;

	create_dialog("dialog-spell-check", &spell);

	spell->init(getDocument());
	
	delete spell;

#endif//ENABLE_ASPELL
}

/*
 *
 */
void SubtitleEditor::on_set_all_end_time()
{
	debug_msg("SubtitleEditor::on_set_all_end_time");

	m_dialogSetAllEndTime.show();

	if(m_dialogSetAllEndTime.run()==Gtk::RESPONSE_OK)
	{
		SubtitleTime time = m_dialogSetAllEndTime.get_time();
		SubtitleColumnRecorder m_column;
		Gtk::TreeIter it = getDocument()->m_subtitleModel->getFirst();
	
		if(m_dialogSetAllEndTime.get_type() == DialogSetAllEndTime::ADD)
		{
			for(; it; ++it)
			{
				SubtitleTime it_time_end((*it)[m_column.end]);

				it_time_end = it_time_end + time;
				(*it)[m_column.end] = it_time_end.str();
				
				// verification il n'est pas trop long avec le prochain st
				Gtk::TreeIter it_next = getDocument()->m_subtitleModel->find_next(it);
				if(it_next)
				{
					SubtitleTime next_time_start = SubtitleTime((*it_next)[m_column.start]);
					
					if(it_time_end > next_time_start)
					{
						it_time_end = next_time_start;
					}
				}
	
				// mise a jour
				(*it)[m_column.end] = it_time_end.str();
			}
		}
		else // SET
		{
			for(; it; ++it)
			{
				SubtitleTime it_time_end((*it)[m_column.start]);

				it_time_end = it_time_end + time;
				(*it)[m_column.end] = it_time_end.str();
				
				// verification il n'est pas trop long avec le prochain st
				Gtk::TreeIter it_next = getDocument()->m_subtitleModel->find_next(it);
				if(it_next)
				{
					SubtitleTime next_time_start = SubtitleTime((*it_next)[m_column.start]);
					
					if(it_time_end > next_time_start)
					{
						it_time_end = next_time_start;
					}
				}
	
				// mise a jour
				(*it)[m_column.end] = it_time_end.str();
			}
		}
	}
	
	m_dialogSetAllEndTime.hide();
	
}

/*
 *	verifie dans une table des erreurs d'encodings.
 */
void SubtitleEditor::on_check_error_encoding()
{
	debug_msg("SubtitleEditor::on_check_error_encoding");

	gchar* array[][2]={
		{"\302\234", "oe"},
		{"\302\222", "'"},
		{"\302\205", "."}, // ??
		{NULL,NULL}};

	SubtitleColumnRecorder m_column;

	bool state=false;

	Gtk::TreeIter iter = getDocument()->m_subtitleModel->getFirst();
	for(; iter; ++iter)
	{
		state = false;
		
		Glib::ustring text = (*iter)[m_column.text];
		
		for(unsigned int i=0; array[i][0] != NULL; ++i)
		{
			if(find_and_replace2(text, array[i][0], array[i][1]))
			{
				state = true;
			}
		}

		if(state)
		{
			(*iter)[m_column.text] = text;
		}

	}
}


/*
 *
 */
void SubtitleEditor::on_about()
{
	debug_msg("SubtitleEditor::on_about");

	DialogAbout about;
	about.run();
}

/*
 *
 */
void SubtitleEditor::on_style_editor()
{
	debug_msg("SubtitleEditor::on_style_editor");

	DialogStyleEditor* m_dialogStyleEditor = NULL;

	create_dialog("dialog-style-editor", &m_dialogStyleEditor);

	m_dialogStyleEditor->execute();

	delete m_dialogStyleEditor;
}


/*
 *
 */
void SubtitleEditor::on_preferences()
{
	debug_msg("SubtitleEditor::on_preferences");

	DialogPreferences* m_dialogPreferences = NULL;

	create_dialog("dialog-preferences", &m_dialogPreferences);

	m_dialogPreferences->run();

	delete m_dialogPreferences;
}


/*
 *
 */
void SubtitleEditor::on_apply_translation()
{
	SubtitleColumnRecorder m_column;

	Gtk::TreeIter iter = getDocument()->m_subtitleModel->getFirst();
	for(; iter; ++iter)
	{
		Glib::ustring translation = (*iter)[m_column.translation];
		
		if(!translation.empty())
			(*iter)[m_column.text] = translation;
	}

	setStatusbar(_("The translation was applied."));
}

void SubtitleEditor::on_split()
{
	debug_msg("SubtitleEditor::on_split");

	DialogSplit* m_dialogSplit= NULL;

	create_dialog("dialog-split", &m_dialogSplit);

	m_dialogSplit->execute();

	delete m_dialogSplit;

}

void SubtitleEditor::on_joint()
{
	DialogOpenSubtitle dialog;
	if(dialog.run() == Gtk::RESPONSE_OK)
	{
		Glib::ustring file = dialog.get_filename();
		Glib::ustring charset = dialog.getEncoding();
		
		if(getDocument()->open(file, charset, false))
		{
			setStatusbar(_("Joint \"%s\" (%s) in this document."), file.c_str(), getDocument()->getCharset().c_str());
		}
		else
		{
			setStatusbar(_("Joint: Error loading file \"%s\" (%s)."), file.c_str(), charset.c_str());
		}

		getDocument()->get_subtitle_model()->rebuild_column_num();
	}
}


/*
 *
 */
template<class T>
bool SubtitleEditor::create_dialog(const Glib::ustring &name, T **dialog)
{
	debug_msg(message("create_dialog %s", name.c_str()));

	if(*dialog == NULL)
	{
		// TODO : try catch...
		Glib::RefPtr<Gnome::Glade::Xml> refXml = Gnome::Glade::Xml::create(get_share_dir(name + ".glade"));
	
		refXml->get_widget_derived(name, *dialog);

		return true;
	}
	return true;
}

