/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include "SubtitleASS.h"
#include "Color.h"
#include "utility.h"

#include <map>
#include <sstream>
#include <iostream>
#include <fstream>
#include <string>
#include <iomanip>


/*
 *
 */
Glib::ustring SubtitleASS::get_name()
{
	return "Advanced Sub Station Alpha";
}

/*
 *
 */
Glib::ustring SubtitleASS::get_extension()
{
	return "ass";
}

/*
 *
 */
bool SubtitleASS::check(const std::string &line)
{
	static RegEx ex("^ScriptType:\\s+(v|V)4.00\\+");

	return ex.exec(line);

}


/*
 * HACK ! recrire a !
 */

bool str2int(const std::string &str);
std::string bool2str(const bool &boolean);


bool SubtitleASS::str2bool(const std::string &str)
{
	//std::cout << "str2bool :[" << str << "]" << std::endl;
	if(str=="0")
		return false;
	return true;
}


/*
 *	hack !
 */
Glib::ustring SubtitleASS::clean_style_name(const Glib::ustring &name)
{
	Glib::ustring str = name;
	Glib::ustring::size_type n = str.find('*');
	if(n != Glib::ustring::npos)
		str.erase(n,1);

	return str;
}


/*
 *	recupere dans un tableau le format a partir de la line
 */
std::vector<std::string> SubtitleASS::build_format(const std::string &text, int column, bool remove_space)
{
	std::string line;

	if(remove_space)
	{
		for(unsigned int i=7; i<text.size(); ++i)
		{
			if(text[i]==' ')
				;
			else
				line+=text[i];
		}
	}
	else
		line = text;

	std::vector<std::string> array;

	std::string::size_type s=0, e=0;

	int i=0;
	do
	{
		if(column > 0 && i+1 == column)
		{
			array.push_back(line.substr(s,std::string::npos));
			break;
		}
		e = line.find(",", s);
		
		array.push_back(line.substr(s,e-s));

		s=e+1;

		++i;
	}while(e != std::string::npos);

	return array;
}






/*
 *	construtor
 */
SubtitleASS::SubtitleASS(Document* doc)
: m_scriptInfo(&doc->m_scriptInfo), m_model(doc->m_subtitleModel), m_style_model(doc->m_styleModel)
{
}

/*
 *
 */
SubtitleASS::~SubtitleASS()
{
}

/*
 *	read subtitle file
 */
bool SubtitleASS::open(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	debug_msg("SubtitleASS::open > start");
	
	SubtitleFormat::open(filename, encoding);
	
	std::ifstream file(filename.c_str());
	
	if(!file)
	{
		throw SubtitleException("SubtitleASS", _("I can't open this file."));
	}

	std::string line;
	
	while(!file.eof() && std::getline(file, line))
	{
		if(line.find("[Script Info]") != std::string::npos)
		{
			readScripInfo(file);
		}
		else if(line.find("[V4+ Styles]") != std::string::npos)
		{
			readStyles(file);
		}
		else if(line.find("[Events]") != std::string::npos)
		{
			readEvents(file);
		}
	}

	file.close();

	debug_msg("SubtitleASS::open > finish");
	return true;
}

bool SubtitleASS::save(const Glib::ustring &filename, const Glib::ustring &encoding)
{
	SubtitleFormat::save(filename, encoding);
	
	std::ofstream file(filename.c_str());

	if(!file)
	{
		throw SubtitleException("SubtitleASS", _("I can't open this file."));
	}

	debug_msg("SubtitleASS::save > start");

	debug_msg("SubtitleASS::save > ScriptInfo");
	
	// ScriptInfo
	{
		file << utf8_to_charset("[Script Info]") << std::endl;
		file << utf8_to_charset("; This script is create by subtitleeditor ") << utf8_to_charset(VERSION) << std::endl;
		file << utf8_to_charset("; http://kitone.free.fr/subtitleeditor/") << std::endl;
	
#define CHECK(label, key) if(m_scriptInfo->key.size() > 0) { file << utf8_to_charset(label) << utf8_to_charset(m_scriptInfo->key) << std::endl; }

		m_scriptInfo->ScriptType="V4.00+";

		CHECK("Title: ", Title);
		CHECK("Original Script: ", OriginalScript);
		CHECK("Original Translation: ", OriginalTranslation);
		CHECK("Original Editing: ", OriginalEditing);
		CHECK("Synch Point: ", SynchPoint);
		CHECK("Script Updated By: ", ScriptUpdatedBy);
		CHECK("Update Details: ", UpdateDetails);
		CHECK("ScriptType: ", ScriptType);
		CHECK("Collisions: ", Collisions);
		CHECK("PlayResX: ", PlayResX);
		CHECK("PlayResY: ", PlayResY);
		CHECK("Timer: ", Timer);
		CHECK("Style: ", Style);
		CHECK("Dialogue: ", Dialogue);
		CHECK("Comment: ", Comment);
		CHECK("Picture: ", Picture);
		CHECK("Sound: ", Sound);
		CHECK("Movie: ", Movie);
		
#undef CHECK

		file << std::endl;
	}

	
	debug_msg("SubtitleASS::save > Style");

	// Style
	{
		StyleColumnRecorder column;
		
		file << "[V4+ Styles]" << std::endl;

		file << "Format: "
				<< "Name, "
				<< "Fontname, "
				<< "Fontsize, "
				
				<< "PrimaryColour, "
				<< "SecondaryColour, "
				<< "OutlineColour, "
				<< "BackColour, "
				
				<< "Bold, "
				<< "Italic, "

				<< "Underline, "
				<< "Strikeout, "
				<< "ScaleX, "
				<< "ScaleY, "
				<< "Spacing, "
				<< "Angle, "

				
				<< "BorderStyle, "
				<< "Outline, "
				<< "Shadow, "
				<< "Alignment, "
				<< "MarginL, "
				<< "MarginR, "
				<< "MarginV, "
				<< "Encoding" << std::endl;

		Gtk::TreeNodeChildren rows = m_style_model->children();
		
		Color primaryColour, secondaryColour, outlineColour, shadowColour;
		
		for(Gtk::TreeIter it = rows.begin(); it; ++it)
		{
			std::ostringstream oss;

			primaryColour.set((*it)[column.primary_colour]);
			secondaryColour.set((*it)[column.secondary_colour]);
			outlineColour.set((*it)[column.outline_colour]);
			shadowColour.set((*it)[column.shadow_colour]);

			oss << "Style: " 
				<< (*it)[column.name] << "," 
				<< (*it)[column.font_name] << "," 
				<< (*it)[column.font_size] << "," 

				<< primaryColour.getRGBA() << "," 
				<< secondaryColour.getRGBA() << "," 
				<< outlineColour.getRGBA() << "," 
				<< shadowColour.getRGBA() << "," 
				
				<< bool2str((*it)[column.bold]) << "," 
				<< bool2str((*it)[column.italic]) << "," 
				<< bool2str((*it)[column.underline]) << "," 
				<< bool2str((*it)[column.strikeout]) << "," 
				
				<< (*it)[column.scale_x] << "," 
				<< (*it)[column.scale_y] << "," 
				<< (*it)[column.spacing] << "," 
				<< (*it)[column.angle] << "," 
				
				<< (*it)[column.border_style] << "," 
				<< (*it)[column.outline] << "," 
				<< (*it)[column.shadow] << "," 
				<< (*it)[column.alignment] << "," 
				<< (*it)[column.margin_l] << "," 
				<< (*it)[column.margin_r] << "," 
				<< (*it)[column.margin_v] << "," 
				<< (*it)[column.encoding] << std::endl; 
				
			file << utf8_to_charset(oss.str());
		}

		file << std::endl;
	}

	debug_msg("SubtitleASS::save > Event");

	// Event
	{
		file << "[Events]" << std::endl;
		// format:
		file << 
			"Format: " <<
			"Layer, " <<
			"Start, " <<
			"End, " <<
			"Style, " <<
			"Name, " <<
			"MarginL, " <<
			"MarginR, " <<
			"MarginV, " <<
			"Effect, " <<
			"Text" << std::endl;
			
		// dialog:
		SubtitleColumnRecorder column;
		Gtk::TreeNodeChildren rows = m_model->children();
		
		for(Gtk::TreeIter it = rows.begin(); it; ++it)
		{
			std::ostringstream oss;

			oss << "Dialogue: "
				<< (*it)[column.layer] << "," 
				<< convert_time_for_save((*it)[column.start]) << "," // convertir , en .
				<< convert_time_for_save((*it)[column.end]) << "," // convertir , en .
				<< utf8_to_charset((*it)[column.style]) << ","
				<< utf8_to_charset((*it)[column.name]) << ","
				<< std::setw(4) << std::setfill('0') << (*it)[column.marginL] << ","
				<< std::setw(4) << std::setfill('0') << (*it)[column.marginR] << ","
				<< std::setw(4) << std::setfill('0') << (*it)[column.marginV] << ","
				<< utf8_to_charset((*it)[column.effect]) << ","
				<< utf8_to_charset((*it)[column.text]) << std::endl;

			file << oss.str();
		}
	}
	
	file.close();

	debug_msg("SubtitleASS::save > finish OK");

	return true;
}

/*
 *	READ BLOCK
 */
	
/*
 *
 */
bool SubtitleASS::readScripInfo(std::ifstream &file)
{
	debug_msg("SubtitleASS::readScripInfo start");

	std::string line;
	while(!file.eof() && std::getline(file, line))
	{
		if(line.size() < 3)
		{
			break;
		}

#define CHECK(label, key) \
		if(line.find(label) != std::string::npos) \
		{	line.erase(0, (std::string(label)).size()); m_scriptInfo->key = check_end_char(charset_to_utf8(line)); }
//		else std::cerr << "CHECK not found: " << label << std::endl;

		CHECK("Title: ", Title);
		CHECK("Original Script: ", OriginalScript);
		CHECK("Original Translation: ", OriginalTranslation);
		CHECK("Original Editing: ", OriginalEditing);
		CHECK("Synch Point: ", SynchPoint);
		CHECK("Script Updated By: ", ScriptUpdatedBy);
		CHECK("Update Details: ", UpdateDetails);
		CHECK("ScriptType: ", ScriptType);
		CHECK("Collisions: ", Collisions);
		CHECK("PlayResX: ", PlayResX);
		CHECK("PlayResY: ", PlayResY);
		CHECK("Timer: ", Timer);
		CHECK("Style: ", Style);
		CHECK("Dialogue: ", Dialogue);
		CHECK("Comment: ", Comment);
		CHECK("Picture: ", Picture);
		CHECK("Sound: ", Sound);
		CHECK("Movie: ", Movie);

#undef CHECK
	}
	
	debug_msg("SubtitleASS::readScripInfo finish");

	return true;
}

/*
 *	read Style
 */
bool SubtitleASS::readStyles(std::ifstream &file)
{
	debug_msg("SubtitleASS::readStyle start");

	std::vector<std::string> formats;
	std::map<std::string, unsigned int> map; 
	
	std::string line;

	
	Color color;
	
	while(!file.eof() && std::getline(file, line))
	{
		if(line.size() < 3)
			break;//return true;

		if(line.find("Style: ") != std::string::npos)
		{
			StyleColumnRecorder column;
			// on donne la ligne sans "Style: " = size - 7
			std::vector<std::string> fmt = build_format(line.substr(7, line.size()-7), formats.size(), false);

			Gtk::TreeIter it = m_style_model->append();

			// on supprime '*' si il existe
			(*it)[column.name]							= clean_style_name( fmt[ map["Name"] ] );
			
			(*it)[column.font_name]					= fmt[ map["Fontname"] ];
			(*it)[column.font_size]					= str2int( fmt[ map["Fontsize"] ] );

			// color
			color.set( str2int( fmt[ map["PrimaryColour"] ]));
			(*it)[column.primary_colour] = color.getRGBA();

			color.set( str2int( fmt[ map["SecondaryColour"] ]));
			(*it)[column.secondary_colour] = color.getRGBA();

			color.set ( str2int( fmt[ map["OutlineColour"] ]));
			(*it)[column.outline_colour] = color.getRGBA();
			
			color.set( str2int( fmt[ map["BackColour"] ]));
			(*it)[column.shadow_colour] = color.getRGBA();

	
			
			(*it)[column.bold]							= str2bool( fmt[ map["Bold"] ] );
			(*it)[column.italic]						= str2bool( fmt[ map["Italic"] ] );

			(*it)[column.underline]					= str2bool( fmt[ map["Underline"] ] );
			(*it)[column.strikeout]					= str2bool( fmt[ map["Strikeout"] ] );

			(*it)[column.scale_x]						= str2int( fmt[ map["ScaleX"] ] );
			(*it)[column.scale_y]						= str2int( fmt[ map["ScaleY"] ] );

			(*it)[column.spacing]						= str2int( fmt[ map["Spacing"] ] );
			(*it)[column.angle]							= str2int( fmt[ map["Angle"] ] );
		
			(*it)[column.border_style]			= str2int( fmt[ map["BorderStyle"] ] );
			(*it)[column.outline]						= str2int( fmt[ map["Outline"] ] );

			(*it)[column.shadow]						= str2int( fmt[ map["Shadow"] ]);
			(*it)[column.alignment]					= str2int( fmt[ map["Alignment"] ]);
			(*it)[column.margin_l]					= str2int( fmt[ map["MarginL"] ]);
			(*it)[column.margin_r]					= str2int( fmt[ map["MarginR"] ]);
			(*it)[column.margin_v]					= str2int( fmt[ map["MarginV"] ]);
			
			(*it)[column.encoding]					= str2int( fmt[ map["Encoding"] ]);
	
		}
		else if(line.find("Format: ") != std::string::npos)
		{
			formats = build_format(line, -1, true);

			for(unsigned int i=0; i<formats.size(); ++i)
				map[formats[i]] = i;
		}
		
	}
	debug_msg("SubtitleASS::readStyle finish");
	return true;
}

/*
 *
 */
bool SubtitleASS::readEvents(std::ifstream &file)
{
	debug_msg("SubtitleASS::readEvents start");
	
	unsigned int num = 1;
	
	std::string line;
	while(!file.eof() && std::getline(file, line))
	{
		if(line.size() < 3)
			break;

		if(/*std::string::size_type n=*/line.find("Dialogue: ") != std::string::npos)
		{
			line.erase(0,10);

			std::vector<std::string> array = build(line, 10);

			Gtk::TreeIter it = m_model->append();
			SubtitleColumnRecorder column;
			
			(*it)[column.num] = num;
			// marked/layer
			(*it)[column.layer]		= array[0];
			(*it)[column.start]		= convert_time_for_subtitleeditor(array[1]);
			(*it)[column.end]			= convert_time_for_subtitleeditor(array[2]);


			(*it)[column.style]		= clean_style_name( charset_to_utf8(array[3]) );
			(*it)[column.name]		= charset_to_utf8(array[4]);
			
			(*it)[column.marginL]	= array[5];
			(*it)[column.marginR]	= array[6];
			(*it)[column.marginV]	= array[7];

			(*it)[column.effect]	= charset_to_utf8(array[8]);

			(*it)[column.text]		= check_end_char(charset_to_utf8(array[9]));

			++num;
		}
	}
	debug_msg("SubtitleASS::readEvents finish");
	return true;
}


/*
 *
 */
std::vector< std::string > SubtitleASS::build(const std::string &line, unsigned int column)
{
	std::vector< std::string > array;

	std::string::size_type s=0, e=0;

	do
	{
		if(column > 0 && array.size()+1 == column)
		{
			array.push_back(line.substr(s,std::string::npos));
			break;
		}

		e = line.find(",", s);
		
		array.push_back(line.substr(s,e-s));

		s=e+1;
	}while(e != std::string::npos);
	return array;
}


/*
 *	convertir le temps utiliser par subtitle editor en tant valide pour le format SSA
 *	h:m:s,ms -> h:m:s.ms
 */
Glib::ustring SubtitleASS::convert_time_for_save(const Glib::ustring &time)
{
	return time;
}

/*
 *	hours:mins:secs.msecs -> hours:mins:secs,msecs
 */
Glib::ustring SubtitleASS::convert_time_for_subtitleeditor(const Glib::ustring &text)
{
	if(SubtitleTime::validate(text))
		return SubtitleTime(text).str();
	
	return "";
}

