/*
 *	subtitle editor
 *
 *	http://kitone.free.fr/subtitleeditor/
 *
 *	Copyright @ 2005-2006, kitone
 *
 *	Contact: kitone at free dot fr
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public
 *	License along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA	02111-1307	USA
 *
 *	See gpl.txt for more information regarding the GNU General Public License.
 *
 *
 *	\file
 *	\brief 
 *	\author kitone (kitone at free dot fr)
 */

#include "Dialog.h"
#include <gtkmm/stock.h>
#include <gtkmm/comboboxtext.h>
#include <iostream>
#include <sstream>
#include <gtkmm.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "utility.h"
#include "Color.h"
#include "Encodings.h"
#include "Config.h"
#include "SubtitleEditor.h"
#include "SubtitleSystem.h"

#include <libglademm/xml.h>

/*
 *	HIG for dialog	
 */
void setDialog(Gtk::Dialog &dialog)
{
	dialog.set_border_width(6);
	dialog.set_resizable (false);
	dialog.set_has_separator (false);
	dialog.set_skip_taskbar_hint (true);
	dialog.get_vbox ()->set_spacing (8);
}


/*
 * GtkSubtitleTimer
 */
GtkSubtitleTime::GtkSubtitleTime()
:Gtk::Table(1,8,true),
 m_labelHours(_("Hours"),0.0,0.5),
 m_labelMins(_("Mins"),0.0,0.5),
 m_labelSecs(_("Secs"),0.0,0.5),
 m_labelMSecs(_("MSecs"),0.0,0.5)
{
	set_col_spacings(2);
 
	attach(m_spinHours, 0,1,0,1);
	attach(m_labelHours, 1,2,0,1);
	
	attach(m_spinMins, 2,3,0,1);
	attach(m_labelMins, 3,4,0,1);
	
	attach(m_spinSecs, 4,5,0,1);
	attach(m_labelSecs, 5,6,0,1);
	
	attach(m_spinMSecs, 6,7,0,1);
	attach(m_labelMSecs, 7,8,0,1);
	
	m_spinHours.set_range(0,24);
	m_spinHours.set_increments(1,1);

	m_spinMins.set_range(0,60);
	m_spinMins.set_increments(1,1);

	m_spinSecs.set_range(0,60);
	m_spinSecs.set_increments(1,1);

	m_spinMSecs.set_range(0,999);
	m_spinMSecs.set_increments(1,1);

	show_all();
}

void GtkSubtitleTime::set_value(int h, int m, int s, int ms)
{
	m_spinHours.set_value(h);
	m_spinMins.set_value(m);
	m_spinSecs.set_value(s);
	m_spinMSecs.set_value(ms);
}
	
void GtkSubtitleTime::set_value(const SubtitleTime &time)
{
	m_spinHours.set_value(time.hours);
	m_spinMins.set_value(time.mins);
	m_spinSecs.set_value(time.secs);
	m_spinMSecs.set_value(time.msecs);
}

SubtitleTime GtkSubtitleTime::get_value()
{
	int h = m_spinHours.get_value_as_int();
	int m = m_spinMins.get_value_as_int();
	int s = m_spinSecs.get_value_as_int();
	int ms= m_spinMSecs.get_value_as_int();
	
	return SubtitleTime(h, m, s, ms);
}






/*
 *
 */
void loadConfigFloder(Gtk::FileChooser *chooser, const Glib::ustring &name)
{
	Config &cfg = Config::getInstance();

	Glib::ustring floder;
	if(cfg.get_value_string("dialog-last-folder", name, floder))
		chooser->set_current_folder_uri(floder);
}

/*
 *
 */
void saveConfigFloder(Gtk::FileChooser *chooser, const Glib::ustring &name)
{
	Config &cfg = Config::getInstance();

	Glib::ustring floder = chooser->get_current_folder_uri();
	
	cfg.set_value_string("dialog-last-folder", name, floder);
}


/*
 *
 */
void DialogOpenFileInitFilter(DialogFileChooser *dialog)
{
	std::list<Glib::ustring> formats = SubtitleSystem::getInstance().get_formats();

	std::list<Glib::ustring>::const_iterator it;
	
	// all supported formats
	Glib::ustring all_supported;
	for(it=formats.begin(); it!=formats.end(); ++it)
	{
		Glib::ustring ext = SubtitleSystem::getInstance().get_extension(*it);
		all_supported += "*."+ext+";";
	}
	dialog->addFilter("All supported formats (*.ass, *.ssa, *.srt, ...)", all_supported);

	// format
	for(it=formats.begin(); it!=formats.end(); ++it)
	{
		Glib::ustring ext = "*." + SubtitleSystem::getInstance().get_extension(*it);
		
		Glib::ustring name = (*it) + " (" + ext + ")";

		dialog->addFilter(name, ext);
	}

	// all files
	dialog->addFilter("All files (*.*)", "*.*");
}


/*
 *	DIALOG OPEN SRT
 */
DialogOpenSubtitle::DialogOpenSubtitle()
:DialogFileChooser(_("Open Subtitle"))
{
	DialogOpenFileInitFilter(this);

	loadConfigFloder(this, "dialog-open-subtitle");
}

DialogOpenSubtitle::~DialogOpenSubtitle()
{
	saveConfigFloder(this, "dialog-open-subtitle");
}

/*
 *	DIALOG SAVE SRT
 */
DialogSaveSubtitle::DialogSaveSubtitle()
:DialogFileChooser(_("Save Subtitle"), Gtk::FILE_CHOOSER_ACTION_SAVE)
{
	DialogOpenFileInitFilter(this);
	
	loadConfigFloder(this, "dialog-save-subtitle");
}

DialogSaveSubtitle::~DialogSaveSubtitle()
{
	saveConfigFloder(this, "dialog-save-subtitle");
}

/*
 *	DIALOG IMPORT TXT
 */
DialogImportText::DialogImportText()
:DialogFileChooser(_("Import Text"))
{
}


/*
 *	DIALOG Export TXT
 */
DialogExportText::DialogExportText()
:DialogFileChooser(_("Export Text"), Gtk::FILE_CHOOSER_ACTION_SAVE, false)
{
}

/*
 *	DIALOG OPEN MOVIE
 */
DialogOpenMovie::DialogOpenMovie()
:Gtk::FileChooserDialog(_("Open Movie"))
{
	Gtk::FileFilter m_filter;
	m_filter.set_name("Movies");
	/*
	m_filter.add_pattern("*.avi");
	m_filter.add_pattern("*.mpg");
	m_filter.add_pattern("*.wma");
	m_filter.add_pattern("*.mkv");
	m_filter.add_pattern("*.mpeg");
	*/
	m_filter.add_mime_type("video/*");
	//...
	add_filter(m_filter);

	Gtk::FileFilter m_filterAll;
	m_filterAll.set_name("ALL");
	m_filterAll.add_pattern("*.*");
	add_filter(m_filterAll);
	
	add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
	set_default_response(Gtk::RESPONSE_OK);
	
	loadConfigFloder(this, "dialog-open-movie");
}

DialogOpenMovie::~DialogOpenMovie()
{
	saveConfigFloder(this, "dialog-open-movie");
}

/*
 *	DIALOG OPEN Media
 */
DialogOpenMedia::DialogOpenMedia()
:Gtk::FileChooserDialog(_("Open Media"))
{
	Gtk::FileFilter m_filter;
	m_filter.set_name("Media, Waveform (*.wf)");
	m_filter.add_mime_type("audio/*");
	m_filter.add_mime_type("video/*");
	m_filter.add_pattern("*.wf");
	add_filter(m_filter);

	
	Gtk::FileFilter m_filterAll;
	m_filterAll.set_name("ALL");
	m_filterAll.add_pattern("*.*");
	add_filter(m_filterAll);
	
	add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
	set_default_response(Gtk::RESPONSE_OK);
	
	loadConfigFloder(this, "dialog-open-media");
}

DialogOpenMedia::~DialogOpenMedia()
{
	saveConfigFloder(this, "dialog-open-media");
}


/*
 *	DIALOG Save Waveform
 */
DialogSaveWaveform::DialogSaveWaveform()
:Gtk::FileChooserDialog(_("Save Waveform"), Gtk::FILE_CHOOSER_ACTION_SAVE)
{
	add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
	set_default_response(Gtk::RESPONSE_OK);
	
	loadConfigFloder(this, "dialog-save-waveform");
}

DialogSaveWaveform::~DialogSaveWaveform()
{
	saveConfigFloder(this, "dialog-save-waveform");
}

/*
 *	DIALOG Move Subtitle
 */
DialogMoveSubtitle::DialogMoveSubtitle(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade)
:Gtk::Dialog(cobject)
{
	//refGlade->get_widget("spinSource1", m_spinSource1);
}

DialogMoveSubtitle::DialogMoveSubtitle()
:Gtk::Dialog(_("Move Subtitle")),
	m_table(4,2,false), 
	//m_labelDest(_("How much time should be added ?"), 0.0,0.5),
	m_labelDest(_("Enter time that you want:"), 0.0,0.5),
	m_labelOption(_("Which subtitle ?"), 0.0,0.5),
	m_radioAll(_("All subtitles")),
	m_radioCurrent(_("Current line"))
{
	setDialog(*this);

	get_vbox()->pack_start(m_table);

	m_table.set_row_spacings(5);
	
	m_table.attach(m_labelDest, 0,2,0,1);
	m_table.attach(m_timer, 0,2,1,2, 
			Gtk::FILL | Gtk::EXPAND,
			Gtk::FILL | Gtk::EXPAND,
			12,0);

	add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
	set_default_response(Gtk::RESPONSE_OK);

	m_table.show_all();
}

void DialogMoveSubtitle::set_value(const SubtitleTime &start)
{
	m_timer.set_value(start);
}

SubtitleTime DialogMoveSubtitle::get_value()
{
	return m_timer.get_value();
}



/*
 *	DIALOG set all end time
 */
DialogSetAllEndTime::DialogSetAllEndTime()
:Gtk::Dialog(_("Set All End Time")), 
	m_table(5,2, false),
	m_labelRadio(_("<b>Option:</b>"),0,.5),
	m_radioAdd(_("Add Time")), 
	m_radioSet(_("Set Time")),
	m_label(_("<b>Please enter time for added or initialized:</b>"),0.0,0.5)
{
	setDialog(*this);

	
	m_label.set_use_markup(true);
	m_labelRadio.set_use_markup(true);
	
	m_table.set_row_spacings(5);
	m_table.set_row_spacing(1,12);

	Gtk::RadioButtonGroup group = m_radioAdd.get_group();

	m_radioSet.set_group(group);
	
	
	get_vbox()->add(m_table);

	m_table.attach(m_label, 0,2,0,1);
	m_table.attach(m_time, 0,2,1,2,
			Gtk::FILL | Gtk::EXPAND,
			Gtk::FILL | Gtk::EXPAND,
			12,0);


	m_table.attach(m_labelRadio, 0,2,2,3);
	m_table.attach(m_radioAdd, 0,1,3,4, 
			Gtk::FILL | Gtk::EXPAND,
			Gtk::FILL | Gtk::EXPAND,
			12,0);
	m_table.attach(m_radioSet, 0,1,4,5,
			Gtk::FILL | Gtk::EXPAND,
			Gtk::FILL | Gtk::EXPAND,
			12,0);


	add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
	add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
	set_default_response(Gtk::RESPONSE_OK);

	m_table.show_all();
}

DialogSetAllEndTime::Type DialogSetAllEndTime::get_type()
{
	if(m_radioAdd.get_active())
		return ADD;
	else 
		return SET;
}

SubtitleTime DialogSetAllEndTime::get_time()
{
	return m_time.get_value();
}



/*
 *
 */
DialogVideoProperties::DialogVideoProperties(BaseObjectType* cobject, const Glib::RefPtr<Gnome::Glade::Xml>& refGlade)
:Gtk::Dialog(cobject)
{
	refGlade->get_widget("label-title", m_label_title);
	refGlade->get_widget("label-duration", m_label_duration);
	refGlade->get_widget("label-dimensions", m_label_dimensions);
	refGlade->get_widget("label-video-codec", m_label_video_codec);
	refGlade->get_widget("label-framerate", m_label_framerate);
	refGlade->get_widget("label-audio-codec", m_label_audio_codec);
}

void DialogVideoProperties::execute()
{
	run();
}

void DialogVideoProperties::info_update()
{
	std::cout << "info update" << std::endl;

	const VideoInfo& info = get_info();

	m_label_title->set_text(info.title);
	m_label_duration->set_text(info.lenght);
	m_label_dimensions->set_text(to_string(info.width) + " x " + to_string(info.height));
	m_label_framerate->set_text(to_string(info.framerate));
}
