use ui;
use core:io;
use progvis:program;
use progvis:data;

/**
 * Root class for the model checker.
 *
 * Contains all state needed for the search of program states.
 *
 * Call 'check' for a simple interface to the entire checking progress.
 *
 * TODO: One interesting approach for getting an estimate of the total time required for checking is
 * to find the maximum execution depth by exexuting one strand to the end, and then use the fanout
 * from that to estimate the total search space.
 */
class Checker on Render {
	// Create.
	init(Url[] files, Progress callback) {
		Program program = Program:load(files);
		Node n(null);

		init {
			program = program;
			world = World(program);
			callback = callback;
			root = n;
			current = n;
			pathDepth = 10;
			exhaustive = false;
		}

		// Don't do sticky threads.
		world.stickyThreads = false;
		program.onError = &this.onProgramError;
	}

	// Program we are checking.
	private Program program;

	// World representing the data in the program.
	private World world;

	// Callback for progress.
	private Progress callback;

	// Root node of the graph. This represents the program state just after starting the program.
	private Node root;

	// Node that represents the current state of the program.
	private Node current;

	// Lookup for states. Finds a node representing a particular state.
	private Str->Node lookup;

	// Number of states yet to explore.
	private Nat toExplore;

	// Total number of edges.
	private Nat edges;

	// Maximum depth of the graph so far.
	private Nat maxDepth;

	// Max number of times we will traverse a single "path" in the graph. We want this fairly large
	// so that we don't have to restart the program very often, but if it is too large, we will miss
	// the opportunity to explore "easy" solutions.
	private Nat pathDepth;

	// Program errors encountered recently.
	private ProgramError? error;

	// Should we make an exhaustive search?
	private Bool exhaustive;

	// Check the loaded program.
	void check() {
		try {
			reset();

			traverse();

			// Terminate when we are done.
			program.terminate();
			world.clear();
		} catch (Exception e) {
			program.terminate();
			world.clear();
			throw e;
		}
	}

	// Find an error. Simply run a BFS until we find an edge with an error.
	Result? result() {
		unless (edge = findError())
			return null;

		// We know that this is true.
		unless (edgeError = edge.error)
			return null;

		try {
			Action[] actions;
			Nat[] path = edge.child.path();

			if (path.empty)
				Result(edgeError, []);

			// Go through the program and resolve the relative thread ID:s. We don't need to keep track
			// of the world here.
			world.clear();
			toStart(program);

			for (Nat i = 0; i < path.count - 1; i++) {
				var t = findSeqThread(program, path[i]);
				if (t.barriersAvailable) {
					actions << Action(t.threadId, true);
					t.resumeBarrier();
				} else {
					actions << Action(t.threadId, false);
					t.resume();
				}

				wait(program);
			}

			// Step the thread in single-step mode until we encounter the error.
			error = null;
			var t = findSeqThread(program, path.last);
			while (t.alive & !t.sleeping) {
				t.resume();
				wait(program);

				actions << Action(t.threadId, false);

				if (e = world.update())
					break;
				if (e = error)
					break;
			}

			program.terminate();
			world.clear();

			Result(edgeError, actions);

		} catch (Exception e) {
			program.terminate();
			world.clear();
			throw e;
		}
	}

	// Find an edge containing an error.
	private Edge? findError() {
		Queue<Node> work;
		Set<Node> traversed;

		work.push(root);
		while (work.any) {
			Node c = work.top; work.pop();
			for (e in c.edges) {
				if (error = e.error)
					return e;

				if (!traversed.has(e.child)) {
					traversed.put(e.child);
					work.push(e.child);
				}
			}
		}

		null;
	}

	// Prepare for checking.
	private void reset() {
		lookup.clear();
		world.clear();
		toStart(program);

		root = getNode(null);
		current = root;
	}

	// Traverse the state space. We try to minimize program restarts, so our strategy is to explore
	// a particular path as far as possible (or until some maximum depth). Then we pick another one
	// (by a BFS search) and update that.
	private void traverse() {
		do {
			// Start by traversing "current" for a while.
			traversePath();

			// Find a new, shallow point to continue traversal. If we can't find one, we are done.
			unless (next = findUnexplored())
				break;

			// If the node has an error edge, we are done.
			if (!exhaustive)
				if (next.hasError)
					return;

			// Update the program execution to match where we wish to be.
			updateCurrent(next);
		}
	}

	// Traverse "current" as far as possible, or until we hit the specified "path length".
	private void traversePath() {
		Nat startDepth = current.depth;
		while (current.unexplored.any & current.depth < startDepth + pathDepth) {
			Nat action = current.unexplored.top;
			current.unexplored.pop();
			toExplore--;

			error = null;
			toNext(program, action);

			Node n = getNode(current);
			current.edges << Edge(action, n, error);
			edges++;
			maxDepth = maxDepth.max(current.depth);
			current = n;
		}

		reportProgress();
	}

	// Find a node with at least one unexplored edge. If an error is known to exist at a depth
	// shorter than any unexplored edges, we return the node containing an error instead. If
	// everything is explored, null is returned.
	private Node? findUnexplored() {
		Queue<Node> work;
		Set<Node> traversed;

		work.push(root);
		while (work.any) {
			Node c = work.top; work.pop();
			if (c.unexplored.any)
				return c;

			for (e in c.edges) {
				if (!exhaustive)
					if (e.error.any)
						return c;

				if (!traversed.has(e.child)) {
					traversed.put(e.child);
					work.push(e.child);
				}
			}
		}

		null;
	}

	// Update 'current' to the desired node. This also means restarting the program and running it
	// to the proper location.
	private void updateCurrent(Node to) {
		world.clear(); // also restarts the program.
		toNode(program, to);

		// Sanity checks that could be enabled during debugging.
		if (true) {
			world.update();
			var found = lookup.find(world.summary);
			if (found == lookup.end) {
				throw CheckError("Failed to replay the program into the desired state.");
			} else if (found.v !is to) {
				throw CheckError("Failed to replay the program into the desired state.");
			}
		}

		current = to;
	}

	// Create a node for the current state.
	private Node getNode(Node? parent) {
		// Note: We let concurrency issues take priority over exceptions.
		if (error = world.update())
			this.error = error;

		Str key = world.summary();

		var found = lookup.find(key);
		if (found == lookup.end) {
			Node created(parent, world);
			lookup.put(key, created);
			toExplore += created.unexplored.count;
			created;
		} else {
			Node n = found.v;
			// Set new parent if needed.
			if (parent)
				n.parent(parent);
			n;
		}
	}

	// Report progress.
	private void reportProgress() {
		callback.progress(maxDepth, lookup.count, edges, toExplore);
	}

	// Called when an error is thrown in the program.
	private void onProgramError(ThreadError error) {
		// Only store the first error.
		if (this.error.empty)
			this.error = error;
	}
}

// Check a program for concurrency errors.
Result? check(Url[] files, Progress callback) on Render {
	Checker checker(files, callback);
	checker.check();
	checker.result();
}
Result? check(Url[] files) on Render {
	check(files, Progress());
}

/**
 * Progress callbacks called by the checker.
 */
class Progress on Render {
	// Called when we have new information.
	// 'states' is the current number of states.
	// 'edges' is the current number of edges.
	// 'unexplored' is the current number of edges that need to be explored.
	void progress(Nat depth, Nat states, Nat edges, Nat unexplored) {}
}

class PrintProgress extends Progress {
	void progress(Nat depth, Nat states, Nat edges, Nat unexplored) : override {
		print("Progress: ${depth} ${states} ${edges} ${unexplored}");
	}
}


/**
 * Result from the checker.
 */
class Result {
	// Create.
	init(ProgramError error, Action[] sequence) {
		init {
			error = error;
			sequence = sequence;
		}
	}

	// Error found.
	ProgramError error;

	// Sequence used to reach the error. We use "barrier steps" if available.
	Action[] sequence;

	// Output.
	void toS(StrBuf to) {
		to << "Sequence: [" << join(sequence, ", ") << "], " << error;
	}
}
