// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


/*
  infodialogs.cc
  This file contains code for the dialog boxes in the StarPlot Stars menu,
  and for the informational popup window.
*/

#include <gtk/gtk.h>
#include "starplot.h"
using std::string;

/* This function displays a popup window of information about the star: */

void my_gtk_star_popup(Star &s)
{
  GtkWidget *popup, *OK, *topbox, *label[6];
  StringList starinfo = s.GetInfo(globals::chart_rules);
  string Name, RA, Dec, Distance, Class, Mag;

  Name = starinfo[1];
  if (globals::chart_rules.StarLabels == NUMBER_LABEL)
    Name = starinfo[0] + ". " + Name;
  if (globals::chart_rules.CelestialCoords) {
    RA = "RA: " + starinfo[2];
    Dec = "Dec: " + starinfo[3];
  }
  else {
    RA = "Long: " + starinfo[2];
    Dec = "Lat: " + starinfo[3];
  }
  Distance = "Dist: " + starinfo[4] + " LY";
  Class = "Class: " + starinfo[5];
  Mag = "Abs. Magnitude: " + starinfo[6];
  
  popup = gtk_dialog_new ();
  gtk_window_set_policy (GTK_WINDOW (popup), false, false, true);
  gtk_window_set_title (GTK_WINDOW (popup), "StarPlot");
  gtk_window_set_position (GTK_WINDOW (popup), GTK_WIN_POS_MOUSE);
  
  topbox = gtk_vbox_new (true, 0);
  gtk_container_set_border_width (GTK_CONTAINER (topbox), 10);
  gtk_widget_show (topbox);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (popup)->vbox), topbox,
		      true, true, 0);
  
  label[0] = gtk_label_new (Name.c_str());
  label[1] = gtk_label_new (RA.c_str());
  label[2] = gtk_label_new (Dec.c_str());
  label[3] = gtk_label_new (Distance.c_str());
  label[4] = gtk_label_new (Class.c_str());
  label[5] = gtk_label_new (Mag.c_str());

  for (unsigned int i = 0; i < 6; i++) {
    gtk_box_pack_start (GTK_BOX (topbox), label[i], true, true, 0);
    gtk_misc_set_alignment(GTK_MISC (label[i]), 0.0F, 0.0F);
    gtk_widget_show (label[i]);
  }

  my_gtk_popup_button (&OK, GTK_DIALOG (popup)->action_area, popup);
  gtk_window_set_focus (GTK_WINDOW (popup), OK);
  gtk_widget_show (popup);
}


/* Here is a function which sets up the column list for starinfo() and
 *  starsearch().  "box" is the box into which the column list should
 *  be packed. */

void my_gtk_list_setup (GtkWidget *box, GtkWidget **clistptr,
			bool searchclist)
{
  GtkWidget *scrolled_window;
  GtkWidget *clist;
  StringList titles;

  // set column titles for the information
  unsigned int i = 0;
  titles.push_back("Star");
  if (searchclist)
    titles.push_back("Also known as");
  if (globals::chart_rules.CelestialCoords) {
    titles.push_back("Right Ascension");
    titles.push_back("Declination");
  }
  else {
    titles.push_back("Gal. Longitude");
    titles.push_back("Gal. Latitude");
  }
  titles.push_back("Distance");
  titles.push_back("Class");
  titles.push_back("Mag.");

  // set up a viewport with scroll bars
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_box_pack_start (GTK_BOX (box), scrolled_window, true, true, 0);
  gtk_widget_show (scrolled_window);

  // create a column list
  *clistptr = gtk_clist_new_with_titles (titles.size(), titles.c_str());
  clist = *clistptr;
  gtk_widget_set_usize (GTK_WIDGET (clist), 565, 300);
  gtk_container_add (GTK_CONTAINER (scrolled_window), clist);
  gtk_widget_show (clist);
  gtk_clist_set_shadow_type (GTK_CLIST (clist), GTK_SHADOW_OUT);

  i = 0;
  gtk_clist_set_column_width (GTK_CLIST (clist), i++, 120);
  if (searchclist)
    gtk_clist_set_column_width (GTK_CLIST (clist), i++, 120);
  gtk_clist_set_column_width (GTK_CLIST (clist), i++, 100);
  gtk_clist_set_column_width (GTK_CLIST (clist), i++, 100);
  gtk_clist_set_column_width (GTK_CLIST (clist), i++, 80);
  gtk_clist_set_column_width (GTK_CLIST (clist), i++, 65);
  gtk_clist_set_column_width (GTK_CLIST (clist), i, 40);
  gtk_clist_set_column_justification (GTK_CLIST (clist), i - 2,
				      GTK_JUSTIFY_RIGHT);
  gtk_clist_set_column_justification (GTK_CLIST (clist), i, GTK_JUSTIFY_RIGHT);

  return;
}


/* Callback to destroy info window dialog */

namespace globals {
  GtkWidget *program_clist = NULL;
  static GtkWidget *program_info_window = NULL;
};

void info_quit()
{
  gtk_widget_destroy(globals::program_info_window);
  globals::program_info_window = NULL;
  globals::program_clist = NULL;
  return;
}

/* Function to update the chart information when necessary */

void update_info(GtkWidget *clist)
{
  if (!clist) return;
  gtk_clist_clear (GTK_CLIST (clist));
  gtk_clist_freeze (GTK_CLIST (clist));

  if (globals::chart_rules.CelestialCoords) {
    gtk_clist_set_column_title (GTK_CLIST (clist), 1, "Right Ascension");
    gtk_clist_set_column_title (GTK_CLIST (clist), 2, "Declination");
  }
  else {
    gtk_clist_set_column_title (GTK_CLIST (clist), 1, "Gal. Longitude");
    gtk_clist_set_column_title (GTK_CLIST (clist), 2, "Gal. Latitude");
  }
  
  citerate (StarArray, *globals::chart_stararray, star_ptr) {
    StringList information;
    StringList infolist = (*star_ptr).GetInfo(globals::chart_rules);
    if (globals::chart_rules.StarLabels == NUMBER_LABEL)
      information.push_back(infolist[0] + ". " + infolist[1]);
    else
      information.push_back(infolist[1]);
    citerate_between (StringList, infolist, info_ptr, 2, 7)
      information.push_back(*info_ptr);
    information[3] += " LY ";
    
    gtk_clist_append (GTK_CLIST (clist), information.c_str());
  }

  gtk_clist_thaw (GTK_CLIST (clist));
  return;
}  

/* This function displays information about the stars shown on the chart. */

void starinfo()
{
  GtkWidget *vbox, *OK;

  // If the info window is already open, do nothing but put it in front
  if (globals::program_info_window) {
    gdk_window_raise(globals::program_info_window->window);
    return;
  }

  // make a window for it
  globals::program_info_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (globals::program_info_window),
			"StarPlot: Chart Data");
  g_signal_connect (G_OBJECT (globals::program_info_window), "destroy", 
		      G_CALLBACK (info_quit), NULL);

  // make a vbox
  vbox = gtk_vbox_new (false, 5);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_container_add (GTK_CONTAINER (globals::program_info_window), vbox);
  gtk_widget_show (vbox);

  // set up the column list and put star information into it
  my_gtk_list_setup (vbox, &globals::program_clist, 
		     false /*chart data, not search results*/);
  update_info(globals::program_clist);
  
  // the button to make the window go away
  my_gtk_popup_button(&OK, vbox, globals::program_info_window);
  gtk_window_set_focus (GTK_WINDOW (globals::program_info_window), OK);
  gtk_widget_show (globals::program_info_window);
  return;
}


/* These functions allow the user to look up star names containing a string.
 *  (admittedly, doing a `grep' of the star data files would work just as
 *  well, but it wouldn't be all nice and GUI...) */

struct sSearchData {
  GtkWidget *casesensitive;
  GtkWidget *entrybox;
  GtkWidget *sclist;
};

void search_callback(GtkWidget *widget, gpointer search_data)
{
  sSearchData data = * (sSearchData *)search_data;
  GtkWidget *sclist = data.sclist;
  string searchstring = gtk_entry_get_text(GTK_ENTRY (data.entrybox));
  bool casesensitive = 
    (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data.casesensitive))
     == true);
  StarArray searchresults;
  StringList information;

  if (starstrings::isempty(searchstring)) return;

  gtk_clist_clear (GTK_CLIST (sclist));
  searchresults.Search(searchstring, globals::chart_rules.ChartFileNames,
		       globals::chart_rules, casesensitive);

  // error checking
  if (searchresults.size() >= MAX_STARS) {
    StringList error1(7, "");
    StringList error2(7, "");
    error1[0] = "** Showing only first";
    error2[0] = "** " + starstrings::itoa(MAX_STARS) + " results";
    gtk_clist_append (GTK_CLIST (sclist), error1.c_str());
    gtk_clist_append (GTK_CLIST (sclist), error2.c_str());
  }
  else if (!searchresults.size()) {
    StringList error(7, "");
    error[0] = "Sorry, nothing found";
    gtk_clist_append (GTK_CLIST (sclist), error.c_str());
    return;
  }

  citerate (StarArray, searchresults, result_ptr) {
    StringList infolist = (*result_ptr).GetInfo(globals::chart_rules);

    // where is the star name which matches the search string in the
    //  list of star characteristics?
    unsigned int nameplace = starmath::atoi(infolist[0]);
    unsigned int infolistplace = nameplace ? nameplace + 9 : 1;
    information.clear();
    information.push_back(infolist[infolistplace]);

    // if this is not the best-known star name, put that in the "aka" column;
    //  else put the second-best-known star name in that column
    information.push_back(nameplace ? infolist[1] : infolist[10]);

    iterate_between (StringList, infolist, j, 2, 7)
      information.push_back(*j);
    information[4] += " LY ";
    gtk_clist_append (GTK_CLIST (sclist), information.c_str());
  }
  return;
}

/* Callback to destroy search window dialog */

GtkWidget *searchwindow = NULL;

void search_quit()
{
  gtk_widget_destroy(searchwindow);
  searchwindow = NULL;
  return;
}

/* create the search window dialog */

void starsearch()
{
  GtkWidget *vbox, *OK, *sclist;
  GtkWidget *hbox, *casesensitive, *entrybox, *searchbutton;
  static sSearchData search_data;
   
  // If the search window is already open, do nothing but put it in front
  if (searchwindow) {
    gdk_window_raise(searchwindow->window);
    return;
  }

  // make a window for it
  searchwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (searchwindow), "StarPlot: Search");
  g_signal_connect (G_OBJECT (searchwindow), "destroy", 
		      G_CALLBACK (search_quit), NULL);
  
  // make a vbox
  vbox = gtk_vbox_new (false, 5);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_container_add (GTK_CONTAINER (searchwindow), vbox);
  gtk_widget_show (vbox);

  // set up the search input field in an hbox
  casesensitive = gtk_check_button_new_with_label ("Case sensitive");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (casesensitive), false);
  gtk_widget_show (casesensitive);

  entrybox = gtk_entry_new();
  gtk_widget_show (entrybox);

  searchbutton = gtk_button_new_with_mnemonic ("_Search");
  gtk_widget_set_usize (searchbutton,
			defaults::program_button_width,
			defaults::program_button_height);
  gtk_widget_show (searchbutton);

  hbox = gtk_hbox_new (false, 5);
  gtk_box_pack_start (GTK_BOX (hbox), casesensitive, false, false, 0);
  gtk_box_pack_start (GTK_BOX (hbox), entrybox, true, true, 0); 
  gtk_box_pack_start (GTK_BOX (hbox), searchbutton, false, false, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, false, false, 0);
  gtk_widget_show (hbox);
  
  // set up the column list
  my_gtk_list_setup (vbox, &sclist, true /* a search clist */);
  
  // set up the search button
  search_data.casesensitive = casesensitive;
  search_data.entrybox = entrybox;
  search_data.sclist = sclist;
  g_signal_connect (G_OBJECT (searchbutton), "clicked", 
		      G_CALLBACK (search_callback),
		      &search_data);
  g_signal_connect (G_OBJECT (entrybox), "activate", 
		      G_CALLBACK (search_callback),
		      &search_data);
  
  // the button to make the window go away
  my_gtk_popup_button (&OK, vbox, searchwindow);
  gtk_window_set_focus (GTK_WINDOW (searchwindow), OK);
  gtk_widget_show (searchwindow);
  return;

}
