#ifndef __SD_LIB_H__
#define __SD_LIB_H__

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <gtk/gtk.h>
#include <string>
#include <vector>

#include "dictziplib.h"

const int MAX_MATCH_ITEM_PER_LIB=100;
const int MAX_FUZZY_DISTANCE= 3; // at most MAX_FUZZY_DISTANCE-1 differences allowed when find similar words

extern bool bIsPureEnglish(const gchar *str);

struct cacheItem {
  guint32 offset;
	gchar *data;
  //write code here to make it inline
  cacheItem() {data= NULL;}
  ~cacheItem() {g_free(data);}
};

const int WORDDATA_CACHE_NUM = 10;
const int INVALID_INDEX=-100;

class DictBase {
public:
	DictBase();
	~DictBase();
  gchar * GetWordData(guint32 idxitem_offset, guint32 idxitem_size);
protected:
	gchar *sametypesequence;
	FILE *dictfile;
	dictData *dictdzfile;
private:
  cacheItem cache[WORDDATA_CACHE_NUM];
	gint cache_cur;	
};

//this structure contain all information about dictionary
struct DictInfo {
  std::string ifo_file_name;
  guint32 wordcount;
  std::string bookname;
  std::string author;
  std::string email;
  std::string website;
  std::string date;
  std::string description;
  guint32 index_file_size;
  std::string sametypesequence;
  bool load_from_ifo_file(const gchar *ifofilename, bool istreedict);
};

#define MEASURE_TIME

class Lib : public DictBase {
public:
  struct WordCoord {
    glong offset;
    glong size;
  };
  struct Character {
    gunichar ch;
    glong index;
    friend bool operator<(const Character& lh, const Character& rh) {
      return lh.ch<rh.ch;
    }
  };
  typedef std::vector<Character> CharList;
private:
  guint32 wordcount;
	gchar *bookname;

	FILE *idxfile;
	union {
		gchar **wordlist;
    WordCoord *wordoffset;
	};

	union {
		gchar *idxdatabuffer;
		glong cur_wordindex;
	};

	gchar wordentry_buf[256]; // The length of "word_str" should be less than 256. See src/tools/DICTFILE_FORMAT.
  guint32 wordentry_offset;
  guint32 wordentry_size;
  CharList alphabet;
	
  bool load_ifofile(const char *ifofilename, gulong *idxfilesize);

	void loadwordlist();
	void loadwordoffset();
public:
#ifdef MEASURE_TIME
  static double middle_lookup_time, measure_time;
#endif
	Lib();
    ~Lib();
  bool load(const char *ifofilename);
  inline glong length() { return wordcount; }
  inline gchar* GetBookname() { return bookname; }
  bool Lookup(const char* sWord,glong *pIndex);
  bool LookupWithRule(GPatternSpec *pspec,glong *aIndex,int iBuffLen);
	gchar * GetWord(glong index);
	gchar * GetWordData(glong index);
};

//============================================================================
class Libs {
private:
	Lib **oLib; // word library.
	gint libcount;
  int iMaxFuzzyDistance;
	
  void LoadDir(const gchar *dirname, const GSList *order_list, const GSList *disable_list);
public:
	Libs();
	~Libs();
  void Load(const gchar *dicts_dir, const GSList *order_list, const GSList *disable_list);
	glong iLength(int iLib);
  gchar* GetBookname(int iLib) { return (oLib[iLib])->GetBookname(); }
  inline gint total_libs() {return libcount;}
  gchar * poGetWord(glong iIndex,int iLib) 
  { 
    return oLib[iLib]->GetWord(iIndex); 
  }
  gchar * poGetWordData(glong iIndex,int iLib) 
  {
    if (iIndex==INVALID_INDEX)
      return NULL;
    return oLib[iLib]->GetWordData(iIndex);
  }
	gchar * poGetCurrentWord(glong * iCurrent);
	gchar * poGetNextWord(const gchar *word,glong * iCurrent);
	gchar * poGetPreWord(glong * iCurrent);
  bool LookupWord(const gchar* sWord,glong& iWordIndex,int iLib) 
  {
    return oLib[iLib]->Lookup(sWord, &iWordIndex);
  }
  bool LookupSimilarWord(const gchar* sWord, glong & iWordIndex, int iLib);
  bool SimpleLookupWord(const gchar* sWord, glong & iWordIndex, int iLib);

  struct Fuzzystruct {
    char * pMatchWord;
    int iMatchWordDistance;
  };
  typedef void (*TProgressFunc)(void);

  bool LookupWithFuzzy(const gchar *sWord, Fuzzystruct oFuzzystruct[], gint fuzzystruct_amount, TProgressFunc ProgressFunc);
  gint LookupWithRule(const gchar *word, TProgressFunc ProgressFunc, gchar **ppMatchWord);
};


class TreeDict : public DictBase
{
private:	
	bool load_ifofile(const char *ifofilename, gulong *tdxfilesize);
	void load_model(gchar **buffer, GtkTreeStore *model, GtkTreeIter *parent, gint count);
public:
	bool load(const char *ifofilename, GtkTreeStore *model);
};

class TreeDicts {
public:
	TreeDicts();
	~TreeDicts();
  GtkTreeStore* Load(const gchar *tree_dicts_dir, const GSList *order_list, const GSList *disable_list);
	gchar * poGetWordData(glong offset, glong size, int iTreeDict);
private:
	TreeDict **oTreeDict;
	gint treedictcount;
	
  void LoadDir(const gchar *dirname, const GSList *order_list, const GSList *disable_list, GtkTreeStore *model);	
};

#endif
