// Copyright (c) 1999-2004  David Muse
// See the file COPYING for more information

#include <sqlrconnection.h>
#include <stdio.h>

// for umask
#include <sys/types.h>
#include <sys/stat.h>

#include <math.h>

#include <rudiments/permissions.h>

#include <config.h>

bool sqlrconnection_svr::getUnixSocket(const char *tmpdir, char *unixsocketptr) {

	#ifdef SERVER_DEBUG
	debugPrint("connection",0,"getting unix socket...");
	#endif

	file	sockseq;
	if (!openSequenceFile(&sockseq,tmpdir,unixsocketptr) ||
						!lockSequenceFile(&sockseq)) {
		return false;
	}
	if (!getAndIncrementSequenceNumber(&sockseq,unixsocketptr)) {
		unLockSequenceFile(&sockseq);
		sockseq.close();
		return false;
	}
	if (!unLockSequenceFile(&sockseq)) {
		sockseq.close();
		return false;
	}
	if (!sockseq.close()) {
		return false;
	}

	#ifdef SERVER_DEBUG
	debugPrint("connection",0,"done getting unix socket");
	#endif

	return true;
}

bool sqlrconnection_svr::openSequenceFile(file *sockseq,
				const char *tmpdir, char *unixsocketptr) {

	// open the sequence file and get the current port number
	size_t	sockseqnamelen=charstring::length(tmpdir)+9;
	char	*sockseqname=new char[sockseqnamelen];
	snprintf(sockseqname,sockseqnamelen,"%s/sockseq",tmpdir);

	#ifdef SERVER_DEBUG
	size_t	stringlen=8+charstring::length(sockseqname)+1;
	char	*string=new char[stringlen];
	snprintf(string,stringlen,"opening %s",sockseqname);
	debugPrint("connection",1,string);
	delete[] string;
	#endif

	mode_t	oldumask=umask(011);
	bool	success=sockseq->open(sockseqname,O_RDWR|O_CREAT,
				permissions::everyoneReadWrite());
	umask(oldumask);

	// handle error
	if (!success) {
		fprintf(stderr,"Could not open: %s\n",sockseqname);
		fprintf(stderr,"Make sure that the file and directory are \n");
		fprintf(stderr,"readable and writable.\n\n");
		unixsocketptr[0]=(char)NULL;

		#ifdef SERVER_DEBUG
		stringlen=14+charstring::length(sockseqname)+1;
		string=new char[stringlen];
		snprintf(string,stringlen,"couldn't open %s",sockseqname);
		debugPrint("connection",1,string);
		delete[] string;
		#endif
	}

	delete[] sockseqname;

	return success;
}

bool sqlrconnection_svr::lockSequenceFile(file *sockseq) {

	#ifdef SERVER_DEBUG
	debugPrint("connection",1,"locking...");
	#endif

	return sockseq->lockFile(F_WRLCK);
}


bool sqlrconnection_svr::getAndIncrementSequenceNumber(file *sockseq,
							char *unixsocketptr) {

	// get the sequence number from the file
	int32_t	buffer;
	if (sockseq->read(&buffer)!=sizeof(int32_t)) {
		buffer=0;
	}
	sprintf(unixsocketptr,"%d",buffer);

	#ifdef SERVER_DEBUG
	size_t	stringlen=21+charstring::length(unixsocketptr)+1;
	char	*string=new char[stringlen];
	snprintf(string,stringlen,"got sequence number: %s",unixsocketptr);
	debugPrint("connection",1,string);
	delete[] string;
	#endif

	// increment the sequence number
	// (the (double) cast is required for solaris with -compat=4)
	if (buffer==pow((double)2,31)) {
		buffer=0;
	} else {
		buffer=buffer+1;
	}

	#ifdef SERVER_DEBUG
	string=new char[50];
	snprintf(string,50,"writing new sequence number: %d",buffer);
	debugPrint("connection",1,string);
	delete[] string;
	#endif

	// write the sequence number back to the file
	if (sockseq->setPositionRelativeToBeginning(0)==-1) {
		return false;
	}
	return (sockseq->write(buffer)==sizeof(int32_t));
}

bool sqlrconnection_svr::unLockSequenceFile(file *sockseq) {

	// unlock and close the file in a platform-independent manner
	#ifdef SERVER_DEBUG
	debugPrint("connection",1,"unlocking...");
	#endif

	return sockseq->unlockFile();
}
