/*
 *  SPL - The SPL Programming Language
 *  Copyright (C) 2004, 2005  Clifford Wolf <clifford@clifford.at>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  hnode.c: C-Library interface for complex nodes
 */

#define _GNU_SOURCE

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>

#include "spl.h"
#include "compat.h"

void spl_hnode_reg(struct spl_vm *vm, const char *name, spl_hnode_function *handler, void *data)
{
	struct spl_hnode *hnode = calloc(1, sizeof(struct spl_hnode));

	hnode->name = strdup(name);
	hnode->handler = handler;
	hnode->data = data;

	hnode->next = vm->hnode_list;
	vm->hnode_list = hnode;
}

void spl_hnode(struct spl_vm *vm, struct spl_node *node, const char *key,
                const char *handler, struct spl_module *mod)
{
        struct spl_node *n = spl_get(0);
        struct spl_task *t = 0;

        if (mod) {
                t = spl_task_create(vm, 0);
                t->module = strdup(mod->name);
        }

	n->hnode_name = strdup(handler);

        spl_create(t, node, key, n, SPL_CREATE_LOCAL);
        if (t) spl_task_destroy(vm, t);
}

void spl_hnode_call(struct spl_task *task, struct spl_vm *vm,
		struct spl_node *node, struct spl_hnode_args *args)
{
	struct spl_hnode *hnode = vm->hnode_list;

	while (hnode) {
		if (!strcmp(node->hnode_name, hnode->name)) {
			hnode->handler(task, vm, node, args, hnode->data);
			return;
		}
		hnode = hnode->next;
	}
}

struct spl_node *spl_hnode_lookup(struct spl_task *task, struct spl_node *node,
		const char *key, int flags)
{
	struct spl_hnode_args args;
	memset(&args, 0, sizeof(args));

	args.action = SPL_HNODE_ACTION_LOOKUP;
	args.key = key;
	args.flags = flags;

	spl_hnode_call(task, task->vm, node, &args);
	return args.value;
}

struct spl_node *spl_hnode_create(struct spl_task *task,
		struct spl_node *node, const char *key, struct spl_node *newnode, int flags)
{
	struct spl_hnode_args args;
	memset(&args, 0, sizeof(args));

	args.action = SPL_HNODE_ACTION_CREATE;
	args.key = key;
	args.value = newnode;
	args.flags = flags;

	spl_hnode_call(task, task->vm, node, &args);
	return args.value;
}

void spl_hnode_delete(struct spl_task *task, struct spl_node *node, const char *key)
{
	struct spl_hnode_args args;
	memset(&args, 0, sizeof(args));

	args.action = SPL_HNODE_ACTION_DELETE;
	args.key = key;

	spl_hnode_call(task, task->vm, node, &args);
}

struct spl_node *spl_hnode_next(struct spl_task *task,
		struct spl_node *node, const char *key)
{
	struct spl_hnode_args args;
	memset(&args, 0, sizeof(args));

	args.action = SPL_HNODE_ACTION_NEXT;
	args.key = key;

	spl_hnode_call(task, task->vm, node, &args);
	return args.value;
}

struct spl_node *spl_hnode_prev(struct spl_task *task,
		struct spl_node *node, const char *key)
{
	struct spl_hnode_args args;
	memset(&args, 0, sizeof(args));

	args.action = SPL_HNODE_ACTION_NEXT;
	args.key = key;

	spl_hnode_call(task, task->vm, node, &args);
	return args.value;
}

struct spl_node *spl_hnode_copy(struct spl_vm *vm, struct spl_node *node)
{
	struct spl_hnode_args args;
	memset(&args, 0, sizeof(args));

	args.action = SPL_HNODE_ACTION_COPY;
	spl_hnode_call(0, vm, node, &args);
	return args.value;
}

void spl_hnode_put(struct spl_vm *vm, struct spl_node *node)
{
	struct spl_hnode_args args;
	memset(&args, 0, sizeof(args));

	args.action = SPL_HNODE_ACTION_PUT;
	spl_hnode_call(0, vm, node, &args);
}

void spl_hnode_dump(struct spl_vm *vm, struct spl_node *node)
{
	struct spl_hnode_args args;
	memset(&args, 0, sizeof(args));

	args.action = SPL_HNODE_ACTION_DUMP;
	spl_hnode_call(0, vm, node, &args);
}

