# Soya 3D
# Copyright (C) 2004 Jean-Baptiste LAMY -- jiba@tuxfamily.org
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# A collider for a Land

cdef class _Land(_soya._Land):
    """Landscape/terrain collider for Soya. This is based on Benoit Chaperot's
    contributed terrain collider for ODE, with some changes to take advantage
    of precomputed normals in Soya's landscape engine.

    How it works: 

    We loop through each "cell," or square of vertices in the heightfield,
    that is underneath the axis-aligned bounding box (AABB) of the geom we're
    testing for collisions with. For rendering, each cell is divided into
    two triangles. We use ray collision to check each edge of one of the 
    triangles, because the other triangle shares edges with adjacent cells
    and we don't want to check edges twice. We make a ray for each direction
    along the edge to get two collision points so we can take the midpoint
    of the two collision points if there is a collision. Right now the normal
    is halfway between the normals computed by Soya for the vertices making
    the edge being tested. This could be weighted, but we don't bother right
    now and it seems to work fine.

    Next, we test the plane of each triangle. We use ODE's own plane collision
    routines to give us a set of collision points on the plane. We then test
    to see if any of the collision points are within the triangle we're testing
    by first seeing if the point is within the cell (a simple range check),
    then testing if it's in the correct isoceles right triangle, which means
    just looking at the sum of the x and z deltas from the upper left vertex
    of the cell. If it's less than the length of one leg, it's in the upper
    triangle. If it's greater, it's in the lower. We keep the normal from any
    plane contact points we keep.

    Ray collision is not currently implemented, but could easily be handled
    using Soya's own raypicking routines. Likewise plane collision, which 
    might be able to use the view frustum culling routines. I don't currently
    plan to implement these because I have ambitions about doing all collision
    detection within Soya rather than using ODE's collision detection.
    """

    cdef void _get_aabb(self, dReal aabb[6]):
        """Get the axis-aligned bounding box. Warning: breaks if the
        land is rotated."""

        aabb[0] = self._matrix[12]
        aabb[1] = self._matrix[12]+self._nb_vertex_width
        aabb[2] = -dInfinity
        aabb[3] = dInfinity
        aabb[4] = self._matrix[14]
        aabb[5] = self._matrix[14]+self._nb_vertex_depth

    cdef int _collide_edge(self, GLfloat *A, GLfloat *B,
                           GLfloat *AB, GLfloat *normalA,
                           GLfloat *normalB,
                           dGeomID o1, dGeomID o2, int max_contacts, 
                           int flags, dContactGeom *contact, 
                           dGetDepthFn *GetDepth):
        """Check for collision with one triangle edge. Uses a normal
        that's halfway between the precomputed normals of the vertices
        that make up the edge."""

        cdef int n, num_contacts, nA, nB
        cdef dContactGeom contactA, contactB
        #cdef GLfloat AB[3]

        #_soya._soya.vector_from_points(AB, A, B)

        # First, do one direction
        dGeomRaySetLength(_land_ray, _soya.vector_length(AB))
        dGeomRaySet(_land_ray, A[0], A[1], A[2], AB[0], AB[1], AB[2])
        nA = dCollide(_land_ray, o2, flags, &contactA, sizeof(dContactGeom))

        # Then the other
        dGeomRaySet(_land_ray, B[0], B[1], B[2], -AB[0], -AB[1], -AB[2])
        nB = dCollide(_land_ray, o2, flags, &contactB, sizeof(dContactGeom))

        if nA and nB:
            contact.pos[0] = (contactA.pos[0] + contactB.pos[0]) / 2.0
            contact.pos[1] = (contactA.pos[1] + contactB.pos[1]) / 2.0
            contact.pos[2] = (contactA.pos[2] + contactB.pos[2]) / 2.0

            # D
            contact.normal[0] = -(normalA[0] + normalB[0]) / 2.0
            contact.normal[1] = -(normalA[1] + normalB[1]) / 2.0
            contact.normal[2] = -(normalA[2] + normalB[2]) / 2.0

            # Get the depth of the contact point in the colliding geom
            contact.depth = GetDepth(o2, contact.pos[0], contact.pos[1], 
                                     contact.pos[2])
            contact.g1 = o1
            contact.g2 = o2

            return 1

        return 0

    cdef int _collide_cell(self, int x, int z, dGeomID o1, 
                           dGeomID o2, int max_contacts, int flags, 
                           dContactGeom *contact, int skip,
                           dGetDepthFn *GetDepth):
        """Test for any collisions within a single cell of the heightfield
        grid."""

        cdef _soya.LandVertex *vA, *vB, *vC, *vD
        cdef GLfloat A[3], B[3], C[3], D[3], 
        cdef GLfloat AB[3], AC[3], BC[3], BD[3], CD[3]
        cdef GLfloat plane[4]
        cdef int nA[3], nB[3], num_contacts, numPlaneContacts
        cdef dContactGeom ContactA[3], ContactB[3], *pContact
        cdef dContactGeom planeContact[10]
        cdef float raynormal[9]

        num_contacts = 0
          
        # Get the four vertices for this cell
        # Pointer arithmetic is evil.
        vA = self._vertices + (x + z * self._nb_vertex_width)
        vB = vA + 1
        vC = vA + self._nb_vertex_width
        vD = vB + self._nb_vertex_width

        # Hmmm, already have normals calculated for both
        # vertices *and* triangles. This could be good.

        # Transform each vertex to the parent's coordsys

        A[0] = vA.coord[0]
        A[1] = vA.coord[1]
        A[2] = vA.coord[2]

        _soya.point_by_matrix(A, self._matrix)

        B[0] = vB.coord[0]
        B[1] = vB.coord[1]
        B[2] = vB.coord[2]

        _soya.point_by_matrix(B, self._matrix)

        C[0] = vC.coord[0]
        C[1] = vC.coord[1]
        C[2] = vC.coord[2]

        _soya.point_by_matrix(C, self._matrix)

        D[0] = vD.coord[0]
        D[1] = vD.coord[1]
        D[2] = vD.coord[2]

        _soya.point_by_matrix(D, self._matrix)

        # Make all the vectors we need
        _soya.vector_from_points(AB, A, B)
        _soya.vector_from_points(AC, A, C)
        _soya.vector_from_points(BC, B, C)
        _soya.vector_from_points(BD, B, D)
        _soya.vector_from_points(CD, C, D)

        # Ray collision to test edges of one triangle
        # Don't need to test the other because adjacent cells will
        num_contacts = num_contacts + self._collide_edge(B, C, BC, vB.normal, 
            vC.normal, o1, o2, max_contacts - num_contacts, flags, 
            <dContactGeom*>(<char*>contact + (num_contacts * skip)), GetDepth)

        if num_contacts == max_contacts:
            return num_contacts

        num_contacts = num_contacts + self._collide_edge(B, D, BD, vB.normal,
            vD.normal, o1, o2, max_contacts - num_contacts, flags, 
            <dContactGeom*>(<char*>contact + (num_contacts * skip)), GetDepth)

        if num_contacts == max_contacts:
            return num_contacts

        num_contacts = num_contacts + self._collide_edge(C, D, CD, vC.normal,
            vD.normal, o1, o2, max_contacts - num_contacts, flags, 
            <dContactGeom*>(<char*>contact + (num_contacts * skip)), GetDepth)

        if num_contacts == max_contacts:
            return num_contacts

        # Now do planes for each of the two triangle planes
        _soya.vector_cross_product(plane, AC, AB)
        _soya.vector_normalize(plane)
        plane[3] = plane[0] * A[0] + plane[1] * A[1] + plane[2] * A[2]
        dGeomPlaneSetParams(_land_plane, plane[0], plane[1], plane[2], plane[3])

        numPlaneContacts = dCollide(o2, _land_plane, flags, planeContact, sizeof(dContactGeom))

        #print "a", numPlaneContacts

        for i from 0 <= i < numPlaneContacts:
            # Figure out if the point is in the triangle.
            # Only need to test x and z coord because we already know it's
            # in the plane and the plane is not vertical

            # First, check if it's in the cell
            if planeContact[i].pos[0] < A[0] or planeContact[i].pos[0] > D[0]:
                continue

            if planeContact[i].pos[2] < A[2] or planeContact[i].pos[2] > D[2]:
                continue

            # Now, check if it's in the correct triangle in the cell.
            # This is made easier since we only support isoceles right triangles
            if (planeContact[i].pos[0] - A[0] + planeContact[i].pos[2] - A[2]) > (D[0] - A[0]):
                continue
            
            # It's in the triangle, so add it to our list of contacts
            pContact = <dContactGeom*>((<char*>contact) + (num_contacts * skip))
            #print <long>contact, <long>pContact, num_contacts, skip
            pContact.pos[0] = planeContact[i].pos[0]
            pContact.pos[1] = planeContact[i].pos[1]
            pContact.pos[2] = planeContact[i].pos[2]
            pContact.normal[0] = -planeContact[i].normal[0]
            pContact.normal[1] = -planeContact[i].normal[1]
            pContact.normal[2] = -planeContact[i].normal[2]
            pContact.depth = planeContact[i].depth
            pContact.g1 = o1
            pContact.g2 = o2

            #print "a", pContact.pos[0], pContact.pos[1], pContact.pos[2], pContact.normal[0], pContact.normal[1], pContact.normal[2], pContact.depth
            num_contacts = num_contacts + 1

            if num_contacts == max_contacts:
                return num_contacts

        # Try the plane for the other triangle
        _soya.vector_cross_product(plane, BD, CD)
        _soya.vector_normalize(plane)
        plane[3] = plane[0] * D[0] + plane[1] * D[1] + plane[2] * D[2]
        dGeomPlaneSetParams(_land_plane, plane[0], plane[1], plane[2], plane[3])

        numPlaneContacts = dCollide(o2, _land_plane, flags, planeContact, sizeof(dContactGeom))

        #print "b", numPlaneContacts

        for i from 0 <= i < numPlaneContacts:
            # Figure out if the point is in the triangle.
            # Only need to test x and z coord because we already know it's
            # in the plane and the plane is not vertical

            # First, check if it's in the cell
            if planeContact[i].pos[0] < A[0] or planeContact[i].pos[0] > D[0]:
                continue

            if planeContact[i].pos[2] < A[2] or planeContact[i].pos[2] > D[2]:
                continue

            # Now, check if it's in the correct triangle in the cell.
            # This is made easier since we only support isoceles right triangles
            if (planeContact[i].pos[0] - A[0] + planeContact[i].pos[2] - A[2]) < (D[0] - A[0]):
                continue
            
            # It's in the triangle, so add it to our list of contacts
            pContact = <dContactGeom*>((<char*>contact) + (num_contacts * skip))
            #print <long>contact, <long>pContact, num_contacts, skip
            pContact.pos[0] = planeContact[i].pos[0]
            pContact.pos[1] = planeContact[i].pos[1]
            pContact.pos[2] = planeContact[i].pos[2]
            pContact.normal[0] = -planeContact[i].normal[0]
            pContact.normal[1] = -planeContact[i].normal[1]
            pContact.normal[2] = -planeContact[i].normal[2]
            pContact.depth = planeContact[i].depth
            pContact.g1 = o1
            pContact.g2 = o2
            
            #print "b", pContact.pos[0], pContact.pos[1], pContact.pos[2], pContact.normal[0], pContact.normal[1], pContact.normal[2], pContact.depth

            num_contacts = num_contacts + 1

            if num_contacts == max_contacts:
                return num_contacts

        return num_contacts


    cdef int _collide(self, dGeomID o1, dGeomID o2, int flags,
                      dContactGeom *contact, int skip,
                      dGetDepthFn *GetDepth):
        cdef int num_contacts, max_contacts, x0, x1, z0, z1, x, z
        cdef dReal aabb[6], depth
        cdef dContactGeom *pContact
        cdef dVector3 lengths
        #cdef dReal *R
        cdef _soya.LandVertex *vA, *vB, *vC, *vD
        #cdef GLfloat mi[19] # Inverse of land's matrix
        #cdef GLfloat *mi # Inverse of land's matrix
        cdef GLfloat BC, BD, CD
        cdef GLfloat plane[4]

        # Get the maximum number of contacts from the flags
        max_contacts = (flags & 0xffff) or 1

        # Only 10 contacts allowed for called collision functions
        flags = (flags & 0xffff0000) | 10

        #R = dGeomGetRotation(o2)

        # Get the AAAB of the geom
        dGeomGetAABB(o2, aabb)

        # Convert the AABB to the land's coordinate system. Assumes the land
        # hasn't been rotated or scaled. 
        # Should at least do scaling. Only really need to prohibit rotation
        # on the Y axis. XXX
        x0 = <int>floor(max(0, aabb[0] - self._matrix[12]))
        x1 = <int>ceil(min(self._nb_vertex_width, aabb[1] - self._matrix[12])) + 1
        z0 = <int>floor(max(0, aabb[4] - self._matrix[14]))
        z1 = <int>ceil(min(self._nb_vertex_depth, aabb[4] - self._matrix[14])) + 1

        num_contacts = 0
        # Test all cells that are under the AABB
        for z from z0 <= z < z1:
            for x from x0 <= x < x1:
                num_contacts = num_contacts + self._collide_cell(x, z, o1, 
                    o2, max_contacts - num_contacts, flags, 
                    <dContactGeom*>((<char*>contact) + (num_contacts*skip)), 
                    skip, GetDepth)

        return num_contacts

    def __new__(self, _soya._World parent=None, SpaceBase space=None):
        cdef dSpaceID sid

        sid = NULL

        if space is not None:
            sid = space.sid

        self.gid = dCreateGeom(dLandClass)

        if space is not None:
            dSpaceAdd(sid, self.gid)

        #_geom_c2py_lut[<long>self.gid] = self
        dGeomSetData(self.gid, <void*>self)

    def __dealloc__(self):
        print "dealloc", self

    def __init__(self, _soya._World parent=None, space=None):
        self.space = space
        _soya._Land.__init__(self, parent)

    def placeable(self):
        return False

    def _id(self):
        cdef long id
        id = <long>self.gid
        return id

    def getBody(self):
        """Land can't have a body, so return environment"""
        return environment


cdef dGeomID _land_plane # Reusable plane geom
cdef dGeomID _land_ray   # Reusable ray geom

_land_plane = dCreatePlane(NULL, 0.0, 1.0, 0.0, 0.0)
_land_ray = dCreateRay(NULL, 1.0)

cdef void LandGetAABB(dGeomID geom, dReal aabb[6]):
    # XXX This is a KLUDGE. It assumes the land hasn't been rotated and
    # multiply_height hasn't been called.
    cdef _Land land
    land = <_Land>dGeomGetData(geom)
    land._get_aabb(aabb)

cdef int LandSphereCollide(dGeomID o1, dGeomID o2, int flags,
                     dContactGeom *contact, int skip):
    cdef _Land land
    land = <_Land>dGeomGetData(o1)
    return land._collide(o1, o2, flags, contact, skip, dGeomSpherePointDepth)

cdef int LandBoxCollide(dGeomID o1, dGeomID o2, int flags,
                        dContactGeom *contact, int skip):
    cdef _Land land
    land = <_Land>dGeomGetData(o1)
    return land._collide(o1, o2, flags, contact, skip, dGeomBoxPointDepth)

cdef int LandCCylinderCollide(dGeomID o1, dGeomID o2, int flags,
                        dContactGeom *contact, int skip):
    cdef _Land land
    land = <_Land>dGeomGetData(o1)
    return land._collide(o1, o2, flags, contact, skip, dGeomCCylinderPointDepth)


cdef dColliderFn * LandGetColliderFn(int gclass):

    if gclass == dSphereClass:
        return LandSphereCollide
    elif gclass == dBoxClass:
        return LandBoxCollide
    elif gclass == dCCylinderClass:
        return LandCCylinderCollide

    return NULL


cdef int LandAABBTest(dGeomID o1, dGeomID o2, dReal aabb2[6]):
    pass


cdef dGeomClass dLandGeomClass

dLandGeomClass.bytes = 0
dLandGeomClass.collider = LandGetColliderFn
dLandGeomClass.aabb = LandGetAABB
dLandGeomClass.aabb_test = NULL # Need to write this function
dLandGeomClass.dtor = NULL
    
cdef int dLandClass

dLandClass = dCreateGeomClass(&dLandGeomClass)



