# Soya 3D tutorial
# Copyright (C) 2003 Bertrand 'blam!' LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


# ------------------------
# Lesson 125: Cell Shading
# ------------------------
#
# Cell-shading is a special lighting effect that replace OpenGL lighting.
# It is used to render Shape with a cartoon like style.

import os, os.path, sys, time, math

import soya
import soya.model
import soya.widget
import soya.soya3d
import soya.sphere


soya.init()


# Create shader material (this is the material used when
# rendering the cell-shaded Shape)
soya.model.Image.PATH = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data", "images")
shader = soya.model.Material()
shader.tex_filename = "shader2.png"
shader.clamp = 0

# Create a world
scene = soya.soya3d.World()
atm = soya.soya3d.Atmosphere()
atm.bg_color = (0.4, 0.4, 1.0, 1.0)
scene.atmosphere = atm

# Create a sphere Shape
# Remember that cell-shaded Shape will look better if it is smoothlit
# (using non smoothlit Shape with a directional Light is a non sens)
shape = soya.sphere.Sphere(None, None, slices = 20, stacks = 20, smooth_lit = 1)


# Cell Shading
# ------------
# all is in the shapify_args
shape.shapify_args = ('cell-shading', { 'shader': shader, 'line_color': (0.0, 0.0, 0.0, 1.0), 'line_width': 10.0 })
# line_color and line_width applies to the outlines. Notice that the
# line_width is only a factor applied to the computed line_width which
# depends on the distance of the Shape to the camera. The line_width
# correspond to the maximal line width when the camera is very near the Shape.
#
# Interest of cell-shading: more contrasted light effects
# Drawbacks: light effect is independant of the light color and of the light intensity

scene.shape = shape.shapify()


# Add a light
light = soya.soya3d.Light(scene)
light.set_xyz(0.0, 1.0, 0.0)

# Add a camera and a loop to render
roow = soya.widget.Group()
camera = soya.soya3d.Camera(scene)
camera.set_xyz(0.0, 0.0, 4.0)
roow.add(camera)

label = soya.widget.Label(roow, " - Cell Shading demo -")
label.resize_style = soya.widget.WIDGET_RESIZE_MAXIMIZE

soya.set_root_widget(roow)


angle = 0.0
radius = 2.0

while(1):
  time.sleep(0.1)

# rotate the light
  angle += 0.1
  if angle > 180.0: angle -= 180.0
  light.x = math.cos(angle) * radius
  light.z = math.sin(angle) * radius

  for event in soya.process_event():

    if event[0] == soya.KEYDOWN:
      sys.exit()

  soya.render()
