# Soya 3D tutorial
# Copyright (C) 2001-2002 Bertrand 'blam!' LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


# ---------------------
# Lesson 123: WaterCube
# ---------------------

import os, os.path, sys, time

import soya
import soya.model
import soya.soya3d
import soya.cube
import soya.watercube
import soya.widget
import soya.idler


soya.init()


# Create a world
scene = soya.soya3d.World()

light = soya.soya3d.Light(scene)
light.ambient = (1.0, 1.0, 1.0, 1.0)
light.set_xyz(0.0, 2.0, 0.0)

# Create WaterCube material
soya.model.Image.PATH = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data", "images")
material = soya.model.Material()
material.tex_filename = "block2.tga"
material.diffuse = (1.0, 1.0, 1.0, 0.5)


# WaterCube creation
# ------------------
water = soya.watercube.WaterCube(scene)
water.material = material

# texture_factor is a float that multiply each texture
# coordinate (works as land.texture_factor)
water.texture_factor = 1.0

water.color = (0.6, 0.7, 0.3, 0.5)

# Adjust WaterCube size
# The following picture explain the meaning of each size.
# Y coordinate 0.0 is the water level when there are no
# waves (and also the uppest point always under water).
# The WaterCube height is below coordinate 0.0 and the
# wave_height is above.
#
#                 |--width--|  _
#   point(0,0,0)_               | wave_height
#            _   \._________.  _|
#           /    /|        /|
#      depth    / |       / |
#         /_   /__|______/  |
#         |    |  |______|__|
#         |    |  /      |  /
#  height |    | /       | /
#         |_   |/________|/
#
water.width  = 20.0
water.height = 10.0
water.depth  = 20.0
water.wave_height = 0.4

# This is the speed of the waves
water.speed = 0.15
# This is the number of possible heights for each point of
# the WaterCube at a given time
water.nb_phases = 7

# point_width and point_height are the number of points
# consituting the WaterCube in the respective 2 dimensions.
# Higher value means more accuracy (smoother WaterCube).
# point_width and point_height must be like this: (2 ^ n) + 1
water.point_width = 9
water.point_depth = 9

# This value affects the level of detail (LOD) of the WaterCube
# in function of the distance to the camera (default value 0.2).
# High value means more less accuracy/details when distance increase.
water.LOD_linear = 0.2



# WaterCube function
# ------------------
# return true if the given Point (or Element3D) is under the water
water.is_underwater(soya.soya3d.Point(scene, 10.0, -1.0, 10.0))

# watercube.get_waterlevel(X, Z, [VECTOR])
#
# VECTOR must be a soya3d.Vector and is optional.
# If VECTOR is None, the function returns a float that is the Y
#   coordinate of the point that is at the surface of the water
#   with the given X and Z coordinate (= waterlevel).
# If VECTOR is not given it will return a 2 elements tuple
#   (waterlevel, normal) where normal is a soya3d.Vector representing
#   the normal of the water surface face at coordinate (X, Z).
# If VECTOR is given and not None, the function returns the waterlevel
#   and VECTOR contains the normal.
water.get_waterlevel(10.0, 10.0)


# Create a cube that will float on the WaterCube like a boat
boat = soya.soya3d.Volume(scene)
cube = soya.cube.Cube()
boat.set_shape(cube.shapify())
boat.set_xyz(14.0, 0.0, 10.0)


# Add a camera and a loop to render
roow = soya.widget.Group()
camera = soya.soya3d.Camera(scene)
# Increase the camera front to have more partial underwater effect
camera.front = 1.0
camera.set_xyz(16.0, 0.0, 16.0)
camera.turn_vertical(-30.0)
roow.add(camera)

label = soya.widget.Label(roow, " - WaterCube demo -\n Left click: upward, Right click: downward")
label.resize_style = soya.widget.WIDGET_RESIZE_MAXIMIZE

soya.set_root_widget(roow)


advance = 0.0


class MyIdler(soya.idler.Idler):
  def begin_round(self):
    global advance
    water.to_surface(boat, boat)
    
    for event in soya.process_event():
      
      if event[0] == soya.KEYDOWN:
        sys.exit()
        
      elif event[0] == soya.MOUSEBUTTONUP:
        advance = 0
        
      elif event[0] == soya.MOUSEBUTTONDOWN:
        if event[1] == soya.BUTTON_LEFT:
          advance = 0.2
        elif event[1] == soya.BUTTON_RIGHT:
          advance = -0.2
          
    if advance: camera.shift(0.0, advance, 0.0)
    camera.look_at(boat)
    
    soya.idler.Idler.begin_round(self)
    
MyIdler(scene).idle()
