# Soya 3D tutorial
# Copyright (C) 2001-2002 Jean-Baptiste LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Lesson 105: Laser

# This lesson includes a red laser ray and 3 rotating cubes.
# Move the mouse to move the laser.
# In particular, lasers have been used while testing raypicking function.

import soya, soya.soya3d as soya3d, soya.model as model, soya.cube as cube, soya.widget as widget, soya.idler as idler
from soya.math3d import Point

# This is the same as lesson 105 (3 rotating cubes)

soya.init()

scene = soya3d.World()

import os, os.path, sys
data_dir = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data")

model.Image   .PATH = os.path.join(data_dir, "images")
model.Material.PATH = os.path.join(data_dir, "materials")
soya3d.World  .PATH = os.path.join(data_dir, "worlds")
model.Shape   .PATH = os.path.join(data_dir, "shapes")

material = model.Material()
material.tex_filename  = "block2.tga"

cube = cube.Cube(None, material)

shape = cube.shapify()

volume_1 = soya3d.Volume(scene, shape)
volume_1.set_xyz(-1.1, 0.5, 0.0)

volume_2 = soya3d.Volume(scene, shape)
volume_2.set_xyz(-1.5, -1.5, 0.0)

volume_3 = soya3d.Volume(scene, shape)
volume_3.set_xyz(1.1, -0.5, 0.0)

light = soya3d.Light(scene)
light.set_xyz(0.0, 0.2, 1.0)

camera = soya3d.Camera(scene)
camera.set_xyz(0.0, -1.0, 3.0)

soya.set_root_widget(widget.Group())
soya.root_widget.add(camera)
soya.root_widget.add(widget.FPSLabel())

def rotate_advance_time(obj, angle):
  return lambda proportion: obj.rotate_incline(0.2 * proportion * angle)

volume_1.advance_time = rotate_advance_time(volume_1,  5.0)
volume_2.advance_time = rotate_advance_time(volume_2, -5.0)
volume_3.advance_time = rotate_advance_time(volume_3,  5.0)


# Creates a red laser, which reflect on walls.

import soya.laser
laser = soya.laser.Laser(scene, reflect = 1)
laser.y = -2.5


# Hide the mouse cursor

soya.cursor_set_visible(0)


def begin_round():
  # Processes the events
  
  for event in soya.process_event():
    if event[0] == soya.MOUSEMOTION:
      
      # For mouse motion event, rotate the laser (quite) toward the mouse.
      # The formulas are empirical; see soya.cursor for a better algorithm
      # if you want to translate mouse positions into 3D coordinates.
      
      mouse = Point(
        scene,
        (float(event[1]) / camera.get_screen_width () - 0.5) *  4.0,
        (float(event[2]) / camera.get_screen_height() - 0.5) * -4.0,
        0.0,
        )
      laser.look_at(mouse)
  
laser.begin_round = begin_round

# Main loop

idler.Idler(scene).idle()


# TODO (left as an exercice):
# Turn this tutorial lesson into a full game, where the player must shoot
# with the laser a specific target (or monsters).
# The levels includes different moving and rotating obstacles.
# A multiplayer version uses two laser, one for each player, and two targets.

# Hint :
# the laser.points list contains all the points that define the laser trajectory.
# use laser.color to change the color of the laser.

# Good luck!
