# Soya 3D tutorial
# Copyright (C) 2001-2002 Jean-Baptiste LAMY
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Lesson 007: Worlds : a solar system

# This lesson is about grouping object. It explains how to include an object
# (called the child) inside an other (called the parent), so as moving the parent
# will move its children.

import sys, soya, soya.soya3d as soya3d, soya.model as model, soya.sphere as sphere

soya.init()

scene = soya3d.World()

import os, os.path, sys
model.Image.PATH = os.path.join(os.path.dirname(os.path.abspath(sys.argv[0])), "data", "images")

# Creates the sun's material

fire = model.Material()
fire.tex_filename  = "lava.png"

# Creates the sun
# The sun is inserted inside the scene

# The soya.sphere module works in a similar way than soya.cube.

sun = soya3d.World(scene)
sun.set_shape(sphere.Sphere(None, fire).shapify())


# Creates the earth's material

ground = model.Material()
ground.diffuse = (0.4, 0.7, 1.0, 1.0)

# Creates the earth
# The earth is inserted inside the sun !
# This implies than moving / rotating the sun moves the earth too!

earth = soya3d.World(sun)
earth.set_shape(sphere.Sphere(None, ground).shapify())
earth.scale(0.6, 0.6, 0.6)
earth.x = 4.0


# Creates the moon's material

crater = model.Material()
crater.diffuse = (0.4, 0.4, 0.4, 1.0)

# Creates the moon
# The moon is inserted inside the earth !
# This implies than moving / rotating the sun and/or the earth moves the moon too!

moon = soya3d.World(earth)
moon.set_shape(sphere.Sphere(None, crater).shapify())
moon.scale(0.4, 0.4, 0.4)
moon.x = -3.0

light = soya3d.Light(scene)
light.set_xyz(0.0, 0.2, 2.0)

camera = soya3d.Camera(scene)
camera.z = 10.0
soya.set_root_widget(camera)


# Makes it less simetric !

sun  .turn_lateral (25.0)
earth.turn_vertical(35.0)


import time

while 1:
  soya.render()
  
  sun  .turn_incline(3.0)
  earth.turn_incline(5.0)
  moon .turn_incline(3.0)
  
  time.sleep(0.02)


# The Soya object hierarchy is different to the branching philosophy used by
# several other 3D engine (such as java 3D).

# It can be compared to an UNIX file system:
# - Worlds are directories
# - Faces, Lights, Cameras,... are files
# - Volumes behave like (soft) links (to a directory): they can be used to have
# the same directory at different location (exactely like, e.g., /usr/src/linux
# is a link to /usr/src/linux-2.4.19-16mdk/ on my Linux file system).

# A World behaves exactely as a list (and technically it IS A list): you can use
# the append, remove,... methods, and use a World in a "for" loop
# (e.g.: for child in world: print child)

# Additionnal methods include:
# - world.recursive() : returns a list of ALL the children in this world,
#   including children of its children, and so forth (e.g. sun.recursive()
#   contains the earth but also the moon).

# Each 3D object can have a "name" property, which may be used to find them.
# The following methods deals with names:
# - world[name], world.__getitem__(name) : gets the child of the given name
# - world.search(name_regexp) : gets all the children whose name match the
#   given regexp
# - world.search_all(name_regexp) : like search, but recursively.
# - world.subitem(name_path) : like __getitem__, but name_path may be composed
#   of several names, separated by dots: "child.sub_child.leaf".

# You can get the World which contains a 3D object by using its "parent" property.
# The get_root() method returns the parent's parent's parent... and so forth
# (= often called "scene" in tutorials)

